/************************************************************************/
/*									*/
/*		applegelo.c						*/
/*									*/
/*	GELO drawing interface for APPLE pictures			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "apple_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_CONTENTS	128
#define MAX_LEVEL	10
#define MAX_USE 	32
#define MIN_STACK	16384
#define PTR_CONVERT	5

#define DUMMY_OBJECT	((GELO_OBJECT) 1)
#define IGNORE_OBJECT	((GELO_OBJECT) 3)

#define NO_ARC		0
#define OBJ_ARC 	1
#define FLD_ARC 	2





/************************************************************************/
/*									*/
/*	Local Types							*/
/*									*/
/************************************************************************/


typedef struct _DRAW_INFO *	DRAW_INFO;
typedef struct _DRAW_NODE *	DRAW_NODE;
typedef struct _DRAW_ARC  *	DRAW_ARC;



typedef struct _DRAW_INFO {
   Universal type;
   String match;
   GELO_OBJECT layout;
   DRAW_INFO layinfo;
   DRAW_INFO oldlayinfo;
   DRAW_NODE node;
   DRAW_ARC arc;
   Short   level;
   Short   ptrlevel;
   Short   usedepth;
   Universal from;
   Universal source;
   APPLE_VALUE_B field;
   Universal index;
   GELO_PORT portal;
   Boolean unique;
   Boolean simple;
   Boolean array;
   Integer base;
   Universal extinfo;
} DRAW_INFO_B;



typedef struct _DRAW_NODE {
   DRAW_NODE next;
   Universal data;
   Universal type;
   GELO_OBJECT object;
} DRAW_NODE_B;



typedef struct _DRAW_ARC {
   DRAW_ARC next;
   Universal from;
   Universal to;
   Universal fromtype;
   Universal totype;
   String label;
   GELO_PORT fport;
   GELO_PORT tport;
   ASH_LINE_STYLE style;
   GELO_ARC_ARROW arrow;
   GELO_ARC_LABEL_LOC label_loc;
   GELO_ARC_LABEL_POS label_pos;
   Boolean label_box;
   GELO_OBJECT arcobj;
   GELO_OBJECT fromobj;
} DRAW_ARC_B;





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/



/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	GELO_OBJECT		create_object();
static	GELO_OBJECT		create_gelo();
static	GELO_OBJECT		create_gelo_box();
static	GELO_OBJECT		create_gelo_layout();
static	GELO_OBJECT		create_gelo_tiled();
static	GELO_OBJECT		create_gelo_field();
static	GELO_OBJECT		create_gelo_pointer();
static	void			create_gelo_arc();
static	GELO_OBJECT		create_gelo_empty();
static	GELO_OBJECT		create_gelo_small();
static	Boolean 		is_appropriate();
static	GELO_OBJECT		draw_component();
static	DRAW_ARC		add_arc();
static	void			draw_arcs();
static	GELO_OBJECT		check_node();
static	void			save_node();
static	void			record_object();
static	Universal		get_data_value();
static	void			free_info();
static	void			define_layout_component();





/************************************************************************/
/*									*/
/*	APPLE_gelo_init -- module initialization			*/
/*									*/
/************************************************************************/


void
APPLE_gelo_init()
{
   ITRACE("APPLE_gelo_init");
};





/************************************************************************/
/*									*/
/*	APPLEgelo_create -- create gelo object using APPLE type 	*/
/*									*/
/************************************************************************/


GELO_OBJECT
APPLEgelo_create(o,typ,match,layfg,ext)
   Universal o;
   Universal typ;
   String match;
   Boolean layfg;
   Universal ext;
{
   register GELO_OBJECT g;
   DRAW_INFO_B info;

   TRACE("APPLEgelo_create 0x%x 0x%x 0x%x %d 0x%x",o,typ,match,layfg,ext);

   info.type = typ;
   info.match = match;
   info.layout = NULL;
   info.layinfo = NULL;
   info.oldlayinfo = NULL;
   info.node = NULL;
   info.arc = NULL;
   info.level = 0;
   info.ptrlevel = 0;
   info.usedepth = 0;
   info.from = NULL;
   info.source = o;
   info.field.name = NULL;
   info.field.value = 0;
   info.index = -1;
   info.array = FALSE;
   info.unique = FALSE;
   info.simple = FALSE;
   info.base = 0;
   info.portal = GELO_PORT_CENTER;
   info.extinfo = ext;

   g = create_object(o,&info,layfg);

   free_info(&info);

   return g;
};





/************************************************************************/
/*									*/
/*	create_object -- create gelo object using APPLE type		*/
/*									*/
/************************************************************************/


static GELO_OBJECT
create_object(o,id,layfg)
   Universal o;
   DRAW_INFO id;
   Boolean layfg;
{
   register APPLE_GELO_OBJECT ago,ag1;
   register GELO_OBJECT go;
   register Sequence l;
   register APPLE_GELO ag;
   Universal ofrom;
   APPLE_VALUE_B flds[2];

   DTRACE("create_object 0x%x 0x%x %d",o,id,layfg);

   if (id->type != NULL) {
      ag1 = APPLE_save_find(id->type);
      if (ag1 == NULL) {
	 ag1 = APPLE_ext_load_picture(NULL,id->type,TRUE);
	 if (ag1 != NULL) APPLE_save_enter(id->type,ag1);
       };
    }
   else
      ag1 = NULL;

   for (ago = ag1;
	ago != NULL && !is_appropriate(ago,o,id,layfg);
	ago = ago->alternative) ;

   if (APPLE_ext_test_null(o,id->type) && ago != NULL && !ago->cond_null &&
	  ago->cond_match == NULL && ago->cond_fct == NULL) {
      if (id->type == NULL ||
		  APPLE_ext_inq_fields(NULL,id->type,FALSE,1,flds) > 0)
	 ago = NULL;
    };

   if (ago == NULL) {
      if (!APPLE_ext_test_null(o,id->type)) ago = APPLE_default_pattern(NULL,id->type);
      else if (id->type != NULL &&
		  APPLE_ext_inq_fields(NULL,id->type,FALSE,1,flds) == 0) {
	 ago = APPLE_default_pattern(NULL,id->type);
       }
      else {
	 ago = APPLE_default_pattern(NULL,NULL);
	 for (ag1 = ago; ag1 != NULL; ag1 = ag1->alternative) ag1->cond_null = TRUE;
       };
      APPLE_save_append(id->type,ago);
      while (ago != NULL && ago->alternative != NULL &&
		!is_appropriate(ago,o,id,layfg))
	 ago = ago->alternative;
    };

   ++ago->inuse;

   if (layfg && ago->layout_obj != NULL)
      go = create_gelo(ago->layout_obj,o,id);
   else
      go = create_gelo(ago->object,o,id);

   if (id->layinfo != NULL && ago->use != NULL && !id->simple &&
	  id->usedepth < MAX_USE) {
      ofrom = id->from;
      id->from = o;
      id->usedepth += 1;
      forin (ag,APPLE_GELO,l,ago->use) {
	 switch (ag->mode) {
	    case APPLE_MODE_FIELD :
	       create_gelo_field(ag,o,id,TRUE);
	       break;
	    case APPLE_MODE_ARC :
	       create_gelo_arc(ag,o,id,FALSE,TRUE);
	       break;
	    default :
	       create_gelo(ag,o,id);
	       break;
	  };
       };
      id->usedepth -= 1;
      id->from = ofrom;
    };

   --ago->inuse;

   return go;
};





/************************************************************************/
/*									*/
/*	create_gelo -- create gelo object given apple description	*/
/*									*/
/************************************************************************/


static GELO_OBJECT
create_gelo(ag,o,id)
   APPLE_GELO ag;
   Universal o;
   DRAW_INFO id;
{
   register GELO_OBJECT g;
   register Boolean badfg;

   DTRACE("create_gelo 0x%x 0x%x 0x%x",ag,o,id);

   if (id->level >= MAX_LEVEL) badfg = TRUE;
   else if (id->usedepth >= MAX_USE) badfg = TRUE;
   else if (USE_THREADS && THREADstackremaining() < MIN_STACK) badfg = TRUE;
   else badfg = FALSE;

   if (badfg) {
      g = create_gelo_small(ag,o,id);
    }
   else switch (ag->mode) {
      case APPLE_MODE_TILED :
	 g = create_gelo_tiled(ag,o,id);
	 break;
      case APPLE_MODE_FIELD :
	 g = create_gelo_field(ag,o,id,FALSE);
	 break;
      case APPLE_MODE_POINTER :
	 g = create_gelo_pointer(ag,o,id);
	 break;
      case APPLE_MODE_LAYOUT :
	 g = create_gelo_layout(ag,o,id);
	 break;
      case APPLE_MODE_BOX :
	 g = create_gelo_box(ag,o,id);
	 break;
      case APPLE_MODE_ARC :
	 create_gelo_arc(ag,o,id,TRUE,FALSE);
	 g = NULL;
	 break;
      case APPLE_MODE_IGNORE :
	 g = NULL;
	 break;
      default :
	 g = NULL;
	 FATAL("Bad apple picture type");
	 break;
    };

   if (g != NULL) {
      if (ag->fix_x) GELOuse_default_x(g,TRUE);
      if (ag->fix_y) GELOuse_default_y(g,TRUE);
    };

   return g;
};





/************************************************************************/
/*									*/
/*	create_gelo_box -- create box-type gelo object			*/
/*									*/
/************************************************************************/


static GELO_OBJECT
create_gelo_box(box,o,id)
   APPLE_GELO_BOX box;
   Universal o;
   DRAW_INFO id;
{
   register GELO_OBJECT g;
   Universal d;
   DRAW_INFO_B info;
   String s;
   Integer idx;

   DTRACE("create_gelo_box 0x%x 0x%x 0x%x",box,o,id);

   g = GELOdefine_data();
   GELOdefine_data_fill(g,box->fill);
   GELOdefine_data_shape(g,box->shape);
   GELOdefine_data_invert(g,box->invert);
   GELOdefine_data_style(g,box->linestyle);
   if (box->font != 0) GELOdefine_data_font(g,box->font-2);

   if (box->value.name != NULL) {
      idx = (id->index < 0 ? -1 : id->index+id->base);
      if (APPLE_ext_text_value(o,id->type,&box->value,id->from,idx,&s)) {
	 GELOset_contents(g,box->value.value);
       };
      GELOdefine_data_text(g,s);
      info = *id;
      d = get_data_value(o,&box->value,&info);
      save_node(g,d,&info);
    };

   define_layout_component(id,g);

   return g;
};





/************************************************************************/
/*									*/
/*	create_gelo_layout -- create layout-type gelo object		*/
/*									*/
/************************************************************************/


static GELO_OBJECT
create_gelo_layout(lay,o,id)
   APPLE_GELO_LAYOUT lay;
   Universal o;
   DRAW_INFO id;
{
   register GELO_OBJECT g;
   register APPLE_GELO_FIELD fld;
   register Sequence l;
   DRAW_INFO_B info;

   DTRACE("create_gelo_layout 0x%x 0x%x 0x%x",lay,o,id);

   g = GELOdefine_layout();

   GELOdefine_layout_method(g,lay->method);
   GELOdefine_layout_conn_method(g,lay->connmethod);
   GELOdefine_layout_box(g,lay->boxed);
   GELOdefine_layout_shrinkage(g,lay->shrinkage);
   GELOdefine_layout_fixed(g,lay->fixed);
   GELOdefine_layout_standard(g,lay->standard);
   GELOdefine_layout_centered(g,lay->centered);
   GELOdefine_layout_white_space(g,lay->white_space);

   define_layout_component(id,g);

   info = *id;
   info.layout = g;
   info.layinfo = &info;
   info.oldlayinfo = &info;
   info.node = NULL;
   info.arc = NULL;
   info.level = id->level + 1;
   info.usedepth = 0;
   info.from = o;
   info.unique = lay->unique;
   info.simple = lay->simple;
   info.portal = GELO_PORT_CENTER;

   save_node(g,o,id);

   forin (fld,APPLE_GELO_FIELD,l,lay->components) {
      create_gelo_field(fld,o,&info,TRUE);
    };

   draw_arcs(g,&info);

   free_info(&info);

   return g;
};





/************************************************************************/
/*									*/
/*	create_gelo_tiled -- create tile-type gelo object		*/
/*									*/
/************************************************************************/


static GELO_OBJECT
create_gelo_tiled(til,o,id)
   APPLE_GELO_TILED til;
   Universal o;
   DRAW_INFO id;
{
   register GELO_OBJECT g,g1;
   register APPLE_GELO_COMPONENT c;
   register Sequence l,la;
   register Integer i,i0,i1;
   register APPLE_GELO_TARC tarc;
   register GELO_CONNECT conn;
   register APPLE_GELO_TCONST cnst;
   DRAW_INFO_B info;
   GELO_OBJECT cmps[MAX_CONTENTS];
   Float shk;

   DTRACE("create_gelo_tiled 0x%x 0x%x 0x%x",til,o,id);

   g = GELOdefine_tiled();

   GELOdefine_tiled_box(g,til->boxed);

   define_layout_component(id,g);

   save_node(g,o,id);

   info = *id;
   info.layinfo = NULL;
   info.node = NULL;
   info.arc = NULL;
   info.level = id->level + 1;
   info.usedepth = 0;
   info.from = o;
   info.unique = FALSE;
   info.simple = FALSE;
   info.portal = GELO_PORT_CENTER;

   i = 0;
   forin (c,APPLE_GELO_COMPONENT,l,til->components) {
      if (c->object != NULL) {
	 g1 = create_gelo(c->object,o,&info);
       }
      else {
	 g1 = create_gelo_empty(&info);
       };

      cmps[i++] = g1;

      if (g1 == NULL) continue;

      if (c->fix_x) GELOuse_default_x(g1,TRUE);
      if (c->fix_y) GELOuse_default_y(g1,TRUE);
      if (c->priority_x != 1.0) GELOset_priority_x(g1,c->priority_x);
      if (c->priority_y != 1.0) GELOset_priority_y(g1,c->priority_y);
      shk = c->shrinkage;
      shk /= 100.0;

      GELOdefine_tiled_component(g,g1,c->lx,c->by,c->rx,c->ty,
				    c->expand_x,c->expand_y,shk);
    };

   forin (tarc,APPLE_GELO_TARC,l,til->arcs) {
      i0 = -1;
      i1 = -1;
      i = 0;
      forin (c,APPLE_GELO_COMPONENT,la,til->components) {
	 if (c == tarc->from) i0 = i;
	 if (c == tarc->to) i1 = i;
	 ++i;
       };
      if (i0 < 0 || i1 < 0) continue;
      conn = GELOnew_connect(cmps[i0],tarc->fport,cmps[i1],tarc->tport);
      GELOconnect_arc_style(conn,tarc->style,tarc->arrow);
      GELOdefine_tiled_connect(g,conn);
    };

   forin (cnst,APPLE_GELO_TCONST,l,til->constraints) {
      i0 = -1;
      i1 = -1;
      i = 0;
      forin (c,APPLE_GELO_COMPONENT,la,til->components) {
	 if (c == cnst->from) i0 = i;
	 if (c == cnst->to) i1 = i;
	 ++i;
       };
      if (i0 < 0 || i1 < 0) continue;
      GELOdefine_tiled_constraint(g,cmps[i0],cmps[i1],
				     cnst->from_y,cnst->to_y,
				     cnst->multiplier,cnst->addend);
    };

   free_info(&info);

   return g;
};





/************************************************************************/
/*									*/
/*	create_gelo_field -- create a field-type object 		*/
/*									*/
/************************************************************************/


static GELO_OBJECT
create_gelo_field(fld,o,id,nnull)
   APPLE_GELO_FIELD fld;
   Universal o;
   DRAW_INFO id;
   Boolean nnull;
{
   register GELO_OBJECT g;
   DRAW_INFO_B info;

   DTRACE("create_gelo_field 0x%x 0x%x 0x%x %d",fld,o,id,nnull);

   info = *id;
   if (fld->match != NULL) {
      info.match = fld->match;
    };
   info.portal = fld->fport;

   g = draw_component(o,&fld->value,&info,nnull,FALSE,NULL);

   return g;
};





/************************************************************************/
/*									*/
/*	create_gelo_pointer -- create a pointer-to-field type object	*/
/*									*/
/************************************************************************/


static GELO_OBJECT
create_gelo_pointer(ptr,o,id)
   APPLE_GELO_ARC ptr;
   Universal o;
   DRAW_INFO id;
{
   register GELO_OBJECT g;
   DRAW_INFO_B info,info1;
   DRAW_ARC arc;
   Universal d,t;
   APPLE_GELO_FIELD_B field;

   DTRACE("create_gelo_pointer 0x%x 0x%x 0x%x",ptr,o,id);

   info1 = *id;
   d = get_data_value(o,&ptr->to,&info1);
   t = info1.type;

   if (APPLE_ext_test_null(d,info1.type)) {
      g = GELOdefine_data();
      GELOdefine_data_fill(g,0);
      GELOdefine_data_shape(g,GELO_SHAPE_NULL_PTR);
      GELOdefine_data_invert(g,FALSE);
      record_object(g,d,&info1);
      define_layout_component(id,g);
    }
   else if (id->layinfo == NULL && id->oldlayinfo == NULL) {
      field.mode = APPLE_MODE_FIELD;
      field.fix_x = ptr->fix_x;
      field.fix_y = ptr->fix_y;
      field.match = ptr->match;
      field.value = ptr->to;
      field.fport = GELO_PORT_CENTER;
      g = create_gelo_field(&field,o,id,FALSE);
    }
   else {
      info = *id;
      if (id->layinfo == NULL) info.layinfo = info.oldlayinfo;
      if (ptr->match != NULL) {
	 info.match = ptr->match;
       };
      info.ptrlevel = info.ptrlevel + 1;
      if ((info.ptrlevel % PTR_CONVERT) != 0) info.level = info.level-1;
      if (info.level < 0) info.level = 0;

      g = draw_component(o,&ptr->to,&info,FALSE,TRUE,NULL);

      arc = add_arc(o,o,d,&info,ptr,FALSE,id->type,t);
      info1.layinfo = id->layinfo;
      if (id->layinfo == NULL) info1.layinfo = info1.oldlayinfo;
      g = GELOdefine_data();
      GELOdefine_data_fill(g,0);
      GELOdefine_data_shape(g,GELO_SHAPE_FLD_PTR);
      GELOdefine_data_invert(g,FALSE);
      define_layout_component(id,g);
      arc->fromobj = g;
      record_object(g,d,&info1);
    };

   return g;
};





/************************************************************************/
/*									*/
/*	create_gelo_arc -- create layout arc				*/
/*									*/
/************************************************************************/


static void
create_gelo_arc(arc,o,id,nullok,seqok)
   APPLE_GELO_ARC arc;
   Universal o;
   DRAW_INFO id;
   Boolean nullok;
   Boolean seqok;
{
   register DRAW_ARC darc;
   register GELO_OBJECT g;
   register Integer i,j;
   Universal from,to;
   Universal ftype,ttype,fbase,tbase;
   Universal fval[MAX_CONTENTS],tval[MAX_CONTENTS];
   Universal fvtyp[MAX_CONTENTS],tvtyp[MAX_CONTENTS];
   Boolean fseq,tseq;
   Integer fct,tct;
   DRAW_INFO_B info,info1;

   DTRACE("create_gelo_arc 0x%x 0x%x 0x%x %d",arc,o,id,nullok);

   if (id->layinfo == NULL) return;

   g = check_node(o,id);

   info = *id;
   info.portal = GELO_PORT_CENTER;
   if (arc->match != NULL) {
      info.match = arc->match;
    };

   from = get_data_value(o,&arc->from,&info);
   ftype = info.type;

   info.type = id->type;
   to = get_data_value(o,&arc->to,&info);
   ttype = info.type;

   if (seqok && APPLE_ext_inq_use_contents(from,NULL,ftype)) {
      fct = APPLE_ext_inq_contents(from,ftype,MAX_CONTENTS,fval,fvtyp,&fbase);
      fseq = TRUE;
    }
   else {
      fct = 1;
      fval[0] = from;
      fvtyp[0] = ftype;
      fseq = FALSE;
    };

   if (seqok && APPLE_ext_inq_use_contents(to,NULL,ttype)) {
      tct = APPLE_ext_inq_contents(to,ttype,MAX_CONTENTS,tval,tvtyp,&tbase);
      tseq = TRUE;
    }
   else {
      tct = 1;
      tval[0] = to;
      tvtyp[0] = ttype;
      tseq = FALSE;
    };

   for (i = 0; i < fct; ++i) {
      info1 = info;
      info1.type = fvtyp[i];
      g = check_node(fval[i],&info1);
      if (!info1.unique || g == NULL) {
	 info1.type = fvtyp[i];
	 info1.source = o;
	 info1.field = arc->from;
	 info1.index = (fseq ? i : -1);
	 info1.base = fbase;
	 g = create_object(fval[i],&info1,FALSE);
	 save_node(g,fval[i],&info1);
       };
      for (j = 0; j < tct; ++j) {
	 info1 = info;
	 info1.type = tvtyp[j];
	 g = check_node(tval[j],&info1);
	 if (!info1.unique || g == NULL) {
	    info1.type = tvtyp[j];
	    info1.source = o;
	    info1.field = arc->to;
	    info1.index = (tseq ? j : -1);
	    info1.array = (info1.index >= 0);
	    info1.base = tbase;
	    g = create_object(tval[j],&info1,FALSE);
	    save_node(g,tval[j],&info1);
	  };
	 if (!APPLE_ext_test_null(fval[i],fvtyp[i]) &&
		!APPLE_ext_test_null(tval[j],tvtyp[i])) {
	    darc = add_arc(o,fval[i],tval[j],id,arc,TRUE,fvtyp[i],tvtyp[i]);
	    record_object(darc->arcobj,o,id);
	  };
       };
    };
};





/************************************************************************/
/*									*/
/*	create_gelo_empty -- create empty box				*/
/*									*/
/************************************************************************/


static GELO_OBJECT
create_gelo_empty(id)
   DRAW_INFO id;
{
   register GELO_OBJECT g;

   DTRACE("create_gelo_empty 0x%x",id);

   g = GELOdefine_data();
   GELOdefine_data_shape(g,GELO_SHAPE_NONE);

   define_layout_component(id,g);

   return g;
};





/************************************************************************/
/*									*/
/*	create_gelo_small -- create empty box for item too small to see */
/*									*/
/************************************************************************/


static GELO_OBJECT
create_gelo_small(ag,o,id)
   APPLE_GELO ag;
   Universal o;
   DRAW_INFO id;
{
   register GELO_OBJECT g;

   DTRACE("create_gelo_small 0x%x 0x%x 0x%x",ag,o,id);

   g = GELOdefine_data();
   GELOdefine_data_shape(g,GELO_SHAPE_RECTANGLE);

   define_layout_component(id,g);

   return g;
};





/************************************************************************/
/*									*/
/*	is_appropriate -- check if this drawing descriptor is to be used*/
/*									*/
/************************************************************************/


static Boolean
is_appropriate(ago,o,id,layfg)
   APPLE_GELO_OBJECT ago;
   Universal o;
   DRAW_INFO id;
   Boolean layfg;
{
   register Boolean fg;
   Integer lv;

   DTRACE("is_appropriate 0x%x 0x%x 0x%x %d",ago,o,id,layfg);

   if (layfg) {
      if (ago->layout_obj != NULL) return TRUE;
      if (ago->object != NULL && ago->object->mode == APPLE_MODE_LAYOUT)
	 return TRUE;
      return FALSE;
    };

   fg = TRUE;

   if (ago->cond_array && (id->index < 0 || !id->array)) fg = FALSE;

   if (ago->cond_layout) {
      if (id->layinfo == NULL) fg = FALSE;
      else if (id->simple && !ago->cond_array) fg = FALSE;
    };

   if (fg && ago->cond_layout && id->simple && ago->object != NULL &&
	  ago->object->mode == APPLE_MODE_ARC) fg = FALSE;

   if (fg && ago->cond_null && !APPLE_ext_test_null(o,id->type)) fg = FALSE;

   if (fg && ago->cond_match != NULL) {
      if (id->match == NULL || re_comp(ago->cond_match) != 0 ||
	     re_exec(id->match) <= 0)
	 fg = FALSE;
    };

   if (fg && ago->cond_fct != NULL) {
      fg = APPLE_ext_test_condition(ago->cond_fct,o);
    };

   if (fg && ago->cond_nested) {
      lv = id->level - (id->ptrlevel/PTR_CONVERT);
      if (id->level == 0) fg = FALSE;
      else if (lv == 1 && id->layinfo != NULL) fg = FALSE;
    };

   if (ago->object == NULL) fg = FALSE;

   return fg;
};





/************************************************************************/
/*									*/
/*	draw_component -- draw subobject				*/
/*									*/
/************************************************************************/


static GELO_OBJECT
draw_component(o,v,id,cntfg,nnull,objp)
   Universal o;
   APPLE_VALUE v;
   DRAW_INFO id;
   Boolean cntfg;
   Boolean nnull;
   Universal * objp;
{
   register GELO_OBJECT g;
   Universal d;
   Universal val[MAX_CONTENTS], vtyp[MAX_CONTENTS];
   DRAW_INFO_B info;
   DRAW_INFO layin;
   register Integer ct,i;
   Integer base;

   DTRACE("draw_component 0x%x 0x%x 0x%x %d",o,v,id,cntfg);

   if (cntfg && v->name != NULL && v->name[0] == '^') cntfg = FALSE;

   info = *id;
   id = &info;
   id->array = FALSE;

   d = get_data_value(o,v,id);
   if (objp != NULL) *objp = d;

   if (APPLE_ext_test_null(d,id->type) && nnull) return NULL;

   if (cntfg && APPLE_ext_inq_use_contents(d,NULL,id->type)) {
      ct = APPLE_ext_inq_contents(d,id->type,MAX_CONTENTS,val,vtyp,&base);
      for (i = 0; i < ct; ++i) {
	 id->type = vtyp[i];
	 id->index = i;
	 id->array = TRUE;
	 id->base = base;
	 if (id->unique) g = check_node(val[i],id);
	 else g = NULL;
	 if (g == NULL) {
	    g = create_object(val[i],id,FALSE);
	    save_node(g,val[i],id);
	  };
       };
      g = NULL;
    }
   else {
      layin = id->layinfo;
      if (layin == NULL) layin = id->oldlayinfo;
      if (layin == NULL) layin = id;
      if (layin->unique) g = check_node(d,id);
      else g = NULL;
      if (g == NULL) {
	 g = create_object(d,id,FALSE);
	 save_node(g,d,id);
	 if (id->layinfo != NULL && g != NULL) {
	    g = NULL;
	  };
       };
    };

   return g;
};





/************************************************************************/
/*									*/
/*	add_arc -- add an automatically generated arc			*/
/*									*/
/************************************************************************/


static DRAW_ARC
add_arc(o,from,to,id,arc,objfg,fromt,tot)
   Universal o;
   Universal from,to;
   DRAW_INFO id;
   APPLE_GELO_ARC arc;
   Boolean objfg;
   Universal fromt,tot;
{
   register DRAW_ARC darc;
   register Boolean fg;
   String s;
   Universal d,type;
   Integer idx;

   DTRACE("add_arc 0x%x 0x%x 0x%x 0x%x 0x%x %d",o,from,to,id,arc,objfg);

   if (id->layinfo == NULL) return NULL;

   darc = PALLOC(DRAW_ARC_B);
   darc->next = id->layinfo->arc;
   id->layinfo->arc = darc;
   darc->from = from;
   darc->to = to;
   darc->fromtype = fromt;
   darc->totype = tot;
   fg = FALSE;

   if (arc->label.name == NULL) darc->label = NULL;
   else {
      d = APPLE_ext_data_value(o,id->type,&arc->label,id->from,id->index,id->level,&type);
      if (APPLE_ext_test_null(d,type)) darc->label = NULL;
      else {
	 idx = (id->index < 0 ? -1 : id->index+id->base);
	 fg = APPLE_ext_text_value(o,id->type,&arc->label,id->from,idx,&s);
	 darc->label = s;
       };
    };

   darc->fport = arc->fport;
   darc->tport = arc->tport;
   darc->style = arc->style;
   darc->arrow = arc->arrow;
   darc->label_loc = arc->label_loc;
   darc->label_pos = arc->label_pos;
   darc->label_box = arc->label_box;
   darc->fromobj = NULL;

   if (objfg) {
      darc->arcobj = GELOdefine_arc();
      if (fg) GELOset_contents(darc->arcobj,arc->label.value);
    }
   else darc->arcobj = NULL;

   return darc;
};





/************************************************************************/
/*									*/
/*	draw_arcs -- draw the arcs we have collected			*/
/*									*/
/************************************************************************/


static void
draw_arcs(g,id)
   GELO_OBJECT g;
   DRAW_INFO id;
{
   register DRAW_ARC darc;
   register DRAW_NODE n;
   register GELO_OBJECT f,t;
   register GELO_CONNECT c;
   register Sequence l,al;

   DTRACE("draw_arcs 0x%x 0x%x",g,id);

   al = NULL;
   for (darc = id->arc; darc != NULL; darc = darc->next) {
      al = CONS(darc,al);
    };

   forin (darc,DRAW_ARC,l,al) {
      f = darc->fromobj;
      t = NULL;
      for (n = id->node; n != NULL; n = n->next) {
	 if (f == NULL && APPLE_ext_inq_equals(darc->from,n->data,darc->fromtype,n->type)) {
	    f = n->object;
	    if (t != NULL) break;
	  };
	 if (t == NULL && APPLE_ext_inq_equals(darc->to,n->data,darc->totype,n->type)) {
	    t = n->object;
	    if (f != NULL) break;
	  };
       };

      DTRACE("\t0x%x 0x%x => 0x%x 0x%x",darc->from,darc->to,f,t);

      if (f == NULL || t == NULL || f == DUMMY_OBJECT || t == DUMMY_OBJECT ||
	     f == IGNORE_OBJECT || t == IGNORE_OBJECT)
	 continue;

      c = GELOnew_connect(f,darc->fport,t,darc->tport);
      GELOconnect_arc_style(c,darc->style,darc->arrow);
      GELOconnect_label_style(c,darc->label_loc,darc->label_pos,darc->label_box);
      GELOconnect_label(c,darc->label_loc,darc->label);

      if (darc->arcobj != NULL) {
	 GELOdefine_arc_connect(darc->arcobj,c);
	 GELOdefine_layout_arc(g,darc->arcobj);
       }
      else {
	 GELOdefine_layout_connect(g,c);
       };
    };

   LFREE(al);
};





/************************************************************************/
/*									*/
/*	check_node -- check if item already exists as node		*/
/*									*/
/************************************************************************/


static GELO_OBJECT
check_node(o,id)
   Universal o;
   DRAW_INFO id;
{
   register DRAW_NODE n;
   DRAW_INFO lay;

   DTRACE("check_node 0x%x 0x%x",o,id);

   lay = id->layinfo;
/* if (lay == NULL) lay = id->oldlayinfo;	*/

   if (lay == NULL) return NULL;

   for (n = lay->node; n != NULL; n = n->next) {
      if (APPLE_ext_inq_equals(n->data,o,n->type,id->type)) break;
    };
   if (n == NULL) {
      save_node(DUMMY_OBJECT,o,id);
      return NULL;
    }
   else if (n->object != NULL && n->object != DUMMY_OBJECT &&
	       id->source != NULL &&
	       id->index >= 0 && GELOinq_source_index(n->object) < 0) {
      record_object(n->object,o,id);
    };

   return n->object;
};





/************************************************************************/
/*									*/
/*	save_node -- save fact that node is being drawn 		*/
/*									*/
/************************************************************************/


static void
save_node(g,o,id)
   GELO_OBJECT g;
   Universal o;
   DRAW_INFO id;
{
   register DRAW_NODE n;
   DRAW_INFO lay;

   DTRACE("save_node 0x%x 0x%x 0x%x (0x%x)",g,o,id,id->layinfo);

   if (g == NULL) return;

   lay = id->layinfo;

   if (lay != NULL) {
      for (n = lay->node; n != NULL; n = n->next) {
	 if (n->data == o) break;
       };
      if (n == NULL) {
	 n = PALLOC(DRAW_NODE_B);
	 n->next = lay->node;
	 lay->node = n;
	 n->data = o;
	 n->type = id->type;
       };
      n->object = g;
    };

   if (g != DUMMY_OBJECT) record_object(g,o,id);
};





/************************************************************************/
/*									*/
/*	record_object -- record structure info for object		*/
/*									*/
/************************************************************************/


static void
record_object(g,o,id)
   GELO_OBJECT g;
   Universal o;
   DRAW_INFO id;
{
   DTRACE("record_object 0x%x 0x%x 0x%x",g,o,id);

   if (g != DUMMY_OBJECT) {
      if (id->extinfo != NULL) GELOset_owner(g,id->extinfo);
      GELOset_user_structure(g,o);
      if (id->source != NULL) {
	 GELOset_source(g,id->source,id->field.value,id->index,id->type);
       };
    };
};





/************************************************************************/
/*									*/
/*	get_data_value -- get value from field/object			*/
/*									*/
/************************************************************************/


static Universal
get_data_value(o,v,id)
   Universal o;
   APPLE_VALUE v;
   DRAW_INFO id;
{
   Universal d,type;

   DTRACE("get_data_value 0x%x 0x%x 0x%x",o,v,id);

   d = APPLE_ext_data_value(o,id->type,v,id->from,id->index,id->level,&type);

   if (d != o) {
      id->source = o;
      id->field = *v;
      id->index = -1;
      id->array = FALSE;
      id->base = 0;
    };

   if (type != NULL) id->type = type;

   return d;
};





/************************************************************************/
/*									*/
/*	free_info -- release accumulated info when done object		*/
/*									*/
/************************************************************************/


static void
free_info(id)
   DRAW_INFO id;
{
   register DRAW_ARC a,a1;
   register DRAW_NODE n,n1;

   DTRACE("free_info 0x%x",id);

   for (a = id->arc; a != NULL; a = a1) {
      a1 = a->next;
      free(a);
    };

   for (n = id->node; n != NULL; n = n1) {
      n1 = n->next;
      free(n);
    };
};





/************************************************************************/
/*									*/
/*	define_layout_component -- define item as layout component	*/
/*									*/
/************************************************************************/


static void
define_layout_component(id,g)
   DRAW_INFO id;
   GELO_OBJECT g;
{
   DTRACE("define_layout_component 0x%x 0x%x",id,g);

   if (id->layinfo != NULL && g != NULL) {
      if (id->portal == GELO_PORT_CENTER) {
	 GELOdefine_layout_component(id->layout,g);
       }
      else {
	 GELOdefine_layout_portal(id->layout,g,id->portal);
       };
    };
};





/* end of applegelo.c */
