/************************************************************************/
/*									*/
/*		appleext.c						*/
/*									*/
/*	External interface for APPLE picture editor			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "apple_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Function_Ptr	load_routine;
static	Function_Ptr	save_routine;
static	Function_Ptr	fields_routine;
static	Function_Ptr	new_field_routine;
static	Function_Ptr	string_routine;
static	Function_Ptr	name_routine;
static	Function_Ptr	notify_routine;
static	Function_Ptr	value_routine;
static	Function_Ptr	content_routine;
static	Function_Ptr	equals_routine;
static	Function_Ptr	condition_routine;
static	Function_Ptr	null_routine;
static	Function_Ptr	pointer_routine;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


/************************************************************************/
/*									*/
/*	APPLE_ext_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
APPLE_ext_init()
{
   ITRACE("APPLE_ext_init");

   load_routine = NULL;
   save_routine = NULL;
   fields_routine = NULL;
   new_field_routine = NULL;
   string_routine = NULL;
   name_routine = NULL;
   notify_routine = NULL;
   value_routine = NULL;
   content_routine = NULL;
   equals_routine = NULL;
   condition_routine = NULL;
   null_routine = NULL;
   pointer_routine = NULL;
};





/************************************************************************/
/*									*/
/*	APPLEregister_external -- set external entries			*/
/*									*/
/************************************************************************/


void
APPLEregister_external(load,save,fields,new,string,name,notify,value,
			  content,eq,cond,null,ptr)
   Function_Ptr load;
   Function_Ptr save;
   Function_Ptr fields;
   Function_Ptr new;
   Function_Ptr string;
   Function_Ptr name;
   Function_Ptr notify;
   Function_Ptr value;
   Function_Ptr content;
   Function_Ptr eq;
   Function_Ptr cond;
   Function_Ptr null;
   Function_Ptr ptr;
{
   ENTER("APPLEregister_external");

   load_routine = load;
   save_routine = save;
   fields_routine = fields;
   new_field_routine = new;
   string_routine = string;
   name_routine = name;
   notify_routine = notify;
   value_routine = value;
   content_routine = content;
   equals_routine = eq;
   condition_routine = cond;
   null_routine = null;
   pointer_routine = ptr;
};





/************************************************************************/
/*									*/
/*	APPLEforce_save -- force a save to occur			*/
/*									*/
/************************************************************************/


int
APPLEforce_save(ad)
   APPLE_DATA ad;
{
   TRACE("APPLEforce_save 0x%x",ad);

   if (ad == NULL || ad->mode == APPLE_MODE_NONE) return FALSE;

   APPLE_save_picture(ad);

   APPLE_get_picture(ad);

   APPLE_menu_setup(ad);
   APPLE_field_menu_setup(ad);
   APPLE_draw_picture(ad);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	APPLEforce_restart -- force an abort and restart to occur	*/
/*									*/
/************************************************************************/


int
APPLEforce_restart(ad)
   APPLE_DATA ad;
{
   TRACE("APPLEforce_restart 0x%x",ad);

   if (ad == NULL) return FALSE;

   APPLE_get_picture(ad);

   APPLE_menu_setup(ad);
   APPLE_field_menu_setup(ad);
   APPLE_draw_picture(ad);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	APPLEforce_fields -- force getting of new fields		*/
/*									*/
/************************************************************************/


int
APPLEforce_fields(ad)
   APPLE_DATA ad;
{
   TRACE("APPLEforce_fields 0x%x",ad);

   if (ad == NULL) return FALSE;

   APPLE_reget_fields(ad);

   APPLE_menu_setup(ad);
   APPLE_field_menu_setup(ad);
   APPLE_draw_picture(ad);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	APPLEallow_new_fields -- allow/disallow new fields		*/
/*									*/
/************************************************************************/


void
APPLEallow_new_fields(ad,fg)
   APPLE_DATA ad;
   Boolean fg;
{
   TRACE("APPLEallow_new_fields 0x%x %d",ad,fg);

   if (ad != NULL) {
      ad->newfields = fg;
      APPLE_field_menu_setup(ad);
    };
};





/************************************************************************/
/*									*/
/*	APPLE_ext_load_picture -- load picture from outside		*/
/*									*/
/************************************************************************/


APPLE_GELO_OBJECT
APPLE_ext_load_picture(h,t,forcefg)
   Universal h;
   Universal t;
   Boolean forcefg;
{
   register APPLE_GELO_OBJECT ag;

   ITRACE("APPLE_ext_load_picture 0x%x 0x%x %d",h,t,forcefg);

   if (load_routine == NULL) {
      ag = NULL;
    }
   else {
      ag = (APPLE_GELO_OBJECT) (*load_routine)(h,t,forcefg);
      APPLE_fixup_load(ag);
    };

   return ag;
};





/************************************************************************/
/*									*/
/*	APPLE_ext_save_picture -- save picture for outside		*/
/*									*/
/************************************************************************/


void
APPLE_ext_save_picture(h,p)
   Universal h;
   APPLE_GELO_OBJECT p;
{
   ITRACE("APPLE_ext_save_picture 0x%x 0x%x",h,p);

   if (save_routine != NULL) {
      (*save_routine)(h,p);
    };
};





/************************************************************************/
/*									*/
/*	APPLE_ext_type_name -- return name of type for handle		*/
/*									*/
/************************************************************************/


String
APPLE_ext_type_name(h,t)
   Universal h;
   Universal t;
{
   register String s;

   ITRACE("APPLE_ext_type_name 0x%x 0x%x",h,t);

   if (name_routine == NULL) {
      s = "<UNKNOWN>";
    }
   else {
      s = (String) (*name_routine)(h,t);
    };

   return s;
};





/************************************************************************/
/*									*/
/*	APPLE_ext_string -- build a string value			*/
/*									*/
/************************************************************************/


void
APPLE_ext_string(v)
   APPLE_VALUE v;
{
   ITRACE("APPLE_ext_string 0x%x",v);

   if (string_routine == NULL) {
      v->value = (Integer) v->name;
    }
   else {
      (*string_routine)(v);
    };
};





/************************************************************************/
/*									*/
/*	APPLE_ext_inq_fields -- return list of fields for type		*/
/*									*/
/************************************************************************/


Integer
APPLE_ext_inq_fields(h,t,selectfg,max,flds)
   Universal h;
   Universal t;
   Boolean selectfg;
   Integer max;
   APPLE_VALUE_B flds[];
{
   register Integer ct;

   ITRACE("APPLE_ext_inq_fields 0x%x %d",h,max);

   if (fields_routine == NULL) {
      ct = 0;
    }
   else {
      ct = (*fields_routine)(h,t,selectfg,max,flds);
    };

   return ct;
};





/************************************************************************/
/*									*/
/*	APPLE_ext_new_field -- add a new field for type 		*/
/*									*/
/************************************************************************/


Boolean
APPLE_ext_new_field(h,v)
   Universal h;
   APPLE_VALUE v;
{
   register Boolean fg;

   ITRACE("APPLE_ext_new_field 0x%x 0x%x",h,v);

   if (new_field_routine == NULL) {
      v->value = (Integer) v->name;
      fg = TRUE;
    }
   else {
      fg = (*new_field_routine)(h,v);
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	APPLE_ext_notify -- tell client that picture was changed	*/
/*									*/
/************************************************************************/


void
APPLE_ext_notify(h)
   Universal h;
{
   ITRACE("APPLE_ext_notify 0x%x",h);

   if (notify_routine != NULL) {
      (*notify_routine)(h);
    };
};





/************************************************************************/
/*									*/
/*	APPLE_ext_data_value -- get data value from client		*/
/*	APPLE_ext_text_value -- get string value from client		*/
/*									*/
/************************************************************************/


Universal
APPLE_ext_data_value(o,typ,v,from,idx,lvl,rtyp)
   Universal o;
   Universal typ;
   APPLE_VALUE v;
   Universal from;
   Integer idx;
   Integer lvl;
   Universal *rtyp;
{
   Universal d;

   ITRACE("APPLE_ext_data_value 0x%x 0x%x 0x%x 0x%x %d 0x%x",o,typ,v,from,lvl,rtyp);

   if (value_routine != NULL) {
      (*value_routine)(o,typ,v,from,idx,lvl,&d,rtyp,NULL);
    }
   else {
      d = v->value;
      if (rtyp != NULL) *rtyp = 0;
    };

   return d;
};





Boolean
APPLE_ext_text_value(o,typ,v,from,idx,cnts)
   Universal o;
   Universal typ;
   APPLE_VALUE v;
   Universal from;
   Integer idx;
   String * cnts;
{
   String s;
   Boolean fg;

   ITRACE("APPLE_ext_text_value 0x%x 0x%x 0x%x 0x%x",o,typ,v,from);

   if (value_routine != NULL) {
      fg = (*value_routine)(o,typ,v,from,idx,0,NULL,NULL,&s);
    }
   else {
      s = v->name;
      fg = FALSE;
    };

   *cnts = s;

   return fg;
};





/************************************************************************/
/*									*/
/*	APPLE_ext_inq_contents -- return contents of object		*/
/*	APPLE_ext_inq_use_contents -- test if we should use contents	*/
/*									*/
/************************************************************************/


Integer
APPLE_ext_inq_contents(obj,typ,max,val,vtyp,base)
   Universal obj;
   Universal typ;
   Integer max;
   Universal val[];
   Universal vtyp[];
   Integer * base;
{
   register Integer ct;

   ITRACE("APPLE_ext_inq_contents 0x%x 0x%x %d",obj,typ,max);

   *base = 0;

   if (content_routine != NULL) {
      ct = (*content_routine)(obj,NULL,typ,max,val,vtyp,base);
    }
   else {
      ct = 0;
    };

   return ct;
};





Boolean
APPLE_ext_inq_use_contents(obj,h,typ)
   Universal obj;
   Universal h;
   Universal typ;
{
   register Integer ct;

   ITRACE("APPLE_ext_inq_use_contents 0x%x 0x%x",obj,typ);

   if (content_routine != NULL) {
      ct = (*content_routine)(obj,h,typ,-1,NULL,NULL,NULL);
    }
   else {
      ct = 0;
    };

   return (ct != 0);
};





/************************************************************************/
/*									*/
/*	APPLE_ext_inq_equals -- check if too values are equal		*/
/*									*/
/************************************************************************/


Boolean
APPLE_ext_inq_equals(v1,v2,t1,t2)
   Universal v1,v2;
   Universal t1,t2;
{
   register Boolean fg;

   ITRACE("APPLE_ext_inq_equals 0x%x 0x%x 0x%x 0x%x",v1,v2,t1,t2);

   if (equals_routine != NULL) {
      fg = (*equals_routine)(v1,v2,t1,t2);
    }
   else {
      fg = (v1 == v2);
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	APPLE_ext_test_null -- check if value is null			*/
/*									*/
/************************************************************************/


Boolean
APPLE_ext_test_null(v,typ)
   Universal v;
   Universal typ;
{
   register Boolean fg;

   ITRACE("APPLE_ext_test_null 0x%x 0x%x",v,typ);

   if (null_routine != NULL) {
      fg = (*null_routine)(v,typ);
    }
   else {
      fg = (v == NULL);
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	APPLE_ext_inq_condition -- request condition from user		*/
/*	APPLE_ext_inq_function -- request function from user		*/
/*	APPLE_ext_test_condition -- check if condition is satisfied	*/
/*									*/
/************************************************************************/


Boolean
APPLE_ext_inq_condition(vp)
   Universal *vp;
{
   register Boolean fg;

   ITRACE("APPLE_ext_inq_condition");

   if (condition_routine != NULL) {
      fg = (*condition_routine)(NULL,NULL,vp,FALSE);
    }
   else {
      fg = FALSE;
    };

   return fg;
};




Boolean
APPLE_ext_inq_function(vp)
   Universal *vp;
{
   register Boolean fg;

   ITRACE("APPLE_ext_inq_function");

   if (condition_routine != NULL) {
      fg = (*condition_routine)(NULL,NULL,vp,TRUE);
    }
   else {
      fg = FALSE;
    };

   return fg;
};





Boolean
APPLE_ext_test_condition(v,o)
   Universal v;
   Universal o;
{
   register Boolean fg;

   ITRACE("APPLE_ext_test_condition 0x%x 0x%x",v,o);

   if (condition_routine != NULL) {
      fg = (*condition_routine)(v,o,NULL,FALSE);
    }
   else {
      fg = FALSE;
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	APPLE_ext_inq_pointer -- check if field is a pointer		*/
/*									*/
/************************************************************************/


Boolean
APPLE_ext_inq_pointer(obj,h,typ,fld)
   Universal obj;
   Universal h;
   Universal typ;
   APPLE_VALUE fld;
{
   Boolean fg;

   ITRACE("APPLE_ext_inq_pointer 0x%x 0x%x 0x%x 0x%x",obj,h,typ,fld);

   if (pointer_routine != NULL) {
      fg = (*pointer_routine)(obj,h,typ,fld);
    }
   else fg = FALSE;

   return fg;
};





/* end of appleext.c */

