/************************************************************************/
/*									*/
/*		appleedit.c						*/
/*									*/
/*	Editor for APPLE type picture editor package			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "apple_local.h"




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/



/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Boolean 		select_block();
static	Boolean 		select_arc();
static	Boolean 		handle_drawing();
static	Boolean 		select_option();
static	Boolean 		handle_typein();
static	Boolean 		draw_new_box();
static	Boolean 		new_box_rtn();
static	Boolean 		move_vert_edge();
static	Boolean 		move_hor_edge();
static	Boolean 		vert_edge_rtn();
static	Boolean 		hor_edge_rtn();
static	APPLE_GELO_COMPONENT	connect_blocks();
static	Boolean 		connect_rtn();
static	Boolean 		request_option();
static	Boolean 		inside_block();





/************************************************************************/
/*									*/
/*	APPLE_edit_init -- module initialization			*/
/*									*/
/************************************************************************/


void
APPLE_edit_init()
{
   ITRACE("APPLE_edit_init");
};





/************************************************************************/
/*									*/
/*	APPLE_window_btn -- handle button press in window		*/
/*									*/
/************************************************************************/


int
APPLE_window_btn(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   register APPLE_DATA ad;
   register Boolean fg;

   ITRACE("APPLE_window_btn %d %d %d 0x%x 0x%x",x,y,ch,btn,rgn);

   ad = (APPLE_DATA) RIPinq_data(rgn);
   if (ad == NULL) return FALSE;

   if (ad->picture == NULL) return FALSE;

   ASHmap(ASHinq_top_window(ad->draw_win),x,y,ad->draw_win,&x,&y);

   if (ad->use_select != USE_SELECT_NONE) {
      ad->use_select = USE_SELECT_NONE;
      ad->usearc_select = NULL;
      APPLE_field_menu_setup(ad);
      APPLE_draw_usearea(ad);
      fg = TRUE;
    }
   else if (ad->usearc_select != NULL) {
      ad->usearc_select = NULL;
      APPLE_field_menu_setup(ad);
      fg = TRUE;
    }
   else fg = FALSE;

   if (btn & RIP_BTN_NONE) {
      fg = handle_typein(ad,x,y,ch);
    }
   else if (ad->mode == APPLE_MODE_TILED) {
      if (btn & RIP_BTN_LEFT) {
	 if (btn & RIP_BTN_UP) return FALSE;
	 fg = handle_drawing(ad,x,y);
       }
      else if (btn & RIP_BTN_MID) {
	 if (btn & RIP_BTN_DOWN) return FALSE;
	 fg = select_block(ad,x,y);
       }
      else if (btn & RIP_BTN_RIGHT) {
	 if (btn & RIP_BTN_UP) return FALSE;
	 fg = select_option(ad,x,y);
       };
    }
   else if (ad->mode == APPLE_MODE_ARC) {
      fg = select_arc(ad,x,y);
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	APPLE_use_btn -- handle hit in USE area 			*/
/*									*/
/************************************************************************/


int
APPLE_use_btn(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   register APPLE_DATA ad;
   Integer xsz,ysz;
   register APPLE_GELO ag;
   register APPLE_VALUE av;
   register Sequence l;
   register Integer chpos,chln;
   register Boolean fg;

   ITRACE("APPLE_use_btn %d %d %d 0x%x 0x%x",x,y,ch,btn,rgn);

   ad = (APPLE_DATA) RIPinq_data(rgn);
   if (ad == NULL) return FALSE;
   if (btn & RIP_BTN_DOWN) return FALSE;

   if (btn & RIP_BTN_NONE) {
      fg = handle_typein(ad,x,y,ch);
      return fg;
    };

   if (ad->selection != NULL) APPLE_draw_select(ad,ad->selection,FALSE);
   ad->selection = NULL;

   ASHinq_text_next(ad->use_window,"X",&xsz,&ysz);
   ASHmap(ASHinq_top_window(ad->use_window),x,y,ad->use_window,&x,&y);
   chpos = x/xsz;
   if (chpos < USE_PREFIX_LENGTH-2) ad->use_select = USE_SELECT_USE;
   else {
      chpos -= USE_PREFIX_LENGTH;
      forin (ag,APPLE_GELO,l,ad->picture->use) {
	 if (chpos < 0) return FALSE;
	 av = APPLE_layout_value(ag);
	 if (av->name != NULL) {
	    chln = strlen(av->name);
	    if (ag->mode == APPLE_MODE_ARC) chln += 3;
	    if (chpos < chln) break;
	    chpos -= chln+1;
	  };
       };
      if (EMPTY(l)) return FALSE;
      ad->usearc_select = ag;
      ad->use_select = USE_SELECT_ONEUSE;
      ASHset_user_data(ad->use_window,-1);
    };

   APPLE_field_menu_setup(ad);
   APPLE_draw_usearea(ad);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	select_block -- select block in tiled drawing			*/
/*									*/
/************************************************************************/


static Boolean
select_block(ad,x,y)
   APPLE_DATA ad;
   Integer x,y;
{
   Integer gx,gy;
   register APPLE_GELO_COMPONENT blk;
   register Sequence l;
   register APPLE_GELO_TILED til;

   DTRACE("select_block 0x%x %d %d",ad,x,y);

   if (ad->mode != APPLE_MODE_TILED) return FALSE;

   APPLE_grid_coords(ad,x,y,&gx,&gy);

   blk = NULL;
   if (ad->selection != NULL) {
      if (inside_block(ad,ad->selection,x,y,gx,gy))
	 blk = ad->selection;
    };

   if (blk == NULL) {
      til = (APPLE_GELO_TILED) ad->picture->object;
      forin (blk,APPLE_GELO_COMPONENT,l,til->components) {
	 if (inside_block(ad,blk,x,y,gx,gy)) break;
       };
      if (EMPTY(l)) blk = NULL;
    };

   APPLE_draw_select(ad,ad->selection,FALSE);
   ad->selection = blk;
   ad->use_select = USE_SELECT_NONE;
   ad->usearc_select = NULL;
   APPLE_menu_prop_setup(ad);
   APPLE_field_menu_setup(ad);
   APPLE_draw_select(ad,blk,TRUE);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	select_arc -- handle arc selection				*/
/*									*/
/************************************************************************/


static Boolean
select_arc(ad,x,y)
   APPLE_DATA ad;
   Integer x,y;
{
   Integer lx,by,rx,ty;
   APPLE_GELO_COMPONENT sel;

   DTRACE("select_arc 0x%x %d %d",ad,x,y);

   sel = SELECT_FROM;
   APPLE_arc_box(ad,sel,&lx,&by,&rx,&ty);
   if (x < lx || x > rx || y < by || y > ty) {
      sel = SELECT_TO;
      APPLE_arc_box(ad,sel,&lx,&by,&rx,&ty);
      if (x < lx || x > rx || y < by || y > ty) {
	 sel = SELECT_LABEL;
	 APPLE_arc_box(ad,sel,&lx,&by,&rx,&ty);
	 if (x < lx || x > rx || y < by || y > ty) {
	    sel = NULL;
	  };
       };
    };

   if (ad->selection != sel) {
      APPLE_draw_select(ad,ad->selection,FALSE);
      ad->selection = sel;
      ad->use_select = USE_SELECT_NONE;
      ad->usearc_select = NULL;
      APPLE_field_menu_setup(ad);
      APPLE_draw_select(ad,sel,TRUE);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	handle_drawing -- user drawing option				*/
/*									*/
/************************************************************************/


static Boolean
handle_drawing(ad,x,y)
   APPLE_DATA ad;
   Integer x,y;
{
   register APPLE_GELO_COMPONENT blk;
   register APPLE_GELO_TILED til;
   register Sequence l;
   register Boolean fg;
   Integer gx,gy;

   DTRACE("handle_drawing 0x%x %d %d",ad,x,y);

   if (ad->mode != APPLE_MODE_TILED) return FALSE;

   APPLE_grid_coords(ad,x,y,&gx,&gy);

   blk = NULL;
   if (ad->selection != NULL) {
      if (gx >= ad->selection->lx && gx <= ad->selection->rx &&
	     (gy == ad->selection->ty || gy == ad->selection->by))
	 blk = ad->selection;
      else if ((gx == ad->selection->lx || gx == ad->selection->rx) &&
	     gy >= ad->selection->ty && gy <= ad->selection->by)
	 blk = ad->selection;
    };

   if (blk == NULL) {
      til = (APPLE_GELO_TILED) ad->picture->object;
      forin (blk,APPLE_GELO_COMPONENT,l,til->components) {
	 if (gx >= blk->lx && gx <= blk->rx && (gy == blk->ty || gy == blk->by))
	    break;
	 else if ((gx == blk->lx || gx == blk->rx) && gy >= blk->ty && gy <= blk->by)
	    break;
       };
      if (EMPTY(l)) blk = NULL;
    };

   if (blk == NULL) {
      fg = draw_new_box(ad,blk,gx,gy);
    }
   else if (gx == blk->lx || gx == blk->rx) {
      if (gy == blk->ty || gy == blk->by) {
	 fg = draw_new_box(ad,blk,gx,gy);
	}
      else {
	 fg = move_hor_edge(ad,blk,gx,gy);
       };
    }
   else if (gy == blk->ty || gy == blk->by) {
      fg = move_vert_edge(ad,blk,gx,gy);
    }
   else {
      fg = FALSE;
    };

   if (fg) {
      APPLE_draw_picture(ad);
      APPLE_menu_setup(ad);
      APPLE_field_menu_setup(ad);
    }
   else {
      fg = select_block(ad,x,y);
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	select_option -- handle optional drawing info (constraint/arc)	*/
/*									*/
/************************************************************************/


static Boolean
select_option(ad,x,y)
   APPLE_DATA ad;
   Integer x,y;
{
   register APPLE_GELO_COMPONENT blk,blk1;
   register APPLE_GELO_TILED til;
   register Sequence l;
   register Boolean fg;
   Integer gx,gy;

   DTRACE("select_option 0x%x %d %d",ad,x,y);

   if (ad->mode != APPLE_MODE_TILED) return FALSE;

   APPLE_grid_coords(ad,x,y,&gx,&gy);

   til = (APPLE_GELO_TILED) ad->picture->object;
   forin (blk,APPLE_GELO_COMPONENT,l,til->components) {
      if (inside_block(ad,blk,x,y,gx,gy)) break;
    };
   if (EMPTY(l)) return FALSE;

   blk1 = connect_blocks(ad,x,y);
   if (blk1 == NULL) return FALSE;

   fg = request_option(ad,blk,blk1);

   if (fg) {
      APPLE_draw_picture(ad);
      APPLE_menu_setup(ad);
      APPLE_field_menu_setup(ad);
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	handle_typein -- handle typein in a box 			*/
/*									*/
/************************************************************************/


static Boolean
handle_typein(ad,x,y,ch)
   APPLE_DATA ad;
   Integer x,y;
   Integer ch;
{
   register Boolean fg;

   DTRACE("handle_typein 0x%x %d %d %d",ad,x,y,ch);

   switch (ch) {
      case 'o' :
      case 'O' :
	 fg = APPLE_option_btn(ad,"TYPEIN","TYPEIN");
	 break;
      case 'b' :
      case 'B' :
	 if (ad->selection != NULL)
	    fg = APPLE_box_type_btn(ad,"TYPEIN","Box");
	 else if (ad->mode == APPLE_MODE_NONE)
	    fg = APPLE_design_btn(ad,"TYPEIN","Box");
	 else fg = FALSE;
	 break;
      case 'l' :
      case 'L' :
	 if (ad->selection != NULL)
	    fg = APPLE_box_type_btn(ad,"TYPEIN","Layout");
	 else if (ad->mode == APPLE_MODE_NONE)
	    fg = APPLE_design_btn(ad,"TYPEIN","Layout");
	 else fg = FALSE;
	 break;
      case 'f' :
      case 'F' :
	 if (ad->selection != NULL) fg = APPLE_box_type_btn(ad,"TYPEIN","Field");
	 else fg = FALSE;
	 break;
      case 't' :
      case 'T' :
	 if (ad->mode == APPLE_MODE_NONE)
	    fg = APPLE_design_btn(ad,"TYPEIN","Tiled");
	 else fg = FALSE;
	 break;
      case 'a' :
      case 'A' :
	 if (ad->mode == APPLE_MODE_NONE)
	    fg = APPLE_design_btn(ad,"TYPEIN","Arc");
	 else fg = FALSE;
	 break;
      case 'd' :
      case 'D' :
	 if (ad->mode == APPLE_MODE_NONE)
	    fg = APPLE_default_btn(ad,"TYPEIN","Default");
	 else fg = FALSE;
	 break;

      default :
	 fg = FALSE;
	 break;
    };

   return fg;
};






/************************************************************************/
/*									*/
/*	draw_new_box -- handle box drawing				*/
/*									*/
/************************************************************************/


typedef struct _box_data {
   Integer	   init_x;
   Integer	   init_y;
   APPLE_DATA	   cur_ad;
   Integer	   last_x;
   Integer	   last_y;
} BOX_DATA;


static Boolean
draw_new_box(ad,blk,x,y)
   APPLE_DATA ad;
   APPLE_GELO_COMPONENT blk;
   Integer x,y;
{
   BOX_DATA b;
   Integer d;

   DTRACE("draw_new_box 0x%x 0x%x %d %d",ad,blk,x,y);

   b.init_x = x;
   b.init_y = y;
   b.last_x = x;
   b.last_y = y;
   b.cur_ad = ad;

   d = ASHinq_user_data(ad->draw_win);
   ASHset_user_data(ad->draw_win,&b);
   ASHdraw_hilite(ad->draw_win,TRUE);
   RIPinput_track_down(ad->draw_win,"",new_box_rtn,3);
   ASHdraw_hilite(ad->draw_win,FALSE);
   ASHset_user_data(ad->draw_win,d);

   if (b.init_x == b.last_x || b.init_y == b.last_y) return FALSE;

   blk = APPLE_add_component(ad,b.init_x,b.init_y,b.last_x,b.last_y);

   if (blk != NULL) ad->selection = blk;
   ad->use_select = USE_SELECT_NONE;
   ad->usearc_select = NULL;

   return TRUE;
};





static Boolean
new_box_rtn(x,y,ct,act,max,w)
   Integer x,y;
   Integer ct;
   RIP_TRANS act;
   Integer max;
   ASH_WINDOW w;
{
   Integer x0,y0,x1,y1,xg,yg,oxg,oyg;
   BOX_DATA * b;

   DTRACE("new_box_rtn %d %d %d %d %d 0x%x",x,y,ct,act,max,w);

   if (act == RIP_TRANS_NONE) return TRUE;

   b = (BOX_DATA *) ASHinq_user_data(w);

   APPLE_grid_coords(b->cur_ad,x,y,&xg,&yg);
   APPLE_ash_coords(b->cur_ad,b->init_x,b->init_y,&x0,&y0);

   oxg = b->last_x;
   oyg = b->last_y;

   if ((b->last_x != b->init_x || b->last_y != b->init_y) &&
	  (oxg != xg || oyg != yg)) {
      APPLE_ash_coords(b->cur_ad,b->last_x,b->last_y,&x1,&y1);
      ASHbox(w,x0,y0,x1,y1);
    };

   b->last_x = xg;
   b->last_y = yg;
   APPLE_ash_coords(b->cur_ad,b->last_x,b->last_y,&x1,&y1);

   if (act != RIP_TRANS_UP) {
      if ((b->last_x != b->init_x || b->last_y != b->init_y) &&
	     (oxg != xg || oyg != yg)) {
	 ASHbox(w,x0,y0,x1,y1);
       };
    }
   else {
      return FALSE;
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	move_vert_edge --- handle edge movement editing (in vert dir)	*/
/*	move_hor_edge -- handle edge movement in hor direction		*/
/*									*/
/************************************************************************/


typedef struct _move_data {
   APPLE_DATA	   cur_ad;
   APPLE_GELO_COMPONENT    cur_blk;
   Integer		   init_edge;
   Integer		   final_edge;
} MOVE_DATA;



static Boolean
move_vert_edge(ad,blk,x,y)
   APPLE_DATA ad;
   APPLE_GELO_COMPONENT blk;
   Integer x,y;
{
   register Boolean fg;
   MOVE_DATA m;
   Integer d;

   DTRACE("move_vert_edge 0x%x 0x%x %d %d",ad,blk,x,y);

   m.cur_ad = ad;
   m.cur_blk = blk;
   m.init_edge = y;
   m.final_edge = y;

   d = ASHinq_user_data(ad->draw_win);
   ASHset_user_data(ad->draw_win,&m);
   ASHdraw_hilite(ad->draw_win,TRUE);
   RIPinput_track_down(ad->draw_win,"",vert_edge_rtn,3);
   ASHdraw_hilite(ad->draw_win,FALSE);
   ASHset_user_data(ad->draw_win,d);

   if (y == blk->ty) {
      fg = APPLE_change_component(ad,blk,blk->lx,blk->by,blk->rx,m.final_edge);
    }
   else {
      fg = APPLE_change_component(ad,blk,blk->lx,m.final_edge,blk->rx,blk->ty);
    };

   if (fg) ad->selection = blk;
   ad->use_select = USE_SELECT_NONE;
   ad->usearc_select = NULL;

   return fg;
};





static Boolean
move_hor_edge(ad,blk,x,y)
   APPLE_DATA ad;
   APPLE_GELO_COMPONENT blk;
   Integer x,y;
{
   register Boolean fg;
   MOVE_DATA m;
   Integer d;

   DTRACE("move_hor_edge 0x%x 0x%x %d %d",ad,blk,x,y);

   m.cur_ad = ad;
   m.cur_blk = blk;
   m.init_edge = x;
   m.final_edge = x;

   d = ASHinq_user_data(ad->draw_win);
   ASHset_user_data(ad->draw_win,&m);
   ASHdraw_hilite(ad->draw_win,TRUE);
   RIPinput_track_down(ad->draw_win,"",hor_edge_rtn,3);
   ASHdraw_hilite(ad->draw_win,FALSE);
   ASHset_user_data(ad->draw_win,d);

   if (x == blk->lx) {
      fg = APPLE_change_component(ad,blk,m.final_edge,blk->by,blk->rx,blk->ty);
    }
   else {
      fg = APPLE_change_component(ad,blk,blk->lx,blk->by,m.final_edge,blk->ty);
    };

   if (fg) ad->selection = blk;
   ad->use_select = USE_SELECT_NONE;
   ad->usearc_select = NULL;

   return fg;
};





static Boolean
vert_edge_rtn(x,y,ct,act,max,w)
   Integer x,y;
   Integer ct;
   RIP_TRANS act;
   Integer max;
   ASH_WINDOW w;
{
   Integer x0,y0,x1,y1,xg,yg;
   register MOVE_DATA * m;

   DTRACE("vert_edge_rtn %d %d %d %d %d 0x%x",x,y,ct,act,max,w);

   if (act == RIP_TRANS_NONE) return TRUE;

   m = (MOVE_DATA *) ASHinq_user_data(w);

   APPLE_grid_coords(m->cur_ad,m->cur_blk->rx,y,&xg,&yg);
   APPLE_ash_coords(m->cur_ad,m->cur_blk->lx,m->init_edge,&x0,&y0);

   APPLE_ash_coords(m->cur_ad,m->cur_blk->rx,m->final_edge,&x1,&y1);
   ASHbox(w,x0,y0,x1,y1);

   m->final_edge = yg;
   APPLE_ash_coords(m->cur_ad,m->cur_blk->rx,m->final_edge,&x1,&y1);

   if (act != RIP_TRANS_UP) {
      ASHbox(w,x0,y0,x1,y1);
    }
   else {
      return FALSE;
    };

   return TRUE;
};






static Boolean
hor_edge_rtn(x,y,ct,act,max,w)
   Integer x,y;
   Integer ct;
   RIP_TRANS act;
   Integer max;
   ASH_WINDOW w;
{
   Integer x0,y0,x1,y1,xg,yg;
   register MOVE_DATA * m;

   DTRACE("hor_edge_rtn %d %d %d %d %d 0x%x",x,y,ct,act,max,w);

   if (act == RIP_TRANS_NONE) return TRUE;

   m = (MOVE_DATA *) ASHinq_user_data(w);

   APPLE_grid_coords(m->cur_ad,x,m->cur_blk->ty,&xg,&yg);
   APPLE_ash_coords(m->cur_ad,m->init_edge,m->cur_blk->by,&x0,&y0);

   APPLE_ash_coords(m->cur_ad,m->final_edge,m->cur_blk->ty,&x1,&y1);
   ASHbox(w,x0,y0,x1,y1);

   m->final_edge = xg;
   APPLE_ash_coords(m->cur_ad,m->final_edge,m->cur_blk->ty,&x1,&y1);

   if (act != RIP_TRANS_UP) {
      ASHbox(w,x0,y0,x1,y1);
    }
   else {
      return FALSE;
    };

   return TRUE;
};






/************************************************************************/
/*									*/
/*	connect_blocks -- handle line connecting boxes drawing		*/
/*									*/
/************************************************************************/


typedef struct _conn_data {
   Integer	   conn_x;
   Integer	   conn_y;
   APPLE_DATA	   conn_ad;
   Integer	   connl_x;
   Integer	   connl_y;
} CONN_DATA;


static APPLE_GELO_COMPONENT
connect_blocks(ad,x,y)
   APPLE_DATA ad;
   Integer x,y;
{
   register APPLE_GELO_COMPONENT blk;
   register APPLE_GELO_TILED til;
   register Sequence l;
   Integer gx,gy;
   CONN_DATA c;
   Integer d;

   DTRACE("connect_blocks 0x%x 0x%x %d %d",ad,x,y);

   c.conn_x = x;
   c.conn_y = y;
   c.connl_x = x;
   c.connl_y = y;
   c.conn_ad = ad;

   d = ASHinq_user_data(ad->draw_win);
   ASHset_user_data(ad->draw_win,&c);
   ASHdraw_hilite(ad->draw_win,TRUE);
   RIPinput_track_down(ad->draw_win,"",connect_rtn,3);
   ASHdraw_hilite(ad->draw_win,FALSE);
   ASHset_user_data(ad->draw_win,d);

   APPLE_grid_coords(ad,c.connl_x,c.connl_y,&gx,&gy);

   til = (APPLE_GELO_TILED) ad->picture->object;
   forin (blk,APPLE_GELO_COMPONENT,l,til->components) {
      if (inside_block(ad,blk,c.connl_x,c.connl_y,gx,gy)) break;
    };
   if (EMPTY(l)) return NULL;

   return blk;
};





static Boolean
connect_rtn(x,y,ct,act,max,w)
   Integer x,y;
   Integer ct;
   RIP_TRANS act;
   Integer max;
   ASH_WINDOW w;
{
   Integer x0,y0;
   CONN_DATA * c;

   DTRACE("connect_rtn %d %d %d %d %d 0x%x",x,y,ct,act,max,w);

   if (act == RIP_TRANS_NONE) return TRUE;

   c = (CONN_DATA *) ASHinq_user_data(w);

   x0 = c->connl_x;
   y0 = c->connl_y;

   if (act == RIP_TRANS_UP ||
	  ((c->connl_x != c->conn_x || c->connl_y != c->conn_y) &&
	      (x0 != x || y0 != y))) {
      ASHline(w,c->conn_x,c->conn_y,c->connl_x,c->connl_y);
    };

   c->connl_x = x;
   c->connl_y = y;

   if (act != RIP_TRANS_UP) {
      if ((c->connl_x != c->conn_x || c->connl_y != c->conn_y) &&
	     (x0 != x || y0 != y)) {
	 ASHline(w,c->conn_x,c->conn_y,c->connl_x,c->connl_y);
       };
    }
   else {
      return FALSE;
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	request_option -- get option info from user			*/
/*									*/
/************************************************************************/


static Boolean
request_option(ad,blk0,blk1)
   APPLE_DATA ad;
   APPLE_GELO_COMPONENT blk0,blk1;
{
   String btns[64];
   Integer i;
   Integer cnst,fport,tport,lstyle,fromy,toy,arrow;
   double mult,add;
   Character mbuf[32],abuf[32];
   register APPLE_GELO_TILED til;
   register APPLE_GELO_TARC arc;
   register APPLE_GELO_TCONST tconst;
   register Sequence l;

   DTRACE("request_option 0x%x 0x%x 0x%x",ad,blk0,blk1);

   til = (APPLE_GELO_TILED) ad->picture->object;

   forin (arc,APPLE_GELO_TARC,l,til->arcs) {
      if (arc->from == blk0 && arc->to == blk1) break;
    };
   if (EMPTY(l)) arc = NULL;

   forin (tconst,APPLE_GELO_TCONST,l,til->constraints) {
      if (tconst->from == blk0 && tconst->to == blk1) break;
    };
   if (EMPTY(l)) tconst = NULL;

   i = 0;
   btns[i++] = "%CTiling Options\n";

   btns[i++] = "%0.0o Add Connection";
   if (arc != NULL) {
      btns[i++] = "%0.2o Remove Connection";
      btns[i++] = "%0.4o Edit Connection";
    };
   btns[i++] = "\n%0.1o Add Constraint";
   if (tconst != NULL) {
      btns[i++] = "%0.3o Remove Constraint";
      btns[i++] = "%0.5o Edit Contraint";
    };
   btns[i++] = "\n   %a%M   %c";
   btns[i] = 0;
   cnst = 0;
   if (!STEMdialog(ad->window,btns,&cnst)) return FALSE;

   if (cnst == 2) {			/* remove connection		*/
      APPLE_tiled_connect_remove(ad,arc);
      return TRUE;
    }
   else if (cnst == 3) {		/* remove constraint		*/
      APPLE_tiled_constraint_remove(ad,tconst);
      return TRUE;
    }
   else if (cnst == 1) tconst = NULL;
   else if (cnst == 0) arc = NULL;
   else if (cnst == 5) cnst = 1;
   else if (cnst == 4) cnst = 0;

   i = 0;

   if (cnst == 0) {			/* add connection		*/
      btns[i++] = "%CConnection Information\n";
      btns[i++] = "   From Port:         %0.9o ANY       %0.14o Computed";
      btns[i++] = "      %0.1o Top Left     %0.2o Top       %0.3o Top Right";
      btns[i++] = "      %0.0o Left         %0.8o Center    %0.4o Right";
      btns[i++] = "      %0.7o Bottom Left  %0.6o Bottom    %0.5o Bottom Right";
      btns[i++] = "      %0.10o Left Sequence     %0.12o Right Sequence";
      btns[i++] = "      %0.11o Top Sequence      %0.13o Bottom Sequence\n";

      btns[i++] = "   To Port:           %1.9o ANY       %1.14o Computed";
      btns[i++] = "      %1.1o Top Left     %1.2o Top       %1.3o Top Right";
      btns[i++] = "      %1.0o Left         %1.8o Center    %1.4o Right";
      btns[i++] = "      %1.7o Bottom Left  %1.6o Bottom    %1.5o Bottom Right";
      btns[i++] = "      %1.10o Left Sequence     %1.12o Right Sequence";
      btns[i++] = "      %1.11o Top Sequence      %1.13o Bottom Sequence\n";

      btns[i++] = "   %2.0o Solid%M%2.1o Dashed";
      btns[i++] = "   %2.2o Dotted%M%2.3o Thick";
      btns[i++] = "   %2.4o Thicker%M%2.5o Invisible\n";
      btns[i++] = "   %3.0o No arrows%M%3.1o Single Arrows";
      btns[i++] = "   %3.2o Double arrows\n";

      btns[i++] = "   %a%M   %c";
      btns[i] = 0;

      if (arc == NULL) {
	 fport = 9;
	 tport = 9;
	 lstyle = 0;
	 arrow = 0;
       }
      else {
	 fport = (int) arc->fport;
	 tport = (int) arc->tport;
	 lstyle = (int) arc->style;
	 arrow = (int) arc->arrow;
       };

      if (!STEMdialog(ad->window,btns,&fport,&tport,&lstyle,&arrow))
	 return FALSE;

      if (arc == NULL) {
	 APPLE_tiled_connect(ad,blk0,blk1,(GELO_PORT) fport,(GELO_PORT) tport,
				lstyle,(GELO_ARC_ARROW) arrow);
       }
      else {
	 arc->fport = (GELO_PORT) fport;
	 arc->tport = (GELO_PORT) tport;
	 arc->style = (ASH_LINE_STYLE) lstyle;
	 arc->arrow = (GELO_ARC_ARROW) arrow;
       };
    }
   else {
      btns[i++] = "%CConstraint Information\n";

      btns[i++] = "   %0.0o From X%M%1.0o To X";
      btns[i++] = "   %0.1o From Y%M%1.1o To Y\n";

      btns[i++] = "   FROM = %2.10t * TO + %3.10t\n\n";

      btns[i++] = "   %a%M   %c";
      btns[i] = 0;

      if (tconst == NULL) {
	 fromy = 0;
	 toy = 0;
	 mbuf[0] = 0;
	 abuf[0] = 0;
       }
      else {
	 fromy = tconst->from_y;
	 toy = tconst->to_y;
	 sprintf(mbuf,"%f",tconst->multiplier);
	 sprintf(abuf,"%f",tconst->addend);
       };

      if (!STEMdialog(ad->window,btns,&fromy,&toy,mbuf,abuf)) return FALSE;

      if (mbuf[0] == 0 && abuf[0] == 0) {
	 mult = 1.0;
	 add = 0.0;
       }
      else {
	 if (mbuf[0] == 0) mult = 0.0;
	 else if (sscanf(mbuf,"%F",&mult) != 1) return FALSE;
	 if (abuf[0] == 0) add = 0.0;
	 else if (sscanf(abuf,"%F",&add) != 1) return FALSE;
       };

      if (tconst == NULL) {
	 APPLE_tiled_constraint(ad,blk0,blk1,fromy,toy,mult,add);
       }
      else {
	 tconst->from_y = fromy;
	 tconst->to_y = toy;
	 tconst->multiplier = mult;
	 tconst->addend = add;
       };
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	inside_block -- check if point is inside component		*/
/*									*/
/************************************************************************/


static Boolean
inside_block(ad,blk,x,y,gx,gy)
   APPLE_DATA ad;
   APPLE_GELO_COMPONENT blk;
   Integer x,y;
   Integer gx,gy;
{
   Integer nlx,nby,nrx,nty;

   DTRACE("inside_block 0x%x 0x%x %d %d %d %d",ad,blk,x,y,gx,gy);

   if (gx >= blk->lx && gx <= blk->rx && gy >= blk->ty && gy <= blk->by) {
      APPLE_ash_coords(ad,blk->lx,blk->by,&nlx,&nby);
      APPLE_ash_coords(ad,blk->rx,blk->ty,&nrx,&nty);
      if (x >= nlx && x <= nrx && y >= nby && y <= nty) return TRUE;
    };

   return FALSE;
};





/* end of appleedit.c */


