/************************************************************************/
/*									*/
/*		appledraw.c						*/
/*									*/
/*	Drawing routines for APPLE type picture editor package		*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "apple_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define HILITE_FILL	0
#define PORTAL_SIZE	5





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		draw_object();
static	void		draw_arc();
static	void		draw_layout();
static	void		draw_tiled();
static	void		draw_box();
static	void		draw_unknown();
static	void		draw_field_object();
static	void		draw_pointer_object();
static	void		output_text();





/************************************************************************/
/*									*/
/*	APPLE_draw_init -- module initialization			*/
/*									*/
/************************************************************************/


void
APPLE_draw_init()
{
   ITRACE("APPLE_draw_init");
};





/************************************************************************/
/*									*/
/*	APPLE_draw_picture -- draw representative picture		*/
/*									*/
/************************************************************************/


void
APPLE_draw_picture(ad)
   APPLE_DATA ad;
{
   Integer lx,by,rx,ty;
   Integer ls,x,y;
   register Integer i;

   ITRACE("APPLE_draw_picture 0x%x",ad);

   if (ad->draw_win == NULL) return;

   ASHbatch_mode(TRUE);

   ASHinq_size(ad->draw_win,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   ASHclear_box(ad->draw_win,lx,by,rx,ty);

   ad->draw_x = abs(rx-lx)+1;
   ad->draw_y = abs(ty-by)+1;

   if (ad->picture != NULL) {
      draw_object(ad->picture->object,ad,lx,by,rx,ty);
      if (ad->selection != NULL)
	 APPLE_draw_select(ad,ad->selection,TRUE);
    };

   if (ad->show_grid) {
      ASHdraw_hilite(ad->draw_win,TRUE);
      ls = ASHline_style(ad->draw_win,ASH_STYLE_DOTTED);
      for (i = 0; i < ad->grid; ++i) {
	 APPLE_ash_coords(ad,i,i,&x,&y);
	 ASHline(ad->draw_win,x,ty,x,by);
	 ASHline(ad->draw_win,lx,y,rx,y);
       };
      ASHline_style(ad->draw_win,ls);
      ASHdraw_hilite(ad->draw_win,FALSE);
    };

   if (ad->use_window != NULL) ASHset_user_data(ad->use_window,-1);

   APPLE_draw_usearea(ad);

   ASHbatch_mode(FALSE);
};





/************************************************************************/
/*									*/
/*	APPLE_grid_coords -- map ash to grid				*/
/*	APPLE_ash_coords -- map grid to ash				*/
/*									*/
/************************************************************************/


void
APPLE_grid_coords(ad,x,y,xp,yp)
   APPLE_DATA ad;
   Integer x,y;
   Integer *xp, *yp;
{
   ITRACE("APPLE_grid_coords 0x%x %d %d",ad,x,y);

   x = (x*ad->grid*2)/ad->draw_x;
   *xp = (x+1)/2;

   y = ((ad->draw_y-y)*ad->grid*2)/ad->draw_y;
   *yp = (y+1)/2;
};





void
APPLE_ash_coords(ad,x,y,xp,yp)
   APPLE_DATA ad;
   Integer x,y;
   Integer *xp,*yp;
{
   ITRACE("APPLE_ash_coords 0x%x %d %d",ad,x,y);

   *xp = (x*ad->draw_x)/ad->grid;
   *yp = ad->draw_y - (y*ad->draw_y)/ad->grid;
};





/************************************************************************/
/*									*/
/*	APPLE_draw_select -- mark selected component of layout		*/
/*									*/
/************************************************************************/


void
APPLE_draw_select(ad,blk,fg)
   APPLE_DATA ad;
   APPLE_GELO_COMPONENT blk;
   Boolean fg;
{
   Integer lx,by,rx,ty;

   ITRACE("APPLE_draw_select 0x%x 0x%x %d",ad,blk,fg);

   if (blk == NULL) return;
   if (ad->draw_win == NULL) return;

   ASHbatch_mode(TRUE);

   if (ad->mode == APPLE_MODE_ARC) {
      APPLE_arc_box(ad,blk,&lx,&by,&rx,&ty);
    }
   else {
      APPLE_ash_coords(ad,blk->lx,blk->by,&lx,&by);
      APPLE_ash_coords(ad,blk->rx,blk->ty,&rx,&ty);
    };

   ASHhilite_box(ad->draw_win,lx,by,rx,ty);

   APPLE_draw_usearea(ad);

   ASHbatch_mode(FALSE);
};




/************************************************************************/
/*									*/
/*	APPLE_redraw_component -- redraw single component		*/
/*									*/
/************************************************************************/


void
APPLE_redraw_component(ad,blk)
   APPLE_DATA ad;
   APPLE_GELO_COMPONENT blk;
{
   Integer lx,by,rx,ty;

   ITRACE("APPLE_redraw_component 0x%x 0x%x",ad,blk);

   if (blk == NULL || ad->mode == APPLE_MODE_ARC) {
      APPLE_draw_picture(ad);
      return;
    };

   if (ad->draw_win == NULL) return;

   ASHbatch_mode(TRUE);

   APPLE_ash_coords(ad,blk->lx,blk->by,&lx,&by);
   APPLE_ash_coords(ad,blk->rx,blk->ty,&rx,&ty);

   ASHclear_box(ad->draw_win,lx+1,by+1,rx-1,ty-1);

   draw_object(blk->object,ad,lx,by,rx,ty);

   if (blk == ad->selection) APPLE_draw_select(ad,blk,TRUE);

   ASHbatch_mode(FALSE);
};





/************************************************************************/
/*									*/
/*	APPLE_arc_box -- return coords of box for arc			*/
/*									*/
/************************************************************************/


void
APPLE_arc_box(ad,sel,lxp,byp,rxp,typ)
   APPLE_DATA ad;
   APPLE_GELO_COMPONENT sel;
   Integer *lxp, *byp, *rxp, *typ;
{
   Integer xi,yi;
   Integer lx,by,rx,ty;

   ITRACE("APPLE_arc_box 0x%x %d",ad,sel);

   ASHinq_size(ad->draw_win,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   lx += 1;
   rx -= 1;
   ty -= 1;
   by += 1;

   xi = (rx-lx)/10;
   yi = (ty-by)/6;

   if (sel == SELECT_FROM) {
      *lxp = lx+xi*1;
      *byp = by+yi*1;
      *rxp = lx+xi*3;
      *typ = by+yi*4;
    }
   else if (sel == SELECT_TO) {
      *lxp = lx+xi*7;
      *byp = by+yi*1;
      *rxp = lx+xi*9;
      *typ = by+yi*4;
    }
   else if (sel == SELECT_LABEL) {
      *lxp = lx+xi*4;
      *byp = by+yi*3;
      *rxp = lx+xi*6;
      *typ = by+yi*5;
    };
};




/************************************************************************/
/*									*/
/*	APPLE_draw_usearea -- draw use region				*/
/*									*/
/************************************************************************/


void
APPLE_draw_usearea(ad)
   APPLE_DATA ad;
{
   Integer lx,by,rx,ty,x,y;
   register APPLE_VALUE av;
   register Sequence l;
   Boolean fg;
   Character buf[1024],pbuf[64];
   register APPLE_GELO_ARC arc;
   register APPLE_GELO ag;
   Integer fx,tx;

   ITRACE("APPLE_draw_usearea 0x%x",ad);

   if (ad->use_window == NULL) return;

   fg = ASHinq_user_data(ad->use_window);
   if (fg != ad->use_select) {
      ASHinq_size(ad->use_window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
      ASHclear_box(ad->use_window,lx,by,rx,ty);
      sprintf(buf,USE_PREFIX);

      fx = tx = -1;

      if (ad->picture != NULL) {
	 forin (ag,APPLE_GELO,l,ad->picture->use) {
	    av = APPLE_layout_value(ag);
	    if (av->name != NULL) {
	       switch (ag->mode) {
		  default :
		  case APPLE_MODE_FIELD :
		     strcpy(pbuf,av->name);
		     break;
		  case APPLE_MODE_ARC :
		     arc = (APPLE_GELO_ARC) ag;
		     if (STREQL(arc->from.name,APPLE_SELF_NAME))
			sprintf(pbuf,"==>%s",arc->to.name);
		     else
			sprintf(pbuf,"<==%s",arc->from.name);
		     break;
		};
	       if (ad->use_select == USE_SELECT_ONEUSE && ad->usearc_select == ag) {
		  ASHinq_text_extent(ad->use_window,buf,&x,&y);
		  fx = x;
		  strcat(buf,pbuf);
		  ASHinq_text_next(ad->use_window,buf,&x,&y);
		  tx = x;
		}
	       else {
		  strcat(buf,pbuf);
		};
	       strcat(buf," ");
	     };
	  };
       };

      ASHinq_text_offset(ad->use_window,buf,&x,&y);
      ASHtext(ad->use_window,lx+2+x,by+1+y,buf);

      if (ad->use_select == USE_SELECT_USE) {
	 ASHinq_text_extent(ad->use_window,USE_PREFIX,&x,&y);
	 ASHhilite_box(ad->use_window,lx,by,lx+x,ty);
       }
      else if (ad->use_select == USE_SELECT_ONEUSE && fx >= 0) {
	 ASHhilite_box(ad->use_window,lx+fx,by,lx+tx,ty);
       };

      ASHset_user_data(ad->use_window,ad->use_select);
    };
};






/************************************************************************/
/*									*/
/*	draw_object -- draw a GELO object view				*/
/*									*/
/************************************************************************/


static void
draw_object(obj,ad,lx,by,rx,ty)
   APPLE_GELO obj;
   APPLE_DATA ad;
   Integer lx,by,rx,ty;
{
   DTRACE("draw_object 0x%x 0x%x %d %d %d %d",obj,ad,lx,by,rx,ty);

   ASHbox(ad->draw_win,lx,by,rx,ty);

   lx += 1;
   rx -= 1;
   ty -= 1;
   by += 1;

   ASHclear_box(ad->draw_win,lx,by,rx,ty);

   if (obj != NULL) {
      switch (obj->mode) {
	 case APPLE_MODE_LAYOUT :
	    draw_layout(obj,ad,lx,by,rx,ty);
	    break;
	 case APPLE_MODE_TILED :
	    draw_tiled(obj,ad,lx,by,rx,ty);
	    break;
	 case APPLE_MODE_BOX :
	    draw_box(obj,ad,lx,by,rx,ty);
	    break;
	 case APPLE_MODE_FIELD :
	    draw_field_object(obj,ad,lx,by,rx,ty);
	    break;
	 case APPLE_MODE_POINTER :
	    draw_pointer_object(obj,ad,lx,by,rx,ty);
	    break;
	 case APPLE_MODE_ARC :
	    draw_arc(obj,ad,lx,by,rx,ty);
	    break;
	 case APPLE_MODE_IGNORE :
	 default :
	    draw_unknown(obj,ad,lx,by,rx,ty);
	    break;
       };
    }
   else {
      draw_unknown(NULL,ad,lx,by,rx,ty);
    };
};





/************************************************************************/
/*									*/
/*	draw_layout -- draw for layout object				*/
/*									*/
/************************************************************************/


static void
draw_layout(lay,ad,lx,by,rx,ty)
   APPLE_GELO_LAYOUT lay;
   APPLE_DATA ad;
   Integer lx,by,rx,ty;
{
   Integer x0,y0,x1,y1,x2,y2,y3;
   register APPLE_GELO_FIELD fld;
   register Sequence l;
   Integer dx;
   Character buf[64];

   DTRACE("draw_layout 0x%x 0x%x %d %d %d %d",lay,ad,lx,by,rx,ty);

   x0 = abs(rx-lx)/4;
   x1 = 10;
   if (x0 > 10) x0 = (abs(rx-lx)-20)/2;
   else x1 = x0;

   y0 = abs(by-ty)/4;
   y1 = 10;
   if (y0 > 10) y0 = (abs(by-ty)-30);
   else y1 = y0;

   x2 = MIN(x1,y1);
   x2 = x2/2-1;

   dx = PORTAL_SIZE;
   if (abs(rx-lx) < 3*PORTAL_SIZE || abs(ty-by) < 3*PORTAL_SIZE) dx = -1;

   ASHbox(ad->draw_win,lx+x1,ty-y1-y1,lx+x1+x0,ty-y1);

   ASHline(ad->draw_win,lx+x1+x1/2,by+y1,lx+x1+x1/2,by+y1+y0);
   ASHline(ad->draw_win,lx+x1+x1/2,by+y1,lx+x1+x1/2-x2,by+y1+x2);
   ASHline(ad->draw_win,lx+x1+x1/2,by+y1,lx+x1+x1/2+x2,by+y1+x2);

   ASHline(ad->draw_win,lx+x1+x0,ty-y1-y1/2,rx-x1,ty-y1-y1/2);
   ASHline(ad->draw_win,rx-x1,ty-y1-y1/2,rx-x1-x2,ty-y1-y1/2+x2);
   ASHline(ad->draw_win,rx-x1,ty-y1-y1/2,rx-x1-x2,ty-y1-y1/2-x2);

   if (dx >= 0) {
      ASHbox(ad->draw_win,lx+1,ty-1,lx+dx,ty-dx);
      ASHbox(ad->draw_win,lx+1,(by+ty+dx)/2,lx+dx,(by+ty-dx)/2);
      ASHbox(ad->draw_win,lx+1,by+1,lx+dx,by+dx);
      ASHbox(ad->draw_win,rx-1,ty-1,rx-dx,ty-dx);
      ASHbox(ad->draw_win,rx-1,(by+ty+dx)/2,rx-dx,(by+ty-dx)/2);
      ASHbox(ad->draw_win,rx-1,by+1,rx-dx,by+dx);
      ASHbox(ad->draw_win,(lx+rx-dx)/2,ty-1,(lx+rx+dx)/2,ty-dx);
      ASHbox(ad->draw_win,(lx+rx-dx)/2,by+1,(lx+rx+dx)/2,by+dx);
    };

   x2 = lx+x1+x0/2+2;
   y2 = ty-y1-y1-2;

   forin (fld,APPLE_GELO_FIELD,l,lay->components) {
      y3 = y2-15;
      if (y3 > by && fld->value.name != NULL) {
	 if (fld->fport == GELO_PORT_CENTER) {
	    strcpy(buf,fld->value.name);
	  }
	 else {
	    sprintf(buf,"*%s*",fld->value.name);
	  };
	 output_text(ad,buf,x2,y3,rx,y2);
	 y2 = y3-1;
       };
    };
};





/************************************************************************/
/*									*/
/*	draw_arc -- draw for an arc object				*/
/*									*/
/************************************************************************/


static void
draw_arc(arc,ad,lx,by,rx,ty)
   APPLE_GELO_ARC arc;
   APPLE_DATA ad;
   Integer lx,by,rx,ty;
{
   Integer xi,yi,x1,y1;

   DTRACE("draw_arc 0x%x 0x%x %d %d %d %d",arc,ad,lx,by,rx,ty);

   xi = (rx-lx)/10;
   yi = (ty-by)/6;

   ASHbox(ad->draw_win,lx+xi*1,by+yi*1,lx+xi*3,by+yi*4);
   if (arc->from.name != NULL) {
      output_text(ad,arc->from.name,lx+xi*1,by+yi*1,lx+xi*3,by+yi*4);
    };

   ASHbox(ad->draw_win,lx+xi*7,by+yi*1,lx+xi*9,by+yi*4);
   if (arc->to.name != NULL) {
      output_text(ad,arc->to.name,lx+xi*7,by+yi*1,lx+xi*9,by+yi*4);
    };

   ASHbox(ad->draw_win,lx+xi*4,by+yi*3,lx+xi*6,by+yi*5);
   if (arc->label.name != NULL) {
      output_text(ad,arc->label.name,lx+xi*4,by+yi*3,lx+xi*6,by+yi*5);
    };

   x1 = lx+xi*7;
   y1 = by+2*yi+yi/2;
   ASHline(ad->draw_win,lx+xi*3,y1,x1,y1);
   ASHline(ad->draw_win,x1-5,y1+5,x1,y1);
   ASHline(ad->draw_win,x1-5,y1-5,x1,y1);
};





/************************************************************************/
/*									*/
/*	draw_tiled -- draw for tiled object				*/
/*									*/
/************************************************************************/


static void
draw_tiled(til,ad,lx,by,rx,ty)
   APPLE_GELO_TILED til;
   APPLE_DATA ad;
   Integer lx,by,rx,ty;
{
   register APPLE_GELO_COMPONENT c;
   register Sequence l;
   Integer nlx,nby,nrx,nty;

   DTRACE("draw_tiled 0x%x 0x%x %d %d %d %d",til,ad,lx,by,rx,ty);

   forin (c,APPLE_GELO_COMPONENT,l,til->components) {
      APPLE_ash_coords(ad,c->lx,c->by,&nlx,&nby);
      APPLE_ash_coords(ad,c->rx,c->ty,&nrx,&nty);
      draw_object(c->object,ad,nlx,nby,nrx,nty);
    };
};







/************************************************************************/
/*									*/
/*	draw_box -- draw for box object 				*/
/*									*/
/************************************************************************/


static void
draw_box(box,ad,lx,by,rx,ty)
   APPLE_GELO_BOX box;
   APPLE_DATA ad;
   Integer lx,by,rx,ty;
{
   register Integer fl,ct;
   Integer x[10],y[10];
   Integer r;

   DTRACE("draw_box 0x%x 0x%x %d %d %d %d",box,ad,lx,by,rx,ty);

   if (box->fill != 0) fl = ASHfill(ad->draw_win,box->fill);
   else fl = ASHfill(ad->draw_win,ASH_FILL_SOLID);

   ct = 0;

   switch (box->shape) {
      case GELO_SHAPE_RECTANGLE :
      default :
	 if (box->fill != 0)
	    ASHrectangle(ad->draw_win,lx,by,rx,ty);
	 else
	    ASHbox(ad->draw_win,lx,by,rx,ty);
	 break;
      case GELO_SHAPE_ROUND_RECTANGLE :
	 if (abs(rx-lx) > abs(ty-by)) r = abs(ty-by)/4;
	 else r = abs(rx-lx)/4;
	 if (box->fill != 0)
	    ASHround_rectangle(ad->draw_win,lx,by,rx,ty,r);
	 else
	    ASHround_box(ad->draw_win,lx,by,rx,ty,r);
	 break;
      case GELO_SHAPE_CIRCLE :
      case GELO_SHAPE_ELLIPSE :
	 if (box->fill != 0)
	    ASHfilled_ellipse(ad->draw_win,(lx+rx)/2,(ty+by)/2,(rx-lx)/2,(ty-by)/2);
	 else
	    ASHellipse(ad->draw_win,(lx+rx)/2,(ty+by)/2,(rx-lx)/2,(ty-by)/2);
	 break;
      case GELO_SHAPE_TRIANGLE_UP :
      case GELO_SHAPE_SEMI_UP :
	 x[0] = (lx+rx)/2; y[0] = ty;
	 x[1] = lx; y[1] = by;
	 x[2] = rx; y[2] = by;
	 ct = 3;
	 break;
      case GELO_SHAPE_TRIANGLE_DOWN :
      case GELO_SHAPE_SEMI_DOWN :
	 x[0] = (lx+rx)/2; y[0] = by;
	 x[1] = lx; y[1] = ty;
	 x[2] = rx; y[2] = ty;
	 ct = 3;
	 break;
      case GELO_SHAPE_TRIANGLE_LEFT :
      case GELO_SHAPE_SEMI_LEFT :
	 x[0] = lx; y[0] = (by+ty)/2;
	 x[1] = rx; y[1] = ty;
	 x[2] = rx; y[2] = by;
	 ct = 3;
	 break;
      case GELO_SHAPE_TRIANGLE_RIGHT :
      case GELO_SHAPE_SEMI_RIGHT :
	 x[0] = rx; y[0] = (by+ty)/2;
	 x[1] = lx; y[1] = by;
	 x[2] = lx; y[2] = ty;
	 ct = 3;
	 break;
      case GELO_SHAPE_DIAMOND :
	 x[0] = (lx+rx)/2; y[0] = ty;
	 x[1] = lx; y[1] = (by+ty)/2;
	 x[2] = (lx+rx)/2; y[2] = by;
	 x[3] = rx; y[3] = (by+ty)/2;
	 ct = 4;
	 break;
      case GELO_SHAPE_PENTAGON :
	 x[0] = (lx+rx)/2; y[0] = ty;
	 x[1] = lx; y[1] = (by+ty)/2;
	 x[2] = (lx+lx+rx)/3; y[2] = by;
	 x[3] = (lx+rx+rx)/3; y[3] = by;
	 x[4] = rx; y[4] = (by+ty)/2;
	 ct = 5;
	 break;
      case GELO_SHAPE_HEXAGON :
	 x[0] = (lx+lx+rx)/3; y[0] = ty;
	 x[1] = lx; y[1] = (ty+by)/2;
	 x[2] = (lx+lx+rx)/3; y[2] = by;
	 x[3] = (lx+rx+rx)/3; y[3] = by;
	 x[4] = rx; y[4] = (ty+by)/2;
	 x[5] = (lx+rx+rx)/3; y[5] = ty;
	 ct = 6;
	 break;
      case GELO_SHAPE_OCTAGON :
	 x[0] = (lx+lx+rx)/3; y[0] = ty;
	 x[1] = lx; y[1] = (ty+ty+by)/3;
	 x[2] = lx; y[2] = (ty+by+by)/3;
	 x[3] = (lx+lx+rx)/3; y[3] = by;
	 x[4] = (lx+rx+rx)/3; y[4] = by;
	 x[5] = rx; y[5] = (ty+by+by)/3;
	 x[6] = rx; y[6] = (ty+ty+by)/3;
	 x[7] = (lx+rx+rx)/3; y[7] = ty;
	 ct = 8;
	 break;
      case GELO_SHAPE_VLINE :
	 if (box->fill != 0)
	    ASHrectangle(ad->draw_win,lx+2,by,lx+4,ty);
	 else
	    ASHbox(ad->draw_win,lx+2,by,lx+4,ty);
	 break;
     };

   if (ct != 0) {
      if (box->fill != 0) {
	 ASHpolygon(ad->draw_win,ct,x,y);
       }
      else {
	 x[ct] = x[0];
	 y[ct] = y[0];
	 ASHpolyline(ad->draw_win,ct+1,x,y);
       };
    };

   ASHfill(ad->draw_win,fl);

   if (box->value.name != NULL) {
      output_text(ad,box->value.name,lx,by,rx,ty);
    };
};





/************************************************************************/
/*									*/
/*	draw_unknown -- draw for unknown object 			*/
/*									*/
/************************************************************************/


static void
draw_unknown(box,ad,lx,by,rx,ty)
   APPLE_GELO_BOX box;
   APPLE_DATA ad;
   Integer lx,by,rx,ty;
{
   register String s;

   DTRACE("draw_unknown 0x%x 0x%x %d %d %d %d",box,ad,lx,by,rx,ty);

   if (box != NULL && box->mode == APPLE_MODE_IGNORE) s = "<NO PICTURE>";
   else s = "?";

   ASHcenter_text(ad->draw_win,s,lx,by,rx,ty);
};





/************************************************************************/
/*									*/
/*	draw_field_object -- draw field-only object			*/
/*									*/
/************************************************************************/


static void
draw_field_object(fld,ad,lx,by,rx,ty)
   APPLE_GELO_FIELD fld;
   APPLE_DATA ad;
   Integer lx,by,rx,ty;
{
   Character buf[256];

   DTRACE("draw_field_object 0x%x 0x%x %d %d %d %d",fld,ad,lx,by,rx,ty);

   if (fld->match == NULL) {
      sprintf(buf,"<%s>",fld->value.name);
    }
   else {
      sprintf(buf,"<%s:%s>",fld->match,fld->value.name);
    };

   output_text(ad,buf,lx,by,rx,ty);
   ASHbox(ad->draw_win,lx,by,rx,ty);
};





/************************************************************************/
/*									*/
/*	draw_pointer_object -- draw pointer-to-field object		*/
/*									*/
/************************************************************************/


static void
draw_pointer_object(arc,ad,lx,by,rx,ty)
   APPLE_GELO_ARC arc;
   APPLE_DATA ad;
   Integer lx,by,rx,ty;
{
   Character buf[256];

   DTRACE("draw_pointer_object 0x%x 0x%x %d %d %d %d",arc,ad,lx,by,rx,ty);

   if (arc->match == NULL) {
      sprintf(buf,"==> <%s>",arc->to.name);
    }
   else {
      sprintf(buf,"==> <%s:%s>",arc->match,arc->to.name);
    };

   output_text(ad,buf,lx,by,rx,ty);
   ASHbox(ad->draw_win,lx,by,rx,ty);
};





/************************************************************************/
/*									*/
/*	output_text -- output text block				*/
/*									*/
/************************************************************************/


static void
output_text(ad,s,lx,by,rx,ty)
   APPLE_DATA ad;
   String s;
   Integer lx,by,rx,ty;
{
   Integer x,y,t;

   DTRACE("output_text 0x%x %s %d %d %d %d",ad,s,lx,by,rx,ty);

   ASHinq_text_extent(ad->draw_win,s,&x,&y);
   x += 2;
   y += 2;

   if (rx > lx+x) {
      t = lx+rx;
      lx = (t-x)/2;
      rx = (t+x+1)/2;
    };

   if (ty > by+y) {
      t = ty+by;
      by = (t-y)/2;
      ty = (t+y+1)/2;
    };

   ASHinq_text_offset(ad->draw_win,s,&x,&y);

   ASHclear_box(ad->draw_win,lx,by,rx,ty);

   ASHclip_region(ad->draw_win,lx,by,rx,ty);
   ASHclip(ad->draw_win,TRUE);
   ASHtext(ad->draw_win,lx+x,by+y,s);
   ASHclip(ad->draw_win,FALSE);
};





/* end of appledraw.c */




