/*
 * Input line editing for atty.
 *
 * Copyright (C) 1989 by Kenneth Almquist.  All rights reserved.
 * This file is part of atty, which is distributed under the terms specified
 * by the Atty General Public License.  See the file named LICENSE.
 */

#include <stdio.h>
#include "attyed.h"
#include "ed.h"
#include "bind.h"
#include "regex.h"
#include <ctype.h>
#include <signal.h>


#define HISTFILLER -1		/* indicates blank record in history file */
#define HISTSIZE 8192		/* size of history; MUST be a power of 2 */
#define LOCMASK (HISTSIZE - 1)	/* mask to get offset in history area */

/* flags to histadvance */
#define SKIPEMPTY 01		/* skip empty history lines */
#define SAMEPRGM 02		/* skip history lines read by other programs */
#define HSEARCH 04		/* skip history lines not matching pattern */


struct histhdr {
      short len;		/* length of history line */
      short prevlen;		/* length of preceding line */
      long prgmid;		/* identifies program */
      /* text of line goes here */
};


struct histline {
      short len;		/* length of history line */
      short prevlen;		/* length of preceding line */
      long prgmid;		/* identifies program */
      char *text;		/* text of line */
};


char curline[MAXLLEN+1];	/* line being edited */
char *point;			/* location of point */
char *mark;			/* location of mark */
char *endcurline;		/* end of line */
char histarea[HISTSIZE];	/* history file */
long endhist;			/* end of history file */
long curhist;			/* location in history file */
char lasthist[MAXLLEN];		/* line after end of history file */
int lasthistlen;		/* length of lasthist */
int histprevlen;		/* length of last entry in history file */
char killbuffer[MAXLLEN];	/* killed text goes here */
int killsize;			/* length of killed text */
int prefixarg;			/* if set, evaluating prefix arg */
int prefixval;			/* value of prefix argument */
int prefixsign;			/* sign of prefix argument */
char cmdcharacter;		/* character that invoked this command */
unsigned char *curmode;		/* base key table for current mode */
unsigned char *curtable;	/* current command parsing table */
unsigned char *keytabs;		/* key tables */
int keyhash;			/* DEBUG */
char syntaxtbl[NSYNTAX][16];	/* syntax tables */
char *edstrings;		/* string table */
char *abbrevtbl;		/* abbreviation table */
extern int version;		/* version of .bindc format */
extern int (*edfunc[])();	/* table of editor functions */
int (*edrecurse)();		/* function to call when recursive edit done */
int (*ednextc)();		/* function to call on next character */
struct re_pattern_buffer pbuf;	/* buffer containing last search pattern */
int savedpattern;		/* true if current pattern set */
char savecurline[MAXLLEN];	/* saved copy of current line */
int savecurlinelen;		/* length of savecurline */
char *savepoint;		/* point saved for recursive edit */
char *savemark;			/* point saved for recursive edit */
int saveprefixval;		/* prefixval saved here during recursive edit */
char editprompt;		/* printed during recursive edit */


#ifdef __STDC__
int getsh(FILE *);
int runfmatch(int);
void zapline(void);
void delregion(char *, int);
int dosearch(int);
int extractword(char *, int, int);
int histadvance(int, int);
void addhist(void);
void loadhistline(long);
void gethist(long, struct histline *);
char *getenv(char *);
char *malloc(unsigned);
#else
int getsh();
int runfmatch();
void zapline();
void delregion();
int dosearch();
int extractword();
int histadvance();
void addhist();
void loadhistline();
void gethist();
char *getenv();
char *malloc();
#endif



void
edinit(file)
      char *file;
      {
      FILE *fp;
      int ktsize;
      int stringsize;
      int abbrevsize;
      char fname[256];
      char *home;
      extern char dftbind[];

      if (file == NULL) {
	    if ((home = getenv("HOME")) != NULL) {
		  strcpy(fname, home);
		  strcat(fname, "/.bindc");
		  if ((fp = fopen(fname, "r")) == NULL)
			file = dftbind;
	    } else {
		  file = dftbind;
	    }
      }
      if (file != NULL) {
	    if ((fp = fopen(file, "r")) == NULL) {
		  perror(file);
		  badinit("Can't read bindings file");
	    }
      }
      if (getsh(fp) != BINDMAGIC) {
	    badinit("Bad magic number on key binding file");
      }
      if (getc(fp) != version) {
	    badinit(".bindc file needs to be recompiled");
      }
      ktsize = getc(fp) << 8;
      stringsize = getsh(fp);
      abbrevsize = getsh(fp);
      if (feof(fp) || ferror(fp))
	    badinit("Key binding file truncated");
      if ((keytabs = (unsigned char *)malloc(ktsize)) == NULL)
	    badinit("Out of space");
      fread((char *)keytabs, ktsize, 1, fp);
      keyhash = genkeyhash();	/*DEBUG*/
      fread((char *)syntaxtbl, sizeof syntaxtbl, 1, fp);
      if (stringsize != 0) {
	    if ((edstrings = malloc(stringsize)) == NULL)
		  badinit("Out of space");
	    fread(edstrings, stringsize, 1, fp);
      }
      if (abbrevsize != 0) {
	    if ((abbrevtbl = malloc(abbrevsize + 1)) == NULL)
		  badinit("Out of space");
	    fread(abbrevtbl, abbrevsize, 1, fp);
	    abbrevtbl[abbrevsize] = '\0';	/* mark end of abbrevs */
      }
      if (feof(fp) || ferror(fp))
	    badinit("Key binding file truncated");
      fclose(fp);
      endcurline = curline;
      point = curline;
      curtable = curmode = keytabs;
      gettermcap();
}



/*
 * Read a short integer from a file.
 */

int
getsh(fp)
      FILE *fp;
      {
      int c;

      c = getc(fp);
      return c | getc(fp) << 8;
}


void
newttychars() {
      /* nothing to do */
}


void
edflush() {
      zapline();
      /* also reset state to start of command */
      curtable = keytabs;
      prefixarg = 0;
}


void
inchar(c)
      char c;
      {
      int cmd;
      int index;
      int status;
      int pfx;
      int (*func)();

      c &= 0177;		/* just in case */
      if (ednextc) {
	    func = ednextc;
	    ednextc = NULL;
	    if ((*func)(c) == 1)
		  needbeep++;
	    goto out;
      }
      if (prefixarg && c >= '0' && c <= '9') {
	    if (prefixarg >= 2)
		  prefixval = 0;
	    prefixarg = 1;
	    prefixval = prefixval * 10 + c - '0';
	    goto out;
      }
      cmd = curtable[c];
      index = curtable[c + 128];
      if (cmd == C_FUNC || cmd == C_INSERT) {
	    if (prefixarg == 0)
		  pfx = 1;
	    else if (prefixarg == 3)
		  pfx = -1;
	    else if (prefixsign)
		  pfx = - prefixval;
	    else
		  pfx = prefixval;
	    cmdcharacter = c;
	    if (cmd == C_FUNC)
		  status = (*edfunc[index])(pfx);
	    else
		  status = insert_string(index, pfx);
	    if (status == 1)
		  needbeep++;
	    if (status != 2)
		  prefixarg = 0;
	    curtable = curmode;
      } else if (cmd == C_PFXTBL) {
	    curtable = keytabs + index * 256;
      } else {
	    needbeep++;
	    curtable = curmode;
      }
out:;
}


void
zapline() {
      point = curline;
      endcurline = curline;
      mark = NULL;
}



universal_argument(n) {
      if (prefixarg == 0) {
	    prefixval = 1;
	    prefixsign = 0;
      }
      prefixarg = 2;
      prefixval *= 4;
      return 2;
}


digit_argument(n) {
      prefixarg = 1;
      prefixval = cmdcharacter - '0';
      if (prefixval > 9 || prefixval < 0)
	    prefixval = 0;
      prefixsign = 0;
      return 2;
}


negative_argument(n) {
      prefixarg = 3;
      prefixval = 0;
      prefixsign = 1;
      return 2;
}


mode_0(n) {
      curmode = keytabs;
      return 0;
}


mode_1(n) {
      curmode = keytabs + 256;
      return 0;
}


newline(n) {
      if (edrecurse)
	    return (*edrecurse)(0);
      if (point == endcurline && expandabbrev() < 0)
	    return 1;
      addhist();
      refresh();
      *endcurline++ = '\n';
      senddata(curline, endcurline - curline);
      movetoend();
      curhist = endhist;
      zapline();
      curmode = keytabs;
      return 0;
}


end_of_file(n) {
      if (edrecurse)
	    return (*edrecurse)(1);
      if (point == endcurline && expandabbrev() < 0)
	    return 1;
      addhist();
      senddata(curline, endcurline - curline);
      freezedisp(1);
      curhist = endhist;
      zapline();
      curmode = keytabs;
      return 0;
}


eof_or_delete_char(n) {
      if (endcurline == curline)
	    return end_of_file(n);
      else
	    return delete_char(n);
}


newline_and_insert(n) {
      if (edrecurse)
	    return 1;
      if (point == endcurline && expandabbrev() < 0)
	    return 1;
      if (curhist == endhist) {
	    lasthistlen = endcurline - curline;
	    bcopy(curline, lasthist, lasthistlen);
      }
      addhist();
      refresh();
      *endcurline++ = '\n';
      senddata(curline, endcurline - curline);
      movetoend();
      zapline();
      curmode = keytabs;
      if (curhist < endhist - HISTSIZE) {
	    curhist = endhist;
	    return 1;
      }
      return histadvance(1, 0);
}


file_complete(n) {
      return runfmatch('c');
}


list_file_completions(n) {
      return runfmatch('l');
}


runfmatch(type) {
      char saveline[MAXLLEN];
      register char *p;
      char *savemark;
      char *savepoint;
      int savelen;
      int status;

      p = point;
      while (p < endcurline && infname(*p))
	    p++;
      if (p == curline || ! infname(p[-1]))
	    return 1;
      savelen = endcurline - curline;
      bcopy(curline, saveline, savelen);
      savemark = mark;
      savepoint = p;
      if (promptset && type == 'l') {
	    refresh();
	    movetoend();
	    promptset = 1;		/* retain prompt */
      }
      point = p;
      delregion(endcurline, 0);
      while (p > curline && infname(p[-1]))
	    p--;
      point = curline;
      delregion(p, 0);
      if (makespace(10)) {
	    status = 1;
	    goto out;
      }
      bcopy("fmatch -x ", curline, 10);
      curline[8] = type;
      if (promptset) {
	    if (makespace(1)) {
		  status = 1;
		  goto out;
	    }
	    curline[0] = '\034';
      } else {
	    refresh();
	    movetoend();
      }
#ifdef notdef /* no longer add to history file */
      atend = (curhist == endhist);
      addhist();
      if (atend)
	    curhist = endhist;
#endif
      *endcurline++ = '\n';
      senddata(curline, endcurline - curline);
      status = 0;
out:
      endcurline = curline + savelen;
      bcopy(saveline, curline, savelen);
      mark = savemark;
      point = savepoint;
      return status;
}

      
tty_intr(n) {
      if (edrecurse)
	    return (*edrecurse)(1);
      sendsig(SIGINT);
      return 0;
}


tty_quit(n) {
      sendsig(SIGQUIT);
      return 0;
}


tty_susp(n) {
      sendsig(SIGTSTP);
      return 0;
}


do_quoted_insert(c) {
      cmdcharacter = c;
      return self_insert(prefixval);
}


quoted_insert(n) {
      ednextc = do_quoted_insert;
      prefixval = n;
}


/*
 * Delete the characters between the point and "end".  If "kill" is true,
 * copy to kill buffer.  Exception:  we don't copy to kill if the region
 * is empty.
 */

void
delregion(end, kill)
      char *end;
      {
      register char *p, *q;
      register char *k;

      if (end > endcurline)
	    end = endcurline;
      if (end < curline)
	    end = curline;
      if (end > point) {
	    p = point;
	    q = end;
      } else {
	    p = end;
	    q = point;
      }
      if (kill && p != q) {
	    end = p;
	    k = killbuffer;
	    while (p < q)
		  *k++ = *p++;
	    killsize = k - killbuffer;
	    p = end;
      }
      point = p;
      if (mark && mark > p) {
	    if (mark > q)
		  mark -= q - p;
	    else
		  mark = p;
      }
      while (q < endcurline)
	    *p++ = *q++;
      endcurline = p;
}


delete_backward_char(n) {
      delregion(point - n, prefixarg);
}


delete_char(n) {
      delregion(point + n, prefixarg);
}


kill_word(n) {
      char *save = point;

      forward_word(n);
      delregion(save, 1);
      return 0;
}


backward_kill_word(n) {
      char *save = point;

      backward_word(n);
      delregion(save, 1);
      return 0;
}


kill_region(n) {
      if (mark == NULL)
	    return 1;
      delregion(mark, 1);
      return 0;
}


kill_line(n) {
      delregion(n <= 0? curline : endcurline, 1);
      return 0;
}


kill_input(n) {
      point = curline;
      delregion(endcurline, 1);
      return 0;
}


copy_region_as_kill(n) {
      char *savepoint = point;
      char *savemark = mark;
      delregion(mark, 1);
      yank(1);
      point = savepoint;
      mark = savemark;
}


forward_char(n) {
      int left;
      left = endcurline - point;
      if (n > left)
	    n = left;
      left = curline - point;
      if (n < left)
	    n = left;
      point += n;
      return 0;
}


backward_char(n) {
      return forward_char(-n);
}


int
inword(c)
      char c;
      {
      c &= 0177;
      return syntaxtbl[0][c >> 3] & 1 << (c & 07);
}


int
infname(c)
      char c;
      {
      c &= 0177;
      return syntaxtbl[1][c >> 3] & 1 << (c & 07);
}


int
inabbrev(c)
      char c;
      {
      c &= 0177;
      return syntaxtbl[2][c >> 3] & 1 << (c & 07);
}


forward_word(n) {
      if (n < 0)
	    return backward_word(-n);
      while (--n >= 0) {
	    while (point < endcurline && ! inword(*point))
		  point++;
	    while (point < endcurline && inword(*point))
		  point++;
      }
      return 0;
}


backward_word(n) {
      if (n < 0)
	    return forward_word(-n);
      while (--n >= 0) {
	    while (--point >= curline && ! inword(*point));
	    point++;
	    while (--point >= curline && inword(*point));
	    point++;
      }
      return 0;
}


end_of_line(n) {
      point = endcurline;
      return 0;
}


beginning_of_line(n) {
      point = curline;
      return 0;
}



/*
 * Make space for n characters after the point.  Return 1 on failure.
 */

int
makespace(n) {
      char *p;

      if (endcurline - curline + n > MAXLLEN)
	    return 1;
      for (p = endcurline ; --p >= point ; )
	    *(p + n) = *p;
      endcurline += n;
      if (mark != NULL && mark > point)
	    mark += n;
      return 0;
}


void
insertchars(p, n)
      char *p;
      {
      if (makespace(n) != 0) {
	    needbeep++;
	    return;
      }
      bcopy(p, point, n);
      point += n;
}


self_insert(n) {
      if (abbrevtbl && ! inabbrev(cmdcharacter)) {
	    if (expandabbrev() < 0)
		  return 1;
      }
      while (--n >= 0) {
	    if (point == endcurline) {
		  if (endcurline == &curline[MAXLLEN])
			return 1;
		  endcurline++;
	    } else {
		  if (makespace(1) != 0)
			return 1;
	    }
	    *point++ = cmdcharacter;
      }
      return 0;
}


yank(n) {
      if (makespace(killsize) != 0)
	    return 1;
      bcopy(killbuffer, point, killsize);
      mark = point +  killsize;
      if (! prefixarg)
	    exchange_point_and_mark(0);
      return 0;
}


insert_string(stringnum, n) {
      char *p;
      int len;

      p = edstrings;
      while (--stringnum >= 0)
	    p = p + *p + 1;
      if (abbrevtbl && *p != 0 &&! inabbrev(p[1])) {
	    if (expandabbrev() < 0)
		  return 1;
      }
      while (--n >= 0) {
	    if (makespace(len = *p) != 0) {
		  return 1;
	    }
	    bcopy(p + 1, point, len);
	    point += len;
      }
      return 0;
}


set_mark(n) {
      mark = point;
      return 0;
}


exchange_point_and_mark(n) {
      char *p;

      if (mark == NULL)
	    return 1;
      p = mark;
      mark = point;
      point = p;
      return 0;
}


/*
 * Convert characters between the point and the specified location to
 * upper case.
 */

void
upcase(end)
      char *end;
      {
      char *p;
      int len;

      if (end > point) {
	    p = point;
	    len = end - point;
      } else {
	    p = end;
	    len = point - end;
      }
      while (--len >= 0) {
	    if (islower(*p))
		  *p = toupper(*p);
	    p++;
      }
}


upcase_char(n) {
      char *save = point;

      forward_char(n);
      upcase(save);
      return 0;
}


upcase_word(n) {
      char *save = point;

      forward_word(n);
      upcase(save);
      return 0;
}


upcase_region(n) {
      upcase(mark);
      return 0;
}


gosling_transpose_chars(n) {
      char c;

      if (prefixarg)
	    return transpose_chars(n);
      if (point < curline + 2)
	    return 1;
      c = point[-1];
      point[-1] = point[-2];
      point[-2] = c;
      return 0;
}


transpose_chars(n) {
      char c;

      if (point == endcurline && point >= curline + 2 && ! prefixarg)
	    point--;
      if (point <= curline || n > 0 && point >= endcurline)
	    return 1;
      c = *(point - 1);
      delregion(point - 1, 0);
      forward_char(n);
      makespace(1);
      *point++ = c;
      return 0;
}


transpose_words(n) {
      char *save = point;
      char *start;
      char *loc;
      int end1, start2, end2;
      char buf[MAXLLEN];

      if (point == endcurline)
	    backward_word(1);
      loc = point;
      backward_word(1);
      start = point;
      if (start >= loc || ! inword(*start))
	    goto fail;
      forward_word(1);
      if (point > loc)
	    point = loc;
      end1 = point - start;
      forward_word(1);
      end2 = point - start;
      backward_word(1);
      if (point < loc) {
	    point = loc;
	    if (! inword(*loc))
		  goto fail;
      }
      start2 = point - start;
      bcopy(start, buf, end2);
      point = start + start2;
      delregion(start + end2, 0);
      point = start;
      delregion(start + end1, 0);
      makespace(end2 - start2);
      bcopy(buf + start2, start, end2 - start2);
      point = start + end2 - end1;
      makespace(end1);
      bcopy(buf, point, end1);
      point += end1;
      if (save == endcurline)
	    point = save;
      return 0;

fail:
      point = save;
      return 1;
}


re_search_backward(n) {
      return re_search_forward(-n);
}


re_search_forward(n) {
      if (edrecurse)
	    return 1;
      edrecurse = dosearch;
      savecurlinelen = endcurline - curline;
      bcopy(curline, savecurline, savecurlinelen);
      savepoint = point;
      savemark = mark;
      saveprefixval = n;
      zapline();
      editprompt = cmdcharacter? cmdcharacter : '/';
      curmode = keytabs;
      return 0;
}


dosearch(flag) {
      edrecurse = NULL;
      editprompt = '\0';
      if (flag)
	    goto fail;
      if (endcurline == curline) {
	    if (! savedpattern)
		  goto fail;
      } else if (re_compile_pattern(curline, endcurline - curline, &pbuf)) {
	    savedpattern = 0;
	    goto fail;
      }
      savedpattern = 1;
      bcopy(savecurline, curline, savecurlinelen);
      endcurline = curline + savecurlinelen;
      point = savepoint;
      mark = savemark;
      return histadvance(saveprefixval, HSEARCH);

fail:
      bcopy(savecurline, curline, savecurlinelen);
      endcurline = curline + savecurlinelen;
      point = savepoint;
      mark = savemark;
      return 1;
}


get_history_word(n) {
      struct histline hline;

      if (endhist == 0)
	    return 1;
      gethist(endhist - histprevlen - sizeof (struct histhdr), &hline);
      if (! prefixarg)
	    n = -1;
      return extractword(hline.text, hline.len, n);
}


last_output_line(n) {
      if (! prefixarg)
	    n = 0;
      return extractword(lastoutline, lastoutlinelen, n);
}


/*
 * Insert the contents of word n of the line.  If n is zero, insert the
 * entire line; if it is negative, count words from the end.  In this
 * routine, words are always delimited by spaces and tabs.
 */

int
extractword(line, len, n)
      char *line;
      {
      char *p;
      char *q;
      char *endline = line + len;
      int wordlen;

      if (n > 0) {
	    p = line;
	    for (;;) {
		  while (p < endline && ! infname(*p))
			p++;
		  q = p;
		  while (q < endline && infname(*q))
			q++;
		  if (--n == 0)
			break;
		  p = q;
	    }
      } else if (n < 0) {
	    q = endline;
	    for (;;) {
		  while (q > line && ! infname(q[-1]))
			q--;
		  p = q;
		  while (p > line && infname(p[-1]))
			p--;
		  if (++n == 0)
			break;
		  q = p;
	    }
      } else { /* n == 0 */
	    p = line;
	    q = endline;
      }
      wordlen = q - p;
      if (makespace(wordlen) != 0)
	    return 1;
      bcopy(p, point, wordlen);
      mark = point;
      point += wordlen;
      return 0;
}


next_history(n) {
      return histadvance(n, SKIPEMPTY);
}


previous_history(n) {
      return histadvance(-n, SKIPEMPTY);
}


end_of_history(n) {
      loadhistline(endhist);
      return 0;
}


beginning_of_history(n) {
      histadvance(-32767, 0);
      return 0;
}


/*
 * Move through the history file, and load the result into curline.
 * N is the number of items to move:  a negative value of n moves
 * backward in the history file and a postive value of n moves forward.
 * The flags control which history lines are counted.  The search starts
 * from the location specified by the curhist variable.
 *
 * The structure of the history file is as follows.  Each line in the
 * history file consists of a header, defined as (struct histhdr), followed
 * by the text of the line.  Locations in the history file are identified
 * by the number of bytes written since the file was first created.  To
 * keep the file from growing without bound, the file wraps around at
 * HISTSIZE bytes.  Thus any location less than (endhist - HISTSIZE) has
 * been overwritten.  (Endhist points to the end of the history file.)
 */

int
histadvance(n, flags) {
      struct histline hline;
      long loc;
      long found;
      int back;
      int oldloc;
      int prgmid = 1;

      if (edrecurse)
	    return 1;
      loc = curhist;
      if (loc == endhist) {
	    lasthistlen = endcurline - curline;
	    bcopy(curline, lasthist, lasthistlen);
      }
      back = 0;
      if (n < 0) {
	    n = -n;
	    back = 1;
      }
      found = loc;
      while (n >= 0) {
	    oldloc = loc;
	    if (back) {
		  if (loc == endhist) {
			loc -= sizeof (struct histhdr) + histprevlen;
		  } else if (loc < 0 || loc < endhist - HISTSIZE) {
			break;
		  } else {
			gethist(loc, &hline);
			loc -= sizeof (struct histhdr) + hline.prevlen;
		  }
	    } else {
		  if (loc == endhist) {
			loc++;
		  } else if (loc > endhist) {
			break;
		  } else {
			gethist(loc, &hline);
			loc += sizeof (struct histhdr) + hline.len;
		  }
	    }
	    if (oldloc == endhist) {
		  hline.prgmid = prgmid;
		  hline.text = lasthist;
		  hline.len = lasthistlen;
	    }
	    if (hline.prgmid != HISTFILLER
	     && ((flags & SKIPEMPTY) == 0 || oldloc == endhist || hline.len > 0)
	     && ((flags & SAMEPRGM) == 0 || hline.prgmid == prgmid)
	     && ((flags & HSEARCH) == 0 || oldloc == curhist ||
			re_search(&pbuf, hline.text, hline.len, 0, hline.len,
				  (struct re_registers *)0) >= 0)) {
		  n--;
		  found = oldloc;
	    }
      }
      if (found != curhist || n < 0)
	    loadhistline(found);
      return n >= 0? 1 : 0;
}


/*
 * Add the current line to the history file.  If echoing is turned off,
 * we store a blank line rather than the actual line so that we don't
 * have passwords in the history file.
 */

void
addhist() {
      struct histhdr hdr;
      int space;
      int len = ttymode.echo? endcurline - curline : 0;
      char *lp;

      hdr.prevlen = histprevlen;
      space = HISTSIZE - (endhist & LOCMASK);
      if (space < len + 2 * sizeof (struct histhdr)) {
	    hdr.len = space - sizeof (struct histhdr);
	    hdr.prgmid = HISTFILLER;
	    bcopy((char *)&hdr, histarea + (endhist & LOCMASK), sizeof hdr);
	    hdr.prevlen = hdr.len;
	    endhist += space;
      }
      lp = histarea + (endhist & LOCMASK);
      hdr.len = len;
      hdr.prgmid = 0;		/* for now */
      bcopy((char *)&hdr, lp, sizeof hdr);
      bcopy(curline, lp + sizeof hdr, len);
      histprevlen = len;
      endhist += sizeof hdr + len;
}


/*
 * Load the contents of a history line into curline.
 */

void
loadhistline(loc)
      long loc;
      {
      struct histline hline;

      if (loc == endhist) {
	    if (curhist == endhist) {
		  return;
	    }
	    hline.text = lasthist;
	    hline.len = lasthistlen;
      } else {
	    gethist(loc, &hline);
      }
      bcopy(hline.text, curline, hline.len);
      endcurline = curline + hline.len;
      point = endcurline;
      mark = NULL;
      curhist = loc;
}


/*
 * Read a history entry into hline.  To make the history file be a real
 * file rather than a block of memory, change this routine and the addhist
 * routine.
 */

void
gethist(loc, hline)
      long loc;
      struct histline *hline;
      {
      char *lp = histarea + (loc & LOCMASK);

      bcopy(lp, (char *)hline, sizeof (struct histhdr));
      hline->text = lp + sizeof (struct histhdr);
}


/*
 * Expand the abbreviation before point, if any.  Returns 1 if expansion
 * occurred, 0 if no expansion occurred, and -1 if the expansion could
 * not be performed because there was not sufficient space.
 */

int
expandabbrev() {
      char *start;
      int len;
      char *p;

      if (abbrevtbl == NULL)
	    return 0;
      start = point;
      while (--start >= curline && inabbrev(*start));
      start++;
      if (start == point)
	    return 0;
      len = point - start;
      p = abbrevtbl;
      for (;;) {
	    if (*p == 0)
		  return 0;		/* end of table */
	    if (*p == len && bcmp(p + 1, start, len) == 0)
		  break;
	    p += *p + 1;
	    p += *p + 1;
      }
      p += *p + 1;
      if (endcurline - curline + *p - len > MAXLLEN)
	    return -1;
      delregion(start, 0);
      if (makespace(*p))
	    return -1;			/* can't happen */
      bcopy(p + 1, point, *p);
      point += *p;
      return 1;
}


/* DEBUG */

int
genkeyhash() {
      int hash;
      unsigned char *p;

      hash = 123456; p = keytabs + 256;
      do { hash = ((hash << 8) + *--p) % 8380087; } while (p >= keytabs);
      return hash;
}


chkkeytab() {
      int hash;
      unsigned char *p;

      hash = 123456; p = keytabs + 256;
      do { hash = ((hash << 8) + *--p) % 8380087; } while (p >= keytabs);
      if (hash != keyhash)
	    abort();
}

/* END DEBUG */
