/* ************************************************************************* *
 * PostScript Interpretor                   Fabien LELAQUAIS                 *
 *                                                                           *
 *   Fichier path.c                                                          *
 *      Path creation and manipulation routines for PSint.                   *
 *                           Version 3.00 on 24/02/89                        *
 * ************************************************************************* *
 *    This document may be distributed, used, or modified, but can NOT be    *
 *  sold nor incorporated in any way in any product.                         *
 *    Permission is granted to distribute modified versions of that software *
 *  under the condition that this notice remains in every source file.       *
 *    Every alteration of the original files should be marked as such.       *
 *    No warranty is assumed by the author on the concequencies of the use   *
 *  of this software. Any defection of this program is at your own risk,     *
 *  you have to assume the cost of any service, installation or repairs      *
 *  this program could generate.                                             *
 *                                                                           *
 *                          Fabien LELAQUAIS - ESIEE - lelaquaf@apo.esiee.fr *
 * ************************************************************************* */
#include "int.h"
#include "graph.h"

#if 0
#include <sgtty.h> /* pour l'echo */
#endif /* 0 */

double sqrt();
#define distance(a, b) sqrt((a.x-b.x)*(a.x-b.x)+(a.y-b.y)*(a.y-b.y))

ps__path free_elements = NULL;

/* ************************************************************************* *
 *        +---+                                                              *
 * path : | 1 |                                                              *
 *        +---+               +---+      +---+                               *
 *          |        returns  | 2 |  and | 1 | is free.                      *
 *          V                 +---+      +---+                               *
 *        +---+                                                              *
 *        | 2 |                                                              *
 *        +---+                                                              *
 * ************************************************************************* */
ps__path
free_element(ps__path path)
{
  ps__path subpath;

  subpath       = path->next;
  path->next    = free_elements;
  free_elements = path;
  return subpath;
  }

/* ************************************************************************* */
void
free_path(ps__path path)
{
  while (path) path = free_element(path);
  }

/* ************************************************************************* *
 *        +---+               +---+                                          *
 * path : | 1 |               | 1 |                                          *
 *        +---+               +---+                                          *
 *          |                   |                                            *
 *          V                   V                                            *
 *                            +---+                                          *
 *          ?        returns  | 2 |  of given graphic type.                  *
 *                            +---+                                          *
 * ************************************************************************* */
ps__path
new_element(ps__path  path, ps_g_type type)
{
  ps__path new;

  if (free_elements) {
    new = free_elements;
    free_elements = new->next;
    }
  else {
    if (memtrace) ps__printf(">>> new_element : ");
    if (!(new = (ps__path)ps_alloc(sizeof(ps__path_element)))) {
      ps__puts("!!! new_element ran out of memory !!!");
      return NULL;
      }
    }
  new->type = type;
  new->next = NULL;
  if (path) path->next = new;
  return new;
  }

/* ************************************************************************* */
ps__path
copy_path(ps__path path)
{
  ps__path otherpath = NULL, pathhead = NULL;

  while (path) {
    otherpath = new_element(otherpath, path->type);
    if (!pathhead) pathhead = otherpath;
    otherpath->element = path->element;
    path               = path->next;
    }
  return pathhead;
  }

/* ************************************************************************* */
void
add_move(ps__path elem, double x, double y)
{
  ps__coord coord;

  coord.x = x;
  coord.y = y;
  coord = user_to_device(state->CTM, coord);
  state->cp = elem->element.move.point = coord;
  if (!state->path) state->path = elem;
  state->pqueue = elem;
  }

/* ************************************************************************* */
void
add_line(ps__path elem, double x, double y)
{
  ps__coord coord;

  coord.x = x;
  coord.y = y;
  coord = user_to_device(state->CTM, coord);
  state->cp = elem->element.line.point = coord;
  state->pqueue = elem;
  }

/* ************************************************************************* */
void
add_curve(ps__path elem, double x1, double y1,
                         double x2, double y2,
                         double x3, double y3)
{
  ps__coord coord;

  coord.x = x1;
  coord.y = y1;
  coord = user_to_device(state->CTM, coord);
  elem->element.curve.point1 = coord;
  coord.x = x2;
  coord.y = y2;
  coord = user_to_device(state->CTM, coord);
  elem->element.curve.point2 = coord;
  coord.x = x3;
  coord.y = y3;
  coord = user_to_device(state->CTM, coord);
  state->cp = elem->element.curve.point3 = coord;
  state->pqueue = elem;
  }

/* ************************************************************************* *
 *    Creates and returns a new path, composed with connected lines, which   *
 *   matches as close as `flatness' requires the given curve element.        *
 *    It returns the head element of new created path (only lines)           *
 * ************************************************************************* */
ps__path
flatten_element(ps__coord origin, ps__path  curve)
{
  double    flatness = to_real(state->flatness), c, s, angle,
            fabs(), sin(), cos(), atan2();
  ps__path  line, line1, line2, othercurve;
  ps__coord otherorigin, p1, p2, p3;

  p1 = curve->element.curve.point1;
  p2 = curve->element.curve.point2;
  p3 = curve->element.curve.point3;

  if (((p3.y==origin.y)&&(p3.x==origin.x)) ||
      ((fabs((s = sin(angle = atan2(p3.y-origin.y, p3.x-origin.x)))*(origin.x-p1.x)
           + (c = cos(angle                                      ))*(p1.y-origin.y))
         <= flatness) &&
       (fabs(s*(origin.x-p2.x) + c*(p2.y-origin.y)) <= flatness))) {
    curve->type               = ps_gt_line;
    curve->element.line.point = p3;
    curve->next = NULL;
    return curve;
    }
  othercurve = new_element((ps__path)NULL, ps_gt_curve);
  otherorigin.x = (origin.x + p3.x + 3*(p1.x+p2.x))/8;
  otherorigin.y = (origin.y + p3.y + 3*(p1.y+p2.y))/8;
  othercurve->element.curve.point1.x = (p1.x+p3.x)/4 + p2.x/2;
  othercurve->element.curve.point1.y = (p1.y+p3.y)/4 + p2.y/2;
  othercurve->element.curve.point2.x = (p2.x+p3.x)/2;
  othercurve->element.curve.point2.y = (p2.y+p3.y)/2;
  othercurve->element.curve.point3   = p3;
  curve->element.curve.point1.x = (origin.x+p1.x)/2;
  curve->element.curve.point1.y = (origin.y+p1.y)/2;
  curve->element.curve.point2.x = (origin.x+p2.x)/4 + p1.x/2;
  curve->element.curve.point2.y = (origin.y+p2.y)/4 + p1.y/2;
                                                     /* Curves are jointives */
  curve->element.curve.point3   = otherorigin;
  line1 = flatten_element(     origin,      curve);
  line2 = flatten_element(otherorigin, othercurve);
                                                                     /* Link */
  for (line = line1; line->next; line = line->next);
  line->next = line2;
  return line1;
  }

/* ************************************************************************* *
 *     Destroy all curves contained in given path, creating an new, 'flat',  *
 *   as defined by flatness factor in state, brain nice path.                *
 *     The older one is destroyed.                                           *
 *     One given curve is subdivided in two until given flatness is reached. *
 * ************************************************************************* */
ps__path
flatten(ps__path path)
{
  ps__path  head, previous = NULL, rest;
  ps__coord oldcp, newcp, lastmove;

  oldcp = state->cp;
  for (head=path; path; ) {
    if (path->type == ps_gt_curve) {
      rest           = path->next;
      previous->next = flatten_element(newcp, path);
      path           = rest;
                      /* Get end of created lines to link with previous path */
      for (rest=previous->next; rest->next; rest=rest->next);
      rest->next = path;
      previous   = rest;
      newcp = rest->element.line.point;
      }
    else {
      newcp = (path->type == ps_gt_close)?lastmove:path->element.move.point;
      if (path->type == ps_gt_move) lastmove = newcp;
      previous = path;
      path     = path->next;
      }
    }
  state->cp = oldcp;
  return head;
  }

/* ************************************************************************* *
 *       Returns a new path defined by line segments according to dash       *
 * description. The given path is supposed to be flat.                       *
 *       The older path doesn't longer exist.                                *
 * ************************************************************************* */
void
resolve_dash()
{
  ps__path  path, nextelem;
  ps__coord current, lastmove;
  int       count = 0, initialcount;
  double    offset, dashdist, initialdashdist, devicedist, linelength,
            angle, s, sin(), c, cos(), fabs(), atan2(), det;
  short     drawing = 1, initialdrawing;

  if (!state->dasharray.size ||
      !(det = fabs(mat(state->CTM, 0)*mat(state->CTM, 3) -
                   mat(state->CTM, 1)*mat(state->CTM, 2)))) return;
  dashdist = mat(state->dasharray, 0);
  offset = to_real(state->dashoffset);
  while (offset > dashdist) {
    offset -= dashdist;
    if (++count == state->dasharray.size) count = 0;
    dashdist = mat(state->dasharray, count);
    drawing = 1-drawing;
    }
  dashdist -= offset;
  initialdashdist = dashdist;
  initialcount    = count;
  initialdrawing  = drawing;
  for (path=state->path; path; path = path->next) {
    switch (path->type) {
      case ps_gt_none : break;
      case ps_gt_move :
        dashdist  = initialdashdist;
        count     = initialcount;
        drawing   = initialdrawing;
        current = lastmove = path->element.move.point;
        break;
      default : /* line or close */
        nextelem = path->next;
        if (linelength = (path->type == ps_gt_line)?
                              distance(current, path->element.line.point):
                              distance(current, lastmove)) {
          s = sin(angle = atan2((path->type == ps_gt_line)?
                                 (path->element.line.point.y-current.y):
                                 (lastmove.y-current.y),
                            (path->type == ps_gt_line)?
                                 (path->element.line.point.x-current.x):
                                 (lastmove.x-current.x)));
          c = cos(angle);
                                                /* Lengths are in user space */
          linelength *= sqrt(mat(state->CTM,3)*mat(state->CTM,1)+
                             mat(state->CTM,3)*mat(state->CTM,3))/det;
          if (!drawing) path->type = ps_gt_move;
          while (linelength > dashdist) {
            linelength -= dashdist;
            devicedist = dashdist*sqrt(mat(state->CTM,0)*mat(state->CTM,0)+
                                       mat(state->CTM,1)*mat(state->CTM,1));
            current.x += devicedist*c;
            current.y += devicedist*s;
            path->element.move.point.x = current.x;
            path->element.move.point.y = current.y;
            drawing = 1-drawing;
            path = new_element(path, drawing?ps_gt_line:ps_gt_move);
            if (++count == state->dasharray.size) count = 0;
            dashdist = mat(state->dasharray, count);
            }
          dashdist -= linelength;
          current.x += linelength*c;
          current.y += linelength*s;
          path->element.move.point.x = current.x;
          path->element.move.point.y = current.y;
          path->next = nextelem;
          }
      }
    }
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__newpath()
{
  free_path(state->path);
  state->path   = NULL;
  state->pqueue = NULL;
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__object pathforall_object;
ps__path   path_forall;
ps__errors
ps__pathforall(ps__object move,  ps__object line,
               ps__object curve, ps__object close)
{
  if (!HAS_X(move) || !HAS_X(line) || !HAS_X(curve) || !HAS_X(close)) 
    return ps_e_invalidaccess;
  path_forall = copy_path(state->path);
  PUSH(execstack, invalid_object);
  PUSH(execstack, close);
  PUSH(execstack, curve);
  PUSH(execstack, line);
  PUSH(execstack, move);
  return PUSH(execstack, pathforall_object);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps___pathforall()
{
  ps__object move, line, curve, close;
  ps__coord  coord;
  int        possible;

  move  = POP(execstack);
  line  = POP(execstack);
  curve = POP(execstack);
  close = POP(execstack);
  if (path_forall) {
    PUSH(execstack, close);
    PUSH(execstack, curve);
    PUSH(execstack, line);
    PUSH(execstack, move);
    PUSH(execstack, pathforall_object);
    switch (path_forall->type) {
      case ps_gt_move  :
        coord = device_to_user(path_forall->element.move.point, state->CTM,
                                                               &possible);
        if (!possible) return ps_e_undefinedresult;
        PUSH(opstack, new_real(coord.x));
        PUSH(opstack, new_real(coord.y));
        PUSH(execstack, move);
        break;
      case ps_gt_line  :
        coord=device_to_user(path_forall->element.line.point,state->CTM,NULL);
        PUSH(opstack, new_real(coord.x));
        PUSH(opstack, new_real(coord.y));
        PUSH(execstack, line);
        break;
      case ps_gt_curve :
        coord=device_to_user(path_forall->element.curve.point1,state->CTM,NULL);
        PUSH(opstack, new_real(coord.x));
        PUSH(opstack, new_real(coord.y));
        coord=device_to_user(path_forall->element.curve.point2,state->CTM,NULL);
        PUSH(opstack, new_real(coord.x));
        PUSH(opstack, new_real(coord.y));
        coord=device_to_user(path_forall->element.curve.point3,state->CTM,NULL);
        PUSH(opstack, new_real(coord.x));
        PUSH(opstack, new_real(coord.y));
        PUSH(execstack, curve);
        break;
      case ps_gt_close :
        PUSH(execstack, close);
        break;
      default :
        ps__puts("** Invalid graphic type in pathforall !! **");
      }
    path_forall = free_element(path_forall);
    }
  else {
    if ((move.flags & DYNAMIC) && !REFS(move))
      ps_destroy_object(move, "pathforall (move)");
    if ((line.flags & DYNAMIC) && !REFS(line))
      ps_destroy_object(line, "pathforall (line)");
    if ((curve.flags & DYNAMIC) && !REFS(move))
      ps_destroy_object(curve, "pathforall (curve)");
    if ((close.flags & DYNAMIC) && !REFS(close))
      ps_destroy_object(close, "pathforall (close)");
    }
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__closepath()
{
  if (state->path && (state->pqueue->type != ps_gt_close))
    state->pqueue = new_element(state->pqueue, ps_gt_close);
  return ps_e_operationok;  
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__flattenpath()
{
  state->path = flatten(state->path);
  if (!state->path) state->pqueue = NULL;
  else
    for (state->pqueue = state->path; state->pqueue->next;
         state->pqueue = state->pqueue->next);
  return ps_e_operationok;  
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__moveto(ps__object x, ps__object y)
{
  ps__path new;

  if (!state->path || (state->pqueue && (state->pqueue->type != ps_gt_move)))
    new = new_element(state->pqueue, ps_gt_move);
  else
    new = state->pqueue;
  add_move(new, to_real(x), to_real(y));
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__lineto(ps__object x, ps__object y)
{
  ps__path new;

  if (!state->path) return ps_e_nocurrentpoint;
  new = new_element(state->pqueue, ps_gt_line);
  add_line(new, to_real(x), to_real(y));
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__curveto(ps__object x1, ps__object y1,
            ps__object x2, ps__object y2,
            ps__object x3, ps__object y3)
{
  ps__path new;

  if (!state->path) return ps_e_nocurrentpoint;
  new = new_element(state->pqueue, ps_gt_curve);
  add_curve(new, to_real(x1), to_real(y1), to_real(x2), to_real(y2),
                                           to_real(x3), to_real(y3));
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__rmoveto(ps__object dx, ps__object dy)
{
  ps__coord coord;
  ps__path  new;

  if (!state->path) return ps_e_nocurrentpoint;
  if (state->pqueue && (state->pqueue->type != ps_gt_move))
    new = new_element(state->pqueue, ps_gt_move);
  else
    new = state->pqueue;
  coord.x = to_real(dx);
  coord.y = to_real(dy);
  coord = duser_to_device(state->CTM, coord);
  coord.x += state->cp.x;
  coord.y += state->cp.y;
  coord = device_to_user(coord, state->CTM, NULL);
  add_move(new, coord.x, coord.y);
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__rlineto(ps__object dx, ps__object dy)
{
  ps__coord coord;
  ps__path  new;

  if (!state->path) return ps_e_nocurrentpoint;
  new = new_element(state->pqueue, ps_gt_line);
  coord.x = to_real(dx);
  coord.y = to_real(dy);
  coord = duser_to_device(state->CTM, coord);
  coord.x += state->cp.x;
  coord.y += state->cp.y;
  coord = device_to_user(coord, state->CTM, NULL);
  add_line(new, coord.x, coord.y);
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__rcurveto(ps__object dx1, ps__object dy1,
             ps__object dx2, ps__object dy2,
             ps__object dx3, ps__object dy3)
{
  ps__coord c1, c2, c3;
  ps__path new;

  if (!state->path) return ps_e_nocurrentpoint;
  new = new_element(state->pqueue, ps_gt_curve);
  c1.x = to_real(dx1);  c1.y = to_real(dy1);
  c1 = duser_to_device(state->CTM, c1);
  c1.x += state->cp.x; c1.y += state->cp.y;
  c1 = device_to_user(c1, state->CTM, NULL);
  c2.x = to_real(dx2);  c2.y = to_real(dy2);
  c2 = duser_to_device(state->CTM, c2);
  c2.x += state->cp.x; c2.y += state->cp.y;
  c2 = device_to_user(c2, state->CTM, NULL);
  c3.x = to_real(dx3);  c3.y = to_real(dy3);
  c3 = duser_to_device(state->CTM, c3);
  c3.x += state->cp.x; c3.y += state->cp.y;
  c3 = device_to_user(c3, state->CTM, NULL);
  add_curve(new, c1.x, c1.y,
                 c2.x, c2.y,
                 c3.x, c3.y);
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__arc(ps__object x0, ps__object y0,
        ps__object rad, ps__object ang1, ps__object ang2)
{
  ps__path    new;
  double      ai, at, x1, y1, x2, y2, r, cx, cy, k, sin(), cos();

  ai = to_real(ang1);
  at = to_real(ang2);
  r  = to_real(rad);
  cy = to_real(y0);
  cx = to_real(x0);
  while ((at-ai) < 0) at += 360;
  at = ((at-ai) >= 360)?360:(at-ai);
  ai  = degrad(ai);
                    /* Traitement du premier point */
  new = new_element(state->pqueue, (state->path)?ps_gt_line:ps_gt_move);
  x1 = cx+r*cos(ai); y1 = cy+r*sin(ai);
  if (!state->path) add_move(new, x1, y1);
  else              add_line(new, x1, y1);
                         /* Calcul de la courbe de Bezier correspondante */
  while (at > 0) { double ar;
    ar = (at > 90)?(_PI_/2):degrad(at);
    at -= 90;
    k = r*ar*.3514141143;
    x1 = r*cos(ar)+k*sin(ar); y1 = r*sin(ar)-k*cos(ar);
    x2 = r*cos(ar);           y2 = r*sin(ar);
    new = new_element(state->pqueue, ps_gt_curve);
    add_curve(new, cx+r*cos(ai)-k*sin(ai),   cy+k*cos(ai)+r*sin(ai),
                   cx+x1*cos(ai)-y1*sin(ai), cy+y1*cos(ai)+x1*sin(ai),
                   cx+x2*cos(ai)-y2*sin(ai), cy+y2*cos(ai)+x2*sin(ai));
    ai += _PI_/2;
    }
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__arcn(ps__object x0, ps__object y0,
         ps__object rad, ps__object ang1, ps__object ang2)
{
  ps__path    new;
  double      ai, at, x1, y1, x2, y2, r, cx, cy, k, sin(), cos();

  ai = to_real(ang1);
  at = to_real(ang2);
  r  = to_real(rad);
  cy = to_real(y0);
  cx = to_real(x0);
  while ((at-ai) > 0) at -= 360;
  at = ((at-ai) <= -360)?360:(ai-at);
  ai  = degrad(ai);
                    /* Traitement du premier point */
  new = new_element(state->pqueue, (state->path)?ps_gt_line:ps_gt_move);
  x1 = cx+r*cos(ai); y1 = cy+r*sin(ai);
  if (!state->path) add_move(new, x1, y1);
  else              add_line(new, x1, y1);
                         /* Calcul de la courbe de Bezier correspondante */
  while (at > 0) { double ar;
    ar = (at > 90)?(_PI_/2):degrad(at);
    at -= 90;
    k =  -r*ar*.3514141143;
    x1 = r*cos(ar)-k*sin(ar); y1 = -r*sin(ar)-k*cos(ar);
    x2 = r*cos(ar);           y2 = -r*sin(ar);
    new = new_element(state->pqueue, ps_gt_curve);
    add_curve(new, cx+r*cos(ai)-k*sin(ai),   cy+k*cos(ai)+r*sin(ai),
                   cx+x1*cos(ai)-y1*sin(ai), cy+y1*cos(ai)+x1*sin(ai),
                   cx+x2*cos(ai)-y2*sin(ai), cy+y2*cos(ai)+x2*sin(ai));
    ai -= _PI_/2;
    }
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__arcto(ps__object ox1, ps__object oy1,
          ps__object ox2, ps__object oy2, ps__object or)
{
  double    x0, y0, x1, y1, x2, y2, x3, y3, r, angle, l,
            sin(), cos(), atan2(), fabs();
  double    a0, a1;
  ps__coord cp, p1, p2;

  if (!state->path) return ps_e_nocurrentpoint;
  cp = device_to_user(state->cp, state->CTM, NULL);
  x0 = cp.x; x1 = to_real(ox1); x2 = to_real(ox2);
  y0 = cp.y; y1 = to_real(oy1); y2 = to_real(oy2);
  r  = to_real(or);
  if (((y2==y1)&&(x2==x1)) ||((y0==y1)&&(x0==x1)))
    return ps_e_undefinedresult;
  a0 = atan2(y2-y1, x2-x1); if (a0<0) a0 += 2*_PI_;
  a1 = atan2(y0-y1, x0-x1); if (a1<0) a1 += 2*_PI_;
  angle  = a0-a1;
  if (angle <  0)      angle += 2*_PI_;
  if (angle >= 2*_PI_) angle -= 2*_PI_;
  if (angle >= _PI_)   angle -= 2*_PI_;
  if ((angle == 0) || (angle == -_PI_)) return ps_e_undefinedresult;
  l = fabs(r / sin(angle/2));
  a0 = angle/2 + a1;
  x3 = l*cos(a0)+x1;
  y3 = l*sin(a0)+y1;
  a0 += _PI_;
  a1 = (angle + ((angle>0)?(-_PI_):_PI_))/2;
  p1.x = x3 + r*cos(a0-a1);
  p1.y = y3 + r*sin(a0-a1);
  p2.x = x3 + r*cos(a0+a1);
  p2.y = y3 + r*sin(a0+a1);
  PUSH(opstack, new_real(p2.x));  PUSH(opstack, new_real(p2.y));
  PUSH(opstack, new_real(p1.x));  PUSH(opstack, new_real(p1.y));
  PUSH(opstack, new_real(x3));    PUSH(opstack, new_real(y3));
  PUSH(opstack, or);
  PUSH(opstack, new_real(180*(a0-a1)/_PI_));
  PUSH(opstack, new_real(180*(a0+a1)/_PI_));
  return PUSH(execstack, cvx(do_name((angle<=0)?"arc":"arcn")));
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__reversepath()
{
  return ps_e_notimplemented;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__pathbbox()
{
  double    llx, lly, urx, ury;
  ps__coord p;
  ps__path  path = state->path;

  if (!path) return ps_e_nocurrentpoint;
  p = device_to_user(path->element.move.point, state->CTM, NULL);
  llx = urx = p.x;
  lly = ury = p.y;
  for (path = path->next; path; path = path->next) {
    switch (path->type) {
      case ps_gt_move :
      case ps_gt_line :
        p = device_to_user(path->element.move.point, state->CTM, NULL);
        llx = MIN(llx, p.x); lly = MIN(lly, p.y);
        urx = MAX(urx, p.x); ury = MAX(ury, p.y);
        break;
      case ps_gt_curve :
        p = device_to_user(path->element.curve.point1, state->CTM, NULL);
        llx = MIN(llx, p.x); lly = MIN(lly, p.y);
        urx = MAX(urx, p.x); ury = MAX(ury, p.y);
        p = device_to_user(path->element.curve.point2, state->CTM, NULL);
        llx = MIN(llx, p.x); lly = MIN(lly, p.y);
        urx = MAX(urx, p.x); ury = MAX(ury, p.y);
        p = device_to_user(path->element.curve.point3, state->CTM, NULL);
        llx = MIN(llx, p.x); lly = MIN(lly, p.y);
        urx = MAX(urx, p.x); ury = MAX(ury, p.y);
        break;
      }
    }
  PUSH(opstack, new_real(llx));
  PUSH(opstack, new_real(lly));
  PUSH(opstack, new_real(urx));
  return PUSH(opstack, new_real(ury));
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__erasepage()
{
  if (state->device) ps__Erase();
  return ps_e_operationok;  
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__copypage()
{
  ps__printf("COPYPAGE> ");
  ps__fflush(stdout);
  while (ps__getchar() != '\n') ;
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__showpage()
{
  extern ps__errors ps__initgraphics();

#if 0
  if (state->device) {  
    struct sgttyb s;
    int tty = open("/dev/tty", 0);
    char c;

    if (tty) {
      ioctl (tty, TIOCGETP, &s);
      s.sg_flags &= ~ECHO;
      ioctl(tty, TIOCSETP, &s);
      while (read(tty, &c, 1) != '\n') ;
      s.sg_flags |= ECHO;
      ioctl(tty, TIOCSETP, &s);
      }
    }
#else
  ps__printf("SHOWPAGE> ");
  ps__fflush(stdout);
  while (ps__getchar() != '\n') ;
  ps__erasepage();
  ps__initgraphics();
#endif /* 0 (better SHOWPAGE) */
  return ps_e_operationok;  
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__gsave()
{
  extern unsigned short save_level;
  extern void           new_state();
  ps__state             oldstate = state;
  short                 loop;

  new_state();
  state->gray         = oldstate->gray;
  state->font         = oldstate->font;         MORE_REFS(state->font);
  state->linewidth    = oldstate->linewidth;
  state->linecap      = oldstate->linecap;
  state->linejoin     = oldstate->linejoin;
  state->flatness     = oldstate->flatness;
  state->miterlimit   = oldstate->miterlimit;
  state->dasharray    = oldstate->dasharray;    MORE_REFS(state->dasharray);
  state->dashoffset   = oldstate->dashoffset;
  state->frameproc    = oldstate->frameproc;    MORE_REFS(state->frameproc);
  state->transferproc = oldstate->transferproc; MORE_REFS(state->transferproc);
  state->device       = oldstate->device;
  state->color        = oldstate->color;
  state->cp           = oldstate->cp;
  state->next         = oldstate;
  state->CTM          = new_array(6);           REFS(state->CTM)++;
  for (loop=0; loop<6; loop++)
    arr_ob(state->CTM, loop) = arr_ob(oldstate->CTM, loop);
  if (state->path = copy_path(oldstate->path))
    for (state->pqueue=state->path; state->pqueue->next;
         state->pqueue = state->pqueue->next);
  else state->pqueue = NULL;
  state->clip         = copy_path(oldstate->clip);
  state->default_clip = copy_path(oldstate->default_clip);
  state->show = NULL;
  state->save_level = save_level;
  return ps_e_operationok;  
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__grestore()
{
  extern void           free_show(), ps_free();
  extern unsigned short save_level;
  ps__state             s;

  if (state->next && (state->save_level >= save_level)) {
    s = state;
    free_path(s->path);
    free_path(s->clip);
    free_path(s->default_clip);
    free_show(s->show);
    CHECK_DESTROY(s->CTM,          "grestore-CTM");
    CHECK_DESTROY(s->font,         "grestore-font");
    CHECK_DESTROY(s->dasharray,    "grestore-dasharray");
    CHECK_DESTROY(s->frameproc,    "grestore-frameproc");
    CHECK_DESTROY(s->transferproc, "grestore-transferproc");
    state = s->next;
    ps_free(s);
    }
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__grestoreall()
{
  extern unsigned short save_level;
  while (state->next && (state->next->save_level >= save_level))
    ps__grestore();
  return ps_e_operationok;
  }

extern ps__object tempCTM;
/* ************************************************************************* */
void
init_path_stuff()
{
  extern void init_fill_stuff();
  extern ps__errors ps__stroke(), ps__strokepath();
  int         loop;

  ps__newpath();
  new_operator("newpath",     0, 0, NONE,                ps__newpath);
  new_operator("pathforall",  4, 0, "a.a.a.a",           ps__pathforall);
  new_operator("moveto",      2, 0, "ir.ir",             ps__moveto);
  new_operator("lineto",      2, 0, "ir.ir",             ps__lineto);
  new_operator("curveto",     6, 0, "ir.ir.ir.ir.ir.ir", ps__curveto);
  new_operator("rmoveto",     2, 0, "ir.ir",             ps__rmoveto);
  new_operator("rlineto",     2, 0, "ir.ir",             ps__rlineto);
  new_operator("rcurveto",    6, 0, "ir.ir.ir.ir.ir.ir", ps__rcurveto);
  new_operator("arc",         5, 0, "ir.ir.ir.ir.ir",    ps__arc);
  new_operator("arcn",        5, 0, "ir.ir.ir.ir.ir",    ps__arcn);
  new_operator("arcto",       5, 4, "ir.ir.ir.ir.ir",    ps__arcto);

  new_operator("closepath",   0, 0, NONE,                ps__closepath);
  new_operator("pathbbox",    0, 4, NONE,                ps__pathbbox);
  new_operator("flattenpath", 0, 0, NONE,                ps__flattenpath);
                     /* Non operationnel */
  new_operator("reversepath", 0, 0, NONE,                ps__reversepath);

  new_operator("stroke",      0, 0, NONE,                ps__stroke);
  new_operator("strokepath",  0, 0, NONE,                ps__strokepath);
  new_operator("erasepage",   0, 0, NONE,                ps__erasepage);
  new_operator("showpage",    0, 0, NONE,                ps__showpage);
  new_operator("copypage",    0, 0, NONE,                ps__copypage);

  new_operator("gsave",       0, 0, NONE,                ps__gsave);
  new_operator("grestore",    0, 0, NONE,                ps__grestore);
  new_operator("grestoreall", 0, 0, NONE,                ps__grestoreall);

  pathforall_object = new_operator(" pathforall", 0,6, NONE, ps___pathforall);

  tempCTM = new_array(6);
  for (loop=0; loop<6; loop++)
    arr_ob(tempCTM, loop) = new_real(0.0);
  }

/* ************************************************************************* */
void
free_path_stuff()
{
  extern void ps_free();
  ps__path    element;

  ps_free(arr_val(tempCTM));
  while (element = free_elements) {
    free_elements = element->next;
    ps_free(element);
    }
  }
