/* ************************************************************************* *
 * PostScript Interpretor                   Fabien LELAQUAIS                 *
 *                                                                           *
 *   Fichier objects.c                                                       *
 *    This contains all routines for objects manipulation and creation.      *
 *   It includes stack management routines, dynamic objects allocations and  *
 * desallocation.                                                            *
 *                           Version 3.50 on 07/02/90                        *
 * ************************************************************************* *
 *    This document may be distributed, used, or modified, but can NOT be    *
 *  sold nor incorporated in any way in any product.                         *
 *    Permission is granted to distribute modified versions of that software *
 *  under the condition that this notice remains in every source file.       *
 *    Every alteration of the original files should be marked as such.       *
 *    No warranty is assumed by the author on the concequencies of the use   *
 *  of this software. Any defection of this program is at your own risk,     *
 *  you have to assume the cost of any service, installation or repairs      *
 *  this program could generate.                                             *
 *                                                                           *
 *                          Fabien LELAQUAIS - ESIEE - lelaquaf@apo.esiee.fr *
 * ************************************************************************* */
#include "int.h"

/* -------------------------***********************------------------------- */
ps__errors
ps___push(ps__stack  *stack, ps__object object)
{
  if (stack->size == stack->maxsize)
    return stack->errov;
  stack->stack[stack->size++] = object;
  MORE_REFS(object);
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__object
ps___pop(ps__stack *stack)
{
  ps__object    object;

  object = stack->stack[--stack->size];
  if (object.flags & DYNAMIC) REFS(object)--;
  return object;
  }

/* ************************************************************************* */
stack_find(ps__object object)
{
  int loop = opstack->size;

  for (; loop && (object.type != opstack->stack[loop-1].type); loop--);
  return loop-1;
  }

/* ************************************************************************* */
ps__object
new_object(ps__type type)
{
  ps__object result;
  result.type      = type;
  result.flags     = READ|WRITE;
  result.size      = 0;
  result.header    = NULL;
  real_val(result) = 0.0;
  return result;
  }

/* ************************************************************************* */
ps__object
new_real(double value)
{
  ps__object result;
  result = new_object(ps_t_real); 
  real_val(result) = value;
  return result;
  }

/* ************************************************************************* */
ps__object
new_int(long value)
{
  ps__object result;
  result = new_object(ps_t_int);
  int_val(result) = value;
  return result;
  }

/* ************************************************************************* */
ps__object
new_boolean(long value)
{
  ps__object result;

  result = new_object(ps_t_bool);
  bool_val(result) = value;
  return result;
  }

/* ************************************************************************* */
ps__object
new_operator(char *name,
             unsigned short argsin, unsigned short argsout, char *argstype,
             ps__errors   (*routine)())
{
  ps__object operator;

  if (argsin > MAX_OP_ARGUMENTS) {
    operator = new_object(ps_t_invalid);
    fprintf(stderr, "`%s' operator has too many arguments...\n", name);
    }
  else {
    operator        = cvx(new_object(ps_t_operator));
    if (memtrace) ps__printf(">>> new_operator : ");
    operator.value.opval = ps_alloc(sizeof(ps__operator));
    block_h(operator) = HEADER(operator.value.opval);
    op_val(operator).argsin   = argsin;
    op_val(operator).argsout  = argsout;
    op_val(operator).argstype = argstype;
    op_val(operator).operator = routine;
    op_val(operator).name     = name;
    dict_store(systemdict, do_name(name), operator);
    }
  return operator;
  }

/* ************************************************************************* */
ps__object
new_array(long size)
{
  extern unsigned short save_level;
  ps__object            result;
  ps__array            *objects;

  result        = new_object(ps_t_array);
  result.flags |= DYNAMIC;
  result.size   = size;
  if (memtrace) ps__printf(">>> new_array (size=%d) : ", size);
  result.size = size;
  objects = arr_val(result) = ps_alloc(size*sizeof(ps__array));
  block_h(result) = HEADER(objects);
  for (; size; size--, objects++) {
    objects->value      = null_object;
    objects->save_level = save_level;
    objects->next       = NULL;
    }
  return result;
  }

/* ************************************************************************* */
ps__object
do_string(char *value)
{
  ps__object result;
  result = new_object(ps_t_stg);
  stg_val(result) = (unsigned char *)value;
  result.size = strlen(value);
  return result;
  }

/* ************************************************************************* */
ps__object
new_string(char *value, unsigned long length)
{
  ps__object result;
  unsigned   char *p;

  result        = new_object(ps_t_stg);
  result.flags |= DYNAMIC;
  if (memtrace) {
    int l = length;
    ps__printf(">>> new_string (");
    for (; l; l--) ps__putchar(value[length-l]);
    ps__printf(") : ");
    }
  p = stg_val(result) = ps_alloc(length);
  block_h(result) = HEADER(p);
  result.size = length;
  for (; length; length--, *p++ = *value++);
  return result;
  }

/* ************************************************************************* */
ps__object
do_name(char *value)
{
 ps__object result;
 result = new_object(ps_t_name);
 stg_val(result) = (unsigned char *)value;
 result.size = strlen(value);
 return result;
 }

/* ************************************************************************* */
ps__object
new_name(char *value, unsigned long length)
{
  ps__object result;
  unsigned   char *p;

  result        = new_object(ps_t_name);
  result.flags |= DYNAMIC;
  if (memtrace) {
    int l = length;
    ps__printf(">>> new_name (");
    for (; l; l--) ps__putchar(value[length-l]);
    ps__printf(") : ");
    }
  p = name_val(result) = ps_alloc(length);
  block_h(result) = HEADER(p);
  result.size = length;
  for (; length; length--, *p++ = *value++);
  return result;
  }

/* ************************************************************************* */
ps__object
new_dict(long size)
{
  extern unsigned short save_level;
  ps__object            result;
  ps__key              *keys;
  int                   loop;

  result        = new_object(ps_t_dict);
  result.flags |= DYNAMIC;
  if (memtrace) ps__printf(">>> new_dict (size=%d) : ", size);
  result.value.dictval = ps_alloc(sizeof(ps__dict));
  result.size = size;
  dict_val(result).size        = 0;      /* Dictionnary is originally empty  */
  keys = dict_val(result).keys = ps_alloc(sizeof(ps__key)*size);
  block_h(result) = HEADER(keys);
  for (loop=0; loop<size; loop++) {
    keys[loop].save_level = save_level;
    keys[loop].name       = null_object; /* As Null is not a valid key entry */
    keys[loop].object     = null_object;
    keys[loop].next       = NULL;
    }
  return result;
  }

/* ************************************************************************* */
ps__object
new_save(char *save)
{
  ps__object result;
  result = new_object(ps_t_save);
  save_val(result) = save;
  return result;
  }

/* ************************************************************************* */
ps__object cvro(ps__object ob)  { CVRO(ob); return ob; }

/* ************************************************************************* */
ps__object cvwo(ps__object ob)  { CVWO(ob); return ob; }

/* ************************************************************************* */
ps__object cvxo(ps__object ob)  { CVXO(ob); return ob; }

/* ************************************************************************* */
ps__object cvx(ps__object ob)   { CVX(ob); return ob; }

/* ************************************************************************* */
ps__object cvlit(ps__object ob) { CVLIT(ob); return ob; }

/* ************************************************************************* */
char *types[] = {
  "invalid",  "null",  "int",  "real",  "bool",  "stg",  "file",  "mark",
  "array",  "operator",  "name",  "dict",  "fontId",  "save"  };
void
ps_destroy_object(ps__object object, char *message)
{
  if (memtrace) {
    ps__printf("-- Destroy_object ");
    switch (object.type) {
      case ps_t_stg      : ps__printf("-stg-"); break;
      case ps_t_name     : ps__printf("-name-"); break;
      case ps_t_operator : ps__printf("-operator-"); break;
      case ps_t_array    : ps__printf("-array-"); break;
      case ps_t_dict     : ps__printf("-dict-"); break;
      default            : ps__printf("type %d", object.type);
      }
    ps__printf(" (%s) refs : %d --\n",
               message,
               (object.flags & DYNAMIC)?REFS(object):99999);
    }
  if ((object.flags & DYNAMIC) && !REFS(object)) {
    int loop;

    switch (object.type) {
      case ps_t_array :
        for (loop=0; loop<object.size; loop++)
          if ((arr_ob(object, loop).flags & DYNAMIC) &&
             !(--REFS(arr_ob(object, loop))))
            ps_destroy_object(arr_ob(object, loop), "rec-array");
      case ps_t_stg      :;
      case ps_t_name     :
        ps_free(BLOCK(object));
        break;        
      case ps_t_dict :
        for (loop=0; loop<dict_val(object).size; loop++) {
          CHECK_DESTROY(dict_val(object).keys[loop].name,   "rec-dict-name");
          if ((dict_val(object).keys[loop].object.type          != ps_t_dict) ||
              (dict_val(object).keys[loop].object.value.dictval
                                        != object.value.dictval))
            CHECK_DESTROY(dict_val(object).keys[loop].object,
                          "rec-dict-object");
          }
        ps_free(BLOCK(object));
        ps_free(object.value.dictval);
        break;
      default : ps__printf("** Error ?? destroy_object %s **\n",
                           types[object.type]);
      }
    }
  }
