/* ************************************************************************* *
 * PostScript Interpretor                   Fabien LELAQUAIS                 *
 *                                                                           *
 *   Fichier memory.c                                                        *
 *      Memory management routines for PSint.                                *
 *                           Version 3.50 on 07/02/90                        *
 * ************************************************************************* *
 *    This document may be distributed, used, or modified, but can NOT be    *
 *  sold nor incorporated in any way in any product.                         *
 *    Permission is granted to distribute modified versions of that software *
 *  under the condition that this notice remains in every source file.       *
 *    Every alteration of the original files should be marked as such.       *
 *    No warranty is assumed by the author on the concequencies of the use   *
 *  of this software. Any defection of this program is at your own risk,     *
 *  you have to assume the cost of any service, installation or repairs      *
 *  this program could generate.                                             *
 *                                                                           *
 *                          Fabien LELAQUAIS - ESIEE - lelaquaf@apo.esiee.fr *
 * ************************************************************************* */
#if AMIGA
#include <exec/types.h>
#include <exec/memory.h>
#define MALLOC(s)  (char *)AllocMem(s, MEMF_PUBLIC)
#define FREE(a, s) FreeMem(a, s)
#else
extern char *malloc();
#ifdef PS_INTERNAL_ALLOC
#define MALLOC ps_internal_alloc
#define FREE   ps_internal_free
#else
#define MALLOC     malloc
#define FREE(p, s) free(p /* s */)
#endif
#endif /* AMIGA */

#include "int.h"

#define MAX_ALLOCATABLE_SIZE  ((1<<24)-1)

#ifdef MEMDEBUG
char memtrace = MEMDEBUG;
#endif

/*
#define CAN_GET_MAX_MEMORY
*/

/* If you wish to use our own allocation routines : */
/*
#define PS_INTERNAL_ALLOC
*/

#if AMIGA
#define ALIGN_SIZE 2                               /* Get sizes word-aligned */
#else
#define ALIGN_SIZE 4                               /* Get sizes long-aligned */
#endif
#define ALIGN_OFFS (ALIGN_SIZE-1)
#define ALIGN(s)   ((s + ALIGN_OFFS) & -ALIGN_SIZE)
ps_size_t allocated_memory = 0;
#if CAN_GET_MAX_MEMORY
ps_size_t maximum_memory   = 0;
#endif /* CAN_GET_MAX_MEMORY */

#if PS_INTERNAL_ALLOC
#define CHUNKSIZE 65536
static char *alloc_base   = NULL,
            *alloc_bot    = NULL,
            *alloc_top    = NULL,
            *alloc_limit  = NULL,
            *malloc_chain = NULL;
ps_size_t total_memory     = 0,
       used_memory      = 0;
#endif /* PS_INTERNAL_ALLOC */

unsigned short save_level = 0;

/* ************************************************************************* *
 *   Memory management routines :                                            *
 *                                                                           *
 * ************************************************************************* */

/* -------------------------***********************------------------------- */
ps__errors
ps__vmstatus()
{
  PUSH(opstack,        new_int(save_level));
  PUSH(opstack,        new_int(allocated_memory));
#if CAN_GET_MAX_MEMORY
  return PUSH(opstack, new_int(maximum_memory));
#else
  return PUSH(opstack, new_int(-1));
#endif /* CAN_GET_MAX_MEMORY */
  }

#if PS_INTERNAL_ALLOC
/* ************************************************************************* */
ps_internal_alloc_add_chunk()
{
  char *chunk = malloc(CHUNKSIZE);

  if (chunk == NULL) return 0;
  used_memory  = (alloc_bot   - alloc_top)  +
                 (alloc_limit - alloc_base) + used_memory;
  total_memory = alloc_limit - alloc_base + total_memory;
  alloc_base   = alloc_bot = chunk;
  alloc_limit  = alloc_top = chunk + CHUNKSIZE;
  return 1;
  }

/* ************************************************************************* */
char *
ps_internal_alloc(ps_size_t size)
{
  extern char *malloc();
  ps_size_t    left = alloc_top - alloc_bot;

  size = ALIGN(size + sizeof(ps_size_t));
  if (3*size >= CHUNKSIZE) {
    char *mblock = malloc(size + sizeof(char *));

    if (mblock != NULL) {
      char *block = mblock + sizeof(char *);

      *(char **)mblock    = malloc_chain;
      *(ps_size_t *)block = size;
      malloc_chain     = mblock;
      return block + sizeof(ps_size_t);
      }
    }
  if ((size > left) && !ps_internal_alloc_add_chunk()) return NULL;
  alloc_top -= size;
  *(ps_size_t *)alloc_top = size;
  return alloc_top + sizeof(ps_size_t);
  }

/* ************************************************************************* */
void
ps_internal_free(char  *addr, ps_size_t size)
{
  unsigned int size;

  addr -= sizeof(ps_size_t);
  size  = *(ps_size_t *)addr;
  if (addr == alloc_top) alloc_top += size;
  else
    if (addr + size == alloc_bot) alloc_bot = addr;
    else
      if ((addr <  alloc_base) || (addr >= alloc_limit)) {
        char **prev, *mblock;

        for (prev = &malloc_chain; *prev; prev = (char **)mblock) {
          mblock = *prev;
          if (mblock + sizeof(char *) == addr) {
            *prev = *(char **)mblock;
            free(mblock);
            return;
            }
          }
        }
  }
#endif /* PS_INTERNAL_ALLOC */

/* ************************************************************************* */
void *
ps_alloc(ps_size_t size)
{
  char *a;

  size = ALIGN(size+sizeof(block_header));
  if (size > MAX_ALLOCATABLE_SIZE) {
    ps__printf("*** Enormous chunk to be allocated... (%d bytes) ***\n", size);
    return NULL;
    }
  a = MALLOC(size);
  if (memtrace) ps__printf("-- Allocating %d bytes at %X\n",
                           size, a);
  if (!a) {
    ps__puts("*** Allocation Error !! ***");
    return NULL;
    }
  else {
    ((block_header *)a)->references =                    0;
    allocated_memory += size;
    ((block_header *)a)->size       = size + (save_level<<24);
    return a+sizeof(block_header);
    }
  }

/* ************************************************************************* */
void
ps_free(void *ptr)
{
  if (memtrace) ps__printf("-- Free of adress %X --\n",
                           (char *)ptr-sizeof(block_header));
  if (!ptr)
    ps__puts("*** Free of NULL pointer ! ***");
  else {
    block_header *header = HEADER(ptr);
    unsigned int size = header->size & MAX_ALLOCATABLE_SIZE;
    if (!(header->references)) {
      allocated_memory -= size;
      FREE(header, size);
      }
    }
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__memstat()
{
  return PUSH(opstack, new_int(allocated_memory));
  }

#ifdef MEMDEBUG
/* --------------------*********************************-------------------- */
ps__errors
ps__memtrace()
{
  memtrace = 1-memtrace;
  return ps_e_operationok;
  }
#endif /* MEMDEBUG */

typedef struct _save_header {
  unsigned short       save_level;
  struct _save_header *next;
  } *save_header;
save_header first_save_header = NULL;

/* --------------------*********************************-------------------- */
ps__errors
ps__save()
{
  extern ps__errors ps__gsave();
  extern ps__object new_save(save_header);
  save_header       header;

  if (save_level > 128) return ps_e_limitcheck;
  if (!(header = ps_alloc(sizeof(struct _save_header)))) return ps_e_VMerror;
                                 /* Scan each stack to save dynamic objects. */
  save_level++;
  header->save_level = save_level;
  header->next       = first_save_header;
  first_save_header  = header;
  ps__gsave();
  return PUSH(opstack, new_save(header));
  }

/* ************************************************************************* *
 *   Verify there are no recent dynamic objects on an stack, so that we can  *
 *  RESTORE properly.                                                        *
 * ************************************************************************* */
check_stack(save_header header, ps__stack *stack)
{
  unsigned short loop;

  for (loop=0; loop<stack->size; loop++)
    if ((stack->stack[loop].flags & DYNAMIC) &&
        ((block_h(stack->stack[loop])->size >> 24) >= header->save_level))
      return 0;
  return 1;
  }

/* ************************************************************************* *
 *  Actually do the real RESTORE. Each dynamic object of the stack is read : *
 *    - ps_t_dict : then we'll have to destroy all key entries, newer than   *
 *                  the current save level.                                  *
 * ************************************************************************* */
void
restore_stack(save_header header, ps__stack *stack)
{
  unsigned short loop, loop2;
  ps__key       *oldkey;

  for (loop=0; loop<stack->size; loop++) {
    if (stack->stack[loop].flags & DYNAMIC) {
      switch (stack->stack[loop].type) {
        case ps_t_dict : {
          ps__dict dict;

          dict = dict_val(stack->stack[loop]);
          for (loop2 = 0; loop2<dict.size; loop2++) {
            while (dict.keys[loop2].save_level >= header->save_level) {
              CHECK_DESTROY(dict.keys[loop2].name, "restore_stack_dict_name");
              CHECK_DESTROY(dict.keys[loop2].object, "restore_stack_dict_obj");
              if (oldkey = dict.keys[loop2].next) {
                dict.keys[loop2] = *oldkey;
                ps_free(oldkey);
                }
              else {
                dict_val(stack->stack[loop]).size--;
                dict.keys[loop2].name = null_object;
                break;
                }
              }
            }
          }
          break;
        }
      }
    }
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__restore(ps__object save)
{
  extern ps__errors ps__grestoreall();
  save_header       header = (save_header)save_val(save), nextheader;
  unsigned short    loop;

                                                     /* Restore saved stacks */
  if ((header->save_level > save_level)  ||
      !check_stack(header, execstack)    ||
      !check_stack(header, dictstack)    ||
      !check_stack(header, opstack)) return ps_e_invalidrestore;
  restore_stack(header, execstack);
  restore_stack(header, dictstack);
  restore_stack(header, opstack);
  while (header != first_save_header) {
    nextheader = first_save_header->next;
    ps_free(first_save_header);
    first_save_header = nextheader;
    }
  first_save_header = header->next;
  ps_free(header);
  save_level = first_save_header ? first_save_header->save_level : 0;
  return ps__grestoreall();
  }

/* ************************************************************************* */
void
init_memory_stuff()
{
  new_operator("vmstatus", 0, 3, NONE, ps__vmstatus);
  new_operator("memstat",  0, 1, NONE, ps__memstat);
#ifdef MEMDEBUG
  new_operator("memtrace", 0, 0, NONE, ps__memtrace);
#endif /* MEMDEBUG */
  new_operator("save",    0, 1, NONE,  ps__save);
  new_operator("restore", 1, 0, "S",   ps__restore);
  }

/* ************************************************************************* */
void
free_memory_stuff()
{
  extern char *getlogin();
  save_header header;

#if PS_INTERNAL_ALLOC
  printf("Memory Used : %d\nTotal Memory : %d\n", used_memory, total_memory);
  printf("malloc_chain : %X\n", malloc_chain);
#endif /* PS_INTERNAL_ALLOC */
  while (header = first_save_header) {
    first_save_header = header->next;
    ps_free(header);
    }
#if APOLLO || AMIGA
#if APOLLO
  if (!strcmp(getlogin(), "lelaquaf"))
#endif /* APOLLO */
  {
    if (allocated_memory) printf(">> %d unfreed bytes << \n", allocated_memory);
    else                  puts("All memory released");
    }
#endif /* APOLLO || AMIGA */
  }
