/* ************************************************************************* *
 * PostScript Interpretor                   Fabien LELAQUAIS                 *
 *                                                                           *
 *   Fichier main.c                                                          *
 * ************************************************************************* *
 *    This document may be distributed, used, or modified, but can NOT be    *
 *  sold nor incorporated in any way in any product.                         *
 *    Permission is granted to distribute modified versions of that software *
 *  under the condition that this notice remains in every source file.       *
 *    Every alteration of the original files should be marked as such.       *
 *    No warranty is assumed by the author on the concequencies of the use   *
 *  of this software. Any defection of this program is at your own risk,     *
 *  you have to assume the cost of any service, installation or repairs      *
 *  this program could generate.                                             *
 *                                                                           *
 *                          Fabien LELAQUAIS - ESIEE - lelaquaf@apo.esiee.fr *
 * ************************************************************************* */
#include "int.h"
#include <signal.h>

#define PARSING_STRING_LENGTH 65536
extern char *int_version,      /* Defined in `version.c' */
            *int_revision,
            *int_date,
            *parsing;          /* Defined in `parse.c'   */

#define int_prompt   "PS> "

#define profilename  "/.ps"
#ifdef ESIEECOM
#define otherprofile "/user/esiee/sys/data/PSintProfile"
#else
#define otherprofile "profile.ps"
#endif /* ESIEECOM */

#define MAX_OPSTACK_SIZE  1500
#define MAX_EXECSTACK_SIZE 300
#define MAX_DICTSTACK_SIZE 100

#define MAX_SYSTEMDICT_SIZE  256
#define MAX_USERDICT_SIZE    256

ps__stack *opstack, *dictstack, *execstack;
ps__object userdict,       systemdict,  stop_object,
           null_object,    true_object, false_object,
           invalid_object, mark_object, closemark_object;

extern void ps_free();
/* Internal special functions */
extern void help(),
            run_interpretor(),
            shut_interpretor(),
            print_more_object(),
            bye();
/* Other modules */
extern void init_memory_stuff(),
            init_exec_stuff(),
            init_image_stuff(),
            init_dict_stuff(),
            init_gstate_stuff(),
            init_file_stuff(),
            init_com_stuff(),
            init_path_stuff(),
            init_fill_stuff(),
            init_font_stuff(),
            init_cache_stuff(),
            init_fonc_stuff(),
            free_cache_stuff(),
            free_font_stuff(),
            free_fill_stuff(),
            free_path_stuff(),
            free_gstate_stuff(),
            free_device_stuff(),
            free_memory_stuff();

/* ************************************************************************* */
void
help()
{
  puts  ("***********************************************************");
  printf("   This is PSint version %s (%s) F.Lelaquais\n",
         int_revision, int_date);
  printf("   It has been developped to emulate most of the Rev.%s Postscript\n",
        int_version);
  puts  ("langage requirements.\n");
  puts  ("   Usage : psint [-H[elp]]");
  puts  ("                 [-S[tartup] [file]]");
  puts  ("                 [-R[un] string]");
  puts  ("                 [files...]\n");
  puts  (" -Help           : Shows this small help");
  puts  (" -Startup [file] : Reads `file' as the startup file,");
  puts  ("                   or none if `file' is empty.");
  printf("                   default startupfile : $HOME%s\n", profilename);
  printf("                   then : %s\n", otherprofile);
  puts  (" -Run string     : Executes `string' as a PostScript langage");
  puts  ("                   command line, after profile loading.");
  puts  (" Files...        : Reads and executes files as PostScript source\n");
  puts  ("  Enjoy the show.");
  }

/* ************************************************************************* */
break_asked()
{
#if !AMIGA
  signal(SIGQUIT, break_asked);
#endif /* !AMIGA */
  signal(SIGINT,  break_asked);
  PUSH(execstack, stop_object);
  }

/* ************************************************************************* *
 *   Main :                                                                  *
 *     initialize interpretor,                                               *
 *     run        interpretor,                                               *
 *     release    interpretor.                                               *
 *                                                                           *
 *  arguments :                                                              *
 *      -h[elp]                : Shows a small help text on PSint arguments  *
 *      -s[tartup_file] [file] : Specifies one ONLY startup file. If no file *
 *                               name is given, then don't load any startup. *
 *      -c[ommand] string      : Executes `string' as a PostScript line.     *
 * ************************************************************************* */
void
main(int argc, char **argv)
{
  ps__object new_file();
  FILE      *profile = NULL, *execfile;
  char      *p, filename[128], *getenv(), load_profile = 1;
  int        loop;
  
  if (!initialize_interpretor()) {
    puts("Sorry... Couldn't start interpretor");
    bye(-1);
    }
  if (init_device_stuff())
    bye("Sorry... Couldn't create device stuff");
  ps__printf("[PostScript Interpretor     Rev.%s    F.L %s]\n",
             int_revision, int_date);
  PUSH(execstack, cvx(mark_object));
  PUSH(execstack, cvx(do_name("executive")));

                              /* Parse PSint arguments to check their syntax */
  for (loop=1; loop<argc; loop++) {
    if (argv[loop][0] == '-') {
      switch (argv[loop][1]) {
        case 'R' : case 'r' :                     /* PostScript command line */
          argv[loop][1] = 'r';                  /* Make sure it's lower case */
          if ((loop+1 == argc) || (argv[++loop][0] == '-')) {
            puts("** Bad syntax in `command' option");
            bye(-1);
            }
          break;
        case 'S' : case 's' :                     /* PostScript Startup file */
          argv[loop][1] = 's';
          if (load_profile != 1) {
            puts("** Error : `startup' option can be used only once");
            bye(-1);
            }
          if ((loop+1 < argc) && (argv[loop+1][0] != '-')) {
            if (!(profile = fopen(argv[++loop], "r"))) {
              puts("** No such startup file");
              bye(-1);
              }
            }
          load_profile = 0;
          break;
        case 'H' : case 'h' :                     /* PSint help              */
          help();
          bye(0);
          break;
        default :
          printf("** Invalid PSint option `%s'\n", argv[loop]);
          bye(-1);
        }
      }
    }

                /* Re-scan arguments to put them on stack in the right order */
  for (loop = argc-1; loop; loop--) {
    if (argv[loop][0] != '-') {
      if ((loop == 1) ||                     /* Argument = File to be loaded */
          ((argv[loop-1][0] != '-') && (argv[loop-1][1] != 's'))) {
        if (execfile = fopen(argv[loop], "r"))
          PUSH(execstack, cvx(cvro(new_file(execfile))));
        else {
          printf("** No such PostScript source file `%s'\n", argv[loop]);
          bye(-1);
          }
        }
      else
        if ((argv[loop-1][0] == '-') && (argv[loop-1][1] == 'r'))
          PUSH(execstack, cvx(cvro(do_string(argv[loop]))));
      }
    }

  if (profile) PUSH(execstack, cvx(cvro(new_file(profile))));
  else
    if (load_profile) {
      strcpy(filename, (p=getenv("HOME"))?p:"~");
      strcat(filename, profilename);
      if (profile = fopen(filename, "r"))
        PUSH(execstack, cvx(cvro(new_file(profile))));
      else {
        if (profile = fopen(otherprofile, "r"))
          PUSH(execstack, cvx(cvro(new_file(profile))));
        else
          ps__puts("** No profile has been found...");
        }
      }
    else ps__puts("** No startup file has been required...");

#if !AMIGA
  signal(SIGQUIT, break_asked);
#endif /* !AMIGA */
  signal(SIGINT,  break_asked);
  run_interpretor();
  bye(0);
  }

/* ************************************************************************* */
void
bye(int error)
{
#ifdef VAX
  if (!error) error = 1;
#endif
  shut_interpretor();
  ps__puts("Bye.");
  free_device_stuff();
  exit(error);
  }

/* ************************************************************************* *
 *  obcmp compares two object, as EQ might do :                              *
 *     return TRUE  if type do not check,                                    *
 *            FALSE (0) if they are equal (same type, same value)            *
 *     any number less than zero if ob1 is 'less' than ob2                   *
 *     any number greater than zero if ob1 is 'greater' than ob2             *
 * ************************************************************************* */
obcmp(ps__object ob1, ps__object ob2)
{
  if ((ob1.type == ps_t_int) && (ob2.type == ps_t_real))
    ob1 = new_real((double)int_val(ob1));
  else
    if ((ob2.type == ps_t_int) && (ob1.type == ps_t_real))
      ob2 = new_real((double)int_val(ob2));
    else {
      if (ob1.type == ps_t_name) ob1.type = ps_t_stg;
      if (ob2.type == ps_t_name) ob2.type = ps_t_stg;
      if (ob1.type != ob2.type) return TRUE;
      }
  switch (ob1.type) {
    case ps_t_null : return FALSE;
    case ps_t_int  :
      return int_val(ob1)-int_val(ob2);
    case ps_t_real :
      return (real_val(ob1)<real_val(ob2))? -1
                   :((real_val(ob1)==real_val(ob2))?FALSE:1);
    case ps_t_stg  :
#if 0
      {
      int i = MIN(ob1.size, ob2.size);
      char *p1=(char *)stg_val(ob1), *p2=(char *)stg_val(ob2);
      for (;i; i--, p1++, p2++) if (*p1 != *p2) return *p1 - *p2;
      return ob1.size-ob2.size;
      }
#else
      {
      int comp = memcmp(stg_val(ob1), stg_val(ob2), MIN(ob1.size, ob2.size));
      return (comp)?comp:((int)(ob1.size-ob2.size));
      }
#endif
    case ps_t_bool  :
      return !(bool_val(ob1)&bool_val(ob2));
    case ps_t_mark  : return FALSE;
    case ps_t_array :
      return !(arr_val(ob1) == arr_val(ob2));
    case ps_t_operator :
      return !cmp(op_val(ob1), op_val(ob2));
    case ps_t_dict :
      return !cmp(dict_val(ob1), dict_val(ob2));
    case ps_t_fontId :
      return !(font_val(ob1) == font_val(ob2));
    case ps_t_save :
      return !(save_val(ob1) == save_val(ob2));
    default :
      ps__printf("!!** Obcmp of two invalid objects !!");
      return FALSE;
    }
  }

/* ************************************************************************* */
char cvsbuf[128];
char *
do_cvs(ps__object object)
{
  switch (object.type) {
    case ps_t_invalid  : case ps_t_null     :
    case ps_t_mark     : case ps_t_file     :
    case ps_t_array    : case ps_t_dict     :
    case ps_t_fontId   : case ps_t_save     :
      sprintf(cvsbuf, "-- nostringval --");
      return cvsbuf;
    case ps_t_int      :
      sprintf(cvsbuf, "%d", int_val(object));
      return cvsbuf;
    case ps_t_real     :
      sprintf(cvsbuf, "%g", real_val(object));
      return cvsbuf;
    case ps_t_bool     :
      return bool_val(object)?"true":"false";
    case ps_t_stg : case ps_t_name     :
      return (char *)stg_val(object);
    case ps_t_operator :
      return op_val(object).name;
    default :
      sprintf(cvsbuf, "--Type %d--", object.type);
      return cvsbuf;
    }
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__cvrs(ps__object number, ps__object radix, ps__object string)
{
  extern char ps_numbers[37];
  extern double log(double);
  int    num, rad, len, l;

  if (((rad = int_val(radix)) < 2) || (rad > 26)) return ps_e_rangecheck;
  if (((num = to_real(number)) == 0) || (num == 1)) {
    if (string.size < 1) return ps_e_rangecheck;
    stg_val(string)[0] = (num == 0)?'0':'1';
    string.size = 1;
    }
  else {     /* Do you have a nicer way ? */
    if (string.size < (len = (.999999999+log((double)num)/log((double)(rad)))))
      return ps_e_rangecheck;
    string.size = len;
    for (l=0; l<len; l++) {
      stg_val(string)[len-l-1] = ps_numbers[num % rad];
      num /= rad;
      }
    }
  return PUSH(opstack, string);
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__cvs(ps__object object, ps__object string)
{
  char           *p=(char *)stg_val(string), *q;
  unsigned short  l;

  if (!HAS_W(string)) return ps_e_invalidaccess;
  if ((object.type == ps_t_name) ||
      (object.type == ps_t_stg)) {
    if (string.size < object.size)     return ps_e_rangecheck;
    string.size = object.size;
    for (q=(char *)stg_val(object);object.size;*p++ = *q++, object.size--);
    }
  else {
    if (strlen(do_cvs(object)) > string.size) return ps_e_rangecheck;
    l = string.size = strlen(cvsbuf);
    for (q=(char *)cvsbuf;l;*p++ = *q++, l--);
    }
  return PUSH(opstack, string);
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__equal(ps__object object)
{
  if ((object.type == ps_t_stg) ||
      (object.type == ps_t_name)) {
    obfcpy(object);
    ps__putchar('\n');
    }
  else
    ps__puts(do_cvs(object));
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__stackop()
{
  int loop = opstack->size;
  for (; loop; loop--)
    ps__equal(opstack->stack[loop-1]);
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__pstack()
{
  int loop = opstack->size;
  for (; loop; loop--) {
    print_more_object(opstack->stack[loop-1]);
    ps__putchar('\n');
    }
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__equalequal(ps__object object)
{
  print_more_object(object);
  ps__putchar('\n');
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__quit()
{
  ps__object exec;

  while (execstack->size) {
    exec = POP(execstack);
    ps_destroy_object(exec, "quit");
    }
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__print(ps__object stg)
{
  obfcpy(stg);
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__shell(ps__object command)
{
  char *line, *p, *getenv();
  
  line = (p = getenv("SHELL"))?ps_alloc(command.size+strlen(p)+2)
                              :ps_alloc(command.size+1);
  if (p) {
    strcpy(line, p);
    strcat(line, " ");
    }
  else
    line[0] = 0;
  strncat(line, stg_val(command), command.size);
printf("Executing `%s'\n", line);
  system(line);
  ps_free(line);
  return ps_e_operationok;
  }

/* ************************************************************************* */
ps__errors
ps__closemark()
{
  int size = stack_find(mark_object);
  ps__object array;

  if (size == -1) return ps_e_unmatchedmark;
  array = new_array(size = opstack->size-1-size);
  for (; size; size--) {
    arr_ob(array, size-1) = POP(opstack);
    MORE_REFS(arr_ob(array, size-1));
    }
  POP(opstack);                /* This should be the first matched marker ?? */
  return PUSH(opstack, array);
  }

/* ************************************************************************* */
ps__stack *
make_stack(unsigned short size, ps__errors errun, ps__errors errov)
{
  ps__stack *newstack;

  if (!(newstack        = ps_alloc(sizeof(ps__stack))) ||
      !(newstack->stack = ps_alloc(sizeof(ps__object)*size))) {
    if (newstack) ps_free(newstack);
    return NULL;
    }
  newstack->errun   = errun;
  newstack->errov   = errov;
  newstack->maxsize = size;
  newstack->size    = 0;
  return newstack;
  }

/* ************************************************************************* */
ps__errors
print_complete_object(ps__object object)
{
  short loop;
  
  switch (object.type) {
    case ps_t_invalid  :
      ps__puts("-- Object : invalid --");
      break;
    case ps_t_null     :
      ps__puts("-- Object : null --");
      break;
    case ps_t_int      :
      ps__puts("-- Object : int --");
      ps__printf("   => %d\n", int_val(object));
      break;
    case ps_t_real     :
      ps__puts("-- Object : real --");
      ps__printf("   => %f\n", real_val(object));
      break;
    case ps_t_bool     :
      ps__puts("-- Object : boolean --");
      ps__printf("   => %s\n", bool_val(object)?"TRUE":"FALSE");
      break;
    case ps_t_stg      :
      ps__puts("-- Object : string --");
      ps__printf("   => ");
      obfcpy(object);
      ps__puts("");
      break;
    case ps_t_file     :
      ps__puts("-- Object : file --");
      ps__printf("   Size : %d\n", object.size);
      ps__printf("   Adr. : %X\n", object.value.fileval);
      break;
    case ps_t_mark     :
      ps__puts("-- Object : mark --");
      break;
    case ps_t_array    :
      ps__puts("-- Object : array --");
      ps__printf("   Size : %d\n", object.size);
      ps__printf("   Adr. : %X\n", arr_val(object));
      for (loop=0; loop<object.size; loop++) {
        switch(arr_ob(object, loop).type) {
          case ps_t_invalid  :  ps__printf("   invalid : "); break;
          case ps_t_null     :  ps__printf("   null   : "); break;
          case ps_t_mark     :  ps__printf("   mark   : "); break;
          case ps_t_int      :  ps__printf("   int    : "); break;
          case ps_t_real     :  ps__printf("   real   : "); break;
          case ps_t_bool     :  ps__printf("   bool   : "); break;
          case ps_t_stg      :  ps__printf("   stg    : "); break;
          case ps_t_name     :  ps__printf("   name   :"); break;
          case ps_t_operator :  ps__printf("   op     : "); break;
          case ps_t_array    :  ps__printf("   array  : "); break;
          case ps_t_file     :  ps__printf("   file   : "); break;
          case ps_t_dict     :  ps__printf("   dict   : "); break;
          case ps_t_fontId   :  ps__printf("   fontId : "); break;
          case ps_t_save     :  ps__printf("   save   : "); break;
          default            :  ps__printf("   ** Unknown ** : ");
          }
        print_more_object(arr_ob(object, loop));
        ps__putchar('\n');
        }
      break;
    case ps_t_operator :
      ps__puts("-- Object : operator --");
      ps__printf("  => %s\n", op_val(object).name);
      break;
    case ps_t_name     :
      ps__puts("-- Object : name --");
      ps__printf("   => ");
      obfcpy(object);
      ps__puts("");
      break;
    case ps_t_dict     :
      ps__puts("-- Object : dictionnary --");
      ps__printf("   Size : %d\n", object.size);
      ps__printf("   Adr. : %X (keys : %X)\n", object.value.dictval,
                                               object.value.dictval->keys);
      break;
    case ps_t_fontId   :
      ps__puts("-- Object : fontid --");
      break;
    case ps_t_save     :
      ps__puts("-- Object : save --");
      break;
    default :
      ps__printf("-- Unknown type : %d --\n", object.type);
    }
  ps__printf("   access : %c%c%c\n",
          HAS_R(object)?'r':'-', HAS_W(object)?'w':'-', HAS_X(object)?'x':'-');
  if (object.flags & DYNAMIC) ps__printf("   DYNAMIC, %d links\n",
                                         REFS(object));
  return ps_e_operationok;
  }

/* ************************************************************************* */
initialize_interpretor()
{
  extern ps__errors ps__stop();

  if (!(parsing = ps_alloc(PARSING_STRING_LENGTH))) return FALSE;
  HEADER(parsing)->references++;
  null_object    = new_object(ps_t_null);
  true_object    = new_boolean(TRUE);
  false_object   = new_boolean(FALSE);
  invalid_object = new_object(ps_t_invalid);
  mark_object    = new_object(ps_t_mark);
  opstack   = make_stack(MAX_OPSTACK_SIZE,   ps_e_stackunderflow,
                         ps_e_stackoverflow);
  if (!opstack) {
    ps__puts("** Unable to create op stack **"); return FALSE; }
  dictstack = make_stack(MAX_DICTSTACK_SIZE, ps_e_dictstackunderflow,
                         ps_e_dictstackoverflow);
  if (!dictstack) {
    ps__puts("** Unable to create dictstack **"); return FALSE; }
  execstack = make_stack(MAX_EXECSTACK_SIZE, ps_e_execstackunderflow,
                         ps_e_execstackoverflow);
  if (!execstack) {
    ps__puts("** Unable to create execstack **"); return FALSE; }
  systemdict = new_dict(MAX_SYSTEMDICT_SIZE);
  dict_store(systemdict, do_name("systemdict"), systemdict);
  userdict = new_dict(MAX_USERDICT_SIZE);
  dict_store(systemdict, do_name("userdict"),     userdict);
  dict_store(systemdict, do_name("null"),         null_object);
  dict_store(systemdict, do_name("mark"),         mark_object);
  dict_store(systemdict, do_name("true"),         true_object);
  dict_store(systemdict, do_name("false"),        false_object);
  dict_store(systemdict, do_name("version"),      do_string(int_version));
  dict_store(systemdict, do_name("promptstring"), do_string(int_prompt));
  dict_store(systemdict, do_name("prompt"),
             cvx(do_string("{promptstring print flush}")));
  dict_store(systemdict, do_name("executive"),
             cvx(do_string("{{prompt exec(%statementedit)(r)file exec}stopped\
{(Stopped\n)print}if}bind loop")));
  PUSH(dictstack, systemdict);
  PUSH(dictstack, userdict);
  stop_object      = new_operator("stop", 0, 0, NONE, ps__stop);
  closemark_object = cvx(do_name(" closemark"));
  new_operator(" closemark", 0, 1, NONE,   ps__closemark);
  new_operator("print",    1, 0, "s",      ps__print);
  new_operator("quit",     0, 0, NONE,     ps__quit);
  new_operator("=",        1, 0, NONE,     ps__equal);
  new_operator("==",       1, 0, NONE,     ps__equalequal);
  new_operator("===",      1, 0, NONE,     print_complete_object);
  new_operator("stack",    0, 0, NONE,     ps__stackop);
  new_operator("pstack",   0, 0, NONE,     ps__pstack);
  new_operator("cvrs",     3, 1, "ir.i.s", ps__cvrs);
  new_operator("cvs",      2, 1, "x.s",    ps__cvs);
  new_operator("shell",    1, 0, "s",      ps__shell);
  init_memory_stuff();                /* Memory management routines          */
  init_file_stuff();                  /* File management routines            */
  init_exec_stuff();                  /* Execution control                   */
  init_fonc_stuff();                  /* All math and logical fonctions      */
  init_com_stuff();                   /* Other commands                      */
  init_dict_stuff();                  /* Dictionnary management              */
  init_gstate_stuff();                /* Gstate manipulation                 */
  init_path_stuff();                  /* Graphic path construction routines  */
  init_fill_stuff();                  /* Fill operators                      */
  init_font_stuff();                  /* PostScript Font Machinery           */
  init_cache_stuff();                 /* PostScript Font Cache Machinery     */
  init_image_stuff();                 /* PostScript Image Processing Package */
  return TRUE;
  }

/* ************************************************************************* */
void
shut_interpretor()
{
  int        loop;
  struct _proc {
    ps__object    object;
    struct _proc *next;
    };
  extern struct _proc *free_procs;
  struct        _proc *oneproc;
  ps__object obj;

  free_cache_stuff();
  free_font_stuff();
  free_fill_stuff();
  free_gstate_stuff();                      /* Must be done BEFORE free_path */
  free_path_stuff();
  while (oneproc = free_procs) {
    ps_free(free_procs);
    free_procs = oneproc->next;
    }
  ps_free(execstack->stack);
  ps_free(execstack);
  while (opstack->size)
    ps_destroy_object(POP(opstack), "shut-opstack");
  ps_free(opstack->stack);
  ps_free(opstack);
          /* Destroy operator                                                */
  for (loop=0; loop<dict_val(systemdict).size; loop++) {
    obj = dict_val(systemdict).keys[loop].object;
    if (obj.type == ps_t_operator) ps_free(BLOCK(obj));
    }
  while (dictstack->size)
    ps_destroy_object(POP(dictstack), "shut-dictstack");
  ps_free(dictstack->stack);
  ps_free(dictstack);
  HEADER(parsing)->references--;
  ps_free(parsing);
  free_memory_stuff();
  }

/* ************************************************************************* */
void
print_more_object(ps__object object)
{
  switch (object.type) {
    case ps_t_invalid  :
      ps__printf("-- invalidtype --");
      break;
    case ps_t_null     :
      ps__printf("-- nulltype --");
      break;
    case ps_t_mark     :
      ps__printf("-- marktype --");
      break;
    case ps_t_int      :
      ps__printf("%d", int_val(object));
      break;
    case ps_t_real     :
      ps__printf("%g", real_val(object));
      break;
    case ps_t_bool     :
      ps__printf("%s", bool_val(object)?"true":"false");
      break;
    case ps_t_stg  :
      ps__putchar('(');
      obfcpy(object);
      ps__putchar(')');
      break;
    case ps_t_name :     
      ps__putchar('/');
      obfcpy(object);
      break;
    case ps_t_operator :
      ps__printf("--%s--", op_val(object).name);
      break;
    case ps_t_array    : { int loop;
      ps__putchar(HAS_X(object)?'{':'[');
      if (object.size) {
        for (loop=0; loop<object.size-1; loop++) {
          print_more_object(arr_ob(object, loop));
          ps__putchar(' ');
          }
        print_more_object(arr_ob(object, loop));
        }
      ps__putchar(HAS_X(object)?'}':']');
      }
      break;
    case ps_t_file     :
      ps__printf("-- filetype --");
      break;
    case ps_t_dict     :
      ps__printf("-- dicttype --");
      break;
    case ps_t_fontId   :
      ps__printf("-- fontidtype --");
      break;
    case ps_t_save   :
      ps__printf("-- savetype --");
      break;
    default :
      ps__printf("-- Unknown type : %d --", object.type);
    }
  }
