/* ************************************************************************* *
 * PostScript Interpretor                   Fabien LELAQUAIS                 *
 *                                                                           *
 *   Fichier Int.h                                                           *
 *                           Version 3.00 on 24/02/89                        *
 * ************************************************************************* *
 *            Fichier de definitions des types de PSINT                      *
 * ************************************************************************* *
 *    This document may be distributed, used, or modified, but can NOT be    *
 *  sold nor incorporated in any way in any product.                         *
 *    Permission is granted to distribute modified versions of that software *
 *  under the condition that this notice remains in every source file.       *
 *    Every alteration of the original files should be marked as such.       *
 *    No warranty is assumed by the author on the concequencies of the use   *
 *  of this software. Any defection of this program is at your own risk,     *
 *  you have to assume the cost of any service, installation or repairs      *
 *  this program could generate.                                             *
 *                                                                           *
 *                          Fabien LELAQUAIS - ESIEE - lelaquaf@apo.esiee.fr *
 * ************************************************************************* */

/* ========================================================================= *
 *  MEMDEBUG was created to get a trace of each allocation or desallocation, *
 *  at debug time.                                                           *
 *  If you leave that define, set it to 1 or 0, depending on if you want     *
 *  memory debugging as soon as PSint is run, or not. A new operator is      *
 *  created, ``memtrace'', that toggles memory debugging on or off.          *
 * ========================================================================= */
/*
#define   MEMDEBUG 0
 */

#ifdef MEMDEBUG
extern char memtrace;
#else
#define memtrace 0
#endif

#include "machine.h"

#include <stdio.h>

#ifndef EOF
#define EOF (-1)
typedef struct _iobuf {
  char *_ptr;
  short _cnt;
  char *_base;
  short _flag;
  short _fd ;
  } FILE ;
#endif

                                      /* Declarations des constantes vitales */
#ifndef NULL
#define NULL 0L
#endif
#ifndef TRUE
#define TRUE  0xff
#define FALSE 0
#endif
#define _PI_  3.141592653589

#if 1 /* Particulary VAX */
#define cmp(a,b) (!memcmp(&(a),&(b),sizeof(a)))
#else /* Worked on APOLLO */
#define cmp(a,b) ((a)==(b))
#endif

#define degrad(x) (_PI_*(x)/180)
#define MIN(a,b) (((a)<(b))?(a):(b))
#define MAX(a,b) (((a)>(b))?(a):(b))

typedef enum {                                 /* Liste des types PostScript */
  ps_t_invalid,                               /* Caractere de codage interne */
  ps_t_null,                         /* 'N' */
  ps_t_int,                          /* 'i' */
  ps_t_real,                         /* 'r' */
  ps_t_bool,                         /* 'b' */
  ps_t_stg,                          /* 's' */
  ps_t_file,                         /* 'F' */
  ps_t_mark,                         /* 'm' */
  ps_t_array,                        /* 'a' */
  ps_t_operator,                     /* 'o' */
  ps_t_name,                         /* 'n' */
  ps_t_dict,                         /* 'd' */
  ps_t_fontId,                       /* 'f' */
  ps_t_save                          /* 'S' */
  } ps__type;
                                              /* Liste des erreurs reconnues */
typedef enum {             
  ps_e_operationok,                    /* Pas d'erreur, Terminaison correcte */
  ps_e_notimplemented,    /* Fonction non encore completement operationnelle */
  ps_e_syntaxerror,                                       /* Mot-cle inconnu */
  ps_e_stackunderflow,          /* Pas assez d'elements dans la pile operand */
  ps_e_stackoverflow,                         /* La pile operand est remplie */
  ps_e_dictstackunderflow,      /* Pas assez de dictionnaires dans dictstack */
  ps_e_dictstackoverflow,    /* Trop de dictionnaires dans dictstack (begin) */
  ps_e_execstackunderflow,                                 /* Pile exec vide */
  ps_e_execstackoverflow,                                /* Pile exec pleine */
  ps_e_typecheck,    /* Le type de l'objet est incompatible avec la fonction */
  ps_e_undefinedresult,        /* Resultat indefini (ex : Division par zero) */
  ps_e_undefined,                                             /* Nom inconnu */
  ps_e_undefinedfilename,              /* Fichier inexistant ou non ouvrable */
  ps_e_unmatchedmark,                           /* Pas de mark correspondant */
  ps_e_limitcheck,                                /* Depassement des limites */
  ps_e_dictfull,                               /* Le dictionnaire est rempli */
  ps_e_VMerror,                                  /* Erreur de Virtual Memory */
  ps_e_IOerror,                                    /* Erreur d'entree-sortie */
  ps_e_invalidexit,                     /* exit hors d'un contexte de boucle */
  ps_e_invalidfont,                                       /* FontId invalide */
  ps_e_invalidrestore,                          /* Restore sans Save courant */
  ps_e_invalidfileaccess,                     /* Acces invalide a un fichier */
  ps_e_invalidaccess,                         /* Acces invalide (R, W, ou X) */
  ps_e_nocurrentpoint,                               /* Pas de point courant */
  ps_e_rangecheck          /* Lecture/ecriture hors des limites d'un tableau */
  } ps__errors;


                       /* Un operateur ne peut pas avoir plus de 8 arguments */
#define MAX_OP_ARGUMENTS 8
              /* En-tete de block memoire alloue dynamiquement pour un objet */
typedef struct {
  unsigned short references;       /* Nombre d'objets pointants sur ce block */
  unsigned long  size;             /* Real aligned space size                */
  } block_header;
                                  /* Attributs divers des objects PostScript */
#define READ    1                          /* Attribut d'acces en lecture    */
#define WRITE   2                          /* Attribut d'acces en ecriture   */
#define EXEC    4                          /* Attribut d'acces en execution  */
#define DYNAMIC 8                          /* Objets alloues dynamiquement   */

/* Le gros morceau :
    ps__object est une structure variable permettant de n'utiliser qu'une seule
   structure pour representer differents types d'objets.
    Ainsi, un objet de ce type peut representer, de maniere interne, soit :
     real - un entier  (32 bits)
     int  - un real    (64 bits)
     bool - un booleen (8 bits)
     stg  - une chaine de caracteres
     arr  - un tableau
     name - un nom PostScript
     lit  - un nom literal (voir Fonc)
     dict - un dictionnaire
     op   - un operateur
     proc - une procedure
     font - une police de caracteres

   Le type de l'objet ainsi defini est place dans le champ `type'.
   La taille occupee (si taille variable) est dans le champs `size'.
   Les attributs de l'objet sont regroupes dans le champs `flags'.

   Le champs `header' est utilise pour lier les objets composites pointants
 vers la meme place memoire. Ceci permet de desallouer ces objets des qu'ils
 ne sont plus references dans aucune pile, ni aucun dictionnaire.
 */
typedef struct ps___object {
  unsigned short flags;
  unsigned int   size;
  ps__type type;
  union {
    double                realval;
    long                  intval;
    char                  boolval;
    unsigned char *       stgval;
    unsigned char *       nameval;
    struct _ps__array    *arrval;
    struct _ps__dict     *dictval;
    FILE                 *fileval;
    struct _ps__operator *opval;
    int                   fontval;
    char                 *saveval;
    } value;
  block_header *header;     /* Real beginning address for a composite object */
  } ps__object;
                                                   /* Accessing the fields : */
#define real_val(ob)  (  (ob).value.realval        )
#define int_val(ob)   (  (ob).value.intval         )
#define bool_val(ob)  (  (ob).value.boolval        )
#define stg_val(ob)   (  (ob).value.stgval         )
#define name_val(ob)  (  (ob).value.nameval        )
#define arr_val(ob)   (  (ob).value.arrval         )
#define arr_ob(ob, n) (  (ob).value.arrval[n].value)
#define dict_val(ob)  (*((ob).value.dictval)       )
#define file_val(ob)  (  (ob).value.fileval        )
#define op_val(ob)    (*((ob).value.opval)         )
#define font_val(ob)  (  (ob).value.fontval        )
#define save_val(ob)  (  (ob).value.saveval        )
#define block_h(ob)   (  (ob).header               )

                       /* Adresse effective de debut de zone memoire occupee */
#define HEADER(s) ((block_header *)((char *)(s)-sizeof(block_header)))
          /* Nombre de references a la memoire occupee par l'objet dynamique */
#define REFS(ob)  (*(unsigned short *)((ob).header))
                       /* Adresse effective de debut de zone memoire occupee */
#define BLOCK(ob) ((char *)((ob).header)+sizeof(block_header))
                       /* Reference qui disparait -> objet a detruire (maybe)*/
#define CHECK_DESTROY(ob, m) if (((ob).flags & DYNAMIC) && !(--REFS(ob)))\
                         ps_destroy_object(ob, m?m:"Unknown");
                       /* One more reference to that dynamic object          */
#define MORE_REFS(ob) if ((ob).flags & DYNAMIC) REFS(ob)++;

                                     /* Pour que new_operator soit plus joli */
#define NONE (char *)0

                                                      /* SubTypes PostScript */
                   /* For the arrays */
typedef struct _ps__array {
  unsigned short     save_level;                  /* Save level when created */
  ps__object         value;                       /* Pointer to objects      */
  struct _ps__array *next;                        /* Next entry for save     */
  } ps__array;
                   /* For the dictionnaries */
typedef struct _ps__key {
  unsigned short    hash;                          /* Key hash value         */
  unsigned short    save_level;                    /* Current save level     */
  ps__object        name, object;                  /* Name and value objects */
  struct _ps__key  *next;                          /* Next entry for save    */
  } ps__key;                            /* Cle = pointeurs vers nom et objet */
typedef struct _ps__dict {
  unsigned short size;                                  /* Longueur courante */
  ps__key       *keys; }     ps__dict;                               /* Cles */
                   /* For the operators */
typedef struct _ps__operator {
  char    *name;
  unsigned short argsin, argsout;
  char    *argstype;
  ps__errors (*operator)(); } ps__operator;

                                         /* Definition d'un stack PostScript */
typedef struct ps___stack {
  unsigned short maxsize, size;
  ps__errors     errun, errov;
  ps__object    *stack;
  } ps__stack;

#if 0
     /* All those were used before version 3.50, when garbage collector
        was first installed. */
                                              /* Manipulation de ces piles : */
#define PUSH(w, o) (((w).size == (w).maxsize)?(w).errov:\
                    ((w).stack[((w).size)++]=(o), ps_e_operationok))
#define POP(w)     ((w).stack[--((w).size)])
#else
extern ps__errors  ps___push(ps__stack *, ps__object);
#define PUSH(w, o) ps___push(w, o)
extern ps__object  ps___pop(ps__stack *);
#define POP(w)     ps___pop(w)
#endif

#define CVRO(o)  (o).flags &=  ~WRITE
#define CVWO(o)  (o).flags &=  ~READ
#define CVXO(o)  (o).flags &= ~(READ|WRITE)
#define CVX(o)   (o).flags |=   EXEC
#define CVLIT(o) (o).flags &=  ~EXEC
#define HAS_R(o) ((o).flags &    READ)
#define HAS_W(o) ((o).flags &    WRITE)
#define HAS_X(o) ((o).flags &    EXEC)

         /* Macro utile pour forcer un objet a retourner une valeur reelle : */
#define to_real(ob) (((ob).type==ps_t_int)?(double)int_val(ob):real_val(ob))
     /* Comparaison d'une chaine de caracteres et d'un object string ou name */
#define obstrcmp(ob, s, l) (((ob).size == l) && !strncmp(stg_val(ob), s, l))
           /* Copie d'un object string ou name dans une chaine de caracteres */
#define obstrcpy(ob, s) { char *p=s, *q=(char *)stg_val(ob);short l;\
                                  for (l=(ob).size;l;*p++ = *q++,l--); *p=0; }
                         /* Copie d'un object string ou name dans un fichier */
#define obfcpy(ob)  { short l; for (l=0;l<(ob).size; \
                      ps__putchar(stg_val(ob)[l++])); }
                     /* Lecture d'un caractere dans une string ou un fichier */
#define getin(o) (((o).type==ps_t_file)?ps__getc(file_val(o)):\
                    (((o).size--)?*stg_val(o)++:(int)EOF))
                 /* 'Delecture' d'un caractere dans une string ou un fichier */
#define getout(o,c) (((o).type==ps_t_file)?ps__ungetc(c,file_val(o)):\
                    ((o).size++, *stg_val(o)--))

extern ps__object  new_object(ps__type),             /* Cree un nouvel objet */
                   new_real(double),
                   new_int(long),
                   new_boolean(long),
                   new_string(char *, unsigned long), do_string(char *),
                   new_name(char *,   unsigned long), do_name(char *),
                   new_array(long),  new_matrix(),
                   new_dict(long),
                   new_operator(char *, unsigned short, unsigned short,
                                char *, ps__errors (*)()),
                   cvro(ps__object), cvwo(ps__object), cvxo(ps__object),
                   cvx(ps__object),  cvlit(ps__object),
                   get_next_command(ps__object *),
                   dict_find_key(ps__object, ps__object),
                   find_key(ps__object),
                   null_object, true_object, false_object,
                   invalid_object, mark_object,
                   systemdict;
extern ps__stack  *opstack, *dictstack, *execstack;

typedef unsigned long ps_size_t;
extern void *ps_alloc(ps_size_t);
extern void ps_destroy_object(ps__object, char *);

