/* ************************************************************************* *
 * PostScript Interpretor                   Fabien LELAQUAIS                 *
 *                                                                           *
 *   Fichier image.c                                                         *
 *       Images management routines for PSint.                               *
 *                           Version 3.00 on 24/02/89                        *
 * ************************************************************************* *
 *    This document may be distributed, used, or modified, but can NOT be    *
 *  sold nor incorporated in any way in any product.                         *
 *    Permission is granted to distribute modified versions of that software *
 *  under the condition that this notice remains in every source file.       *
 *    Every alteration of the original files should be marked as such.       *
 *    No warranty is assumed by the author on the concequencies of the use   *
 *  of this software. Any defection of this program is at your own risk,     *
 *  you have to assume the cost of any service, installation or repairs      *
 *  this program could generate.                                             *
 *                                                                           *
 *                          Fabien LELAQUAIS - ESIEE - lelaquaf@apo.esiee.fr *
 * ************************************************************************* */
#include "int.h"
#include "graph.h"

extern void ps_free();

short       bits;
int         width, height, image_x;
unsigned    char *image_buffer = NULL, *image_p, image_mask;
unsigned    long image_size, current_size;
double      matrix_det, image_matrix[6];

ps__object image_object;

/* --------------------*********************************-------------------- */
ps__errors
ps__image(ps__object w, ps__object h, ps__object b,
          ps__object matrix, ps__object proc)
{
  image_mask = 0;
  if (!is_a_matrix(matrix)) return ps_e_typecheck;
  bits   = int_val(b);
  width  = int_val(w);
  height = int_val(h);
  image_size   = width*height;
  image_matrix[0] = mat(matrix, 0);
  image_matrix[1] = mat(matrix, 1);
  image_matrix[2] = mat(matrix, 2);
  image_matrix[3] = mat(matrix, 3);
  image_matrix[4] = mat(matrix, 4);
  image_matrix[5] = mat(matrix, 5);
  current_size = 0;
  image_x      = 0;
  matrix_det   = image_matrix[0]*image_matrix[3]
               - image_matrix[1]*image_matrix[2];
  if ((matrix_det == 0)             ||
      (width <= 0) || (height <= 0) ||
      ((bits != 1) && (bits != 2) && (bits != 4) && (bits != 8)))
    return ps_e_undefinedresult;
  if (image_buffer) ps_free(image_buffer);
  if (!(image_p = image_buffer = (unsigned char *)ps_alloc(image_size)))
    return ps_e_VMerror;
  PUSH(execstack, proc);
  PUSH(execstack, image_object);
  PUSH(execstack, proc);
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__imagemask(ps__object w, ps__object h, ps__object b,
              ps__object matrix, ps__object proc)
{
  if (!is_a_matrix(matrix)) return ps_e_typecheck;
  image_mask   = bool_val(b) ? 1 : 2;
  width        = int_val(w);
  height       = int_val(h);
  image_size   = width*height;
  image_matrix[0] = mat(matrix, 0);
  image_matrix[1] = mat(matrix, 1);
  image_matrix[2] = mat(matrix, 2);
  image_matrix[3] = mat(matrix, 3);
  image_matrix[4] = mat(matrix, 4);
  image_matrix[5] = mat(matrix, 5);
  current_size = 0;
  image_x      = 0;
  matrix_det   = image_matrix[0]*image_matrix[3]
               - image_matrix[1]*image_matrix[2];
  if ((matrix_det == 0)             ||
      (width <= 0) || (height <= 0))
    return ps_e_undefinedresult;
  if (image_buffer) ps_free(image_buffer);
  if (!(image_p = image_buffer = (unsigned char *)ps_alloc(image_size)))
    return ps_e_VMerror;
  PUSH(execstack, proc);
  PUSH(execstack, image_object);
  PUSH(execstack, proc);
  return ps_e_operationok;
  }

#if SECMAI
double microdots = 0;
#endif /* SECMAI */
/* ************************************************************************* */
ps__errors
ps___image(ps__object string)
{
  unsigned char *p = stg_val(string);

  if (!string.size) {
    while (current_size < image_size) {
      *image_p++ = 0;
      current_size++;
      }
    }
  for (; string.size && (current_size < image_size); string.size--, p++) {
    if (image_mask) {
      short mask = 0x80, loop,
            pixel = (image_mask == 1) ? *p : (255- *p);       /* Inversion ? */
      for (loop=0; loop < 8; loop++, mask >>= 1) {
        *image_p++ = (pixel & mask) ? 255 : 0;
        current_size++;
        image_x++;
        if (image_x == width) {
          image_x = 0;
          break;
          }
        }
      }
    else {
      switch (bits) {
        case 1 : {
          short mask = 0x80, loop, pixel = *p;
          for (loop=7; loop >= 0; loop--, mask >>= 1) {
            *image_p++ =  (pixel & mask) ? 255 : 0;
            current_size++;
            image_x++;
            if (image_x == width) {
              image_x = 0;
              break;
              }
            }
          } break;
        case 2 : {
          short mask = 0xC0, loop, pixel = *p;
          for (loop=0; loop < 4; loop++, mask >>= 2) {
            *image_p++ = ((pixel & mask)<<(loop<<1)) | 0x3F;
            current_size++;
            image_x++;
            if (image_x == width) {
              image_x = 0;
              break;
              }
            }
          } break;
        case 4 :
          *image_p++ =  (*p & 0xF0)     | 0xF;
          current_size++;
          image_x++;
          if (image_x == width)   image_x = 0;
          else {
            *image_p++ = ((*p & 0x0F)<<4) | 0xF;
            current_size++;
            image_x++;
            if (image_x == width) image_x = 0;
            }
          break;
        case 8 :
          *image_p++ = *p;
          current_size++;
        }
      }
    }
  string = POP(execstack);
  if (current_size < image_size) {
    PUSH(execstack, string);
    PUSH(execstack, image_object);
    PUSH(execstack, string);
    }
  else {
  /* Affichage de l'image : ************************************************ *
   *        Le buffer 'image_buffer' doit etre relu, selon un procede        *
   *     de reechantillonnage au plus proche voisin, et transmis,            *
   *     colonne par colonne, a la routine ps__WritePixels, qui va les       *
   *     inscrire sur le buffer de sortie.                                   *
   *        Cet appel contient :                                             *
   *           - les cotes X et Y de debut                                   *
   *           - La longueur du buffer passe                                 *
   *           - Un pointeur sur le premier pixel du buffer,                 *
   *             ou les pixels sont codes entre 0 (noir) et 255 (blanc)      *
   *        Pour ce faire, il faut concatener la CTM avec l'inverse          *
   *     de la matrice de l'image, pour se replacer dans le repere image,    *
   *     puis revenir en repere device.                                      *
   ************************************************************************* */
                                                /* Inverse de image_matrix : */
    double a =   image_matrix[3]/matrix_det,
           b =  -image_matrix[1]/matrix_det,
           c =  -image_matrix[2]/matrix_det,
           d =   image_matrix[0]/matrix_det,
           e =  (image_matrix[2]*image_matrix[5]
                -image_matrix[3]*image_matrix[4])/matrix_det,
           f =  (image_matrix[1]*image_matrix[4]
                -image_matrix[0]*image_matrix[5])/matrix_det;
#if SECMAI
    double x0, y0, x1, y1, x2, y2, x3, y3, xmin, xmax, ymin, ymax;
#else
    int    x0, y0, x1, y1, x2, y2, x3, y3, xmin, xmax, ymin, ymax;
#endif /* SECMAI */
    int    buffer_length;
    char   plotting;
    unsigned char *pixel_buffer, *p;

    if ((string.type != ps_t_stg) &&
        (string.flags & DYNAMIC)     && !REFS(string))
      ps_destroy_object(string, "image");
    image_matrix[0] = mat(state->CTM, 0)*a + mat(state->CTM, 2)*b;
    image_matrix[1] = mat(state->CTM, 1)*a + mat(state->CTM, 3)*b;
    image_matrix[2] = mat(state->CTM, 0)*c + mat(state->CTM, 2)*d;
    image_matrix[3] = mat(state->CTM, 1)*c + mat(state->CTM, 3)*d;
    image_matrix[4] = mat(state->CTM, 0)*e + mat(state->CTM, 2)*f
                    + mat(state->CTM, 4);
    image_matrix[5] = mat(state->CTM, 1)*e + mat(state->CTM, 3)*f
                    + mat(state->CTM, 5);
       /* *********************************************** *
        *              Dans le repere device :            *
        *           0                                     *
        *         0 0---------------1                     *
        *           |               |                     *
        *           |               |                     *
        *           |               |                     *
        *           |               |                     *
        *           2---------------3                     *
        * *********************************************** */
#if SECMAI
  /* ------------------------------------------------------------ *
   *  En version 4.00, et a cause des limitations presentes de la *
   * machine Secmai (mars 1990),  les images ne peuvent etre lues *
   * QUE si  leurs pixels sont carres.  Il en decoule  une limite *
   * quant aux matrices utilisees.                                *
   *   Celles-ci doivent etre necessairement etre de la forme :   *
   *    [ +-a   0 ]                  [   0 +-a ]                  *
   *    [   0 +-a ]        ou bien   [ +-a   0 ]                  *
   *    [  X    X ]                  [   X   Y ]                  *
   * ------------------------------------------------------------ */
    if (image_matrix[0]) { /* Forme [x 0   0 +-x X Y] ? */
      if ((image_matrix[1] != 0.0) ||
          (image_matrix[2] != 0.0) ||
          ((image_matrix[0] !=  image_matrix[3]) &&
           (image_matrix[0] != -image_matrix[3]))) {
        ps__puts("** Couldn't draw image (no square pixels) **");
        ps_free(image_buffer);
        image_buffer = NULL;
        return ps_e_operationok;
        }
      microdots = image_matrix[0];
      if (microdots < 0) microdots = -microdots;
      }
    else {                 /* Forme [0 x +-x   0 X Y] ? */
      if ((image_matrix[1] == 0.0) ||
          (image_matrix[3] != 0.0) ||
          ((image_matrix[1] !=  image_matrix[2]) &&
           (image_matrix[1] != -image_matrix[2]))) {
        ps__puts("** Couldn't draw image (no square pixels) **"); 
        ps_free(image_buffer);
        image_buffer = NULL;
        return ps_e_operationok;
        }
      microdots = image_matrix[1];
      if (microdots < 0) microdots = -microdots;
      }
    ps__printf("** Image %d x %d (%f) micro dots per dot\n",
               width, height, microdots);
#endif /* SECMAI */

    xmax = xmin = x0 = image_matrix[4];
    ymax = ymin = y0 = image_matrix[5];
    if ((x1=image_matrix[0]*width+x0)  > xmax) xmax=x1; if(x1<xmin) xmin=x1;
    if ((y1=image_matrix[1]*width+y0)  > ymax) ymax=y1; if(y1<ymin) ymin=y1;
    if ((x2=image_matrix[2]*height+x0) > xmax) xmax=x2; if(x2<xmin) xmin=x2;
    if ((y2=image_matrix[3]*height+y0) > ymax) ymax=y2; if(y2<ymin) ymin=y2;
    if ((x3 = x1+x2-x0) > xmax) xmax = x3; if (x3 < xmin) xmin = x3;
    if ((y3 = y1+y2-y0) > ymax) ymax = y3; if (y3 < ymin) ymin = y3;
    matrix_det = image_matrix[0]*image_matrix[3] 
               - image_matrix[1]*image_matrix[2];
    a =   image_matrix[3]/matrix_det;
    b =  -image_matrix[1]/matrix_det;
    c =  -image_matrix[2]/matrix_det;
    d =   image_matrix[0]/matrix_det;
    e =  (image_matrix[2]*image_matrix[5]
         -image_matrix[3]*image_matrix[4])/matrix_det;
    f =  (image_matrix[1]*image_matrix[4]
         -image_matrix[0]*image_matrix[5])/matrix_det;
    if (image_mask) {                        /* Cas de l'operateur imagemask */
#if SECMAI
      for (x0 = xmin; x0 <= xmax; x0 += microdots)
#else
      for (x0 = xmin; x0 <= xmax; x0++)
#endif /* SECMAI */
        {
#if SECMAI
        for (buffer_length = 0, y0 = ymin; y0 <= ymax; y0 += microdots)
#else
        for (buffer_length = 0, y0 = ymin; y0 <= ymax; y0++)
#endif /* SECMAI */
          {
          if ((((x3 = (int)(a*x0 + c*y0 + e)) < 0) || (x3 >= width) ||
               ((y3 = (int)(b*x0 + d*y0 + f)) < 0) || (y3 >= height))) {
            if (buffer_length) break;
            }
          else {                                 /* Nous sommes dans l'image */
            if (image_buffer[(int)(y3*width+x3)]) { /* Point a conserver     */
              if (buffer_length) buffer_length++;
              else {
                x1 = x0;
                y1 = y0;
                buffer_length = 1;
                }
              }
            else
              if (buffer_length) {
                ps__WritePixels((int)x1, (int)y1, buffer_length, NULL);
                buffer_length = 0;
                }
            }
          } /* Fin du for y */
        if (buffer_length) ps__WritePixels((int)x1, (int)y1,
                                           buffer_length, NULL);
        } /* Fin du for x */
      ps__WritePixels(0,0,0,NULL);                   /* To say it's finished */
      }
    else {
      if (!(pixel_buffer = (unsigned char *)ps_alloc(ymax-ymin+1))) {
        ps_free(image_buffer);
        image_buffer = NULL;
        return ps_e_VMerror;
        }
#if SECMAI
      for (x0 = xmin; x0 <= xmax; x0 += microdots)
#else
      for (x0 = xmin; x0 <= xmax; x0++)
#endif /* SECMAI */
        {
        plotting      = 0;
        p             = pixel_buffer;
#if SECMAI
        for (buffer_length = 0, y0 = ymin; y0 <= ymax; y0 += microdots)
#else
        for (buffer_length = 0, y0 = ymin; y0 <= ymax; y0++)
#endif /* SECMAI */
          {
          if (((x3 = (int)(a*x0 + c*y0 + e))< 0) || (x3 >= width) ||
              ((y3 = (int)(b*x0 + d*y0 + f))< 0) || (y3 >= height)) {
            if (plotting) break;                       /* On sort de l'image */
            }
          else if (!plotting) {                     /* On entre dans l'image */
            plotting = 1;
            x1 = x0;
            y1 = y0;
            }
          if (plotting) {
            *p++ = image_buffer[(int)(y3*width+x3)];
            buffer_length++;
            }
          }
        if (buffer_length) ps__WritePixels((int)x1,(int)y1,
                                           buffer_length, pixel_buffer);
        }
      ps__WritePixels(0,0,0,NULL);                   /* To say it's finished */
#if SECMAI
      microdots = 0;
#endif /* SECMAI */
      ps_free(pixel_buffer);
      }
    ps_free(image_buffer);
    image_buffer = NULL;
    }
  return ps_e_operationok;
  }

/* ************************************************************************* */
void
init_image_stuff()
{
  new_operator("image",     5, 0, "i.i.i.a.x",   ps__image);
  new_operator("imagemask", 5, 0, "i.i.b.a.x",   ps__imagemask);
  image_object = new_operator(" image", 1, 0, "s", ps___image);
  }
