/* ************************************************************************* *
 * PostScript Interpretor                   Fabien LELAQUAIS                 *
 *                                                                           *
 *   Fichier gstate.c                                                        *
 *        Graphic state management routines for PSint.                       *
 *                           Version 2.00 on 16/02/88                        *
 * ************************************************************************* *
 *    This document may be distributed, used, or modified, but can NOT be    *
 *  sold nor incorporated in any way in any product.                         *
 *    Permission is granted to distribute modified versions of that software *
 *  under the condition that this notice remains in every source file.       *
 *    Every alteration of the original files should be marked as such.       *
 *    No warranty is assumed by the author on the concequencies of the use   *
 *  of this software. Any defection of this program is at your own risk,     *
 *  you have to assume the cost of any service, installation or repairs      *
 *  this program could generate.                                             *
 *                                                                           *
 *                          Fabien LELAQUAIS - ESIEE - lelaquaf@apo.esiee.fr *
 * ************************************************************************* */
#include "int.h"
#include "graph.h"

#define COLORVERSION  /* Not operationnal */

extern void free_path();
ps__state  state;
ps__object default_matrix;

typedef struct _ps__matrix_t {
  ps__object            matrix;
  struct _ps__matrix_t *next;
  } ps__matrix_t, *ps__matrix_p;
ps__matrix_p free_matrixes = NULL,
             new_matrixes  = NULL;

/* ************************************************************************* */
ps__object
new_matrix()
{
  ps__object matrix;
  return matrix;
  }

/* ************************************************************************* */
void
free_matrix()
{
  }

/* ************************************************************************* */
is_a_possible_matrix(ps__object object)
{
  return ((object.type == ps_t_array) && (object.size == 6));
  }

/* ************************************************************************* */
is_a_matrix(ps__object object)
{
  int      loop;
  ps__type type;

  if (!is_a_possible_matrix(object)) return FALSE;
  for (loop=0; loop<6; loop++)
    if (((type = arr_ob(object, loop).type) != ps_t_int) &&
        (type != ps_t_real)) return FALSE;
  return TRUE;
  }

/* ************************************************************************* */
ps__object
do_matrix(ps__object m, double a, double b,
                        double c, double d,
                        double e, double f)
{
  arr_ob(m, 0) = new_real(a);  arr_ob(m, 1) = new_real(b);
  arr_ob(m, 2) = new_real(c);  arr_ob(m, 3) = new_real(d);
  arr_ob(m, 4) = new_real(e);  arr_ob(m, 5) = new_real(f);
  return m;
  }

/* ************************************************************************* */
void
new_state()
{
  ps__state new = (ps__state)ps_alloc(sizeof(ps__state_t));
  new->next = state;
  state = new;
  }

/* ************************************************************************* */
void
do_concat(ps__object dest, ps__object source,
          double a, double b,
          double c, double d,
          double e, double f)
{
  double aa, bb, cc, dd;
  aa = mat(source, 0)*a + mat(source, 1)*c;
  bb = mat(source, 0)*b + mat(source, 1)*d;
  cc = mat(source, 2)*a + mat(source, 3)*c;
  dd = mat(source, 2)*b + mat(source, 3)*d;
  arr_ob(dest, 4) = new_real(mat(source, 4)*a + mat(source, 5)*c + e);
  arr_ob(dest, 5) = new_real(mat(source, 4)*b + mat(source, 5)*d + f);
  arr_ob(dest, 0) = new_real(aa);
  arr_ob(dest, 1) = new_real(bb);
  arr_ob(dest, 2) = new_real(cc);
  arr_ob(dest, 3) = new_real(dd);
  }

/* ************************************************************************* */
ps__coord
user_to_device(ps__object m, ps__coord user)
{
 ps__coord device;
 device.x = user.x*mat(m, 0)+user.y*mat(m,2)+ mat(m,4);
 device.y = user.x*mat(m, 1)+user.y*mat(m,3)+ mat(m,5);
 return device;
 }

/* ************************************************************************* */
ps__coord
duser_to_device(ps__object m, ps__coord user)
{
 ps__coord device;
 device.x = user.x*mat(m, 0)+user.y*mat(m,2);
 device.y = user.x*mat(m, 1)+user.y*mat(m,3);
 return device;
 }

/* ************************************************************************* */
ps__coord
device_to_user(ps__coord device, ps__object matrix, int *ok)
{
 ps__coord user;
 double det;
#if AMIGA
 double t0 = mat(matrix, 0),
        t1 = mat(matrix, 1),
        t2 = mat(matrix, 2),
        t3 = mat(matrix, 3); 
#endif /* AMIGA */

 if (ok) *ok = 0;
#if AMIGA
 det = t0*t3 - t1*t2;
 if (det != 0.0) {
   user.x = (device.x*t3 - device.y*t2 +
             t2*mat(matrix, 5)-t3*mat(matrix, 4))/det;
   user.y = (device.y*t0 - device.x*t1 + 
             t1*mat(matrix, 4)-t0*mat(matrix, 5))/det;
   if (ok) *ok = 1;
   }
#else
 det = mat(matrix, 0)*mat(matrix, 3) - mat(matrix, 1)*mat(matrix, 2);
 if (det != 0.0) {
   user.x = (device.x*mat(matrix, 3) - device.y*mat(matrix, 2) +
             mat(matrix, 2)*mat(matrix, 5)-mat(matrix, 3)*mat(matrix, 4))/det;
   user.y = (device.y*mat(matrix, 0) - device.x*mat(matrix, 1) + 
             mat(matrix, 1)*mat(matrix, 4)-mat(matrix, 0)*mat(matrix, 5))/det;
   if (ok) *ok = 1;
   }
#endif /* AMIGA */
 return user;
 }

/* ************************************************************************* */
ps__coord
ddevice_to_user(ps__coord device, ps__object matrix, int *ok)
{
 ps__coord user;
 double det;
#if AMIGA
 double t0 = mat(matrix, 0),
        t1 = mat(matrix, 1),
        t2 = mat(matrix, 2),
        t3 = mat(matrix, 3); 
#endif /* AMIGA */

 if (ok) *ok = 0;
#if AMIGA
 det = t0*t3 - t1*t2;
 if (det != 0.0) {
   user.x = (device.x*t3 - device.y*t2)/det;
   user.y = (device.y*t0 - device.x*t1)/det;
   if (ok) *ok = 1;
   }
#else
 det = mat(matrix, 0)*mat(matrix, 3) - mat(matrix, 1)*mat(matrix, 2);
 if (det != 0.0) {
   user.x = (device.x*mat(matrix, 3) - device.y*mat(matrix, 2))/det;
   user.y = (device.y*mat(matrix, 0) - device.x*mat(matrix, 1))/det;
   if (ok) *ok = 1;
   }
#endif /* AMIGA */
 return user;
 }

/* -------------------------***********************------------------------- */
ps__errors
ps__initgraphics()
{
  extern ps__path copy_path();
  extern void     free_show();
  short           i;

  free_path(state->path);
  free_path(state->clip);
  state->path   = NULL;
  state->pqueue = NULL;
  state->clip   = copy_path(state->default_clip);
  free_show(state->show);
  state->show   = NULL;
  CHECK_DESTROY(state->CTM, "initgraphics-CTM");
  state->CTM = new_array(6);
  MORE_REFS(state->CTM);
  for (i=0; i<6; i++)
    arr_ob(state->CTM, i) = arr_ob(default_matrix, i);
  state->color.r    = 0;
  state->color.g    = 0;
  state->color.b    = 0;
  state->linewidth  = new_real(1.0);
  state->linecap    = new_int(CAP_BUTT);
  state->linejoin   = new_int(JOIN_MITER);
  state->flatness   = new_real(1.0);
  state->miterlimit = new_real(10.0);
  CHECK_DESTROY(state->dasharray, "initgraphics-dasharray");
  state->dasharray  = new_array(0);
  MORE_REFS(state->dasharray);
  state->dashoffset = new_int(0);
  state->gray       = new_int(0);
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__matrix()
{
  return PUSH(opstack, do_matrix(new_array(6), 1.0, 0.0, 0.0, 1.0, 0.0, 0.0));
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__defaultmatrix(ps__object matrix)
{
  short i;

  if (!is_a_possible_matrix(matrix)) return ps_e_typecheck;
  for (i=0; i<6; i++)
    arr_ob(matrix, i) = arr_ob(default_matrix, i);
  return PUSH(opstack, matrix);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__currentmatrix(ps__object matrix)
{
  short loop;

  if (!is_a_possible_matrix(matrix)) return ps_e_typecheck;
  for (loop = 0; loop < 6; loop++)
    arr_ob(matrix, loop) = arr_ob(state->CTM, loop);
  return PUSH(opstack, matrix);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__concat(ps__object m)
{
  if (!is_a_matrix(m)) return ps_e_typecheck;
  do_concat(state->CTM, m,
      mat(state->CTM, 0), mat(state->CTM, 1),
      mat(state->CTM, 2), mat(state->CTM, 3),
      mat(state->CTM, 4), mat(state->CTM, 5));
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__concatmatrix(ps__object m1, ps__object m2, ps__object m3)
{
  if (!is_a_matrix(m1) || !is_a_matrix(m2) || !is_a_possible_matrix(m3))
    return ps_e_typecheck;
  do_concat(m3, m1,  mat(m2, 0), mat(m2, 1), mat(m2, 2),
                     mat(m2, 3), mat(m2, 4), mat(m2, 5));
  return PUSH(opstack, m3);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__identmatrix(ps__object matrix)
{
  arr_ob(matrix, 0) = new_real(1.0);  arr_ob(matrix, 1) = new_real(0.0);
  arr_ob(matrix, 2) = new_real(0.0);  arr_ob(matrix, 3) = new_real(1.0);
  arr_ob(matrix, 4) = new_real(0.0);  arr_ob(matrix, 5) = new_real(0.0);
  return PUSH(opstack, matrix);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__setmatrix(ps__object matrix)
{
  arr_ob(state->CTM, 0) = arr_ob(matrix, 0);
  arr_ob(state->CTM, 1) = arr_ob(matrix, 1);
  arr_ob(state->CTM, 2) = arr_ob(matrix, 2);
  arr_ob(state->CTM, 3) = arr_ob(matrix, 3);
  arr_ob(state->CTM, 4) = arr_ob(matrix, 4);
  arr_ob(state->CTM, 5) = arr_ob(matrix, 5);
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__invertmatrix(ps__object source, ps__object destination)
{
#if AMIGA
  double t0  = mat(source, 0),
         t1  = mat(source, 1),
         t2  = mat(source, 2),
         t3  = mat(source, 3),
         det = t0*t3-t1*t2;
#else
  double det = mat(source, 0)*mat(source, 3) - mat(source, 1)*mat(source, 2);
#endif

  if (det == 0.0) return ps_e_undefinedresult;
#if AMIGA
  arr_ob(destination, 0) = new_real( t3/det);
  arr_ob(destination, 1) = new_real(-t1/det);
  arr_ob(destination, 2) = new_real(-t2/det);
  arr_ob(destination, 3) = new_real( t0/det);
  arr_ob(destination, 4) = new_real((t2*mat(source, 5)-
                                     t3*mat(source, 4))/det);
  arr_ob(destination, 5) = new_real((t1*mat(source, 4)-
                                     t0*mat(source, 5))/det);
#else
  arr_ob(destination, 0) = new_real( mat(source, 3)/det);
  arr_ob(destination, 1) = new_real(-mat(source, 1)/det);
  arr_ob(destination, 2) = new_real(-mat(source, 2)/det);
  arr_ob(destination, 3) = new_real( mat(source, 0)/det);
  arr_ob(destination, 4) = new_real((mat(source, 2)*mat(source, 5)-
                                     mat(source, 3)*mat(source, 4))/det);
  arr_ob(destination, 5) = new_real((mat(source, 1)*mat(source, 4)-
                                     mat(source, 0)*mat(source, 5))/det);
#endif /* AMIGA */
  return PUSH(opstack, destination);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__initmatrix()
{
  arr_ob(state->CTM, 0) = new_real(mat(default_matrix, 0));
  arr_ob(state->CTM, 1) = new_real(mat(default_matrix, 1));
  arr_ob(state->CTM, 2) = new_real(mat(default_matrix, 2));
  arr_ob(state->CTM, 3) = new_real(mat(default_matrix, 3));
  arr_ob(state->CTM, 4) = new_real(mat(default_matrix, 4));
  arr_ob(state->CTM, 5) = new_real(mat(default_matrix, 5));
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__rotate(ps__object angle)
{
  double sin(), cos(), m0, m1, c, s;

  if (is_a_matrix(angle)) {
    ps__object matrix;

    matrix = angle;
    if (!opstack->size) return ps_e_stackunderflow;
    angle = POP(opstack);
    if ((angle.type != ps_t_real) && (angle.type != ps_t_int)) {
      PUSH(opstack, angle);
      return ps_e_typecheck;
      }
    arr_ob(matrix, 2) =
    arr_ob(matrix, 0) = new_real(cos(degrad(to_real(angle))));
    arr_ob(matrix, 1) = new_real(s  = sin(degrad(to_real(angle))));
    arr_ob(matrix, 2) = new_real(-s);
    return PUSH(opstack, matrix);
    }
  else {
    if ((angle.type != ps_t_real) && (angle.type != ps_t_int))
      return ps_e_typecheck;
    c  = cos(degrad(to_real(angle)));
    s  = sin(degrad(to_real(angle)));
    m0 = mat(state->CTM, 0)*c + mat(state->CTM, 2)*s;
    m1 = mat(state->CTM, 1)*c + mat(state->CTM, 3)*s;
    arr_ob(state->CTM, 2) = new_real(mat(state->CTM, 2)*c -
                                     mat(state->CTM, 0)*s);
    arr_ob(state->CTM, 3) = new_real(mat(state->CTM, 3)*c -
                                     mat(state->CTM, 1)*s);
    arr_ob(state->CTM, 0) = new_real(m0);
    arr_ob(state->CTM, 1) = new_real(m1);
    return ps_e_operationok;
    }
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__scale(ps__object sx, ps__object sy)
{
  if (is_a_matrix(sy)) {
    ps__object matrix;

    matrix = sy;
    if (!opstack->size) return ps_e_stackunderflow;
    sy = sx;
    sx = POP(opstack);
    if ((sx.type != ps_t_real) && (sx.type != ps_t_int)) {
      PUSH(opstack, sx);
      return ps_e_typecheck;
      }
    arr_ob(matrix, 0) = sx;
    arr_ob(matrix, 3) = sy;
    arr_ob(matrix, 1) = arr_ob(matrix, 2) = 
    arr_ob(matrix, 4) = arr_ob(matrix, 5) = new_int(0);
    return PUSH(opstack, matrix);
    }
  else {
    if ((sy.type != ps_t_real) && (sy.type != ps_t_int))
      return ps_e_typecheck;
    arr_ob(state->CTM, 0) = new_real(to_real(sx)*
                            to_real(arr_ob(state->CTM, 0)));
    arr_ob(state->CTM, 1) = new_real(to_real(sx)*
                            to_real(arr_ob(state->CTM, 1)));
    arr_ob(state->CTM, 2) = new_real(to_real(sy)*
                            to_real(arr_ob(state->CTM, 2)));
    arr_ob(state->CTM, 3) = new_real(to_real(sy)*
                            to_real(arr_ob(state->CTM, 3)));
    return ps_e_operationok;
    }
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__translate(ps__object tx, ps__object ty)
{
  ps__coord t;

  if (is_a_matrix(ty)) {
    ps__object matrix;

    matrix = ty;
    if (!opstack->size) return ps_e_stackunderflow;
    ty = tx;
    tx = POP(opstack);
    if ((tx.type != ps_t_real) && (tx.type != ps_t_int)) {
      PUSH(opstack, tx);
      return ps_e_typecheck;
      }
    arr_ob(matrix, 4) = tx;
    arr_ob(matrix, 5) = ty;
    arr_ob(matrix, 0) = arr_ob(matrix, 1) = 
    arr_ob(matrix, 2) = arr_ob(matrix, 3) = new_int(0);
    return PUSH(opstack, matrix);
    }
  else {
    if ((ty.type != ps_t_real) && (ty.type != ps_t_int))
      return ps_e_typecheck;
    t.x = to_real(tx);
    t.y = to_real(ty);
    t = duser_to_device(state->CTM, t);
    arr_ob(state->CTM, 4) = new_real(mat(state->CTM, 4)+t.x);
    arr_ob(state->CTM, 5) = new_real(mat(state->CTM, 5)+t.y);
    return ps_e_operationok;
    }
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__transform(ps__object x, ps__object y)
{
  ps__coord us, dev;
  ps__object matrix;

  if (is_a_matrix(y)) {
    matrix = y;
    if (!opstack->size) return ps_e_stackunderflow;
    y = x;
    x = POP(opstack);
    if ((x.type != ps_t_real) && (x.type != ps_t_int)) {
      PUSH(opstack, x);
      return ps_e_typecheck;
      }
    }
  else {
    if ((y.type != ps_t_real) && (y.type != ps_t_int))
      return ps_e_typecheck;
    matrix = state->CTM;
    }
  us.x = to_real(x);
  us.y = to_real(y);
  dev = user_to_device(matrix, us);
  PUSH(opstack, new_real(dev.x));
  return PUSH(opstack, new_real(dev.y));
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__dtransform(ps__object x, ps__object y)
{
  ps__coord us, dev;
  ps__object matrix;

  if (is_a_matrix(y)) {
    matrix = y;
    if (!opstack->size) return ps_e_stackunderflow;
    y = x;
    x = POP(opstack);
    if ((x.type != ps_t_real) && (x.type != ps_t_int)) {
      PUSH(opstack, x);
      return ps_e_typecheck;
      }
    }
  else {
    if ((y.type != ps_t_real) && (y.type != ps_t_int))
      return ps_e_typecheck;
    matrix = state->CTM;
    }
  us.x = to_real(x);
  us.y = to_real(y);
  dev = duser_to_device(matrix, us);
  PUSH(opstack, new_real(dev.x));
  return PUSH(opstack, new_real(dev.y));
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__itransform(ps__object x, ps__object y)
{
  ps__coord  dev, us;
  ps__object matrix;
  int        possible;

  if (is_a_matrix(y)) {
    matrix = y;
    if (!opstack->size) return ps_e_stackunderflow;
    y = x;
    x = POP(opstack);
    if ((x.type != ps_t_real) && (x.type != ps_t_int)) {
      PUSH(opstack, x);
      return ps_e_typecheck;
      }
    }
  else {
    if ((y.type != ps_t_real) && (y.type != ps_t_int))
      return ps_e_typecheck;
    matrix = state->CTM;
    }
  dev.x = to_real(x);
  dev.y = to_real(y);
  us = device_to_user(dev, matrix, &possible);
  if (!possible) return ps_e_undefinedresult;
  PUSH(opstack, new_real(us.x));
  return PUSH(opstack, new_real(us.y));
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__idtransform(ps__object x, ps__object y)
{
  ps__coord  dev, us;
  ps__object matrix;
  int        possible;

  if (is_a_matrix(y)) {
    matrix = y;
    if (!opstack->size) return ps_e_stackunderflow;
    y = x;
    x = POP(opstack);
    if ((x.type != ps_t_real) && (x.type != ps_t_int)) {
      PUSH(opstack, x);
      return ps_e_typecheck;
      }
    }
  else {
    if ((y.type != ps_t_real) && (y.type != ps_t_int))
      return ps_e_typecheck;
    matrix = state->CTM;
    }
  dev.x = to_real(x);
  dev.y = to_real(y);
  dev.x = to_real(x);
  dev.y = to_real(y);
  us = ddevice_to_user(dev, matrix, &possible);
  if (!possible) return ps_e_undefinedresult;
  PUSH(opstack, new_real(us.x));
  return PUSH(opstack, new_real(us.y));
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__currentdash()
{
  PUSH(opstack, state->dasharray);
  return PUSH(opstack, state->dashoffset);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__setdash(ps__object array, ps__object offset)
{
  CHECK_DESTROY(state->dasharray, "setdash");
  state->dasharray  = array;
  MORE_REFS(array);
  state->dashoffset = offset;
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__currentflat()
{
  return PUSH(opstack, state->flatness);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__setflat(ps__object flat)
{
  state->flatness = flat;
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__currentgray()
{
  return PUSH(opstack, state->gray);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__setgray(ps__object gray)
{
  state->gray = gray;
  return ps_e_operationok;
  }

#ifdef COLORVERSION
/* -------------------------***********************------------------------- */
ps__errors
ps__currenthsbcolor()
{
  PUSH(opstack, new_real(0));
  PUSH(opstack, new_real(0));
  return PUSH(opstack, state->gray);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__sethsbcolor(ps__object h, ps__object s, ps__object b)
{
  state->gray = b;
  return ps_e_operationok;
  }
#endif /* COLORVERSION */

/* -------------------------***********************------------------------- */
ps__errors
ps__currentrgbcolor()
{
  PUSH(opstack, new_real(state->color.r));
  PUSH(opstack, new_real(state->color.g));
  return PUSH(opstack, new_real(state->color.b));
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__setrgbcolor(ps__object r, ps__object g, ps__object b)
{
  state->color.r = to_real(r);
  state->color.g = to_real(g);
  state->color.b = to_real(b);
  state->gray = new_real((to_real(r)+to_real(g)+to_real(b)) / 3);
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__currentlinecap()
{
  return PUSH(opstack, state->linecap);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__setlinecap(ps__object linecap)
{
  state->linecap = linecap;
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__currentlinejoin()
{
  return PUSH(opstack, state->linejoin);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__setlinejoin(ps__object linejoin)
{
  state->linejoin = linejoin;
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__currentlinewidth()
{
  return PUSH(opstack, state->linewidth);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__setlinewidth(ps__object linewidth)
{
  state->linewidth = linewidth;
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__currentmiterlimit()
{
  return PUSH(opstack, state->miterlimit);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__setmiterlimit(ps__object miterlimit)
{
  state->miterlimit = miterlimit;
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__currenttransfer()
{
  return PUSH(opstack, state->transferproc);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__settransfer(ps__object proc)
{
  if (!HAS_X(proc)) return ps_e_invalidaccess;
  CHECK_DESTROY(state->transferproc, NULL);
  state->transferproc = proc;
  MORE_REFS(proc);
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__currentscreen()
{
  PUSH(opstack, new_int(0));
  PUSH(opstack, new_int(0));
  return PUSH(opstack, cvx(new_array(0)));
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__setscreen(ps__object freq, ps__object angle, ps__object proc)
{
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__currentpoint()
{
  ps__coord  user;
  int        possible;

  if (!state->path) return ps_e_nocurrentpoint;
  user = device_to_user(state->cp, state->CTM, &possible);
  if (!possible) return ps_e_undefinedresult;
  PUSH(opstack, new_real(user.x));
  return PUSH(opstack, new_real(user.y));
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__nulldevice()
{
  if (state->device) ps__KillWindow(state->device);
  state->device = NULL;
  arr_ob(default_matrix, 0) = new_real(1.0);
  arr_ob(default_matrix, 1) = new_real(0.0);
  arr_ob(default_matrix, 2) = new_real(0.0);
  arr_ob(default_matrix, 3) = new_real(1.0);
  arr_ob(default_matrix, 4) = new_real(0.0);
  arr_ob(default_matrix, 5) = new_real(0.0);
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__framedevice(ps__object matrix, ps__object width, ps__object height,
                ps__object proc)
{
  ps__path new;

  if (!(is_a_matrix(matrix))) return ps_e_typecheck;
  if (!HAS_R(matrix) || !HAS_X(proc)) return ps_e_invalidaccess;
  if (state->device) ps__KillWindow(state->device);
  arr_ob(default_matrix, 0) = arr_ob(matrix, 0);
  arr_ob(default_matrix, 1) = arr_ob(matrix, 1);
  arr_ob(default_matrix, 2) = arr_ob(matrix, 2);
  arr_ob(default_matrix, 3) = arr_ob(matrix, 3);
  arr_ob(default_matrix, 4) = arr_ob(matrix, 4);
  arr_ob(default_matrix, 5) = arr_ob(matrix, 5);
  state->device       = ps__CreateWindow(8*int_val(width), int_val(height));
  free_path(state->default_clip);
  state->default_clip = new = new_element(NULL, ps_gt_move);
#if SECMAI
#define OFFSET -32768
#else
#define OFFSET 0
#endif
  new->element.move.point.x = OFFSET;
  new->element.move.point.y = OFFSET;
  new = new_element(new, ps_gt_line);
  new->element.move.point.x = 8*int_val(width) - 1 + OFFSET;
  new->element.move.point.y =                        OFFSET;
  new = new_element(new, ps_gt_line);
  new->element.move.point.x = 8*int_val(width)  - 1 + OFFSET;
  new->element.move.point.y =   int_val(height) - 1 + OFFSET;
  new = new_element(new, ps_gt_line);
  new->element.move.point.x =                       OFFSET;
  new->element.move.point.y = int_val(height) - 1 + OFFSET;
#undef OFFSET
  new_element(new, ps_gt_close);
  CHECK_DESTROY(state->frameproc, "framedevice");
  state->frameproc    = proc;
  MORE_REFS(proc);
  return ps_e_operationok;
  }

/* ************************************************************************* */
void
init_gstate_stuff()
{
  extern void new_state();

  default_matrix = new_array(6);
  new_state();
  state->CTM = null_object;
  ps__nulldevice();
  state->font         = invalid_object;
  state->path         = NULL;
  state->clip         = NULL;
  state->default_clip = NULL;
  state->show         = NULL;
  state->device       = NULL;
  state->frameproc    = cvx(new_array(0));
  MORE_REFS(state->frameproc);
  state->transferproc = cvx(new_array(0));
  MORE_REFS(state->transferproc);
  ps__initgraphics();

  new_operator("initgraphics",      0, 0, NONE,         ps__initgraphics);
  new_operator("currentpoint",      0, 2, NONE,         ps__currentpoint);
  new_operator("matrix",            0, 1, NONE,         ps__matrix);
  new_operator("defaultmatrix",     1, 1, "a",          ps__defaultmatrix);
  new_operator("currentmatrix",     1, 1, NONE,         ps__currentmatrix);
  new_operator("concat",            1, 0, "a",          ps__concat);
  new_operator("concatmatrix",      3, 1, "a.a.a",      ps__concatmatrix);
  new_operator("identmatrix",       1, 1, "a",          ps__identmatrix);
  new_operator("initmatrix",        0, 0, NONE,         ps__initmatrix);
  new_operator("invertmatrix",      2, 1, "a.a",        ps__invertmatrix);
  new_operator("rotate",            1, 0, "x",          ps__rotate);
  new_operator("scale",             2, 0, "ir.x",       ps__scale);
  new_operator("translate",         2, 0, "ir.x",       ps__translate);
  new_operator("setmatrix",         1, 0, "a",          ps__setmatrix);
  new_operator("transform",         2, 2, "ir.x",       ps__transform);
  new_operator("dtransform",        2, 2, "ir.x",       ps__dtransform);
  new_operator("itransform",        2, 2, "ir.x",       ps__itransform);
  new_operator("idtransform",       2, 2, "ir.x",       ps__idtransform);
  new_operator("currentdash",       0, 2, NONE,         ps__currentdash);
  new_operator("setdash",           2, 0, "a.ir",       ps__setdash);
  new_operator("currentflat",       0, 1, NONE,         ps__currentflat);
  new_operator("setflat",           1, 0, "ir",         ps__setflat);
  new_operator("currentgray",       0, 1, NONE,         ps__currentgray);
  new_operator("setgray",           1, 0, "ir",         ps__setgray);
#ifdef COLORVERSION
  new_operator("currenthsbcolor",   0, 3, NONE,         ps__currenthsbcolor);
  new_operator("sethsbcolor",       3, 0, "ir.ir.ir",   ps__sethsbcolor);
#endif /* COLORVERSION */
  new_operator("currentrgbcolor",   0, 1, NONE,         ps__currentrgbcolor);
  new_operator("setrgbcolor",       3, 0, "ir.ir.ir",   ps__setrgbcolor);
  new_operator("currentlinecap",    0, 1, NONE,         ps__currentlinecap);
  new_operator("setlinecap",        1, 0, "i",          ps__setlinecap);
  new_operator("currentlinejoin",   0, 1, NONE,         ps__currentlinejoin);
  new_operator("setlinejoin",       1, 0, "i",          ps__setlinejoin);
  new_operator("currentlinewidth",  0, 1, NONE,         ps__currentlinewidth);
  new_operator("setlinewidth",      1, 0, "ir",         ps__setlinewidth);
  new_operator("currentmiterlimit", 0, 1, NONE,         ps__currentmiterlimit);
  new_operator("setmiterlimit",     1, 0, "ir",         ps__setmiterlimit);

  new_operator("currenttransfer",   0, 1, NONE,         ps__currenttransfer);
  new_operator("settransfer",       1, 0, "a",          ps__settransfer);
  new_operator("currentscreen",     0, 3, NONE,         ps__currentscreen);
  new_operator("setscreen",         3, 0, "ir.ir.a",    ps__setscreen);
  new_operator("nulldevice",        0, 0, NONE,         ps__nulldevice);
  new_operator("framedevice",       4, 0, "a.i.i.a",    ps__framedevice);
  }

/* ************************************************************************* */
void
free_gstate_stuff()
{
  extern void       free_show(), ps_free();
  extern ps__errors ps_grestore();

  while (state->next) ps__grestore();
  ps__newpath();
  free_path(state->clip);
  free_path(state->default_clip);
  free_show(state->show);
  if (state->device) ps__KillWindow(state->device);
  CHECK_DESTROY(state->CTM,          "free-gstate-stuff-CTM");
  CHECK_DESTROY(state->font,         "free-gstate-stuff-font");
  CHECK_DESTROY(state->dasharray,    "free-gstate-stuff-dasharray");
  CHECK_DESTROY(state->frameproc,    "free-gstate-stuff-frameproc");
  CHECK_DESTROY(state->transferproc, "free-gstate-stuff-transferproc");
  ps_free(state);
  ps_free(BLOCK(default_matrix));
  }

