/* ************************************************************************* *
 * PostScript Interpretor                   Fabien LELAQUAIS                 *
 *                                                                           *
 *   Fichier font.c                                                          *
 *       Fonts management routines for PSint.                                *
 *                           Version 2.00 on 16/02/88                        *
 * ************************************************************************* *
 *    This document may be distributed, used, or modified, but can NOT be    *
 *  sold nor incorporated in any way in any product.                         *
 *    Permission is granted to distribute modified versions of that software *
 *  under the condition that this notice remains in every source file.       *
 *    Every alteration of the original files should be marked as such.       *
 *    No warranty is assumed by the author on the concequencies of the use   *
 *  of this software. Any defection of this program is at your own risk,     *
 *  you have to assume the cost of any service, installation or repairs      *
 *  this program could generate.                                             *
 *                                                                           *
 *                          Fabien LELAQUAIS - ESIEE - lelaquaf@apo.esiee.fr *
 * ************************************************************************* */
#include "int.h"
#include "graph.h"

     /* MAX_FONTDIRECTORY_SIZE is defined in graph.h as it's used in cache.c */

ps__object FontDirectory, StandardEncoding, notdef,      FontMatrix,
           FontType,      FontBBox,         Encoding,    FontName,
           PaintType,     Metrics,          StrokeWidth, CharStrings,
           UniqueID, BuildChar;
ps__errors ps__gsave(), ps__grestore(), ps__newpath();

char *standard_encoding[] = {
  NULL,             NULL,            NULL,           NULL,
  NULL,             NULL,            NULL,           NULL,
  NULL,             NULL,            NULL,           NULL,
  NULL,             NULL,            NULL,           NULL,
  NULL,             NULL,            NULL,           NULL,
  NULL,             NULL,            NULL,           NULL,
  NULL,             NULL,            NULL,           NULL,
  NULL,             NULL,            NULL,           NULL,
  "space",          "exclam",        "quotedbl",     "numbersign",
  "dollar",         "percent",       "ampersand",    "quoteright",
  "parenleft",      "parenright",    "asterisk",     "plus",
  "comma",          "hyphen",        "period",       "slash",
  "zero",           "one",           "two",          "three",
  "four",           "five",          "six",          "seven",
  "eight",          "nine",          "colon",        "semicolon",
  "less",           "equal",         "greater",      "question",
  "at",             "A",             "B",            "C",
  "D",              "E",             "F",            "G",
  "H",              "I",             "J",            "K",
  "L",              "M",             "N",            "O",
  "P",              "Q",             "R",            "S",
  "T",              "U",             "V",            "W",
  "X",              "Y",             "Z",            "bracketleft",
  "backslash",      "bracketright",  "asciicircum",  "underscore",
  "quoteleft",      "a",             "b",            "c",
  "d",              "e",             "f",            "g",
  "h",              "i",             "j",            "k",
  "l",              "m",             "n",            "o",
  "p",              "q",             "r",            "s",
  "t",              "u",             "v",            "w",
  "x",              "y",             "z",            "braceleft",
 "bar",             "braceright",    "asciitilde",   NULL,
  NULL,             NULL,            NULL,           NULL,
  NULL,             NULL,            NULL,           NULL,
  NULL,             NULL,            NULL,           NULL,
  NULL,             NULL,            NULL,           NULL,
  NULL,             NULL,            NULL,           NULL,
  NULL,             NULL,            NULL,           NULL,
  NULL,             NULL,            NULL,           NULL,
  NULL,             NULL,            NULL,           NULL,
  NULL,            "exclamdown",     "cent",         "sterling",
  "fraction",      "yen",            "florin",       "section",
  "currency",      "quotesingle",    "quotedblleft", "guillemotleft",
  "guilsinglleft", "guilsinglright", "fi",           "fl",
  NULL,            "endash",         "dagger",       "daggerdbl",
  "periodcentered", NULL,            "paragraph",    "bullet",
  "quotesinglbase", "quotedblbase",  "quotedblright", "guillemotright",
  "ellipsis",      "perthousand",    NULL,           "questiondown",
  NULL,            "grave",          "acute",        "circumflex",
  "tilde",         "macron",         "breve",        "dotaccent",
  "dieresis",      NULL,             "ring",         "cedilla",
  NULL,            "hungarumlaut",   "ogonek",       "caron",
  "emdash",        NULL,             NULL,           NULL,
  NULL,            NULL,             NULL,           NULL,
  NULL,            NULL,             NULL,           NULL,
  NULL,            NULL,             NULL,           NULL,
  NULL,            "AE",             NULL,           "ordfeminine",
  NULL,            NULL,             NULL,           NULL,
  "Lslash",        "Oslash",         "OE",           "ordmasculine",
  NULL,            NULL,             NULL,           NULL,
  NULL,            "ae",             NULL,           NULL,
  NULL,            "dotlessi",       NULL,           NULL,
  "lslash",        "oslash",         "oe",           "germandbls",
  NULL,            NULL,             NULL,           NULL
  } ;

int      fid = 0;
ps__path charpath = NULL, charpath_queue = NULL;

#define USERFONT 3

/* ************************************************************************* */
void
free_show(ps__show_p show)
{
  extern void ps_free();
  ps__show_p  show1;

  while (show) {
    show1 = show->next;
    ps_free(show);
    show = show1;
    }
  }

#if 0
/* ************************************************************************* */
void
print_matrix(char      *t, ps__object m)
{
  short loop;
  ps__printf(t);
  ps__printf(" : [ ");
  for (loop=0; loop<6; loop++)
    ps__printf("%f ", mat(m, loop));
  ps__printf("]\n");
  }
#endif /* 0 */

/* ************************************************************************* */
is_a_font(ps__object font)
{
  ps__object object;

  object = dict_find_key(font, FontType);
  if ((object.type == ps_t_invalid) ||
      (object.type != ps_t_int)    ||
      (int_val(object) != USERFONT))   return FALSE;
  object = dict_find_key(font, FontMatrix);
  if ((object.type == ps_t_invalid) ||
      !is_a_matrix(object)) return FALSE;
  object = dict_find_key(font, FontBBox);
  if ((object.type == ps_t_invalid) ||
      (object.type != ps_t_array) ||
      (object.size != 4))   return FALSE;
  object = dict_find_key(font, Encoding);
  if ((object.type == ps_t_invalid) ||
      (object.type != ps_t_array)) return FALSE;
  object = dict_find_key(font, BuildChar);
  if (object.type == ps_t_invalid) return FALSE;
  return TRUE;            
  }

typedef struct _font {
  ps__object    font;
  struct _font *next;
  } font_struct, *font_desc;
font_desc known_fonts = NULL;
/* ************************************************************************* */
ps__object
new_font(ps__object font, double sx, double xy,
                          double yx, double sy,
                          double tx, double ty)
{
  extern void do_concat();
  ps__object  newmat, oldmat, orimat;
  int         oldid;
  font_desc   current, same = NULL;
  short       loop;

  oldid = font_val(dict_find_key(font, UniqueID));
  newmat = new_array(6);
  orimat = dict_find_key(font, FontMatrix);
  do_concat(newmat, orimat, sx, xy, yx, sy, tx, ty);
  for (current = known_fonts; current && !same; current = current->next)
    if (oldid ==  font_val(dict_find_key(current->font, UniqueID))) {
      oldmat = dict_find_key(current->font, FontMatrix);
      for (loop=0; loop<6; loop++)
        if (mat(newmat, loop) != mat(oldmat, loop)) break;
      if (loop == 6) same = current;
      }
  if (same)
    ps_destroy_object(newmat, "new_font");
  else {                                             /* Font must be created */
    same = (font_desc)ps_alloc(sizeof(font_struct));
    same->next = known_fonts;
    known_fonts   = same;
    same->font = new_dict(dict_val(font).size);
    MORE_REFS(same->font);
    for (loop = 0; loop<dict_val(font).size; loop++) {  /* Copy Font dict    */
      dict_val(same->font).keys[loop] = dict_val(font).keys[loop];
      MORE_REFS(dict_val(font).keys[loop].name);
      MORE_REFS(dict_val(font).keys[loop].object);
      }
    dict_val(same->font).size = loop;
    dict_store(same->font, FontMatrix, newmat);
    }
  return same->font;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__definefont(ps__object key, ps__object font)
{
  if (!is_a_font(font)) return ps_e_invalidfont;
  
  if (((dict_find_index(FontDirectory, font) == -1) &&
       (FontDirectory.size == dict_val(FontDirectory).size)) ||
      ((dict_find_index(font, UniqueID) == -1) &&
       (font.size == dict_val(font).size))) return ps_e_dictfull;
  dict_store(font,          UniqueID, new_int(fid++));
  dict_store(FontDirectory, key,      font);
  return PUSH(opstack, font);
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__findfont(ps__object key)
{
  ps__object font;

  font = dict_find_key(FontDirectory, key);
  switch (font.type) {
    case ps_t_invalid : return ps_e_invalidfont;
    case ps_t_dict    : return PUSH(opstack, font);
    default           : return PUSH(execstack, font);
    }
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__scalefont(ps__object font, ps__object size)
{
  if (!is_a_font(font)) return ps_e_invalidfont;
  return PUSH(opstack, new_font(font, to_real(size), 0.0, 0.0, to_real(size),
                                      0.0,                0.0));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__makefont(ps__object font, ps__object matrix)
{
#if AMIGA
  double t0 = mat(matrix, 0), t1 = mat(matrix, 1),
         t2 = mat(matrix, 2), t3 = mat(matrix, 3);
#endif

  if (!is_a_font(font))     return ps_e_invalidfont;
  if (!is_a_matrix(matrix)) return ps_e_typecheck;
#if AMIGA
  return PUSH(opstack, new_font(font, t0, t1, t2, t3,
                                      mat(matrix, 4), mat(matrix, 5)));
#else
  return PUSH(opstack, new_font(font, mat(matrix, 0), mat(matrix, 1),
                                      mat(matrix, 2), mat(matrix, 3),
                                      mat(matrix, 4), mat(matrix, 5)));
#endif /* AMIGA */
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__setfont(ps__object font)
{
  if (!is_a_font(font)) return ps_e_invalidfont;
  CHECK_DESTROY(state->font, "setfont");
  state->font = font;
  MORE_REFS(font);
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__currentfont()
{
  return (state->font.type == ps_t_invalid) ? ps_e_invalidfont:
                                              PUSH(opstack,state->font);
  }

/* ************************************************************************* *
 *    ShowChar draws a char using current font.                              *
 *  It is used for each show operator : show, ashow, widthshow, awidthshow,  *
 *  and kshow.                                                               *
 *   It pops from ExecStack :                                                *
 *    - The string to be shown                                               *
 *    - The extra spacing x and y to be used for all letters (0.0, 0.0)      *
 *    - an optional char code used by a- routines                            *
 *    - The extra spacing x and y to be used when printing this character    *
 *    - The kshow procedure (or null_object)                                 *
 *    The character is then printed thanks to BuildChar, and values are      *
 *   pushed again on the stack.                                              */
ps__object        showchar_object, showend_object, show_proc, show_stg;
ps__coord         showcursor, showwidth, extraspacing;
int               showcharcode;
extern ps__cache *currentcache;
extern int        lowcachex, lowcachey;
char              showing = 0;
ps__errors
ps___showchar()
{
  extern void do_concat();
  ps__object  string, cx, cy, code, character, ax, ay, proc;
  ps__cache  *c;

  show_proc = proc = POP(execstack);
  cy           = POP(execstack);
  cx           = POP(execstack);
  code         = POP(execstack);
  ay           = POP(execstack);
  ax           = POP(execstack);
  string       = POP(execstack);
  showcharcode = stg_val(string)[0];
  character    = new_int(showcharcode);
  string.size--;
  stg_val(string)++;
  show_stg = string;
  if (string.size) {
    PUSH(execstack, string);
    PUSH(execstack, ax); PUSH(execstack, ay);
    PUSH(execstack, code);
    PUSH(execstack, cx); PUSH(execstack, cy);
    PUSH(execstack, proc);
    PUSH(execstack, showchar_object);
    PUSH(execstack, showend_object);
    }
  else 
    if ((string.flags & DYNAMIC) && !REFS(string))
      ps_destroy_object(string, "showchar");
  if ((code.type == ps_t_int) &&
      (showcharcode == int_val(code))) {
    extraspacing.x = to_real(cx);
    extraspacing.y = to_real(cy);
    }
  else {
    extraspacing.x = to_real(ax);
    extraspacing.y = to_real(ay);
    }
  extraspacing = duser_to_device(state->CTM, extraspacing);
  ps__gsave();
  do_concat(state->CTM, dict_find_key(state->font, FontMatrix),
                        mat(state->CTM, 0), mat(state->CTM, 1),
                        mat(state->CTM, 2), mat(state->CTM, 3),
                        mat(state->CTM, 4), mat(state->CTM, 5));
  showcursor = state->cp;
  showing = 1;
  ps__newpath();
  arr_ob(state->CTM, 4) = new_real(showcursor.x);
  arr_ob(state->CTM, 5) = new_real(showcursor.y);
  if (!charpath && (c = char_is_in_cache())) {      /* Is character cached ? */
    if (state->device) ps__DoCache(c, showcursor);
    showwidth.x = c->wx;
    showwidth.y = c->wy;
    }
  else {                                           /* If not, call BuildChar */
    ps__newpath();
    PUSH(opstack,   state->font);
    PUSH(opstack,   character);
    PUSH(execstack, dict_find_key(state->font, BuildChar));
    }
  return ps_e_operationok;
  }

/* ************************************************************************* */
ps__errors
ps___showend()
{
  ps__cache *c;

  if (c = currentcache) {
    currentcache = NULL;
    ps__CloseCache(c);
    if (state->device) ps__DoCache(c, showcursor);
    }
  if (charpath) {
    if (state->path) {
      charpath_queue->next = state->path;
      charpath_queue       = state->pqueue;
      state->path = NULL;
      }
    }
  ps__grestore();
  state->cp.x = showcursor.x + showwidth.x + extraspacing.x;
  state->cp.y = showcursor.y + showwidth.y + extraspacing.y;
  if (!state->pqueue || state->pqueue->type != ps_gt_move)
    state->pqueue = new_element(state->pqueue, ps_gt_move);
  state->pqueue->element.move.point = state->cp;
  showing = 0;
  if ((show_proc.type != ps_t_null) && (show_stg.size))      /* For Kshow... */
    PUSH(execstack, show_proc);
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__setcharwidth(ps__object x, ps__object y)
{
  if (!showing) return ps_e_undefined;
  showwidth.x = to_real(x)*mat(state->CTM, 0)+
                to_real(y)*mat(state->CTM, 2);
  showwidth.y = to_real(x)*mat(state->CTM, 1)+
                to_real(y)*mat(state->CTM, 3);
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__show(ps__object string)
{
  if (!HAS_R(string))                   return ps_e_invalidaccess;
  if (state->font.type == ps_t_invalid) return ps_e_invalidfont;
  if (!state->path)                     return ps_e_nocurrentpoint;
  if (string.size) {
    PUSH(execstack, showend_object);
    PUSH(execstack, string);
    PUSH(execstack, new_real(0.0)); PUSH(execstack, new_real(0.0));
    PUSH(execstack, null_object);
    PUSH(execstack, new_real(0.0)); PUSH(execstack, new_real(0.0));
    PUSH(execstack, null_object);
    PUSH(execstack, showchar_object);
    }
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__ashow(ps__object ax, ps__object ay, ps__object string)
{
  if (!HAS_R(string))                   return ps_e_invalidaccess;
  if (state->font.type == ps_t_invalid) return ps_e_invalidfont;
  if (!state->path)                     return ps_e_nocurrentpoint;
  if (string.size) {
    PUSH(execstack, showend_object);
    PUSH(execstack, string);
    PUSH(execstack, new_real(to_real(ax)));
    PUSH(execstack, new_real(to_real(ay)));
    PUSH(execstack, null_object);
    PUSH(execstack, new_real(0.0)); PUSH(execstack, new_real(0.0));
    PUSH(execstack, null_object);
    PUSH(execstack, showchar_object);
    }
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__widthshow(ps__object cx, ps__object cy, ps__object c, ps__object string)
{
  if (!HAS_R(string))                   return ps_e_invalidaccess;
  if (state->font.type == ps_t_invalid) return ps_e_invalidfont;
  if (!state->path)                     return ps_e_nocurrentpoint;
  if (string.size) {
    PUSH(execstack, showend_object);
    PUSH(execstack, string);
    PUSH(execstack, new_real(0.0)); PUSH(execstack, new_real(0.0));
    PUSH(execstack, c);
    PUSH(execstack, new_real(to_real(cx)));
    PUSH(execstack, new_real(to_real(cy)));
    PUSH(execstack, null_object);
    PUSH(execstack, showchar_object);
    }
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__awidthshow(ps__object cx, ps__object cy,
               ps__object c,
               ps__object ax, ps__object ay, ps__object string)
{
  if (!HAS_R(string))                   return ps_e_invalidaccess;
  if (state->font.type == ps_t_invalid) return ps_e_invalidfont;
  if (!state->path)                     return ps_e_nocurrentpoint;
  if (string.size) {
    PUSH(execstack, showend_object);
    PUSH(execstack, string);
    PUSH(execstack, new_real(to_real(ax)));
    PUSH(execstack, new_real(to_real(ay)));
    PUSH(execstack, c);
    PUSH(execstack, new_real(to_real(cx)));
    PUSH(execstack, new_real(to_real(cy)));
    PUSH(execstack, null_object);
    PUSH(execstack, showchar_object);
    }
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__kshow(ps__object proc, ps__object string)
{
  if (!HAS_R(string))                   return ps_e_invalidaccess;
  if (state->font.type == ps_t_invalid) return ps_e_invalidfont;
  if (!state->path)                     return ps_e_nocurrentpoint;
  if (string.size) {
    PUSH(execstack, showend_object);
    PUSH(execstack, string);
    PUSH(execstack, new_real(0.0)); PUSH(execstack, new_real(0.0));
    PUSH(execstack, null_object);
    PUSH(execstack, new_real(0.0)); PUSH(execstack, new_real(0.0));
    PUSH(execstack, proc);
    PUSH(execstack, showchar_object);
    }
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
ps__object stringwidthend_object;
ps__path  oldpath, oldpqueue, oldclip;
char     *olddevice;
ps__coord oldcp, newcp;
ps__errors
ps___stringwidthend()
{
  extern void free_path();
  int         possible;

  newcp         = state->cp;
  free_path(state->path);
  state->path   = oldpath;
  state->pqueue = oldpqueue;
  free_path(state->clip);
  state->clip   = oldclip;
  state->device = olddevice;
  state->cp     = oldcp;
  newcp = device_to_user(newcp, state->CTM, &possible);
  PUSH(opstack, new_real(newcp.x));
  return PUSH(opstack, new_real(newcp.y));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__stringwidth(ps__object string)
{
  extern void     add_move();
  extern ps__path copy_path();

  if (!HAS_R(string))                   return ps_e_invalidaccess;
  if (state->font.type == ps_t_invalid) return ps_e_invalidfont;
  if (string.size) {
    oldpath   = state->path;
    oldclip   = state->clip;
    oldpqueue = state->pqueue;
    olddevice = state->device;
    oldcp     = state->cp;
    state->device = NULL;
    state->pqueue = state->path = new_element(NULL, ps_gt_move);
    add_move(state->path, 0.0, 0.0);
    state->clip = copy_path(state->default_clip);
    PUSH(execstack, stringwidthend_object);
    PUSH(execstack, showend_object);
    PUSH(execstack, string);
    PUSH(execstack, new_real(0.0)); PUSH(execstack, new_real(0.0));
    PUSH(execstack, new_int(0));
    PUSH(execstack, new_real(0.0)); PUSH(execstack, new_real(0.0));
    PUSH(execstack, null_object);
    PUSH(execstack, showchar_object);
    }
  else {
    PUSH(opstack, new_real(0.0));
    PUSH(opstack, new_real(0.0));
    }
  return ps_e_operationok;
  }

/* ************************************************************************* */
ps__object charpathend_object;
ps__errors
ps___charpathend()
{
  for (state->pqueue->next = charpath; state->pqueue->next;
                                       state->pqueue = state->pqueue->next);
  charpath = NULL;
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__charpath(ps__object string, ps__object bool)
{
  if (!state->path)                     return ps_e_nocurrentpoint;
  if (state->font.type == ps_t_invalid) return ps_e_invalidfont;
  charpath = charpath_queue = new_element(NULL, ps_gt_move);
  charpath->element.move.point = state->cp;
  PUSH(execstack, charpathend_object);
  if (bool_val(bool)) PUSH(execstack, cvx(do_name("strokepath")));
  PUSH(opstack,   string);
  return PUSH(execstack, cvx(do_name("show")));
  }

/* --------------------*********************************-------------------- */
#define HERSHEYMOVE 0x80
ps__errors
ps__hershey(ps__object font, ps__object charname, ps__object charstring)
{
  extern void add_line(), add_move();
  ps__object  metrics, move;
  unsigned    char op1, op2;
  double      x, y;
  ps__path    new;
  ps__errors  ps__stroke();

  metrics = dict_find_key(font, Metrics);
  if (metrics.type == ps_t_invalid)         return ps_e_invalidfont;
  move = dict_find_key(metrics, charname);
  if (move.type == ps_t_invalid)
    showwidth.x = showwidth.y = 0;
  switch (move.type) {
    case ps_t_int  :
    case ps_t_real :
      showwidth.x = to_real(move)*mat(state->CTM, 0);
      showwidth.y = to_real(move)*mat(state->CTM, 1);
      break;
    case ps_t_array :
      switch(move.size) {
        case 2 :                               
          showwidth.x = to_real(arr_ob(move, 1))*mat(state->CTM, 0);
          showwidth.y = to_real(arr_ob(move, 1))*mat(state->CTM, 1);
          break;
        case 4 :
          showwidth.x = to_real(arr_ob(move, 2))*mat(state->CTM, 0)+
                        to_real(arr_ob(move, 3))*mat(state->CTM, 2);
          showwidth.y = to_real(arr_ob(move, 2))*mat(state->CTM, 1)+
                        to_real(arr_ob(move, 3))*mat(state->CTM, 3);
          break;
        default : return ps_e_invalidfont;
        }
      break;
    default :
      return ps_e_invalidfont;
    }
  for (; charstring.size; charstring.size -= 2) {
    op1 = *stg_val(charstring)++;
    op2 = *stg_val(charstring)++;
    if (!state->path)
        state->path = state->pqueue =
                      new = new_element(NULL, (op1&HERSHEYMOVE)?
                                               ps_gt_move:ps_gt_line);
      else
        new = new_element(state->pqueue, (op1&HERSHEYMOVE)?
                                   ps_gt_move:ps_gt_line);
    x = (double)(int)((op1 & ~HERSHEYMOVE)-64);
    y = (double)(int)((op2 & ~HERSHEYMOVE)-73);
    (*((op1&HERSHEYMOVE)?add_move:add_line))(new, x, y);
    }
  return charpath?ps_e_operationok:ps__stroke();
  }

/* ************************************************************************* */
void
init_font_stuff()
{
  short loop;

  FontDirectory    = new_dict(MAX_FONTDIRECTORY_SIZE);
  dict_store(systemdict, do_name("FontDirectory"), FontDirectory);
  notdef = do_name(".notdef");
  StandardEncoding = new_array(256);
  for (loop=0; loop<256; loop++)
    arr_ob(StandardEncoding, loop) = standard_encoding[loop] ?
                                 do_name(standard_encoding[loop]) :
                                 notdef;
  dict_store(systemdict, do_name("StandardEncoding"), StandardEncoding);
  FontMatrix  = do_name("FontMatrix");
  FontType    = do_name("FontType");
  FontBBox    = do_name("FontBBox");
  Encoding    = do_name("Encoding");
  FontName    = do_name("FontName");
  PaintType   = do_name("PaintType");
  Metrics     = do_name("Metrics");
  StrokeWidth = do_name("StrokeWidth");
  CharStrings = do_name("CharStrings");
  UniqueID    = do_name("UniqueID");
  BuildChar   = do_name("BuildChar");
  new_operator("definefont",  2, 1, "x.d",  ps__definefont);
  new_operator("findfont",    1, 1, "x",    ps__findfont);
  new_operator("scalefont",   2, 1, "d.ir", ps__scalefont);
  new_operator("makefont",    2, 1, "d.a",  ps__makefont);
  new_operator("setfont",     1, 0, "d",    ps__setfont);
  new_operator("currentfont", 0, 1, NONE,   ps__currentfont);

  showchar_object = new_operator(" showchar", 0, 0, NONE, ps___showchar);
  showend_object  = new_operator(" showend",  0, 0, NONE, ps___showend);
  new_operator("show",       1, 0, "s",                ps__show);
  new_operator("ashow",      3, 0, "ir.ir.s",          ps__ashow);
  new_operator("awidthshow", 6, 0, "ir.ir.i.ir.ir.s",  ps__awidthshow);
  new_operator("kshow",      2, 0, "a.s",              ps__kshow);
  new_operator("widthshow",  4, 0, "ir.ir.i.s",        ps__widthshow);

  stringwidthend_object  = new_operator(" stringwidthend", 0, 0, NONE,
                                           ps___stringwidthend);
  new_operator("stringwidth",  1, 2, "s",      ps__stringwidth);
  new_operator("setcharwidth", 2, 0, "ir.ir",  ps__setcharwidth);

  new_operator("hershey",      3, 0, "d.n.s",  ps__hershey);

  charpathend_object  = new_operator(" charpathend", 0, 0, NONE,
                                     ps___charpathend);
  new_operator("charpath", 2, 0, "s.b", ps__charpath);
  }

/* ************************************************************************* */
void
free_font_stuff()
{
  extern void ps_free();
  font_desc   font, next;

  for (font=known_fonts; font; font=next) {
    next = font->next;
    CHECK_DESTROY(font->font, "free_font_stuff");
    ps_free(font);
    }
  }

