/* ************************************************************************* *
 * PostScript Interpretor                   Fabien LELAQUAIS                 *
 *                                                                           *
 *   Fichier fonc.c                                                          *
 *       PostScript functions implementation for PSint.                      *
 *                           Version 3.00 on 24/02/89                        *
 * ************************************************************************* *
 *    This document may be distributed, used, or modified, but can NOT be    *
 *  sold nor incorporated in any way in any product.                         *
 *    Permission is granted to distribute modified versions of that software *
 *  under the condition that this notice remains in every source file.       *
 *    Every alteration of the original files should be marked as such.       *
 *    No warranty is assumed by the author on the concequencies of the use   *
 *  of this software. Any defection of this program is at your own risk,     *
 *  you have to assume the cost of any service, installation or repairs      *
 *  this program could generate.                                             *
 *                                                                           *
 *                          Fabien LELAQUAIS - ESIEE - lelaquaf@apo.esiee.fr *
 * ************************************************************************* */
#include "int.h"

/* #define RAND_IS_16_BITS */

int randseed = 1;

/* --------------------*********************************-------------------- */
ps__errors
ps__length(ps__object object)
{
  if (!HAS_R(object)) return ps_e_invalidaccess;
  switch (object.type) {
    case ps_t_array :
    case ps_t_stg   : return PUSH(opstack, new_int(object.size));
    case ps_t_dict  : return PUSH(opstack, new_int(dict_val(object).size));
    }
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__maxlength(ps__object object)
{
  return HAS_R(object)?PUSH(opstack, new_int(object.size)):ps_e_invalidaccess;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__add(ps__object a, ps__object b)
{
  ps__object result;

  result = new_object(((a.type == ps_t_int) && (b.type == ps_t_int))?
                 ps_t_int:ps_t_real);
  if (result.type == ps_t_int)
    int_val(result) = int_val(a)+int_val(b);
  else
    real_val(result) = to_real(a) + to_real(b);
  return PUSH(opstack, result);
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__sub(ps__object a, ps__object b)
{
  ps__object result;

  result = new_object(((a.type == ps_t_int) && (b.type == ps_t_int))?
                 ps_t_int:ps_t_real);
  if (result.type == ps_t_int)
    int_val(result) = int_val(a)-int_val(b);
  else
    real_val(result) = to_real(a)-to_real(b);
  return PUSH(opstack, result);
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__mul(ps__object a, ps__object b)
{
  ps__object result;

  result = new_object(((a.type == ps_t_int) && (b.type == ps_t_int))?
                 ps_t_int:ps_t_real);
  if (result.type == ps_t_int)
    int_val(result) = int_val(a)*int_val(b);
  else
    real_val(result) = to_real(a)*to_real(b);
  return PUSH(opstack, result);
 }

/* --------------------*********************************-------------------- */
ps__errors
ps__div(ps__object a, ps__object b)
{
  if (!to_real(b)) return ps_e_undefinedresult;
  return PUSH(opstack, new_real(to_real(a)/to_real(b)));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__idiv(ps__object a, ps__object b)
{
  if (!int_val(b)) return ps_e_undefinedresult;
  return PUSH(opstack, new_int(int_val(a)/int_val(b)));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__mod(ps__object a, ps__object b)
{
  if (!int_val(b)) return ps_e_undefinedresult;
  return PUSH(opstack, new_int(int_val(a)%int_val(b)));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__neg(ps__object a)
{
  if (a.type==ps_t_int) int_val(a)  *= -1;
  else                  real_val(a) *= -1;
  return PUSH(opstack, a);
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__not(ps__object a)
{
  if (a.type == ps_t_int)
    return PUSH(opstack, new_int(~int_val(a)));
  else
    return PUSH(opstack, (bool_val(a))?false_object:true_object);
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__and(ps__object a, ps__object b)
{
  if (a.type != b.type) return(ps_e_typecheck);
  return PUSH(opstack, (a.type==ps_t_int)
                       ?new_int(    int_val (a)  & int_val (b))
                       :new_boolean(bool_val(a) && bool_val(b)));
 }

/* --------------------*********************************-------------------- */
ps__errors
ps__or(ps__object a, ps__object b)
{
  if (a.type != b.type) return(ps_e_typecheck);
  return PUSH(opstack, (a.type==ps_t_int)
                                  ?new_int(    int_val (a)  | int_val (b))
                                  :new_boolean(bool_val(a) || bool_val(b)));
 }


/* --------------------*********************************-------------------- */
ps__errors
ps__xor(ps__object a, ps__object b)
{
  if (a.type != b.type) return(ps_e_typecheck);
  return PUSH(opstack, (a.type==ps_t_int)
                                  ?new_int(    int_val (a) ^ int_val (b))
                                  :new_boolean(bool_val(a) ^ bool_val(b)));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__bitshift(ps__object n, ps__object shift)
{
  return PUSH(opstack, new_int((int_val(shift)<0)?
                                     int_val(n) >> -int_val(shift)
                                    :int_val(n) <<  int_val(shift)));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__sin(ps__object a)
{
  double sin();
  return PUSH(opstack, new_real(sin(degrad(to_real(a)))));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__cos(ps__object a)
{
  double cos();
  return PUSH(opstack, new_real(cos(degrad(to_real(a)))));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__atan(ps__object num, ps__object den)
{
  double atan2(), angle;
  if (!to_real(num) && !to_real(den)) return ps_e_undefinedresult;
  if ((angle = 180*atan2(to_real(num), to_real(den))/_PI_)<0) angle += 360;
  return PUSH(opstack, new_real(angle));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__exp(ps__object base, ps__object ex)
{
  double exp(), log(), b, e = to_real(ex);

  return  (((b = to_real(base)) < 0.0) &&
           ((e = to_real(ex))!= (double)(int)e)) ?
    ps_e_undefinedresult :
    PUSH(opstack, new_real((b==0) ? 0.0 :
                           ((b<0)?(exp(e)*log(-b)):(exp(e)*log(b)))));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__log(ps__object a)
{
  double log();
  if (to_real(a) <= 0)  return ps_e_undefinedresult;
  return PUSH(opstack, new_real(log(to_real(a))/log(10.0)));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__ln(ps__object a)
{
  double log();
  if (to_real(a) <= 0)  return ps_e_undefinedresult;
  return PUSH(opstack, new_real(log(to_real(a))));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__sqrt(ps__object a)
{
  double sqrt();
  if (to_real(a) < 0) return ps_e_undefinedresult;
  return PUSH(opstack, new_real(sqrt(to_real(a))));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__abs(ps__object a)
{
  return PUSH(opstack, (a.type == ps_t_int)?
    new_int((int_val(a)  >0)?int_val(a) :-int_val(a)):
    new_real((real_val(a)>0)?real_val(a):-real_val(a)));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__truncate(ps__object a)
{
  return PUSH(opstack, (a.type==ps_t_int)?a:
                        new_real((double)((int)real_val(a))));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__round(ps__object a)
{
  return PUSH(opstack, (a.type==ps_t_int)?a:new_int((int)(real_val(a)+.5)));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__floor(ps__object a)
{
  double floor();
  return PUSH(opstack, (a.type==ps_t_int)?a:new_real(floor(real_val(a))));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__ceiling(ps__object a)
{
  double ceil();
  return PUSH(opstack, (a.type==ps_t_int)?a:new_real(ceil(real_val(a))));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__eq(ps__object a, ps__object b)
{
  return PUSH(opstack, new_boolean(!obcmp(a, b)));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__ne(ps__object a, ps__object b)
{
  return PUSH(opstack, new_boolean(obcmp(a, b)));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__le(ps__object a, ps__object b)
{
  return PUSH(opstack, new_boolean((obcmp(a, b)<=0)));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__lt(ps__object a, ps__object b)
{
  return PUSH(opstack, new_boolean((obcmp(a, b)<0)));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__ge(ps__object a, ps__object b)
{
  return PUSH(opstack, new_boolean((obcmp(a, b)>=0)));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__gt(ps__object a, ps__object b)
{
  return PUSH(opstack, new_boolean((obcmp(a, b)>0)));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__rand()
{
#ifdef RAND_IS_16_BITS
  return PUSH(opstack, new_int(randseed = rand()+rand()));  /* Make it 32... */
#else
  return PUSH(opstack, new_int(randseed = rand()));
#endif /* RAND_IS_16_BITS */
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__rrand()
{
  return PUSH(opstack, new_int(randseed));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__srand(ps__object seed)
{
  srand(randseed = int_val(seed));
  return ps_e_operationok;
  }

/* ************************************************************************* */
void
init_fonc_stuff()
{
  new_operator("length",    1, 1, "das",     ps__length);
  new_operator("maxlength", 1, 1, "d",       ps__maxlength);
  new_operator("add",       2, 1, "ir.ir",   ps__add);
  new_operator("sub",       2, 1, "ir.ir",   ps__sub);
  new_operator("mul",       2, 1, "ir.ir",   ps__mul);
  new_operator("div",       2, 1, "ir.ir",   ps__div);
  new_operator("idiv",      2, 1, "i.i",     ps__idiv);
  new_operator("mod",       2, 1, "i.i",     ps__mod);
  new_operator("neg",       1, 1, "ir",      ps__neg);
  new_operator("not",       1, 1, "ib",      ps__not);
  new_operator("and",       2, 1, "ib.ib",   ps__and);
  new_operator("or",        2, 1, "ib.ib",   ps__or);
  new_operator("xor",       2, 1, "ib.ib",   ps__xor);
  new_operator("bitshift",  2, 1, "i.i",     ps__bitshift);
  new_operator("sin",       1, 1, "ir",      ps__sin);
  new_operator("cos",       1, 1, "ir",      ps__cos);
  new_operator("atan",      2, 1, "ir.ir",   ps__atan);
  new_operator("exp",       2, 1, "ir.ir",   ps__exp);
  new_operator("log",       1, 1, "ir",      ps__log);
  new_operator("ln",        1, 1, "ir",      ps__ln);
  new_operator("sqrt",      1, 1, "ir",      ps__sqrt);
  new_operator("abs",       1, 1, "ir",      ps__abs);
  new_operator("truncate",  1, 1, "ir",      ps__truncate);
  new_operator("round",     1, 1, "ir",      ps__round);
  new_operator("floor",     1, 1, "ir",      ps__floor);
  new_operator("ceiling",   1, 1, "ir",      ps__ceiling);
  new_operator("eq",        2, 1, NONE,      ps__eq);
  new_operator("ne",        2, 1, NONE,      ps__ne);
  new_operator("ge",        2, 1, "irs.irs", ps__ge);
  new_operator("gt",        2, 1, "irs.irs", ps__gt);
  new_operator("le",        2, 1, "irs.irs", ps__le);
  new_operator("lt",        2, 1, "irs.irs", ps__lt);
  new_operator("rand",      0, 1, NONE,      ps__rand);
  new_operator("rrand",     0, 1, NONE,      ps__rrand);
  new_operator("srand",     1, 0, "i",       ps__srand);
  }
