/* ************************************************************************* *
 * PostScript Interpretor                   Fabien LELAQUAIS                 *
 *                                                                           *
 *   Fichier file.c                                                          *
 *      Files management routines for PSint.                                 *
 *                           Version 1.00 on 15/01/88                        *
 * ************************************************************************* *
 *    This document may be distributed, used, or modified, but can NOT be    *
 *  sold nor incorporated in any way in any product.                         *
 *    Permission is granted to distribute modified versions of that software *
 *  under the condition that this notice remains in every source file.       *
 *    Every alteration of the original files should be marked as such.       *
 *    No warranty is assumed by the author on the concequencies of the use   *
 *  of this software. Any defection of this program is at your own risk,     *
 *  you have to assume the cost of any service, installation or repairs      *
 *  this program could generate.                                             *
 *                                                                           *
 *                          Fabien LELAQUAIS - ESIEE - lelaquaf@apo.esiee.fr *
 * ************************************************************************* */
#include "int.h"
#if UNIX
#include <sgtty.h>

struct sgttyb inittty, newtty;
#endif /* UNIX */

#define MAX_FILENAME_LENGTH 80
#define MAX_EDITBUF_SIZE   256

FILE *fopen();

/* Due to an APOLLO ioctl bug, I've had to define two RETURN chars !! */
#define RETNORM   '\n'
#if APOLLO
#define RETNOECHO 13
#else
#define RETNOECHO RETNORM
#endif /* APOLLO */
int retchar = RETNORM;

ps__object stdin_object, stdout_object, stderr_object;

/* ************************************************************************* */
ps__object
new_file(FILE *filedes)
{
  ps__object result;

  result = new_object(ps_t_file);
  file_val(result) = filedes;
  result.size      = 1;
  return result;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__echo(ps__object flag)
{
#if !UNIX
  ps__puts("Can't suppress echo...");
#else
  if (bool_val(flag)) { 
    ioctl(0, TIOCSETP, &inittty);
    retchar = RETNORM;
    }
  else {
    ioctl(0, TIOCSETP, &newtty);
    retchar = RETNOECHO;
    }
#endif /* !UNIX */
  return ps_e_operationok;
  }

char editbuffer[MAX_EDITBUF_SIZE];
#define editgetchar (((c = ps__getchar())==RETNORM)?EOF:c)
/* --------------------*********************************-------------------- */
ps__errors
ps__editstatement()
{
  char *p = editbuffer;
  int   c, braces = 0, parenthesis = 0;

  while (((c=editgetchar) != EOF) || braces || parenthesis) {
    if ((c == '{') && !parenthesis) { *p++ = '{'; braces++; continue; }
    if ((c == '}') && !parenthesis) {
      if (!braces) return ps_e_syntaxerror;
      *p++ = '}';
      braces--;
      continue; }
    if (c == '(') parenthesis++;
    if (c == ')') {
      if (parenthesis) parenthesis--;
      else             return ps_e_syntaxerror;
      }
    *p++ = (c==EOF)?(parenthesis?'\n':' '):c;
    if (p-editbuffer > MAX_EDITBUF_SIZE) return ps_e_limitcheck;
    }
  *p = 0;
  return PUSH(opstack, cvx(do_string(editbuffer)));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__editline()
{
  char *p = editbuffer;
  *p = 0;
  return PUSH(opstack, cvx(do_string(editbuffer)));
  }

/* --------------------*********************************-------------------- *
 *   Pushes the opened file object corresponding to the given filename, with *
 *  the given access.                                                        *
 *  returns  : invalidfileaccess if :                                        *
 *                   'filename' has a length of zero,                        *
 *                or 'access' is neighter (r) nor (w)                        *
 *                or 'access' doesn't match the special file access, as      *
 *                    (w) on stdin or (r) on stdout                          *
 *             undefinedfilename if :                                        *
 *                   'filename' begins with '%' and is neighter              *
 *                              stdin nor stdout                             *
 *                   'filename' doesn't exist or has a length bigger         *
 *                              than MAX_FILENAME_LENGTH                     */
ps__errors
ps__file(ps__object filename, ps__object access)
{
  FILE      *fp;
  char       name[MAX_FILENAME_LENGTH+1], filemode[2], openmode;

  if (!filename.size || (access.size != 1) ||      
      ((openmode = *stg_val(access)) != 'r' && (openmode != 'w')))
    return ps_e_invalidfileaccess;
  if (*stg_val(filename) == '%') {
    if (obstrcmp(filename, "%stdin",  6))
      return (openmode=='r')?PUSH(opstack, stdin_object ):
                             ps_e_invalidfileaccess;
    if (obstrcmp(filename, "%stdout", 7))
      return (openmode=='w')?PUSH(opstack, stdout_object):
                             ps_e_invalidfileaccess;
    if (obstrcmp(filename, "%stderr", 7))
      return (openmode=='w')?PUSH(opstack, stderr_object):
                             ps_e_invalidfileaccess;
    if (obstrcmp(filename, "%statementedit", 14))
      return (openmode=='r')?ps__editstatement():ps_e_invalidfileaccess;
    if (obstrcmp(filename, "%lineedit", 9))
      return (openmode=='r')?ps__editline():ps_e_invalidfileaccess;
    return ps_e_undefinedfilename;
    }
  else {
    if (filename.size > MAX_FILENAME_LENGTH) return ps_e_undefinedfilename;
      filemode[0] = openmode;
    filemode[1] = 0;
    obstrcpy(filename, name);
    return (fp = fopen(name, filemode))?
      PUSH(opstack,
	   (openmode=='r')?cvro(new_file(fp)):cvwo(new_file(fp)))
           :ps_e_undefinedfilename;
    }
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__bytesavailable(ps__object file)
{
#ifdef VAX
  return PUSH(opstack, new_int(-1));
#else
  return PUSH(opstack, new_int(HAS_R(file)?(file_val(file)->_cnt):-1));
#endif /* VAX */
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__currentfile()
{
  int loop = execstack->size;

  for (; loop && (execstack->stack[loop-1].type != ps_t_file); loop--);
  return PUSH(opstack, loop?execstack->stack[loop-1]:stdin_object);
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__status(ps__object file)
{
  return PUSH(opstack, new_boolean((long)file_val(file)));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__resetfile(ps__object file)
{
  if (HAS_W(file) && (file.type == ps_t_file) && file_val(file))
    ps__fflush(file_val(file));
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__flush()
{
  ps__fflush(stdout);
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__flushfile(ps__object file)
{
  if (HAS_W(file) && (file.type == ps_t_file) && file_val(file))
    ps__fflush(file_val(file));
  else
    if (HAS_R(file)) while (getin(file) != EOF);
 return ps_e_operationok;
 }

/* --------------------*********************************-------------------- */
ps__errors ps__closefile(ps__object file)
{
  fclose(file_val(file));
  file_val(file) = NULL;
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__run(ps__object filename)
{
  FILE      *fp;
  char       name[MAX_FILENAME_LENGTH+1];

  if (filename.size > MAX_FILENAME_LENGTH) return ps_e_undefinedfilename;
  obstrcpy(filename, name);
  if (fp = fopen(name,"r")) {
    PUSH(execstack, invalid_object);
    return PUSH(execstack, cvx(cvro(new_file(fp))));
    }
  return ps_e_undefinedfilename;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__read(ps__object file)
{
  int c;

  if (!HAS_R(file)) return ps_e_invalidaccess;
  if (!file_val(file)) return ps_e_IOerror;
  if ((c = getin(file)) == EOF)
    return PUSH(opstack, false_object);
  PUSH(opstack, new_int(c));
  return PUSH(opstack, true_object);
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__readhexstring(ps__object file, ps__object string)
{
  int c, index = 0, code, digits;
  unsigned char *p = stg_val(string);

  if (!HAS_R(file) || !HAS_W(string)) return ps_e_invalidaccess;
  if (!file_val(file))                return ps_e_IOerror;
  if (!string.size)                   return ps_e_rangecheck;
  code = digits = 0;
  while (1) {
    c = getin(file);
    if (c == EOF) {
      string.size = index;
      fclose(file_val(file));
      PUSH(opstack, string);
      return PUSH(opstack, false_object);
      }
    if ((c>='0')&&(c<='9')) c -= '0';
    else 
      if ((c>='A')&&(c<='F')) c -= 'A'-10;
      else 
        if ((c>='a')&&(c<='f')) c -= 'f'-10;
        else continue;
    if (digits) {
      digits = 0;
      code += c;
      *p++ = code;
      index++;
      if (index == string.size) {
        PUSH(opstack, string);
        return PUSH(opstack, true_object);
        }
      }
    else {
      code = c<<4;
      digits = 1;
      }
    }
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__readline(ps__object file, ps__object string)
{
  int      c, index = 0;
  unsigned char *p = stg_val(string);

  if (!HAS_R(file) || !HAS_W(string)) return ps_e_invalidaccess;
  if (!file_val(file)) return ps_e_IOerror;
  if (!string.size)                   return ps_e_rangecheck;
  while (1) {
    c = getin(file);
    if (c == EOF) {
      string.size = index;
      fclose(file_val(file));
      PUSH(opstack, string);
      return PUSH(opstack, false_object);
      }
    if (c == RETNORM) {
      string.size = index;
      PUSH(opstack, string);
      return PUSH(opstack, true_object);
      }
    *p++ = c;
    index++;
    if (index > string.size)
      return ps_e_rangecheck;
    }
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__readstring(ps__object file, ps__object string)
{
  int      c, index = 0;
  unsigned char *p = stg_val(string);

  if (!HAS_R(file) || !HAS_W(string)) return ps_e_invalidaccess;
  if (!file_val(file)) return ps_e_IOerror;
  if (!string.size)                   return ps_e_rangecheck;
  while (1) {
    c = getin(file);
    if (c == EOF) {
      string.size = index;
      fclose(file_val(file));
      PUSH(opstack, string);
      return PUSH(opstack, false_object);
      }
    *p++ = c;
    index++;
    if (index == string.size) {
      PUSH(opstack, string);
      return PUSH(opstack, true_object);
      }
    }
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__write(ps__object file, ps__object code)
{
  if (!HAS_W(file)) return ps_e_invalidaccess;
  if (!file_val(file) ||
      (putc(int_val(code)%256,file_val(file)) == EOF)) return ps_e_IOerror;
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
#define tohex(x) (((x)<10)?((x)+'0'):((x)-10+'a'))
ps__errors
ps__writehexstring(ps__object file, ps__object string)
{
  int index, c;

  if (!HAS_W(file)) return ps_e_invalidaccess;
  if (!file_val(file)) return ps_e_IOerror;
  for (index=0; index<string.size; index++) {
     c = stg_val(string)[index]%256;
     if ((putc(tohex(c>>4), file_val(file)) == EOF) ||
         (putc(tohex(c&15), file_val(file)) == EOF)) return ps_e_IOerror;
     }
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__writestring(ps__object file, ps__object string)
{
  int index;

  if (!HAS_W(file)) return ps_e_invalidaccess;
  if (!file_val(file)) return ps_e_IOerror;
  for (index=0; index<string.size; index++)
     if (putc(stg_val(string)[index], file_val(file)) == EOF)
       return ps_e_IOerror;
  return ps_e_operationok;
  }

/* ************************************************************************* */
void
init_file_stuff()
{
#if UNIX
  ioctl(0, TIOCGETP, &inittty); 
  newtty = inittty;
  newtty.sg_flags &= ~ECHO;
#endif /* UNIX */

  stdin_object  = cvro(new_file(stdin));
  stdout_object = cvwo(new_file(stdout));
  stderr_object = cvwo(new_file(stderr));
  dict_store(systemdict, do_name("stdin"),  stdin_object);
  dict_store(systemdict, do_name("stdout"), stdout_object);
  dict_store(systemdict, do_name("stderr"), stderr_object);

  new_operator("echo",           1, 0, "b",   ps__echo);

  new_operator("file",           2, 1, "s.s", ps__file);
  new_operator("status",         1, 1, "F",   ps__status);
  new_operator("bytesavailable", 1, 1, "F",   ps__bytesavailable);
  new_operator("resetfile",      1, 0, "F",   ps__resetfile);
  new_operator("flush",          0, 0, NONE,  ps__flush);
  new_operator("flushfile",      1, 0, "F",   ps__flushfile);
  new_operator("currentfile",    0, 1, NONE,  ps__currentfile);
  new_operator("closefile",      1, 0, "F",   ps__closefile);
  new_operator("run",            1, 0, "s",   ps__run);

  new_operator("read",           1, 2, "F",   ps__read);
  new_operator("readhexstring",  2, 2, "F.s", ps__readhexstring);
  new_operator("readline",       2, 2, "F.s", ps__readline);
  new_operator("readstring",     2, 2, "F.s", ps__readstring);

  new_operator("write",          2, 0, "F.i", ps__write);
  new_operator("writehexstring", 2, 0, "F.s", ps__writehexstring);
  new_operator("writestring",    2, 0, "F.s", ps__writestring);
  }
