/* ************************************************************************* *
 * PostScript Interpretor                   Fabien LELAQUAIS                 *
 *                                                                           *
 *   Fichier exec.c                                                          *
 *      Execstack management routines for PSint.                             *
 *                           Version 1.00 on 15/01/88                        *
 * ************************************************************************* *
 *    This document may be distributed, used, or modified, but can NOT be    *
 *  sold nor incorporated in any way in any product.                         *
 *    Permission is granted to distribute modified versions of that software *
 *  under the condition that this notice remains in every source file.       *
 *    Every alteration of the original files should be marked as such.       *
 *    No warranty is assumed by the author on the concequencies of the use   *
 *  of this software. Any defection of this program is at your own risk,     *
 *  you have to assume the cost of any service, installation or repairs      *
 *  this program could generate.                                             *
 *                                                                           *
 *                          Fabien LELAQUAIS - ESIEE - lelaquaf@apo.esiee.fr *
 * ************************************************************************* */
#include "int.h"

extern ps__object check_value();
ps__object op_arguments[MAX_OP_ARGUMENTS];
short      op_nargs;

/* ************************************************************************* */
ps__errors
execute_stg_or_file(ps__object object)
{
 ps__object subobject;

  subobject = get_next_command(&object);
  if (subobject.type == ps_t_invalid) return ps_e_syntaxerror;
  if (subobject.type != ps_t_null) {
    PUSH(execstack, object);
    if ((subobject.type==ps_t_name) && HAS_X(subobject))
      return PUSH(execstack, subobject);
    else
      return PUSH(opstack, subobject);
    }
  return ps_e_operationok;
  }

/* ************************************************************************* */
ps__errors
execute_name(ps__object object)
{
  ps__object subobject;

  subobject = find_key(object);
  if (subobject.type == ps_t_invalid)
    return ps_e_undefined;
  return PUSH(HAS_X(subobject)?execstack:opstack, subobject);
  }

/* ************************************************************************* */
ps__errors
execute_array(ps__object object)
{
  ps__object subobject;

  if (object.size) {
    subobject = arr_ob(object, 0);
    if ((subobject.flags & DYNAMIC) &&
        (object.flags    & DYNAMIC) &&
        !REFS(object))
      REFS(subobject)--;
    arr_val(object)++;
    object.size--;
    PUSH(execstack, object);
    if (subobject.type == ps_t_invalid) return ps_e_undefined;
/* BUG ??
    if ((subobject.type == ps_t_name) && HAS_X(subobject))
      return PUSH(execstack, subobject);
    else
      return PUSH(opstack,   subobject);
 remplace par */
    if (HAS_X(subobject) && (subobject.type != ps_t_array))
      return PUSH(execstack, subobject);
    else
      return PUSH(opstack,   subobject);
    }
  return ps_e_operationok;
  }

/* ************************************************************************* */
#define fonc (op_val(operator).operator)
ps__errors
execute_operator(ps__object operator)
{
  ps__errors error;
  short      loop;
  char       chartype, *p;

  if (memtrace)
    ps__printf("*** Executing operator `%s' ***\n", op_val(operator).name);
              /* Is Operand stack ready to stack-unstack routine arguments ? */
  op_nargs = op_val(operator).argsin;
  if (op_nargs > opstack->size) {
    op_nargs = 0;
    return ps_e_stackunderflow;
    }
  if (op_val(operator).argsout+opstack->size > opstack->maxsize) {
    op_nargs = 0;
    return ps_e_stackoverflow;
    }
                    /* Lets's load those arguments in the op_arguments array */
  for (loop=op_nargs; loop; loop--)
    op_arguments[loop-1] = POP(opstack);
  if (p = op_val(operator).argstype)        /* Checking those arguments type */
    for (loop=0; loop<op_nargs; loop++) {
      switch (op_arguments[loop].type) {
        case ps_t_null     : chartype = 'N'; break;
        case ps_t_int      : chartype = 'i'; break;
        case ps_t_real     : chartype = 'r'; break;
        case ps_t_bool     : chartype = 'b'; break;
        case ps_t_stg      : chartype = 's'; break;
        case ps_t_file     : chartype = 'F'; break;
        case ps_t_mark     : chartype = 'm'; break;
        case ps_t_array    : chartype = 'a'; break;
        case ps_t_operator : chartype = 'o'; break;
        case ps_t_name     : chartype = 'n'; break;
        case ps_t_dict     : chartype = 'd'; break;
        case ps_t_fontId   : chartype = 'f'; break;
        case ps_t_save     : chartype = 'S'; break;
        }
      for (; *p && (*p != '.') && (*p != 'x') && (*p != chartype); p++);
        if ((*p != chartype) && (*p != 'x')) return ps_e_typecheck;
        for (; *p && (*p != '.'); p++); if (*p) p++;
        }
#if 0
  switch (op_nargs) {
    case 0 : return (*fonc)();
    case 1 : return (*fonc)(op_arguments[0]);
    case 2 : return (*fonc)(op_arguments[0],
                            op_arguments[1]);
    case 3 : return (*fonc)(op_arguments[0],
                            op_arguments[1],
                            op_arguments[2]);
    case 4 : return (*fonc)(op_arguments[0],
                            op_arguments[1],
                            op_arguments[2],
                            op_arguments[3]);
    case 5 : return (*fonc)(op_arguments[0],
                            op_arguments[1],
                            op_arguments[2],
                            op_arguments[3],
                            op_arguments[4]);
    case 6 : return (*fonc)(op_arguments[0],
                            op_arguments[1],
                            op_arguments[2],
                            op_arguments[3],
                            op_arguments[4],
                            op_arguments[5]);
    default: return (*fonc)(op_arguments[0],
                            op_arguments[1],
                            op_arguments[2],
                            op_arguments[3],
                            op_arguments[4],
                            op_arguments[5],
                            op_arguments[6]);
    }
#else
  error = (*fonc)(op_arguments[0], op_arguments[1],
                  op_arguments[2], op_arguments[3],
                  op_arguments[4], op_arguments[5],
                  op_arguments[6], op_arguments[7]);
#endif
  return error;
  }

/* ************************************************************************* */
void
run_interpretor()
{
  ps__errors error;
  ps__object exec;
  short      loop;

  while (execstack->size) {
    exec = POP(execstack);
    op_nargs = 0;
    if (HAS_X(exec)) {
      switch (exec.type) {
        case ps_t_operator :
          error = execute_operator(exec);
          break;
        case ps_t_stg :
        case ps_t_file :
          error = execute_stg_or_file(exec);
          break;
        case ps_t_name :
          error = execute_name(exec);
          break;
        case ps_t_array :                /* Executing a PostScript procedure */
          error = execute_array(exec);
          break;
        case ps_t_mark  :
          error = PUSH(opstack, false_object);
          break;
        default :
          error = PUSH(opstack, exec);
        }
      }
    else
      error = (exec.type == ps_t_invalid)
                 ? ps_e_operationok
                 : PUSH(opstack, exec);
                                                /* Returned error processing */
    if (error != ps_e_operationok) {
      extern void process_error();

      for (loop=0; loop<op_nargs; loop++)
        PUSH(opstack, op_arguments[loop]);
      process_error(error, exec);
      }
    else {
      for (loop=0; loop<op_nargs; loop++)
        if ((op_arguments[loop].flags & DYNAMIC) && !REFS(op_arguments[loop]))
          ps_destroy_object(op_arguments[loop], "execute_operator");
      }
    if ((exec.flags & DYNAMIC) && !REFS(exec))
      ps_destroy_object(exec, "run_interpretor");
    }
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__exec(ps__object object)
{
  PUSH(execstack, object);
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__object loop_object;
ps__errors
ps__loop(ps__object proc)
{
  if (!(HAS_X(proc))) return ps_e_invalidaccess;
  PUSH(execstack, invalid_object);
  PUSH(execstack, proc);
  PUSH(execstack, loop_object);
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps___loop()
{
  ps__object proc;
  proc = POP(execstack);
  PUSH(execstack, proc);
  PUSH(execstack, loop_object);
  PUSH(execstack, proc);
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__cvi(ps__object object)
{
  extern void ps_free();

  switch (object.type) {
    case ps_t_int    : return PUSH(opstack, object);
    case ps_t_real   : return PUSH(opstack, new_int((int)real_val(object)));
    case ps_t_stg : {
      char *s = ps_alloc(object.size+1), *q = s, *p = (char *)stg_val(object);
      int   loop;
      if (!(HAS_R(object))) return ps_e_invalidaccess;
      for (loop=0; loop<object.size; loop++, *q++ = *p++);
      *p = 0;
      object = check_value(s);
      ps_free(s);
      if (object.type == ps_t_invalid) return ps_e_syntaxerror;
      return (object.type!=ps_t_int)?PUSH(opstack, new_real(to_real(object)))
                                    :PUSH(opstack, object);
      }
    }
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__cvr(ps__object object)
{
  extern void ps_free();

  switch (object.type) {
    case ps_t_int    : return PUSH(opstack, new_real((double)int_val(object)));
    case ps_t_real   : return PUSH(opstack, object);
    case ps_t_stg    : {
      char *s = ps_alloc(object.size+1), *q = s, *p = (char *)stg_val(object);
      int   loop;
      if (!(HAS_R(object))) return ps_e_invalidaccess;
      for (loop=0; loop<object.size; loop++, *q++ = *p++);
      *p = 0;
      object = check_value(s);
      ps_free(s);
      if (object.type == ps_t_invalid) return ps_e_syntaxerror;
      return (object.type==ps_t_int)?PUSH(opstack, new_real(to_real(object)))
                                    :PUSH(opstack,object);
      }
    }
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__cvlit(ps__object object)
{
  return PUSH(opstack, cvlit(object));
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__cvn(ps__object object)
{
  object.type = ps_t_name;
  return PUSH(opstack, object);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__cvx(ps__object object)
{
  return PUSH(opstack, cvx(object));
  }

/* --------------------*********************************-------------------- */
ps__errors
ps__countexecstack()
{
 return PUSH(opstack, new_int(execstack->size));
 }

/* -------------------------***********************------------------------- */
ps__errors
ps__execstack(ps__object array)
{
  int loop;

  if (!HAS_W(array))                return ps_e_invalidaccess;
  if (array.size < execstack->size) return ps_e_rangecheck;
  for (loop=0; loop <= execstack->size; loop++) {
    arr_ob(array, loop) = execstack->stack[loop];
    MORE_REFS(arr_ob(array, loop));
    }
  return PUSH(opstack, array);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__bind(ps__object proc)
{
  int        loop;
  ps__object subobject;

  for (loop=0; loop<proc.size; loop++) {
    switch (arr_ob(proc, loop).type) {
      case ps_t_array :
        ps__bind(arr_ob(proc, loop));
        arr_ob(proc, loop) = POP(opstack);
        break;
      case ps_t_name  :
        if (HAS_X(arr_ob(proc, loop))) {
          subobject = find_key(arr_ob(proc, loop));
          if (subobject.type == ps_t_operator) {
            CHECK_DESTROY(arr_ob(proc, loop), "bind");
            arr_ob(proc, loop) = subobject;
            }
          }
      }
    }
  return PUSH(opstack, proc);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__if(ps__object condition, ps__object proc)
{
  if (bool_val(condition)) return PUSH(execstack, proc);
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__ifelse(ps__object condition, ps__object proctrue, ps__object procfalse)
{
  return PUSH(execstack, (bool_val(condition))?proctrue:procfalse);
  }

/* -------------------------***********************------------------------- */
ps__object repeat_object;
ps__errors
ps__repeat(ps__object count, ps__object proc)
{
  if (!(HAS_X(proc)))     return ps_e_invalidaccess;
  if (int_val(count) < 0) return ps_e_rangecheck;
  PUSH(execstack, invalid_object);
  PUSH(execstack, count);
  PUSH(execstack, proc);
  PUSH(execstack, repeat_object);
  return ps_e_operationok;
  }

/* ************************************************************************* */
ps__errors
ps___repeat()
{
  ps__object proc, count;
  proc  = POP(execstack);
  count = POP(execstack);
  if (int_val(count)) {
    PUSH(execstack, new_int(int_val(count)-1));
    PUSH(execstack, proc);
    PUSH(execstack, repeat_object);
    PUSH(execstack, proc);
    }
  else
    if ((proc.flags & DYNAMIC) && !REFS(proc))
      ps_destroy_object(proc, "repeat");
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__object for_object;
ps__errors
ps__for(ps__object init, ps__object increm, ps__object end, ps__object proc)
{
  if (!(HAS_X(proc))) return ps_e_invalidaccess;
  if ((init.type   != ps_t_int) ||
      (increm.type != ps_t_int) ||
      (end.type    != ps_t_int)) {
    if (init.type   == ps_t_int) init   = new_real((double)int_val(init));
    if (increm.type == ps_t_int) increm = new_real((double)int_val(increm));
    if (end.type    == ps_t_int) end    = new_real((double)int_val(end));
    }
  PUSH(execstack, invalid_object);
  PUSH(execstack, init);
  PUSH(execstack, increm);
  PUSH(execstack, end);
  PUSH(execstack, proc);
  PUSH(execstack, for_object);
  return ps_e_operationok;
  }

/* ************************************************************************* */
ps__errors
ps___for()
{
  ps__object proc, init, increm, end;
  proc   = POP(execstack);
  end    = POP(execstack);
  increm = POP(execstack);
  init   = POP(execstack);
  if (init.type == ps_t_int) {
    if (int_val(increm) > 0) {
      if (int_val(init) > int_val(end)) init = invalid_object;
      else {
	PUSH(opstack, init);
	int_val(init) += int_val(increm);
	}
      }
    else {
      if (int_val(init) < int_val(end)) init = invalid_object;
      else {
	PUSH(opstack, init);
	int_val(init) += int_val(increm);
	}
      }                                       
    }
  else {
    if (real_val(increm) > 0) {
      if (real_val(init) > real_val(end)) init = invalid_object;
      else {
	PUSH(opstack, init);
	real_val(init) += real_val(increm);
	}
      }
    else {
      if (real_val(init) < real_val(end)) init = invalid_object;
      else {
	PUSH(opstack, init);
	real_val(init) += real_val(increm);
	}
      }
    }
  if (init.type != ps_t_invalid) {  /* Looping be keep on */
    PUSH(execstack, init);
    PUSH(execstack, increm);
    PUSH(execstack, end);
    PUSH(execstack, proc);
    PUSH(execstack, for_object);
    PUSH(execstack, proc);
    }
  else
    if ((proc.flags & DYNAMIC) && !REFS(proc))
      ps_destroy_object(proc, "for");
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__object forall_object;
ps__errors
ps__forall(ps__object object, ps__object proc)
{
  if (!HAS_R(object) || !HAS_X(proc)) return ps_e_invalidaccess;
  PUSH(execstack, invalid_object);
  PUSH(execstack, object);
  PUSH(execstack, new_int(0));
  PUSH(execstack, proc);
  return PUSH(execstack, forall_object);
  }

/* ************************************************************************* */
ps__errors
ps___forall()
{
  ps__object proc, index, object;

  proc   = POP(execstack);
  index  = POP(execstack);
  object = POP(execstack);
  if (((object.type != ps_t_dict)&&(int_val(index) == object.size)) ||
      ((object.type == ps_t_dict)&&(int_val(index) == dict_val(object).size))){
    if ((proc.flags & DYNAMIC) && !REFS(proc))
      ps_destroy_object(proc, "forall");
    return ps_e_operationok;
    }
  switch (object.type) {
   case ps_t_array :
     PUSH(opstack, arr_ob(object, int_val(index)));
     break;
   case ps_t_stg :
     PUSH(opstack, new_int(stg_val(object)[int_val(index)]));
     break;
   default : 
     PUSH(opstack, dict_val(object).keys[int_val(index)].name);
     PUSH(opstack, dict_val(object).keys[int_val(index)].object);
    }
  PUSH(execstack, object);
  PUSH(execstack, new_int(int_val(index)+1));
  PUSH(execstack, proc);
  PUSH(execstack, forall_object);  
  return PUSH(execstack, proc);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__exit()
{
  ps__object dummy;

  int loop = execstack->size;
  for (; loop && (execstack->stack[loop-1].type != ps_t_invalid); loop--);
  for (; loop && (execstack->stack[loop-1].type != ps_t_invalid); loop--);
  if (!loop) return ps_e_invalidexit;
  do {
    dummy = POP(execstack);
    if ((dummy.flags & DYNAMIC) && !REFS(dummy))
      ps_destroy_object(dummy, "exit");
    } while (dummy.type != ps_t_invalid);
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__start()
{
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__stop()
{
  int        loop = execstack->size;
  ps__object dummy;
  ps__errors ps__exit();
               /* Search last STOPPED context and close opened running files */
  for (; loop; loop--) {
    if ((execstack->stack[loop-1].type      == ps_t_file)     &&
        (execstack->stack[loop-1].flags     == (READ | EXEC)) &&
        (file_val(execstack->stack[loop-1]) != stdin))
      fclose(file_val(execstack->stack[loop-1]));
    else
      if (execstack->stack[loop-1].type == ps_t_mark) break; /* Stopped Ctxt */
    }
  if (loop) {
    while (execstack->size != loop-1) {
      dummy = POP(execstack);
      if ((dummy.flags & DYNAMIC) && !REFS(dummy))
        ps_destroy_object(dummy, "stop");
      }
    return PUSH(opstack, true_object);
    }
  return ps__exit();
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__stopped(ps__object proc)
{
  PUSH(execstack, cvx(mark_object));
  return PUSH(execstack, proc);
  }

/* ************************************************************************* */
void
init_exec_stuff()
{
  new_operator("start",          0, 0, NONE,         ps__start);
  new_operator("stopped",        1, 0, "a",          ps__stopped);

  new_operator("loop",           1, 0, "a",          ps__loop);
  new_operator("exec",           1, 0, NONE,         ps__exec);
  new_operator("cvi",            1, 1, "irs",        ps__cvi);
  new_operator("cvr",            1, 1, "irs",        ps__cvr);
  new_operator("cvn",            1, 1, "s",          ps__cvn);
  new_operator("cvx",            1, 1, NONE,         ps__cvx);
  new_operator("cvlit",          1, 1, NONE,         ps__cvlit);
  new_operator("countexecstack", 0, 1, NONE,         ps__countexecstack);
  new_operator("execstack",      1, 1, "a",          ps__execstack);
  new_operator("bind",           1, 1, "a",          ps__bind);
  new_operator("if",             2, 0, "b.a",        ps__if);
  new_operator("ifelse",         3, 0, "b.a.a",      ps__ifelse);
  new_operator("repeat",         2, 0, "i.a",        ps__repeat);
  new_operator("for",            4, 0, "ir.ir.ir.a", ps__for);
  new_operator("forall",         2, 0, "ads.a",      ps__forall);
  new_operator("exit",           0, 0, NONE,         ps__exit);

  loop_object   = new_operator(" loop",     0, 0, NONE, ps___loop);
  repeat_object = new_operator(" repeat",   0, 0, NONE, ps___repeat);
  for_object    = new_operator(" for",      0, 1, NONE, ps___for);
  forall_object = new_operator(" forall",   0, 1, NONE, ps___forall);
  }
