/* ************************************************************************* *
 * PostScript Interpretor                   Fabien LELAQUAIS                 *
 *                                                                           *
 *   Fichier dict.c                                                          *
 *      Dictionnaries management routines for PSint.                         *
 *                           Version 3.00 on 24/02/89                        *
 * ************************************************************************* *
 *    This document may be distributed, used, or modified, but can NOT be    *
 *  sold nor incorporated in any way in any product.                         *
 *    Permission is granted to distribute modified versions of that software *
 *  under the condition that this notice remains in every source file.       *
 *    Every alteration of the original files should be marked as such.       *
 *    No warranty is assumed by the author on the concequencies of the use   *
 *  of this software. Any defection of this program is at your own risk,     *
 *  you have to assume the cost of any service, installation or repairs      *
 *  this program could generate.                                             *
 *                                                                           *
 *                          Fabien LELAQUAIS - ESIEE - lelaquaf@apo.esiee.fr *
 * ************************************************************************* */
#include "int.h"


/* ************************************************************************* */
compute_hash_value(ps__object key)
{
  short i, hash = 0;
  for (i=0; i<key.size; hash += stg_val(key)[i++]);
  return (hash == -1)?0:hash;
  }

/* ************************************************************************* *
 *   Returns the index where keyname is stored in dict (or -1 if not found)  *
 *                                                                           *
 *   Very first version used obcmp to compare the given key with all stored  *
 * keys in given dictionnary. This was quite CPU time consuming.             *
 *   I replaced this in version 3.50 with a specific comparaison routine,    *
 * using a most simple hashing process, in case key is a string or name.     *
 *   Just have a quick look at the `comput_hash_value' routine...            *
 * ************************************************************************* */
dict_find_index(ps__object dict, ps__object key)
{
  int index, hash;

  hash = ((key.type == ps_t_stg) || (key.type == ps_t_name))
           ? compute_hash_value(key)
           : -1;
  for (index = 0; index<dict_val(dict).size; index++)
#if 0
    if (!obcmp(dict_val(dict).keys[index].name, key)) break;
#else
    if ((hash == dict_val(dict).keys[index].hash) &&
        !obcmp(dict_val(dict).keys[index].name, key)) break;
#endif
  return (index==dict_val(dict).size)?-1:index;
  }

/* ************************************************************************* *
 *    Stores key and its value in given dictionnary, erasing older value.    *
 *    Returns 0 on success, 1 otherwize (dict is full)                       *
 *                                                                           *
 *    Save level is controlled by the `save_level' field in each key entry.  *
 *                                                                           *
 *  dict is assumed to be a dictionnary, key a name, and object, anything... *
 * ************************************************************************* */
dict_store(ps__object dict, ps__object key, ps__object object)
{
  int                   index;
  extern unsigned short save_level;

  if ((index = dict_find_index(dict, key)) == -1) {             /* New entry */
    if (dict_val(dict).size == dict.size) return 1;  /* Not enough free keys */
    index = dict_val(dict).size++;
    dict_val(dict).keys[index].hash =
      ((key.type == ps_t_stg) || (key.type == ps_t_name))
           ? compute_hash_value(key)
           : -1;
    dict_val(dict).keys[index].name = key;
    MORE_REFS(key);
    dict_val(dict).keys[index].save_level = save_level;
    }
  else {
    if (save_level != dict_val(dict).keys[index].save_level) { /* New save ? */
      ps__key *newkey;

      newkey = (ps__key *)ps_alloc(sizeof(ps__key));
      *newkey = dict_val(dict).keys[index];
      dict_val(dict).keys[index].next       = newkey;
      dict_val(dict).keys[index].save_level = save_level;
      MORE_REFS(dict_val(dict).keys[index].name);
      }
    if ((dict_val(dict).keys[index].object.flags & DYNAMIC) &&
        !(--REFS(dict_val(dict).keys[index].object))        &&
         (BLOCK(dict_val(dict).keys[index].object) != BLOCK(object)))
       /* Cas particulier :  on ecrase un objet dynamique libre par lui meme */
        ps_destroy_object(dict_val(dict).keys[index].object, "dict_store");
    }
  dict_val(dict).keys[index].object = object;
  if ((object.type          != ps_t_dict) ||
      (object.value.dictval != dict.value.dictval))
    MORE_REFS(object);
  return 0;
  }

/* ************************************************************************* *
 *   Returns value of the given key if found in given dictionnary,           *
 *  or invalid_object if not found                                           *
 * ************************************************************************* */
ps__object
dict_find_key(ps__object dict, ps__object key)
{
  int index;
  return ((index = dict_find_index(dict, key))== -1)?invalid_object:
                         dict_val(dict).keys[index].object;
  }

/* ************************************************************************* *
 *   Returns value of the given key if found in any dictionnary stored in    *
 *  dictstack, or invalid_object if not found                                *
 * ************************************************************************* */
ps__object
find_key(ps__object key)
{
  int        loop, index;
  ps__object dict;

  for (loop = dictstack->size; loop; loop--) {
    dict=dictstack->stack[loop-1];
    if ((index = dict_find_index(dict, key)) != -1)
      return dict_val(dict).keys[index].object;
    }
  return invalid_object;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__def(ps__object key, ps__object value)
{
  ps__object dict;

  dict = dictstack->stack[dictstack->size-1];
  if (!(HAS_W(dict))) return ps_e_invalidaccess;
  if (key.type == ps_t_null) return ps_e_typecheck;
  return (dict_store(dict, key, value))?ps_e_dictfull:ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__load(ps__object key)
{
  ps__object value;

  if (key.type == ps_t_null) return ps_e_typecheck;
  value = find_key(key);
  return (value.type == ps_t_invalid)?ps_e_undefined:PUSH(opstack, value);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__dictop(ps__object object)
{
  int size = int_val(object);

  if (size < 1) return ps_e_rangecheck;
  object = new_dict(size);
  return (object.type == ps_t_invalid)?ps_e_VMerror:PUSH(opstack, object);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__countdictstack()
{
  return PUSH(opstack, new_int(dictstack->size));
  }

/* -------------------------***********************------------------------- */
ps__errors ps__begin(dict)
ps__object dict;
{
  return HAS_R(dict)?PUSH(dictstack, dict):ps_e_invalidaccess;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__end()
{
  ps__object dict;

  if (dictstack->size > 2) {
    dict = POP(dictstack);
    ps_destroy_object(dict, "end");
    return ps_e_operationok;
    }
  return ps_e_dictstackunderflow;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__currentdict()
{
  return PUSH(opstack, dictstack->stack[dictstack->size-1]);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__known(ps__object dict, ps__object key)
{
  if (!(HAS_R(dict))) return ps_e_invalidaccess;
  return (key.type == ps_t_null)?ps_e_typecheck
           :PUSH(opstack,
                  ((dict_find_index(dict, key) == -1)?false_object:true_object));
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__where(ps__object key)
{
  unsigned short loop;
  ps__object     dict;

  if (key.type == ps_t_null) return ps_e_typecheck;
  for (loop = dictstack->size; loop; loop--) {
    dict=dictstack->stack[loop-1];
    if (dict_find_index(dict, key) != -1) {
      PUSH(opstack, dict);
      return PUSH(opstack, true_object);
      }
    }
  return PUSH(opstack, false_object);
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__store(ps__object key, ps__object value)
{
  int        loop, index;
  ps__object dict;

  if (key.type == ps_t_null) return ps_e_typecheck;
  for (loop = dictstack->size; loop; loop--) {
    dict=dictstack->stack[loop-1];
    if ((index = dict_find_index(dict, key)) != -1) {
      MORE_REFS(key);
      CHECK_DESTROY(dict_val(dict).keys[index].name, "store-name");
      dict_val(dict).keys[index].name   = key;
      if ((value.type          != ps_t_dict) ||
          (value.value.dictval != dict.value.dictval))
        MORE_REFS(value);
      if ((dict_val(dict).keys[index].object.type          != ps_t_dict) ||
          (dict_val(dict).keys[index].object.value.dictval != dict.value.dictval))
        CHECK_DESTROY(dict_val(dict).keys[index].object, "store-object");
      dict_val(dict).keys[index].object = value;
      return ps_e_operationok;
      }
    }
  if (dict_val(dict).size == dict.size) return ps_e_dictfull;
  MORE_REFS(key);
  CHECK_DESTROY(dict_val(dict).keys[index].name, "store-name");
  dict_val(dict).keys[dict_val(dict).size].name   = key;
  if ((value.type          != ps_t_dict) ||
      (value.value.dictval != dict.value.dictval))
    MORE_REFS(value);
  CHECK_DESTROY(dict_val(dict).keys[index].object, "store-object");
  dict_val(dict).keys[dict_val(dict).size].object = value;
  dict_val(dict).size++;
  return ps_e_operationok;
  }

/* -------------------------***********************------------------------- */
ps__errors
ps__dictstack(ps__object array)
{
  int loop;

  if (!HAS_W(array))               return ps_e_invalidaccess;
  if (array.size < dictstack->size) return ps_e_rangecheck;
  for (loop=0; loop<dictstack->size; loop++) {
    CHECK_DESTROY(arr_ob(array, loop), "dictstack-array");
    arr_ob(array, loop) = dictstack->stack[loop];
    MORE_REFS(dictstack->stack[loop]);
    }
  return PUSH(opstack, array);
  }

/* ************************************************************************* */
void
init_dict_stuff()
{
  new_operator("def",            2, 0, NONE,  ps__def);
  new_operator("load",           1, 1, NONE,  ps__load);
  new_operator("dict",           1, 1, "i",   ps__dictop);
  new_operator("countdictstack", 0, 1, NONE,  ps__countdictstack);
  new_operator("begin",          1, 0, "d",   ps__begin);
  new_operator("end",            0, 0, NONE,  ps__end);
  new_operator("currentdict",    0, 1, NONE,  ps__currentdict);
  new_operator("known",          2, 1, "d.x", ps__known);
  new_operator("where",          1, 2, NONE,  ps__where);
  new_operator("store",          2, 0, NONE,  ps__store);
  new_operator("dictstack",      1, 1, "a",   ps__dictstack);
  }
