/* ************************************************************************* *
 * PostScript Interpretor                   Fabien LELAQUAIS                 *
 *                                                                           *
 *   Fichier com.c                                                           *
 *       Implementation of PostScript commands for PSint.                    *
 *                           Version 3.00 on 24/02/89                        *
 * ************************************************************************* *
 *    This document may be distributed, used, or modified, but can NOT be    *
 *  sold nor incorporated in any way in any product.                         *
 *    Permission is granted to distribute modified versions of that software *
 *  under the condition that this notice remains in every source file.       *
 *    Every alteration of the original files should be marked as such.       *
 *    No warranty is assumed by the author on the concequencies of the use   *
 *  of this software. Any defection of this program is at your own risk,     *
 *  you have to assume the cost of any service, installation or repairs      *
 *  this program could generate.                                             *
 *                                                                           *
 *                          Fabien LELAQUAIS - ESIEE - lelaquaf@apo.esiee.fr *
 * ************************************************************************* */
#include "int.h"

ps__object arraytype,          nametype,
           booleantype,        nulltype,
           dicttype,           operatortype,
           filetype,           realtype,
           fonttype,           savetype,
           integertype,        stringtype,
           marktype;

/* --------------------*********************************-------------------- *
 *                                  any POP -                                *
 * --------------------*********************************-------------------- */
ps__errors ps__pop(ps__object object)
{
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- *
 *                              any DUP any any                              *
 * --------------------*********************************-------------------- */
ps__errors ps__dup(ps__object object)
{
  PUSH(opstack, object);
  return PUSH(opstack, object);
  }

/* --------------------*********************************-------------------- *
 *                        any1 any2 EXCH any2 any1                           *
 * --------------------*********************************-------------------- */
ps__errors ps__exch(ps__object o1, ps__object o2)
{
  PUSH(opstack, o2);
  return PUSH(opstack, o1);
  }

/* --------------------*********************************-------------------- *
 *              anyn .. any1 any0 i INDEX anyn .. any1 any0 anyi             *
 * --------------------*********************************-------------------- */
ps__errors ps__index(ps__object index)
{
  int        i = int_val(index);
  ps__object o;

  if (i<0)              return ps_e_rangecheck;
  if (i>=opstack->size) return ps_e_stackunderflow;
  o = opstack->stack[opstack->size-i-1];
  return PUSH(opstack, o);
  }

/* --------------------*********************************-------------------- *
 *                          any ... any CLEAR  |-                            *
 * --------------------*********************************-------------------- */
ps__errors ps__clear()
{
  while (opstack->size)
    ps_destroy_object(POP(opstack), "clear");
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- *
 *                   any mark anyn ... any0 CLEARTOMARK any                  *
 * --------------------*********************************-------------------- */
ps__errors ps__cleartomark()
{
  int        index = stack_find(mark_object);

  if (index != -1) {
    index = opstack->size - index;
    while (index--)
      ps_destroy_object(POP(opstack), "cleartomark");
    return ps_e_operationok;
    }
  return ps_e_unmatchedmark;
  }

/* --------------------*********************************-------------------- *
 *            |- anyn ... any1 any0 COUNT anyn ... any1 any0 n               *
 * --------------------*********************************-------------------- */
ps__errors ps__count()
{
  return PUSH(opstack, new_int(opstack->size));
  }

/* --------------------*********************************-------------------- *
 *     mark anyn ... any1 any0 COUNTTOMARK mark anyn ... any1 any0 n         *
 * --------------------*********************************-------------------- */
ps__errors ps__counttomark()
{
  int index = stack_find(mark_object);
  return (index != -1)?PUSH(opstack, new_int(opstack->size-1-index))
                      :ps_e_unmatchedmark;
  }

/* --------------------*********************************-------------------- *
 *                             size ARRAY array                              *
 * --------------------*********************************-------------------- */
ps__errors ps__oparray(ps__object size)
{
  if (int_val(size)<0) return ps_e_rangecheck;
  return PUSH(opstack, new_array(int_val(size)));
  }

/* --------------------*********************************-------------------- *
 *                       array ALOAD a0 a1 ... an array                      *
 * --------------------*********************************-------------------- */
ps__errors ps__aload(ps__object array)
{
  int size = array.size, loop;

  if (!HAS_R(array))                         return ps_e_invalidaccess;
  if (opstack->size+size >= opstack->maxsize) return ps_e_stackoverflow;
  for (loop=0; loop<size; loop++) PUSH(opstack, arr_ob(array, loop));
  return PUSH(opstack, array);
  }

/* --------------------*********************************-------------------- *
 *                      a0 a1 ... an array ASTORE array                      *
 * --------------------*********************************-------------------- */
ps__errors ps__astore(ps__object array)
{
  int size = array.size;

  if (!HAS_W(array))       return ps_e_invalidaccess;
  if (opstack->size < size) return ps_e_stackunderflow;
  for (; size; size--) CHECK_DESTROY(arr_ob(array, size-1), "astore");
  for (size = array.size; size; size--) {
    arr_ob(array, size-1) = POP(opstack);
    MORE_REFS(arr_ob(array, size-1));
    }
  return PUSH(opstack, array);  
  }

/* --------------------*********************************-------------------- *
 *                           size STRING string                              *
 * --------------------*********************************-------------------- */
ps__errors ps__string(ps__object sizeob)
{
  ps__object string;
  unsigned   char *p;
  int        size = int_val(sizeob);

  if (size<0) return ps_e_rangecheck;
  string = new_object(ps_t_stg);
  string.flags |= DYNAMIC;
  if (!(p = stg_val(string) = (unsigned char *)ps_alloc(size)))
    return ps_e_VMerror;
  block_h(string) = HEADER(p);
  string.size = size;
  for (; size; size--) *p++ = 0;
  return PUSH(opstack, string);
  }

/* --------------------*********************************-------------------- *
 *               anyn ... any1 n COPY anyn ... any1 anyn ... any1            *
 *                     stg1 stg2 COPY substg                                 *
 *                 array1 array2 COPY subarray                               *
 *                   dict1 dict2 COPY subdict                                *
 * --------------------*********************************-------------------- */
ps__errors ps__copy(ps__object destination)
{
  ps__object source;
  int        l;

  if (destination.type != ps_t_int) {
    if (!opstack->size) return ps_e_stackunderflow;
    source = POP(opstack);
    if (destination.type != source.type) {
      PUSH(opstack, source);
      return ps_e_typecheck;
      }
    if (!HAS_W(destination) || !HAS_R(source)) {
      PUSH(opstack, source);
      return ps_e_invalidaccess;
      }
    switch (destination.type) {
      case ps_t_array :
        if ((l = source.size)> destination.size) return ps_e_rangecheck;
        for (; l; l--) {
          CHECK_DESTROY(arr_ob(destination, l-1), "copy-array");
          arr_ob(destination, l-1) = arr_ob(source, l-1);
          MORE_REFS(arr_ob(destination, l-1));
          }
        break;
      case ps_t_stg   :
        if ((l = source.size)> destination.size) return ps_e_rangecheck;
        for (; l; l--) stg_val(destination)[l-1] = stg_val(source)[l-1];
        break;
      case ps_t_dict  :
        if (((l = dict_val(source).size) > destination.size) ||
                  dict_val(destination).size) return ps_e_rangecheck;
        dict_val(destination).size = l;
        for (; l; l--) {
          CHECK_DESTROY(dict_val(destination).keys[l-1].name,
                        "copy-dict-name");
          CHECK_DESTROY(dict_val(destination).keys[l-1].object,
                        "copy-dict-object");
          dict_val(destination).keys[l-1] = dict_val(source).keys[l-1];
          MORE_REFS(dict_val(destination).keys[l-1].name);
          MORE_REFS(dict_val(destination).keys[l-1].object);
          }
        break;
      }
    return PUSH(opstack, destination);
    }
  else {
    int i = int_val(destination);
    ps__object ob;

    if ((l=i) > opstack->size)              return ps_e_stackunderflow;
    if (opstack->size+i > opstack->maxsize) return ps_e_stackoverflow;
    for (; i; i--) { 
      ob = POP(opstack);
      opstack->stack[opstack->size+l] = ob;
      if (ob.flags & DYNAMIC) REFS(ob) += 2;
      }
    opstack->size += 2*l;
    return ps_e_operationok;
    }
  }

/* --------------------*********************************-------------------- *
 *     anyn-1 ... any0 n j ROLL any(j-1)modn ... any0 anyn-1 .. anyjmodn     *
 * --------------------*********************************-------------------- */
ps__errors ps__roll(ps__object no, ps__object jo)
{
  extern void  ps_free();
  int          n = int_val(no), j = int_val(jo), loop;
  ps__object *objects;

  if (n<0)               return ps_e_rangecheck;
  if (n > opstack->size) return ps_e_stackunderflow;
  if (n) {
    if (!(objects = (ps__object *) ps_alloc(n*sizeof(ps__object))))
      return ps_e_VMerror;
    for (loop = n; loop; loop--) objects[loop-1] = POP(opstack);
    while (j<=0) j += n;
    while (j>n)  j -= n;
    for (loop = n-j; loop<n; loop++) PUSH(opstack, objects[loop]);
    for (loop = 0; loop<n-j; loop++) PUSH(opstack, objects[loop]);
    ps_free(objects);
    }
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- *
 *                         array  index any  PUT                             *
 *                         string index code PUT                             *
 *                         dict   key   any  PUT                             *
 * --------------------*********************************-------------------- */
ps__errors ps__put(ps__object object, ps__object index, ps__object value)
{
  if (!HAS_W(object)) return ps_e_invalidaccess;
  switch (object.type) {
    case ps_t_array : {
      int i;
      if (index.type != ps_t_int) return ps_e_typecheck;
      if (((i = int_val(index))<0) || (i>=object.size)) return ps_e_rangecheck;
      if ((arr_ob(object, i).flags & DYNAMIC) &&
          !(--REFS(arr_ob(object, i))))
       /* Cas particulier :  on ecrase un objet dynamique libre par lui meme */
        if (BLOCK(arr_ob(object, i)) != BLOCK(object))
          ps_destroy_object(arr_ob(object, i), "put");
      arr_ob(object, i) = value;
      MORE_REFS(value);
      return ps_e_operationok;
      }
    case ps_t_stg : {
      int i;
      if ((index.type != ps_t_int) ||
          (value.type != ps_t_int)) return ps_e_typecheck;
      if (((i = int_val(index))<0) || (i>=object.size)) return ps_e_rangecheck;
      stg_val(object)[i] = int_val(value);
      return ps_e_operationok;
      }
    case ps_t_dict :
      if (index.type == ps_t_null) return ps_e_typecheck;
      if (dict_store(object, index, value)) return ps_e_dictfull;
      return ps_e_operationok;
    }
  }

/* --------------------*********************************-------------------- *
 *                           array  index GET any                            *
 *                           string index GET code                           *
 *                           dict   key   GET any                            *
 * --------------------*********************************-------------------- */
ps__errors ps__get(ps__object object, ps__object index)
{
  if (!HAS_R(object)) return ps_e_invalidaccess;
  switch (object.type) {
    case ps_t_array : {
      int i;
      if (index.type != ps_t_int) return ps_e_typecheck;
      if (((i = int_val(index))<0) || (i>=object.size)) return ps_e_rangecheck;
      return PUSH(opstack, arr_ob(object, i));
      }
    case ps_t_stg : {
      int i;
      if (index.type != ps_t_int) return ps_e_typecheck;
      if (((i = int_val(index))<0) || (i>=object.size)) return ps_e_rangecheck;
      return PUSH(opstack, new_int(stg_val(object)[i]));
      }
    case ps_t_dict : {
      int where;
      if (index.type == ps_t_null) return ps_e_typecheck;
      return ((where = dict_find_index(object, index)) == -1)?ps_e_undefined
            :PUSH(opstack, dict_val(object).keys[where].object);
      }
    }
  }

/* --------------------*********************************-------------------- *
 *                             any RCHECK bool                               *
 * --------------------*********************************-------------------- */
ps__errors ps__rcheck(ps__object object)
{                            
  return PUSH(opstack, new_boolean(HAS_R(object)));
  }

/* --------------------*********************************-------------------- *
 *                             any WCHECK bool                               *
 * --------------------*********************************-------------------- */
ps__errors ps__wcheck(ps__object object)
{
  return PUSH(opstack, new_boolean(HAS_W(object)));
  }

/* --------------------*********************************-------------------- *
 *                             any XCHECK bool                               *
 * --------------------*********************************-------------------- */
ps__errors ps__xcheck(ps__object object)
{
  return PUSH(opstack, new_boolean(HAS_X(object)));
  }

/* --------------------*********************************-------------------- *
 *                             any READONLY any                              *
 * --------------------*********************************-------------------- */
ps__errors ps__readonly(ps__object object)
{
  return (HAS_W(object))?PUSH(opstack, cvro(object)):ps_e_invalidaccess;
  }

/* --------------------*********************************-------------------- *
 *                           any EXECUTEONLY any                             *
 * --------------------*********************************-------------------- */
ps__errors ps__executeonly(ps__object object)
{
  return (HAS_W(object))?PUSH(opstack, cvxo(object)):ps_e_invalidaccess;
  }

/* --------------------*********************************-------------------- *
 *                             any NOACCESS any                              *
 * --------------------*********************************-------------------- */
ps__errors ps__noaccess(ps__object object)
{
  return (HAS_W(object))?PUSH(opstack, cvlit(cvxo(object))):ps_e_invalidaccess;
  }

/* --------------------*********************************-------------------- *
 *                             any TYPE typename                             *
 * --------------------*********************************-------------------- */
ps__errors ps__typeop(ps__object object)
{
  switch(object.type) {
    case ps_t_array    : return PUSH(opstack, arraytype);
    case ps_t_name     : return PUSH(opstack, nametype);
    case ps_t_bool     : return PUSH(opstack, booleantype);
    case ps_t_null     : return PUSH(opstack, nulltype);
    case ps_t_dict     : return PUSH(opstack, dicttype);
    case ps_t_operator : return PUSH(opstack, operatortype);
    case ps_t_file     : return PUSH(opstack, filetype);
    case ps_t_real     : return PUSH(opstack, realtype);
    case ps_t_fontId   : return PUSH(opstack, fonttype);
    case ps_t_save     : return PUSH(opstack, savetype);
    case ps_t_int      : return PUSH(opstack, integertype);
    case ps_t_stg      : return PUSH(opstack, stringtype);
    case ps_t_mark     : return PUSH(opstack, marktype);
    default : return PUSH(opstack, invalid_object);
    }
  }

/* --------------------*********************************-------------------- *
 *                  string seek ANCHORSEARCH post match true                 *
 *           or                              string false                    *
 * --------------------*********************************-------------------- */
ps__errors ps__anchorsearch(ps__object string, ps__object seek)
{
  unsigned char *p = stg_val(string), *q = stg_val(seek);
  unsigned short l = seek.size;

  if (!HAS_R(string)) return ps_e_invalidaccess;
  if (l <= string.size) {
    while (l && (*p == *q)) l--, p++, q++;
    if (!l) {
      PUSH(opstack, new_string((char *)p,
                               (unsigned long)string.size-seek.size));
      PUSH(opstack, seek);
      return PUSH(opstack, true_object);
      }
    }
  PUSH(opstack, string);
  return PUSH(opstack, false_object);
  }

/* --------------------*********************************-------------------- *
 *                   string seek SEARCH post match pre true                  *
 *           or                         string false                         *
 * --------------------*********************************-------------------- */
ps__errors ps__search(ps__object string, ps__object seek)
{
  unsigned short lstring = string.size, lseek = seek.size,
                 lse;
  unsigned char *st = stg_val(string);
  char          *se = (char *)stg_val(seek),
                *pst,
                *pse;

  if (!HAS_R(string)) return ps_e_invalidaccess;
  while (lstring >= lseek) {
    lse = lseek;
    pst = (char *)st;
    pse = se;
    while (lse && (*pst == *pse))
      lse--, pst++, pse++;
    if (lse) lstring--, st++;
    else {
      PUSH(opstack, new_string(pst, (unsigned long)(lstring-lseek)));
      PUSH(opstack, seek);
      PUSH(opstack, new_string((char *)stg_val(string),
                               (unsigned long)(st-stg_val(string))));
      return PUSH(opstack, true_object);
      }
    }
  PUSH(opstack, string);
  return PUSH(opstack, false_object);
  }

/* --------------------*********************************-------------------- *
 *                       array1 index array2  PUTINTERVAL                    *
 *           or         string1 index string2 PUTINTERVAL                    *
 * --------------------*********************************-------------------- */
ps__errors ps__putinterval(ps__object ob1, ps__object index, ps__object ob2)
{
  int idx = int_val(index), i;

  if ((ob1.type != ob2.type) ||
      ((ob1.type != ps_t_stg) && (ob1.type != ps_t_array)))
                                  return ps_e_typecheck;
  if (!HAS_R(ob2) || !HAS_W(ob1)) return ps_e_invalidaccess;
  if (idx+ob2.size > ob1.size)    return ps_e_rangecheck;
  if (ob1.type == ps_t_stg)
    for (i=0; i<ob2.size; i++) stg_val(ob1)[i+idx] = stg_val(ob2)[i];
  else
    for (i=0; i<ob2.size; i++) {
      CHECK_DESTROY(arr_ob(ob1, i+idx), "putinterval");
      arr_ob(ob1, i+idx) = arr_ob(ob2, i);
      MORE_REFS(arr_ob(ob2, i));
      }
  return ps_e_operationok;
  }

/* --------------------*********************************-------------------- *
 *                   array index count GETINTERVAL subarray                  *
 *           or     string index count GETINTERVAL substring                 *
 * --------------------*********************************-------------------- */
ps__errors ps__getinterval(ps__object object, ps__object index,
                                              ps__object count)
{
  int        idx = int_val(index), cnt = int_val(count);

  if (!HAS_R(object))              return ps_e_invalidaccess;
  if ((object.type != ps_t_stg) &&
      (object.type != ps_t_array)) return ps_e_typecheck;
  if (idx+cnt > object.size)       return ps_e_rangecheck;
  if (object.type == ps_t_stg)
    return PUSH(opstack, new_string((char *)(stg_val(object)+idx), cnt));
  else {
    ps__object result;
    int        i;
    result = new_array(cnt);
    for (i=0; i<cnt; i++) {
      arr_ob(result, i) = arr_ob(object, i+idx);
      MORE_REFS(arr_ob(result, i));
      }
    return PUSH(opstack, result);
    }
  }

/* --------------------*********************************-------------------- *
 *                               USERTIME int                                *
 * --------------------*********************************-------------------- */
ps__errors ps__usertime()
{
  long time();
  return PUSH(opstack, new_int((time(0)%10000) * 1000));
  }

/* ************************************************************************* */
void init_com_stuff()
{
  arraytype    = do_name("arraytype");
  nametype     = do_name("nametype");
  booleantype  = do_name("booleantype");
  nulltype     = do_name("nulltype");
  dicttype     = do_name("dicttype");
  operatortype = do_name("operatortype");
  filetype     = do_name("filetype");
  realtype     = do_name("realtype");
  fonttype     = do_name("fonttype");
  savetype     = do_name("savetype");
  integertype  = do_name("integertype");
  stringtype   = do_name("stringtype");
  marktype     = do_name("marktype");
  new_operator("pop",          1, 0, NONE,      ps__pop);
  new_operator("dup",          1, 2, NONE,      ps__dup);
  new_operator("exch",         2, 2, NONE,      ps__exch);
  new_operator("index",        1, 1, "i",       ps__index);
  new_operator("clear",        0, 0, NONE,      ps__clear);
  new_operator("cleartomark",  0, 0, NONE,      ps__cleartomark);
  new_operator("count",        0, 1, NONE,      ps__count);
  new_operator("counttomark",  0, 1, NONE,      ps__counttomark);
  new_operator("array",        1, 1, "i",       ps__oparray);
  new_operator("aload",        1, 1, "a",       ps__aload);
  new_operator("astore",       1, 1, "a",       ps__astore);
  new_operator("string",       1, 1, "i",       ps__string);
  new_operator("copy",         1, 1, "iasd",    ps__copy);
  new_operator("roll",         2, 0, "i.i",     ps__roll);
  new_operator("put",          3, 0, "ads.x.x", ps__put);
  new_operator("get",          2, 1, "ads.x",   ps__get);
  new_operator("rcheck",       1, 1, "adFs",    ps__rcheck);
  new_operator("wcheck",       1, 1, "adFs",    ps__wcheck);
  new_operator("xcheck",       1, 1, NONE,      ps__xcheck);
  new_operator("readonly",     1, 1, "adFs",    ps__readonly);
  new_operator("executeonly",  1, 1, "aFs",     ps__executeonly);
  new_operator("noaccess",     1, 1, "adFs",    ps__noaccess);
  new_operator("type",         1, 1, NONE,      ps__typeop);
  new_operator("anchorsearch", 2, 3, "s.s",     ps__anchorsearch);
  new_operator("search",       2, 4, "s.s",     ps__search);
  new_operator("putinterval",  3, 0, "x.i.x",   ps__putinterval);
  new_operator("getinterval",  3, 1, "x.i.i",   ps__getinterval);
  new_operator("usertime",     0, 1, NONE,      ps__usertime);
  }
