/* ************************************************************************* *
 * PostScript Interpretor                   Fabien LELAQUAIS                 *
 *                                                                           *
 *   Fichier Borow.c                                                         *
 *    Create PostScript Interpretor routines so that APOLLO stations can     *
 *  manage all graphic routines in Borrow mode.                              *
 *    This mode uses the 'mmlib', creating and managing menus and/or windows *
 *                           Version 3.50 on 18/12/90                        *
 * ************************************************************************* *
 *    This document may be distributed, used, or modified, but can NOT be    *
 *  sold nor incorporated in any way in any product.                         *
 *    Permission is granted to distribute modified versions of that software *
 *  under the condition that this notice remains in every source file.       *
 *    Every alteration of the original files should be marked as such.       *
 *    No warranty is assumed by the author on the concequencies of the use   *
 *  of this software. Any defection of this program is at your own risk,     *
 *  you have to assume the cost of any service, installation or repairs      *
 *  this program could generate.                                             *
 *                                                                           *
 *                          Fabien LELAQUAIS - ESIEE - lelaquaf@apo.esiee.fr *
 * ************************************************************************* */
#include "int.h"
#include "graph.h"
#include <apollo/base.h>
#include <apollo/gpr.h>
#include <apollo/kbd.h>
#include <apollo/error.h>
#include "/user/lelaquaf/sys/ins/mm.h"

#ifdef REV9_7
#define STATUS  status
#else
#define STATUS &status
#endif /* REV9_7 */

#define Gpr_triangle(x, y, z) gpr_$triangle(x, y, z, STATUS)
#define Gpr_set_bitmap(x)     gpr_$set_bitmap((gpr_$bitmap_desc_t  ) x, STATUS)

extern ps__cache *currentcache;
typedef struct _bitmap {
  int                   sx, sy;
  gpr_$bitmap_desc_t    bitmap;
  gpr_$attribute_desc_t attributes;
  } *bitmap;

short page_x, page_y;

char hide_command_done = 0;

status_$t             status;
gpr_$bitmap_desc_t    bigbm, screenbm;
gpr_$offset_t         bigsize,  screensize;
gpr_$attribute_desc_t bigattrib;
gpr_$keyset_t         buttonset, keyset, smallkeyset;
gpr_$window_t         dispwin;
gpr_$position_t       disppos;
gpr_$rgb_plane_t      planes;
short                 colors;


/* ========================================================================= */
void
cs(char *m)
{
  if (status.all) {
    puts(m);
    error_$print(status);
    }
  }


#define LINE_NUMBER 20
#define LINE_LENGTH 80
#define DWINX 800
#define DWINY 450
mm_$window_p       present_window, dialogue_window;
mm_$line_p         dialogue_lines[LINE_NUMBER];
typedef char       dialogue_text[LINE_LENGTH];
dialogue_text      my_texts[LINE_NUMBER];
char              *dialogue_texts[LINE_LENGTH];
mm_$gadget_p       dialogue_gadget;
mm_$string_info_t  dialogue_info;
mm_$menu_p         main_menu;
mm_$color_vector_t color_vector;
gpr_$position_t    dialogue_window_pos;
short              dialogue_window_shown = 0,
                   current_dialogue_x = 0;
char              *input_buffer, *bpointer;
typedef struct {
  mm_$text_t               text;
  short                    length;
  short                    pos_x, pos_y;
  mm_$gadget_type_t        type;
  mm_$gadget_flags_t       flags;
  char                    *info;
  gpr_$offset_t            size;
  } *special_gadget;

/* ************************************************************************* */
void
display_page()
{
  gpr_$set_cursor_active(false, STATUS);
  if (bigbm) {
    if (page_x<0) page_x = 0;
    else if (page_x+screensize.x_size > bigsize.x_size)
      page_x = bigsize.x_size-screensize.x_size;
    if (page_y<0) page_y = 0;
    else if (page_y+screensize.y_size > bigsize.y_size)
      page_y = bigsize.y_size-screensize.y_size;
    dispwin.window_base.x_coord = page_x;
    dispwin.window_base.y_coord = page_y;
    gpr_$bit_blt(bigbm, dispwin, 0, disppos, planes, STATUS);
    }
  else
    gpr_$clear(0L, STATUS);
  gpr_$set_cursor_active(true, STATUS);
  }

/* ************************************************************************* */
ps__errors
ps__hide()
{
  hide_dialogue_window();
  return ps_e_operationok;
  }

/* ************************************************************************* */
ps__errors
ps__display()
{
  show_dialogue_window();
  return ps_e_operationok;
  }

/* ************************************************************************* */
ps__errors
ps__borrowshowpage()
{
  extern ps__errors showpage();

  ps__hide();
  return ps__showpage();
  }

/* ************************************************************************* */
init_device_stuff()
{  
  short                 old_page_x, old_page_y, i, j, font;
  char                  stg[20];
  gpr_$color_vector_t   cmap;
  gpr_$display_config_t config;

  gpr_$inq_config(&config, STATUS);
  switch(config) {         /* Graphisme en fonction du type de visu utilise */
    case gpr_$color_1024x800x4  :                              /* DN550/560 */
    case gpr_$color2_1024x800x4 :                                /* DN3000C */
      screensize.x_size = 1024; screensize.y_size =  800;
      colors = 15;
      planes = 3;
      break;
    case gpr_$color_1024x800x8  :                              /* DN550/560 */
    case gpr_$color1_1024x800x8 :                              /* DN570     */
    case gpr_$color2_1024x800x8 :                              /* DN3000E   */
      screensize.x_size = 1024; screensize.y_size =  800;
      planes = 7;
      colors = 255;
      break;
    case gpr_$bw_1280x1024      :                                /* DN3000M */
      screensize.x_size = 1280; screensize.y_size = 1024;
      planes = 0;
      colors = 0;
      break;
    default :
      printf("Sorry... Non-registred screen type (%d).\n\n", config);
      return -1;
    }
  gpr_$init(gpr_$borrow, 1, screensize, planes, &screenbm, STATUS);
  lib_$init_set(keyset, 256);
  for (i=0; i< 256; i++) lib_$add_to_set(keyset, 256, i);
  lib_$init_set(smallkeyset, 256);
  lib_$add_to_set(smallkeyset, 256, 'Q');
  lib_$add_to_set(smallkeyset, 256, 'q');
  lib_$add_to_set(smallkeyset, 256, KBD_$POP);
  lib_$add_to_set(smallkeyset, 256, KBD_$HELP);
  lib_$add_to_set(smallkeyset, 256, KBD_$MOVE);
  lib_$add_to_set(smallkeyset, 256, KBD_$UP_ARROW);
  lib_$add_to_set(smallkeyset, 256, KBD_$DOWN_ARROW);
  lib_$add_to_set(smallkeyset, 256, KBD_$LEFT_ARROW);
  lib_$add_to_set(smallkeyset, 256, KBD_$RIGHT_ARROW);
  if (planes) {
    cmap[        0] = 0x000000;
    cmap[1<<planes] = 0xF0F0F0;
    gpr_$set_color_map(0L, colors, cmap, STATUS);
    }
  disppos.x_coord = 0;
  disppos.y_coord = 0;
  page_x          = 0;
  page_y          = 0;
  bigbm           = 0;
  return init_menus();
  }

/* ************************************************************************* *
 * Menus :                                                                   *
 *                                                                           *
 *    Hide Text Window                                                       *
 *    Leave Interpretor                                                      *
 * ************************************************************************* */
init_menus()
{
  short             i, j;
  char             *p;
  mm_$event_t       mm_ev;
  mm_$gadget_p      mm_gadget;
  short             mm_data;
  unsigned char     ed;
  gpr_$event_t      et;
  gpr_$position_t   epos;

  color_vector.grey      = 1;
  color_vector.dark_grey = 2;
  color_vector.white     = 3;
  color_vector.yellow    = 4;
  mm_$init(true, color_vector, STATUS); cs("mm init");
  if (!planes) mm_$tell_display_mode(gpr_$direct, STATUS);
  mm_$draw_cursor(true, STATUS);
  lib_$init_set(buttonset,256);
  lib_$add_to_set(buttonset,256,'a');
  lib_$add_to_set(buttonset,256,'b');
  lib_$add_to_set(buttonset,256,'c');
  lib_$add_to_set(buttonset,256,'A');
  lib_$add_to_set(buttonset,256,'B');
  lib_$add_to_set(buttonset,256,'C');

  mm_$open_menu(STATUS);
  mm_$add_option_to_menu(mm_$create_option("Exit Interpretor",
                                           0, 2, mm_$no_sub_menu, 0, STATUS),
                         STATUS);
  mm_$add_option_to_menu(mm_$create_option("Show Text Window",
                                           0, 1, mm_$no_sub_menu, 0, STATUS),
                         STATUS);
  main_menu = mm_$close_menu(STATUS);

  mm_$open_window("BorrowPSint presentation", 0, 500, 220, STATUS);
  mm_$add_line_to_window(mm_$create_line("  This version is the 'Borrow' version of",
                                         0, 5, 40, 0, STATUS), STATUS);
  mm_$add_line_to_window(mm_$create_line("PSint (revision 3), which is a PostScript",
                                         0, 5, 60, 0, STATUS), STATUS);
  mm_$add_line_to_window(mm_$create_line("Interpretor.",  0, 5, 80, 0, STATUS),
                         STATUS);
  mm_$add_line_to_window(mm_$create_line("  It has been  entirely written at  ESIEE",
                                         0, 5, 105, 0, STATUS), STATUS);
  mm_$add_line_to_window(mm_$create_line("(France) by F.Lelaquais.",  0, 5, 125, 0,
                                         STATUS),
                         STATUS);
  mm_$add_line_to_window(mm_$create_line("  Windows and menus are Lelaquais's too.",
                                         0, 5, 150, 0, STATUS), STATUS);
  mm_$add_line_to_window(mm_$create_line("  Hope you'll enjoy your time.",
                                         0, 5, 175, 0, STATUS), STATUS);
  present_window = mm_$close_window(STATUS);

  mm_$open_window("BorrowPSint", 0, DWINX, DWINY, STATUS);
  for (i=0; i<LINE_NUMBER; i++) {
    dialogue_texts[i] = my_texts[i];
    for (p=dialogue_texts[i], j=0; j<LINE_LENGTH; j++, *p++ = ' ');
    dialogue_lines[i] = mm_$create_line(dialogue_texts[i], LINE_LENGTH,
                                        5, 25+20*i, 0, STATUS);
    mm_$add_line_to_window(dialogue_lines[i], STATUS);
    }
  dialogue_window = mm_$close_window(STATUS);
  mm_$desable_event(dialogue_window, mm_$window_left, STATUS);

  dialogue_gadget = mm_$create_gadget(NULL, 0, 0, 0, 80, 0, mm_$gadget_string,
                                      STATUS);
  mm_$create_string_info(dialogue_gadget, dialogue_info, STATUS);
  bpointer = input_buffer = dialogue_info.text;
  *bpointer = 0;

  dialogue_window_pos.x_coord = screensize.x_size>>1;
  dialogue_window_pos.y_coord = screensize.y_size>>1;
  mm_$draw_window(present_window, dialogue_window_pos, STATUS);
  do mm_$event_wait(&mm_ev, &mm_gadget, STATUS);
  while (mm_ev != mm_$window_left);
  mm_$erase_window(STATUS);
  show_dialogue_window();
  return 0;
  }

/* ************************************************************************* */
user_cmd()
{
  mm_$event_t       mm_ev;
  mm_$gadget_p      mm_gadget;
  short             mm_data,
                    old_page_x = -1,
                    old_page_y = -1;
  mm_$option_code_t code;
  unsigned char     ed;
  gpr_$event_t      et;
  gpr_$position_t   epos;

  while (!dialogue_window_shown) {
    if ((old_page_x != page_x) || (old_page_y != page_y)) display_page();
    old_page_x = page_x;
    old_page_y = page_y;
    gpr_$set_cursor_active(true, STATUS);
    gpr_$enable_input(gpr_$buttons, keyset, STATUS);
    gpr_$event_wait(&et, &ed, &epos, STATUS);
    switch(et) {
      case gpr_$buttons :
        if (ed == 'b') {
          code = mm_$start_menu(main_menu, epos, STATUS);
          switch(code) {
            case 1 :
              show_dialogue_window();
              break;
            case 2 :
              while (execstack->size) POP(execstack);
              return '\n';
            }
          }
      case gpr_$keystroke :
        switch (ed) {
          case 'q' : case 'Q'   : show_dialogue_window(); break;
          case KBD_$POP         : break;
          case KBD_$HELP        : break;
          case KBD_$MOVE        : break;
          case KBD_$UP_ARROW    : page_y -= 100; break;
          case KBD_$DOWN_ARROW  : page_y += 100; break;
          case KBD_$LEFT_ARROW  : page_x -= 100; break;
          case KBD_$RIGHT_ARROW : page_x += 100; break;
          }
        break;
      }
    }
  gpr_$disable_input(gpr_$buttons,   STATUS);
  return ps__getchar();
  }

/* ------------------------------------------------------------------------- */
void
show_dialogue_window()
{
  if (!dialogue_window_shown) {
    mm_$draw_window(dialogue_window, dialogue_window_pos, STATUS);
    gpr_$disable_input(gpr_$keystroke, STATUS);
    gpr_$enable_input(gpr_$keystroke, keyset, STATUS);
    dialogue_window_shown = 1;
    }
  }

/* ------------------------------------------------------------------------- */
void
hide_dialogue_window()
{
  if (dialogue_window_shown) {
    mm_$erase_window(STATUS);
    gpr_$disable_input(gpr_$keystroke, STATUS);
    gpr_$enable_input(gpr_$keystroke, smallkeyset, STATUS);
    dialogue_window_shown = 0;
    }
  }

/* ------------------------------------------------------------------------- */
gpr_$pixel_value_t
choose_color()
{
  gpr_$pixel_value_t gray;

  gray = (currentcache)? 1L
                       : ((to_real(state->gray)<.5) ? 0 : 1);
  gpr_$set_draw_value(gray, STATUS);
  gpr_$set_fill_value(gray, STATUS);
  return gray;
  }

/* ************************************************************************* */
void
ps__WritePixels(int x, int y, int length, unsigned char *buf)
{
  unsigned char      *p;
  gpr_$window_t       window;
  gpr_$pixel_value_t *pbuf, *pp, *malloc();

  if (x < 0) {
    puts("** Negative X coordinate writing pixels **");
    return;
    }
  if (!length) return; /* Fin d'image */
  if (buf) {                                                        /* Image */
    if (pbuf = malloc(length*sizeof(gpr_$pixel_value_t))) {
      Gpr_set_bitmap(bigbm);
      window.window_base.x_coord = x;
      window.window_base.y_coord = y;
      window.window_size.x_size  = 1;
      window.window_size.y_size  = length;
      for (p = buf, pp = pbuf; length; length--)
        *pp++ = ((*p++)>128)?1:0;
      gpr_$write_pixels(pbuf, window, STATUS);
      cs("WritePixels (WritePixels)");
      free(pbuf);
      }
    }
  else {            /* Ligne verticale de pixels de meme couleur (imagemask) */
    if (currentcache) Gpr_set_bitmap(((bitmap)(currentcache->device))->bitmap);
    else              Gpr_set_bitmap(bigbm);
    choose_color();
    gpr_$move((short)x, (short)y, STATUS);
    gpr_$line((short)x, (short)(y+length), STATUS);
    }
  Gpr_set_bitmap(screenbm);
  }


/* ------------------------------------------------------------------------- */
void
ps__DoTrapeze(int ul, int ll, int ur, int lr, int left, int right)
{
  gpr_$position_t    p1, p2, p3;

  if (currentcache) Gpr_set_bitmap(((bitmap)(currentcache->device))->bitmap);
  else              Gpr_set_bitmap(bigbm);
  choose_color();
  if (ul == ll) {
    p1.x_coord = left;   p1.y_coord = ul;
    p2.x_coord = right;  p2.y_coord = ur;
    p3.x_coord = right;  p3.y_coord = lr;
    }
  else {
    p1.x_coord = left;   p1.y_coord = ul;
    p2.x_coord = left;   p2.y_coord = ll;
    p3.x_coord = right;  p3.y_coord = ur;
    if (ur != lr) {
      Gpr_triangle(p1, p2, p3);
      p1.x_coord = right; p1.y_coord = lr;
      }
    }
  Gpr_triangle(p1, p2, p3);
  Gpr_set_bitmap(screenbm);
  }

/* ------------------------------------------------------------------------- */
void
ps__DoLine(ps__coord p0, ps__coord p1)
{
  if (currentcache) Gpr_set_bitmap(((bitmap)(currentcache->device))->bitmap);
  else              Gpr_set_bitmap(bigbm);
  choose_color();
  gpr_$move((short)p0.x, (short)p0.y, STATUS);
  gpr_$line((short)p1.x, (short)p1.y, STATUS);
  Gpr_set_bitmap(screenbm);
  }

/* ------------------------------------------------------------------------- */
void *
ps__CreateWindow(int sx, int sy)
{
  bigsize.x_size = sx;
  bigsize.y_size = sy;
  gpr_$allocate_attribute_block(&bigattrib, STATUS); cs("allocate_attribute_block");
  gpr_$allocate_bitmap(bigsize, 0, bigattrib, &bigbm, STATUS); cs("allocate_bitmap");

  Gpr_set_bitmap(bigbm);
  gpr_$clear((gpr_$pixel_value_t)1, STATUS);
  gpr_$set_draw_value(0, STATUS);
  gpr_$move(               0,                0, STATUS);
  gpr_$line(bigsize.x_size-1,                0, STATUS);
  gpr_$line(bigsize.x_size-1, bigsize.y_size-1, STATUS);
  gpr_$line(               0, bigsize.y_size-1, STATUS);
  gpr_$line(               0,                0, STATUS);
  Gpr_set_bitmap(screenbm);
  dispwin.window_size.x_size = (bigsize.x_size < screensize.x_size) ?
                                bigsize.x_size : screensize.x_size;
  dispwin.window_size.y_size = (bigsize.y_size < screensize.y_size) ?
                                bigsize.y_size : screensize.y_size;  
  return bigbm;
  }

/* ------------------------------------------------------------------------- */
void
ps__KillWindow(gpr_$bitmap_desc_t bm)
{
  if (bm != bigbm)
    puts("** Strange.... KillWindow of un unknown window **");
  else {
    if (bigbm) {
      gpr_$deallocate_bitmap(bigbm, STATUS);
      gpr_$deallocate_attribute_block(bigattrib, STATUS);
      bigbm = 0;
      }
    }
  }

/* ************************************************************************* */
void *
ps__CreateCache(int sx, int sy)
{
  bitmap             cache;
  gpr_$offset_t      size;

  if (!(cache = (bitmap)ps_alloc(sizeof(struct _bitmap))))
    return NULL;
  size.x_size = cache->sx = sx;
  size.y_size = cache->sy = sy;
#ifdef REV9_7
  gpr_$allocate_attribute_block(cache->attributes, STATUS);
#else
  gpr_$allocate_attribute_block(&cache->attributes, STATUS);
#endif /* REV9_7 */
  cs("AllocateAttributeBlock (CreateCache)");
  gpr_$allocate_bitmap(size, 0, cache->attributes,
#ifdef REV9_7
                       cache->bitmap,
#else
                       &cache->bitmap,
#endif /* REV9_7 */
                       STATUS);
  cs("AllocateBitmap (CreateCache)");
  Gpr_set_bitmap(cache->bitmap);
  gpr_$clear(0L, STATUS);
  Gpr_set_bitmap(screenbm);
  return cache;
  }

/* ************************************************************************* */
void
ps__KillCache(bitmap b)
{
  extern void ps_free();

  gpr_$deallocate_bitmap(b->bitmap, STATUS);
  cs("DeallocateBitmap (KillCache)");
  gpr_$deallocate_attribute_block(b->attributes, STATUS);
  cs("DeallocateAttributeBlock (KillCache)");
  ps_free(b);
  }

/* ************************************************************************* */
void
ps__CloseCache()
{
  }

/* ************************************************************************* */
void
ps__DoCache(ps__cache *c, ps__coord coord)
{
  gpr_$window_t      window;
  gpr_$position_t    position;
  gpr_$pixel_value_t color;

  /* Grand coup de pixel_blt :
       Le bitmap du cache n'a qu'un plan.
       On suppose que chaque pixel pouvant etre modifie est a '1'
     les autres, a 0.
  Lors du pixel_blt, les operations suivantes doivent etre programmees :

   Pour chaque bit 'bit' de la couleur courante, 'src' etant le bitmap cache :
            bit    src     dst    resultat
          +-----+-------+-------++---------+
          |  0  |   0   |   0   ||    0    |
          |  0  |   0   |   1   ||    1    | soit : !src && dst
          |  0  |   1   |   0   ||    0    |
          |  0  |   1   |   1   ||    0    |
          +--------------------------------+
          |  1  |   0   |   0   ||    0    |
          |  1  |   0   |   1   ||    1    | soit :  src || dst
          |  1  |   1   |   0   ||    1    |
          |  1  |   1   |   1   ||    1    |
          +-----+-------+-------++---------+
   */
  Gpr_set_bitmap(bigbm);
  color = choose_color();
  window.window_base.x_coord = window.window_base.y_coord = 0;
  window.window_size.x_size = ((bitmap)(c->device))->sx;
  window.window_size.y_size = ((bitmap)(c->device))->sy;
  position.x_coord = coord.x+c->cx;
  position.y_coord = coord.y+c->cy;
  if (color) gpr_$set_raster_op(0, gpr_$rop_src_or_dst,      STATUS);
  else       gpr_$set_raster_op(0, gpr_$rop_not_src_and_dst, STATUS);
  gpr_$bit_blt(((bitmap)(c->device))->bitmap, window, 0, position, 0, STATUS);
  gpr_$set_raster_op(planes, gpr_$rop_src, STATUS);
  Gpr_set_bitmap(screenbm);
  }

/* ------------------------------------------------------------------------- */
void
ps__Erase()
{
  Gpr_set_bitmap(bigbm);
  gpr_$clear((gpr_$pixel_value_t)1, STATUS);
  Gpr_set_bitmap(screenbm);
  }

/* ------------------------------------------------------------------------- */
void
free_device_stuff()
{
  mm_$terminate(STATUS);
  if (bigbm) {
    gpr_$deallocate_bitmap(bigbm, STATUS);
    gpr_$deallocate_attribute_block(bigattrib, STATUS);
    bigbm = 0;
    }
  gpr_$terminate(true, STATUS);
  }

/* ------------------------------------------------------------------------- */
ps__getchar()
{
  char        *p;
  int          gadget_length;
  mm_$event_t  mm_ev;
  mm_$gadget_p mm_gadget;

  if (!hide_command_done) {
    ps__object showpage;

    showpage = dict_find_key(systemdict, do_name("showpage"));
    op_val(showpage).operator = ps__borrowshowpage;
    new_operator("hide",    0, 0, NONE,  ps__hide);
    new_operator("display", 0, 0, NONE,  ps__display);
    hide_command_done = 1;
    }
  if (!dialogue_window_shown) return user_cmd();
  if (!(*bpointer)) {
    mm_$clear_string_info(dialogue_gadget, STATUS);
    gadget_length = 80-current_dialogue_x;
    dialogue_info.max_length     = gadget_length;
    dialogue_info.max_box_length = gadget_length;
    ((special_gadget)(dialogue_gadget))->pos_x =
                                          4+mm_$font_width*current_dialogue_x;
    ((special_gadget)(dialogue_gadget))->pos_y =
                                               LINE_NUMBER*mm_$font_height+26;
    mm_$add_gadget(dialogue_window, dialogue_gadget, STATUS);
    mm_$move_cursor_into_gadget(dialogue_gadget, STATUS);
    do mm_$event_wait(&mm_ev, &mm_gadget, STATUS);
    while (mm_ev != mm_$gadget_released);
    mm_$remove_gadget(dialogue_window, dialogue_gadget, STATUS);
    input_buffer[dialogue_info.length]   = '\n';
    input_buffer[dialogue_info.length+1] =   0;
    for (p = bpointer = input_buffer; *p; p++) ps__putchar(*p);
    }
  return *bpointer++;
  }

/* ------------------------------------------------------------------------- */
ps__getc(FILE *f)
{
  if (f == stdin) return ps__getchar();
  else            return getc(f);
  }

/* ------------------------------------------------------------------------- */
ps__ungetc(int c, FILE *f)
{
  if (f != stdin) return ungetc(c,f);
  if (bpointer != input_buffer) bpointer--;
  return *bpointer;
  }

/* ------------------------------------------------------------------------- */
ps__fflush(FILE *f)
{
  if ((f == stdout) && (dialogue_window_shown)) {
    mm_$change_line(dialogue_lines[LINE_NUMBER-1],
                    dialogue_texts[LINE_NUMBER-1], LINE_LENGTH, -1, STATUS);
    return 1;
    }
  else
    return fflush(f);
  }

/* ------------------------------------------------------------------------- */
ps__putchar(int c)
{
  char *line;
  short i;

  if (((c&127)<32) && (c != '\n')) c = '?';
  if ((c == '\n') || (current_dialogue_x == LINE_LENGTH)) {
    line = dialogue_texts[0];
    for (i=0; i<LINE_NUMBER-1; i++) {
      dialogue_texts[i] = dialogue_texts[i+1];
      if (dialogue_window_shown)
        mm_$change_line(dialogue_lines[i],
                        dialogue_texts[i], LINE_LENGTH, -1, STATUS);
      }
    dialogue_texts[LINE_NUMBER-1] = line;
    for (i=0; i<LINE_LENGTH; i++) dialogue_texts[LINE_NUMBER-1][i] = ' ';
    if (dialogue_window_shown)
        mm_$change_line(dialogue_lines[LINE_NUMBER-1],
                        dialogue_texts[LINE_NUMBER-1],
                        LINE_LENGTH, -1, STATUS);
    current_dialogue_x = 0;
    }
  else dialogue_texts[LINE_NUMBER-1][current_dialogue_x++] = c;
  return;
  }

/* ------------------------------------------------------------------------- */
ps__putc(int c, FILE *f)
{ if ((f == stdout) || (f == stderr))
    return ps__putchar(c);
  else
    return putc(c, f);
  }

/* ------------------------------------------------------------------------- */
ps__puts(char *m)
{
  char *p = m;

  for (; *p; p++) ps__putchar(*p);
  ps__putchar('\n');
  }

/* ------------------------------------------------------------------------- */
ps__printf(char *format, int arguments)
{
  char *args = (char *)&arguments, buffer[1024], *p = buffer;

  vsprintf(buffer, format, args);
  for (; *p; p++) ps__putchar(*p);
  }

