/* ************************************************************************* *
 * PostScript Interpretor                   Fabien LELAQUAIS                 *
 *                                                                           *
 *   Fichier Apollo.c                                                        *
 *    Create PostScript Interpretor routines so that APOLLO stations can     *
 *  manage all graphic routines and windowing.                               *
 *                                                                           *
 *  Works on every station I know, under rev 10 OR 9... Monochrome or not... * 
 *                                                                           *
 *                           Version 3.00 on 24/02/89                        *
 * ************************************************************************* *
 *    This document may be distributed, used, or modified, but can NOT be    *
 *  sold nor incorporated in any way in any product.                         *
 *    Permission is granted to distribute modified versions of that software *
 *  under the condition that this notice remains in every source file.       *
 *    Every alteration of the original files should be marked as such.       *
 *    No warranty is assumed by the author on the concequencies of the use   *
 *  of this software. Any defection of this program is at your own risk,     *
 *  you have to assume the cost of any service, installation or repairs      *
 *  this program could generate.                                             *
 *                                                                           *
 *                          Fabien LELAQUAIS - ESIEE - lelaquaf@apo.esiee.fr *
 * ************************************************************************* */
#ifdef REV9_7
#include "/sys/ins/base.ins.c"
#include "/sys/ins/gpr.ins.c"
#include "/sys/ins/pad.ins.c"
#include "/sys/ins/streams.ins.c"
#include "/sys/ins/error.ins.c"
#else
#include <apollo/base.h>
#include <apollo/gpr.h>
#include <apollo/pad.h>
#include <apollo/ios.h>
#include <apollo/error.h>
#endif /* REV9_7 */


#ifdef REV9_7
#define STATUS status
#define Gpr_inq_bitmap(x)     gpr_$inq_bitmap(x, STATUS)
#else
#define STATUS &status
#define Gpr_inq_bitmap(x)     gpr_$inq_bitmap((gpr_$bitmap_desc_t *)&x, STATUS)
#endif /* REV9_7 */

#define Gpr_set_bitmap(x)     gpr_$set_bitmap((gpr_$bitmap_desc_t  )x, STATUS)
#define Gpr_acquire_display() gpr_$acquire_display(STATUS)
#define Gpr_release_display() gpr_$release_display(STATUS)
#define Gpr_triangle(x, y, z) gpr_$triangle(x, y, z, STATUS)

extern char *malloc();

typedef struct _window {
  int                   sx, sy;
  gpr_$bitmap_desc_t    bitmap;
  stream_$id_t          stream;
  } *window;
typedef struct _bitmap {
  int                   sx, sy;
  gpr_$bitmap_desc_t    bitmap;
  gpr_$attribute_desc_t attributes;
  } *bitmap;
gpr_$color_vector_t init_cmap, new_cmap;

#define checkstat(m) if (status.all) { puts(m); error_$print(status); }

status_$t status;
gpr_$rgb_plane_t planes;
short     colors, cmap_changed,
          screen_xsize, screen_ysize;

#include "int.h"
#include "graph.h"

extern ps__cache *currentcache;

#ifndef NULL
#define NULL 0L
#endif /* NULL */

/* ************************************************************************* */
void *
ps__CreateWindow(int sx, int sy)
{
  window             win;
  gpr_$offset_t      size;
  gpr_$bitmap_desc_t new_bitmap;
  pad_$window_desc_t new_window;
  stream_$id_t       new_stream;
  pad_$window_list_t windowlst;
  short              graphic_nb, lstsize, i;

  if (!(win=(window)malloc(sizeof(struct _window))))
    return NULL;
  win->sx     = sx;
  win->sy     = sy;
  new_window.top    = 0; new_window.height = sy;
  new_window.left   = 0; new_window.width  = sx;
  pad_$create_window(0L, 0, pad_$transcript, 1, new_window,
#ifdef REV9_7
                     new_stream,
#else
                     &new_stream,
#endif /* REV9_7 */
                     STATUS);
  checkstat("CreateWindow (CreateWindow)");
  pad_$inq_windows(new_stream, windowlst, lstsize,
#ifdef REV9_7
                   graphic_nb,
#else
                  &graphic_nb,
#endif /* REV9_7 */
                   STATUS);
  checkstat("InqWindows (CreateWindow)");
  pad_$set_border(new_stream, graphic_nb, false, STATUS);
  checkstat("SetBorder (CreateWindow)");
  pad_$set_auto_close(new_stream, graphic_nb, true, STATUS);
  checkstat("SetAutoClose (CreateWindow)");
  size.x_size = sx;
  size.y_size = sy;
  gpr_$init(gpr_$direct, new_stream, size, planes,
#ifdef REV9_7
            new_bitmap,
#else
           &new_bitmap,
#endif
            STATUS);
  checkstat("Init (CreateWindow)");
  gpr_$set_obscured_opt(gpr_$block_if_obs, STATUS);
  checkstat("SetObscuredOpt (CreateWindow)");
  gpr_$set_auto_refresh(true, STATUS);
  checkstat("SetAutoRefresh (CreateWindow)");
  win->stream = new_stream;
  win->bitmap = new_bitmap;
  if (planes) {
    gpr_$inq_color_map(0L, (short)(colors+1),
#ifdef REV9_7
                       init_cmap[0],
#else
                       init_cmap,
#endif /* REV9_7 */
                       STATUS);
    checkstat("InqColorMap (CreateWindow)");
    for (i=0; i<=colors; i++)
     new_cmap[i] = ((255*i/colors)<<16)|((255*i/colors)<<8)|(255*i/colors);
    }
  Gpr_acquire_display();
  switch (cmap_changed = planes) {
    case 3 :
      gpr_$set_color_map(2L, 14,
#ifdef REV9_7
                         new_cmap[2],
#else
                         &(new_cmap[2]),
#endif /* REV9_7 */
                         STATUS);
      checkstat("SetColorMap (CreateWindow)");
      gpr_$clear(15L, STATUS);
      checkstat("Clear  (CreateWindow)");
      gpr_$set_draw_value(2L, STATUS);
      checkstat("SetDrawValue (CreateWindow)");
      gpr_$set_fill_value(2L, STATUS);
      checkstat("SetFillValue (CreateWindow)");
      break;
    case 7 :
      gpr_$set_color_map(16L, 240,
#ifdef REV9_7
                         new_cmap[16],
#else
                         &(new_cmap[16]),
#endif /* REV9_7 */
                         STATUS);
      checkstat("SetColorMap (CreateWindow)");
      gpr_$clear(255L, STATUS);
      checkstat("Clear (CreateWindow)");
      gpr_$set_draw_value(16L, STATUS);
      checkstat("SetDrawValue (CreateWindow)");
      gpr_$set_fill_value(16L, STATUS);
      checkstat("SetFillValue (CreateWindow)");
      break;
    default : /* 0 ?? */
      gpr_$clear(0L, STATUS);
      checkstat("Clear (CreateWindow)");
      gpr_$set_draw_value(0L, STATUS);
      checkstat("SetDrawValue (CreateWindow)");
      gpr_$set_fill_value(0L, STATUS);
      checkstat("SetFillValue (CreateWindow)");
    }
  Gpr_release_display();
  return (char *)win;
  }

/* ************************************************************************* */
void
ps__KillWindow(void *win)
{
#define Win ((window)win)
  if (planes) {
    cmap_changed = 0;
    Gpr_acquire_display();
    checkstat("Acquire Display (KillWindow)");
    gpr_$set_color_map(0L, (short)(colors+1),
#ifdef REV9_7
                       init_cmap[0],
#else
                       init_cmap,
#endif /* REV9_7 */
                       STATUS);
    checkstat("SetColorMap (KillWindow)");
    Gpr_release_display();
    checkstat("ReleaseDisplay (KillWindow)");
    }
#ifdef REV9_7
  stream_$close(Win->stream, STATUS);
  checkstat("Stream Close (KillWindow)");
#else
  ios_$close(Win->stream, STATUS);
  checkstat("CloseFrame (KillWindow)");
#endif /* REV9_7 */
  gpr_$deallocate_bitmap(Win->bitmap, STATUS);
  checkstat("DeallocateBitmap (KillWindow)");
  gpr_$terminate(true, STATUS);
 /* checkstat("Terminate (KillWindow)"); */
  free(win);
#undef Win
  }

/* ************************************************************************* */
void *
ps__CreateCache(int sx, int sy)
{
  bitmap             cache;
  gpr_$offset_t      size;
  gpr_$bitmap_desc_t bm;

  if (!(cache = (bitmap)malloc(sizeof(struct _bitmap))))
    return NULL;
  size.x_size = cache->sx = sx;
  size.y_size = cache->sy = sy;
#ifdef REV9_7
  gpr_$allocate_attribute_block(cache->attributes, STATUS);
#else
  gpr_$allocate_attribute_block(&cache->attributes, STATUS);
#endif /* REV9_7 */
  checkstat("AllocateAttributeBlock (CreateCache)");
  gpr_$allocate_bitmap(size, 0, cache->attributes,
#ifdef REV9_7
                       cache->bitmap,
#else
                       &cache->bitmap,
#endif /* REV9_7 */
                       STATUS);
  checkstat("AllocateBitmap (CreateCache)");
  Gpr_inq_bitmap(bm);
  checkstat("InqBitmap (CreateCache)");
  Gpr_set_bitmap(cache->bitmap);
  checkstat("SetBitmap (CreateCache) cache");
  gpr_$clear(0L, STATUS);
  checkstat("Clear (CreateCache)");
  gpr_$set_draw_value(1L, STATUS);
  checkstat("SetDrawValue (CreateCache)");
  Gpr_set_bitmap(bm);
  checkstat("SetBitmap (CreateCache) main");
  return (char *)cache;
  }

/* ************************************************************************* */
void
ps__KillCache(void *b)
{
#define B ((bitmap)b)
  gpr_$deallocate_bitmap(B->bitmap, STATUS);
  checkstat("DeallocateBitmap (KillCache)");
  gpr_$deallocate_attribute_block(B->attributes, STATUS);
  checkstat("DeallocateAttributeBlock (KillCache)");
  free(b);
#undef B
  }

/* ************************************************************************* */
void
ps__CloseCache()
{
  }

/* ************************************************************************* */
void
ps__Erase()
{
  Gpr_acquire_display();
  checkstat("Acquire Display (Erase)");
  gpr_$clear(planes?colors:0L, STATUS);
  checkstat("Clear (Erase)");
  Gpr_release_display();
  checkstat("ReleaseDisplay (Erase)");
  }

/* ************************************************************************* */
gpr_$pixel_value_t choose_color()
{
  gpr_$pixel_value_t gray;

  if (currentcache)
    gray = 1L;
  else {
    switch (planes) {
      case 0 : gray = (to_real(state->gray)>.5)?0:1; break;
      case 3 : gray = 2+13*to_real(state->gray);     break;
      case 7 : gray = 16+239*to_real(state->gray);   break;
      }
    }
  gpr_$set_draw_value(gray, STATUS);
  checkstat("SetDrawValue (choose_color)");
  gpr_$set_fill_value(gray, STATUS);
  checkstat("SetFillValue (choose_color)");
  return gray;
  }

/* ************************************************************************* */
void
ps__DoCache(ps__cache *c, ps__coord coord)
{
  gpr_$window_t      window;
  gpr_$position_t    position;
  gpr_$pixel_value_t color;
  gpr_$rgb_plane_t   loop;

  /* Grand coup de pixel_blt :
       Le bitmap du cache n'a qu'un plan.
       On suppose que chaque pixel pouvant etre modifie est a '1'
     les autres, a 0.
  Lors du pixel_blt, les operations suivantes doivent etre programmees :

   Pour chaque bit 'bit' de la couleur courante, 'src' etant le bitmap cache :
            bit    src     dst    resultat
          +-----+-------+-------++---------+
          |  0  |   0   |   0   ||    0    |
          |  0  |   0   |   1   ||    1    | soit : !src && src
          |  0  |   1   |   0   ||    0    |
          |  0  |   1   |   1   ||    0    |
          +--------------------------------+
          |  1  |   0   |   0   ||    0    |
          |  1  |   0   |   1   ||    1    | soit :  src || dst
          |  1  |   1   |   0   ||    1    |
          |  1  |   1   |   1   ||    1    |
          +-----+-------+-------++---------+
   */
  color = choose_color();
  window.window_base.x_coord = window.window_base.y_coord = 0;
  window.window_size.x_size = ((bitmap)(c->device))->sx;
  window.window_size.y_size = ((bitmap)(c->device))->sy;
  position.x_coord = coord.x+c->cx;
  position.y_coord = coord.y+c->cy;
  Gpr_acquire_display();
  for (loop = 0; loop <= planes; loop++) {
    if (color & (1<<loop))
      gpr_$set_raster_op(loop, gpr_$rop_src_or_dst,      STATUS);
    else
      gpr_$set_raster_op(loop, gpr_$rop_not_src_and_dst, STATUS);
    checkstat("SetRasterOp (DoCache) set");
    gpr_$bit_blt(((bitmap)(c->device))->bitmap, window, 0,
                 position, loop, STATUS);
    checkstat("BitBlt (DoCache)");
    gpr_$set_raster_op(loop, gpr_$rop_src, STATUS);
    checkstat("SetRasterOp (DoCache) reset");
    }
  Gpr_release_display();
  }

/* ************************************************************************* */
void
ps__DoLine(ps__coord p0, ps__coord p1)
{
  gpr_$bitmap_desc_t bm;

  if (currentcache) {
    Gpr_inq_bitmap(bm);
    Gpr_set_bitmap(((bitmap)(currentcache->device))->bitmap);
    }
  else
    Gpr_acquire_display();
  choose_color();
  gpr_$move((short)p0.x, (short)p0.y, STATUS);
  checkstat("Move (DoLine)");
  gpr_$line((short)p1.x, (short)p1.y, STATUS);
  checkstat("Line (DoLine)");
  if (currentcache) Gpr_set_bitmap(bm);
  else              Gpr_release_display();
  }

/* ************************************************************************* */
void
ps__DoTrapeze(int ul, int ll, int ur, int lr, int left, int right)
{
  gpr_$position_t    p1, p2, p3;
  gpr_$bitmap_desc_t bm;

  if (currentcache) {
    Gpr_inq_bitmap(bm);
    Gpr_set_bitmap(((bitmap)(currentcache->device))->bitmap);
    }
  else Gpr_acquire_display();
  choose_color();
  p1.x_coord = left;   p1.y_coord = ul;
  if (ul == ll) {
    p2.x_coord = right;  p2.y_coord = ur;
    p3.x_coord = right;  p3.y_coord = lr;
    }
  else {
    p2.x_coord = left;   p2.y_coord = ll;
    p3.x_coord = right;  p3.y_coord = ur;
    if (ur != lr) {
      Gpr_triangle(p1, p2, p3);
      p1.x_coord = right;
      p1.y_coord = lr;
      }
    }
  Gpr_triangle(p1, p2, p3);
  if (currentcache) Gpr_set_bitmap(bm);
  else              Gpr_release_display();
  }

/* ************************************************************************* *
 *   Writes vertical pixel buffer on screen                                  *
 * ************************************************************************* */
void
ps__WritePixels(int x, int y, int length, unsigned char *buf)
{
  gpr_$bitmap_desc_t  bm;
  unsigned char      *p;
  gpr_$window_t       window;
  gpr_$pixel_value_t *pbuf, *pp;

  if (x < 0) {
    printf("** Negative X coordinate writing pixels **");
    return;
    }
  if (!length) return; /* Fin d'image */
  if (buf) {                                                        /* Image */
    if (pbuf = (gpr_$pixel_value_t *)
               malloc(length*sizeof(gpr_$pixel_value_t))) {
      window.window_base.x_coord = x;
      window.window_base.y_coord = y;
      window.window_size.x_size  = 1;
      window.window_size.y_size  = length;
      for (p = buf, pp = pbuf; length; length--) {
        switch (planes) {
          case 0 : *pp++ = ((*p++)>128)?0:1;   break;
          case 3 : *pp++ = 2+13* (*p++)/255;   break;
          case 7 : *pp++ = 16+239* (*p++)/255; break;
          }
        }
      Gpr_acquire_display();
      gpr_$write_pixels(pbuf, window, STATUS);
      free(pbuf);
      checkstat("WritePixels (WritePixels)");
      Gpr_release_display();
      }
    }
  else {            /* Ligne verticale de pixels de meme couleur (imagemask) */
    if (currentcache) {
      Gpr_inq_bitmap(bm);
      Gpr_set_bitmap(((bitmap)(currentcache->device))->bitmap);
      }
    else
      Gpr_acquire_display();
    choose_color();
    gpr_$move((short)x, (short)y, STATUS);
    checkstat("Move (DoLine)");
    gpr_$line((short)x, (short)(y+length), STATUS);
    checkstat("Line (DoLine)");
    if (currentcache) Gpr_set_bitmap(bm);
    else              Gpr_release_display();
    }
  }

/* ************************************************************************* */
init_device_stuff()
{  
  gpr_$display_config_t config;

#ifdef REV9_7
  gpr_$inq_config(config, STATUS);
#else
  gpr_$inq_config(&config, STATUS);
#endif /* REV9_7 */
  switch(config) {          /* Graphisme en fonction du type de visu utilise */
    case gpr_$color_1024x800x4  :                               /* DN550/560 */
    case gpr_$color2_1024x800x4 :                               /* DN3000C   */
      screen_xsize = 1024; screen_ysize =  800; colors =  15; planes = 3;
      break;
    case gpr_$color_1024x800x8  :                               /* DN550/560 */
    case gpr_$color1_1024x800x8 :                               /* DN570     */
    case gpr_$color2_1024x800x8 :                               /* DN3000E   */
      screen_xsize = 1024; screen_ysize =  800; colors = 255; planes = 7;
      break;
    case gpr_$bw_1280x1024      :                               /* DN3000M   */
      screen_xsize = 1280; screen_ysize = 1024; colors =   0; planes = 0;
      break;
    case 15                     :                               /* DN2500    */
      screen_xsize = 1024; screen_ysize =  800; colors =   0; planes = 0;
      break;
    default :
      printf("Sorry... Non-registred screen type (%d).\n\n", config);
      return -1;
    }
  pad_$isa_dm_pad(1, &status);                         /* Is this a dm pad ? */
  if (status.all) {
    puts("Sorry... Display not acquired. Remote ? Running X ?");
    return -1;
    }
  return 0;
  }

/* ************************************************************************* */
void
free_device_stuff()
{
  if (cmap_changed) {
    Gpr_acquire_display();
    checkstat("Acquire Display (free device stuff)");
    gpr_$set_color_map(0L, (short)(colors+1),
#ifdef REV9_7
                       init_cmap[0],
#else
                       init_cmap,
#endif /* REV9_7 */
                       STATUS);
    checkstat("gpr_$set_color_map");
    Gpr_release_display();
    checkstat("Release Display (free device stuff)");
    }
  }
