/* Simple Transaction Recovery Manager. 

   Reliably records a list of committed transactions. A list of aborted 
   transactions is not needed since we use the presumed abort technique.

   Currently if a transaction completes (commits or aborts) with
   none of the participants failing this service is not used at all.
   When a participant fails (during transaction termination) 
   this service is used to record the outcome for that participant to
   find what happened when it recovers.

   It is likely that in situations where a participant dies the whole transaction
   is aborted anyway (especially in the common case of only one participant), so
   we use the presumed abort strategy to reduce the storage and communication
   requirements of transactions. Thus only commits are recorded. If this service
   doesn't know about a completed transaction it must have aborted.

   Things to do:
   Currently a copy of this service is started up on every site. It should be
   started up on say five sites, and generate extra copies of itself (using
   rexec) if some copies die. 

   Currently all sites reply to a get_outcome message. We could have only
   one site reply, and the caller retry on failure.

   I'm not sure whether we should do manual flushing of the lmgr log, because
   there seems to be a long time between automatic flushes. However xmgr is
   very reliable and its unlikely to experience total failure.
*/

char xmgr_rcsid[] = "$Revision: 1.34 $$Date: 89/01/31 09:56:00 $$Source: /usr/fsys/bullwinkle/b/isis/distrib/util/RCS/xmgr.c,v $";
#include "isis.h"

typedef struct {
    x_id *ids; 
    int max;     /* Elements 0 to max-1 of ids exist. */
    int n;       /* Elements 0 to n-1 of ids are defined, n <= max. */
} idlist;
#define increment 20 /* Amount by which to increase ids when it gets full. */

/* Replicated state of recovery manager. */
idlist committed; /* Ids of committed transactions. */

void
add_to_list(list, id)
  idlist *list;
  x_id id;
  /* Add id to list. id should not already exist in list. */
{
    if (list-> n >= list-> max) {
        /* Increase the size of the array by increment. */
        list-> max += increment;
        list-> ids = (x_id *) realloc(list-> ids, sizeof(x_id) * (list-> max));
    }
    list-> ids[(list-> n)++] = id;
}

bool
search_list(list, id)
  idlist *list;
  x_id id;
  /* Test whether id is a member of list. */
{
    register int i;
    register int n = list-> n;
    register x_id *ids = list-> ids;
    
    for (i = 0; i < n; i++) {
        if (xid_cmp(id, ids[i]) == 0) {
            return(TRUE);
        }
    }
    return(FALSE);
}  

void
send_state(locator, gaddr)
  int locator;
  address gaddr;
{
    xfer_out(1, "%d", committed.n);
    xfer_out(2, "%A", committed.ids, committed.n);
}

void
receive_state(locator, msg)
  int locator;
  message *msg;
{
    if (locator == 1) {
#       ifdef trans_debug
        print("Transaction Recovery Manager: restart");
#       endif
        msg_get(msg, "%d", &(committed.n));
        committed.max = committed.n + increment;
        committed.ids = (x_id *) malloc(sizeof(x_id) * committed.max);

    } else if (locator == 2) {
        msg_get(msg, "%A", committed.ids, (int *) 0);

    } else {
        panic("XMgr, receive_state: got bad locator");
    }
}

void
init_lists(gaddr)
  address gaddr; /* Ignored. */
{
#   ifdef trans_debug
    print("Transaction Recovery Manager: total startup\n");    
#   endif
    committed.n = 0;
    committed.max = increment;
    committed.ids = (x_id *) malloc(sizeof(x_id) * increment);
}

void
transaction_service() {
    address gaddr;
    gaddr = pg_join(xmgr_service,
                    PG_INIT, init_lists,
                    PG_LOGGED, xmgr_service, 0, L_AUTO, NULL,
                    PG_XFER, 1, send_state, receive_state,
                    0);
    if (addr_isnull(gaddr)) {
        isis_perror("XMgr, pg_join");
        panic("");
    }
    isis_logentry(gaddr, XR_SAVE_OUTCOME);
    isis_start_done();
    
#   ifndef trans_debug
    freopen("xmgr.log", "w", stdout); /* Save error messages in a log. */
#   endif
}

void
handle_save_outcome(msg)
  message *msg;
  /* Input: "%a %d" id, outcome (must be X_COMMIT)
     Output: ""

     Record the outcome of the transaction with this id. Since we presume abort
     we only allow X_COMMIT outcomes to be recorded.
  */
{
    int i;
    x_id id;
    int outcome;
    msg_get(msg, "%a %d", &id, &outcome);

    if (outcome == X_ABORT) {
        print("XMgr: save_outcome got obsolete X_ABORT outcome for transaction ");
        paddr(id); print("\n");
        reply(msg, "");
    } 

    /* Check it doesn't already exist. */
    if (search_list(&committed, id)) {
#       ifdef trans_debug
        print("save_outcome: "); paddr(id); print(" duplicate commit\n");
#       endif            
    } else {
#       ifdef trans_debug
        print("save_outcome: "); paddr(id); print(" commit\n");
#       endif            
        add_to_list(&committed, id);
    }
    reply(msg, "");
}

handle_get_outcome(msg)
  message *msg;
/* Input: "%a" id
   Output: "%d" outcome (one of X_COMMIT, X_ABORT).

   Return the saved outcome of the transaction with this id. We presume
   a transaction aborted if we don't know anything about it.
*/
{
    int i;
    x_id id;
    msg_get(msg, "%a", &id);

    if (search_list(&committed, id)) {
#       ifdef trans_debug
        print("get_outcome: "); paddr(id); print(" commit\n");
#       endif            
        reply(msg, "%d", X_COMMIT);
        return;
    } else {
#       ifdef trans_debug
        print("get_outcome: "); paddr(id); print(" abort\n");
#       endif            
        reply(msg, "%d", X_ABORT);
        return;
    }
}

main (argc, argv)
  int argc;
  char *argv[];
{
    int isis_port = 0;

    if (argc != 2) {
        fprintf(stderr, "usage: %s port\n", argv[0]);
        exit(-1);
    }
    isis_port = atoi(argv[1]);
    if (isis_port == 0) {
        fprintf(stderr, "usage: %s port\n", argv[0]);
        exit(-1);
    }
    my_process_id = XMGR; /* Special negative process id to ease identification. */
        /* Need to modify cmd to recognize this process. */

    isis_init(isis_port);
 
    isis_task(transaction_service, "transaction_service");
    isis_entry(XR_SAVE_OUTCOME, handle_save_outcome, "handle_save_outcome");
    isis_entry(XR_GET_OUTCOME, handle_get_outcome, "handle_get_outcome");

    isis_mainloop(transaction_service);
};
