/*
 * ISIS System V1.0 (8/15/88).   Report problems to isis-bugs@gvax.cs.cornell.edu 
 * Note: Export restrictions apply.  License required for export to ``non-free world''
 * countries (USDC terminology).  Agreement to respect this export restriction required
 * for export to all other countries.
 */
/******************************************************************************
 *
 *  pp_post_parse.c - module to create the result files after parsing
 *                      the source file(s).
 *
 *  by : Cheong, Weng Seng
 *       Dept of Computer Science
 *       Cornell University
 *
 *****************************************************************************/
#include <stdio.h>
#include <strings.h>
#include "pp_const.h"
#include "pp_errhandle.e"
#include "pp_global.h"
#include "pp_patch.h"

static FILE     *libccfile;     /* **.libc.c : _call routines */
static FILE     *libcefile;     /* **.libc.e : _call routines externs*/
static FILE     *librcfile;     /* **.libr.c : _rcv routines */
static FILE     *librefile;     /* **.libr.e : _rcv routines externs*/
static FILE     *libfnfile;     /* **.libfn  : rpc function names */
static char     textbuf[PP_MAX_FILE_CHAR];
                                /* buffer to form function and filenames,
                                 * it has the size of
                                 * max{PP_MAX_FILE_CHAR,PP_MAX_FUNCTION_CHAR} */


/* 
 * delete the redundant file which is named in the parameter
 * If error, print message. ( Error will be unusual because the file
 * being delete is redundant and therefore should exist)
 * 
 * function assume file is already closed
 */
static
void    delete_redundant(fname)
        char    *fname;
{
        if(unlink(fname) != PP_OK)
                sys_error("%s: Can't delete redundant file ", fname);
}

/*
 * routine to create entry point definition files, *.sh and *.m
 */
int     create_entry_point_files(common, want_mfile, want_shfile)
     char       *common;        /* name of the source file w/o suffix */
     bool       want_mfile, want_shfile;
{
  register FILE *mfile; /* rpc function's isis_entry() statements */ 
  register FILE *shfile;/* rpc function's entry point definitions */
  char  *filename = textbuf;
  char  functionname[PP_MAX_FUNCTION_CHAR + 1];
  char  constantname[PP_MAX_FUNCTION_CHAR + 1], *constptr;
  int   i;
  
  if (want_mfile) 
    {
      /* create .m file */
      sprintf(filename, "%s.m", common);
      if((mfile = fopen(filename, "w")) == NULL) 
        {
          sys_error("%s: Can't open file ", filename);
          return (PP_ERROR);
        }
      fprintf(mfile,"/*\n * %s.m\n */\n", common);
    }
  if (want_shfile) 
    {
      /* create .sh file */
      sprintf(filename, "%s.sh", common);
      if((shfile = fopen(filename, "w")) == NULL) 
        {
          sys_error("%s: Can't open file ", filename);
          return (PP_ERROR);
        }
      fprintf(shfile,"/*\n * %s.sh\n */\n", common);
    }
  
  if (want_shfile || want_mfile) 
    {
      /* open the .fn file for reading */
      sprintf(filename, "%s.fn", common);
      if((pp_fnfile = fopen(filename, "r")) == NULL) 
        {
          sys_error("%s: Can't open file ", filename);
          return (PP_ERROR);
        }
      
      /* write to the .sh and .m files */
      for(i = 1; fscanf(pp_fnfile, "%s", functionname) != EOF; i++)
        {
          constptr = invert_lettercase(strcpy(constantname,&functionname[1])); 

          /* entry points from exported functions only */
          if(mfile != NULL && (functionname[0] == PP_FROM_EXPORT ||
                               functionname[0] == PP_FROM_IM_EXPORT))
            fprintf(mfile, INIT_ENTRY_POINT,constptr, &functionname[1], 
                    &functionname[1]);

          /* all function included */
          if(shfile != NULL) 
            fprintf(shfile, DEFINE_ENTRY_POINT, constptr, i);                   
        }
      
      /* close the files */
      if(mfile != NULL && fclose(mfile) == EOF)
        sys_error("'%s.m': Can't close file", common);
      if(shfile != NULL && fclose(shfile) == EOF)
        sys_error("'s.sh': Can't close file", common);
      if(fclose(pp_fnfile) == EOF)
        sys_error("'%s.fn': Can't close file", common);
    }
  sprintf(filename, "%s.fn", common);
  delete_redundant(filename);
  return(PP_OK);
}
                
/*
 * compile the library files from preprocessed files (.fn , .call and .rcv)
 * of the library specification file (.l)
 *
 * Library spec file            concat          library file
 * ^^^^^^^^^^^^^^^^^            ^^^^^^          ^^^^^^^^^^^^
 * .fn files                    ----->          **.libfn
 * .call files                  ----->          **.libc.c
 * .rcv files                   ----->          **.libr.c
 */
void    compile_library_files(libname, filenamevect, filecount)
        char    *libname;       /* name of the library to be compile */
        char    **filenamevect; /* vect to pointer to the source file names */
        int     filecount;              /* # of source files */
{
        int     i;
        char    *strptr;
        char    *functionname = textbuf;        
        
        /* create the library files */
        if(create_lib_files(libname) != PP_OK)
                halt("Fatal error: Library incomplete");

        /* concat the spec files */
        for(i = 0; i < filecount; i++) {
                strptr = strncpy(pp_genname, filenamevect[i], 
                                strlen(filenamevect[i]) - 3);
                if(concat_lib_files(strptr) != PP_OK)
                        halt("Fatal error: Library incomplete");
        }       
        
        /* close the library spec files */
        if(close_lib_files() != PP_OK)
                halt("Fatal error: Can't close library specification files");
        
        /* open the library extern files */
        if(open_lib_extern(libname) != PP_OK)
                halt("Fatal error: Can't create library extern files");
        
        /* read each functionname from <libname>.libfn, and appropriate
         * extern into <libname>.libc.e and <libname>.libr.e */
        while(fscanf(libfnfile, "%s", functionname) != EOF) {
                fprintf(libcefile, EXTERN_CALL, functionname);
                fprintf(librefile, EXTERN_RCV, functionname);
        }
        
        /* close the library extern files */
        if(close_lib_extern() != PP_OK)
                halt("Fatal error: Can't close library extern files");
}


/*
 * create the library's .c and .libfn files 
 */
static
int     create_lib_files(libname)
        char    *libname;
{
        char    *filename = textbuf;    
        
        sprintf(filename, "%s.libfn",libname);
        if ((libfnfile = fopen(filename, "w")) == NULL) {
                sys_error("%s: Can't open file ", filename);
                return (PP_ERROR);
        }
        
        sprintf(filename, "%s.libc.c",libname);
        if ((libccfile = fopen(filename, "w")) == NULL) {
                sys_error("%s: Can't open file ", filename);
                return (PP_ERROR);
        }
        fprintf(libccfile,"/*\n * Filename : %s.libc.c\n */\n", libname);
        
        sprintf(filename, "%s.libr.c",libname);
        if ((librcfile = fopen(filename, "w")) == NULL) {
                sys_error("%s: Can't open file ", filename);
                return (PP_ERROR);
        }
        fprintf(librcfile,"/*\n * Filename : %s.libr.c\n */\n", libname);
        
        return(PP_OK);
}

/*
 * concat each source file's .call , .rcv and .fn files into
 * the library's .c's and .libfn
 */
static
int     concat_lib_files(common)
        char    *common;
{
        char    filenamebuffer[PP_MAX_FILE_CHAR], *filename = filenamebuffer;
        FILE    *sourcefile;
        
        /* concat <common>.fn to libfnfile */
        sprintf(filename, "%s.fn", common);
        if((sourcefile = fopen(filename, "r")) == NULL) {
                sys_error("%s: Can't open file ", filename);
                return (PP_ERROR);
        }
        if (concat_files(libfnfile, sourcefile) != PP_OK) return(PP_ERROR);
        if(fclose(sourcefile) == EOF) {
                        sys_error("%s: Can't close file ", filename);
                        return (PP_ERROR);
        }
        delete_redundant(filename);
        
        /* concat <common>.call to libccfile */
        sprintf(filename, "%s.call", common);
        if((sourcefile = fopen(filename, "r")) == NULL) {
                sys_error("%s: Can't open file ", filename);
                return (PP_ERROR);
        }
        if (concat_files(libccfile, sourcefile) != PP_OK) return(PP_ERROR);
        if(fclose(sourcefile) == EOF) {
                        sys_error("%s: Can't close file ", filename);
                        return (PP_ERROR);
        }
        delete_redundant(filename);
        
        /* concat <common>.rcv to librcfile */
        sprintf(filename, "%s.rcv", common);
        if((sourcefile = fopen(filename, "r")) == NULL) {
                sys_error("%s: Can't open file ", filename);
                return (PP_ERROR);
        }
        if (concat_files(librcfile, sourcefile) != PP_OK) return(PP_ERROR);
        if(fclose(sourcefile) == EOF) {
                        sys_error("%s: Can't close file ", filename);
                        return (PP_ERROR);
        }
        delete_redundant(filename);

        return(PP_OK);
}

/* 
 * concat file fil1 and fil2 by appending fil2 to fil1
 */
static
int     concat_files(fil1, fil2)
        FILE    *fil1, *fil2;
{
        char    buffer[PP_BLKSIZE];
        int     char_read;
        
        while ((char_read = fread(buffer, sizeof(*buffer), PP_BLKSIZE, fil2)) != 0) {
                if(fwrite(buffer, sizeof(*buffer), char_read, fil1) == 0) {
                        sys_error("Block write error", PP_NULLSTR);
                        return(PP_ERROR);
                }
        }
        return(PP_OK);
}

/* 
 * close the library spec file
 */
static
int     close_lib_files() 
{
        if(fclose(librcfile) == EOF) {
                        sys_error("(Error) Can't close file ", PP_NULLSTR);
                        return (PP_ERROR);
        }
        if(fclose(libccfile) == EOF) {
                        sys_error("(Error) Can't close file ", PP_NULLSTR);
                        return (PP_ERROR);
        }
        if(fclose(libfnfile) == EOF) {
                        sys_error("(Error) Can't close file ", PP_NULLSTR);
                        return (PP_ERROR);
        }
        return(PP_OK);
}
/* 
 * open **.libfn and create new **.libr.e and **.libc.e
 */
static
int     open_lib_extern(libname)
        char    *libname;
{
        char    *filename = textbuf;

        sprintf(filename,"%s.libr.e", libname);
        if((librefile = fopen(filename, "w")) == NULL) {
                        sys_error("%s: Can't open file ", filename);
                        return (PP_ERROR);
        }
        fprintf(librefile,"/*\n * Filename : %s\n */\n",filename );
        
        sprintf(filename,"%s.libc.e", libname);
        if((libcefile = fopen(filename, "w")) == NULL) {
                        sys_error("%s: Can't open file ", filename);
                        return (PP_ERROR);
        }
        fprintf(libcefile,"/*\n * Filename : %s\n */\n",filename );
        
        sprintf(filename,"%s.libfn", libname);
        if((libfnfile = fopen(filename, "r")) == NULL) {
                        sys_error("%s: Can't open file ", filename);
                        return (PP_ERROR);
        }

        return(PP_OK);
}

/* 
 * close **.libfn , **.libr.e and **.libc.e
 */
static
int     close_lib_extern()
{
        if(fclose(librefile) == EOF) {
                        sys_error("(Error) Can't close file ", PP_NULLSTR);
                        return (PP_ERROR);
        }
        if(fclose(libcefile) == EOF) {
                        sys_error("(Error) Can't close file ", PP_NULLSTR);
                        return (PP_ERROR);
        }
        if(fclose(libfnfile) == EOF) {
                        sys_error("(Error) Can't close file ", PP_NULLSTR);
                        return (PP_ERROR);
        }
        return(PP_OK);
}
