/*
 * ISIS System V1.0 (8/15/88).   Report problems to isis-bugs@gvax.cs.cornell.edu 
 * Note: Export restrictions apply.  License required for export to ``non-free world''
 * countries (USDC terminology).  Agreement to respect this export restriction required
 * for export to all other countries.
 */
/******************************************************************************
 *
 *  pp_main.c - main routine and argument checker of ISIS_pre
 *
 *  by : Cheong, Weng Seng
 *       Dept of Computer Science
 *       Cornell University
 *
 ******************************************************************************/
#include <stdio.h>
#include <strings.h>
#include "pp_const.h"
#include "pp_errhandle.e"
#include "pp_fname_tab.e"
#include "pp_post_parse.e"
#define USAGE   \
        "\nUsage: isis_pre [-l libname] specfilename [specfilename] ...\n"
        
typedef char    bool;
        
/* global variables */
FILE    *pp_cfile = NULL, *pp_callfile = NULL, 
        *pp_rcvfile = NULL, *pp_fnfile = NULL;
int     pp_lineno = 1;
char    *pp_sourcename, pp_genname[PP_MAX_FILE_CHAR];
int     pp_mode = PP_NORMAL;
bool    pp_mfile_included, pp_shfile_included;

static  int     sourcefileno = 0;
static  char    **sourcefilevect;       /* starting point of the 
                                         * sourcefile vector */
static  char    *libraryname = NULL;
 
/* 
 * return the boolean value :
 *      " Is str a valid source specification filename ? "
 *
 * Source file is denoted with a the suffix ".df"
 */
static
bool    isSourceSpec(str)
        char    *str;
{
        char    *ptr;
        
        ptr = rindex(str, '.');
        return(strcmp(ptr, ".df") != PP_EQUAL ? PP_FALSE : PP_TRUE);
}

/* 
 * return the boolean value :
 *      " Is str a valid library specification filename ? "
 *
 * Source file is denoted with a the suffix ".lf"
 */
static
bool    isLibSpec(str)
        char    *str;
{
        char    *ptr;
        
        ptr = rindex(str, '.');
        return(strcmp(ptr, ".lf") != PP_EQUAL ? PP_FALSE : PP_TRUE);
}


/* 
 * return the boolean value :
 *      " Is str a valid library name ? "
 *
 * library name has no '.'
 */
static
bool    isLibname(str)
        char    *str;
{       
        return(index(str, '.') == 0 ? PP_TRUE : PP_FALSE);
}

/* 
 * check whether the vector namevect of count elements
 * consists of valid sourcefile names depending on the value of pp_mode  
 */
static
int     check_sourcefile_name(namevect, count)
        char    **namevect;
        int     count;
{
        register int    i;
                
        switch(pp_mode) {
        case PP_NORMAL:
                for(i = 0; i < count; i++){
                        if( !isSourceSpec(namevect[i]) ) {
                                fprintf(stderr, "Error: Source specification\
 file must end with \".df\"\n");
                                fprintf(stderr, USAGE);
                                return (PP_ERROR);
                        }
                }
                break;
        case PP_MAKELIB:
                for(i = 0; i < count; i++){
                        if (!isLibSpec(namevect[i]) ) {
                                fprintf(stderr, "Error: Library specification\
 file must end with \".lf\"\n");
                                fprintf(stderr, USAGE);
                                return (PP_ERROR);
                        }
                }
                break;
        }
        return (PP_OK);
}

/* 
 * process the arguments from the command line
 * return PP_OK if successful, else return PP_ERROR
 * 
 * Initializes :        pp_mode
 *                      libraryname
 *                      sourcefileno
 *                      sourcefilevect
 */
static
int     process_arguments(argcount, argvect)
        int     argcount;
        char    **argvect;
{
        int     i;
        int     last_source_index = -1; /* index of the last source file in 
                                         * argvect */
        
        for (i = 1; i < argcount; i++) {
                /* inspect every argument */    
                if (argvect[i][0] == '-') {
                        /* option indicator */
                        switch (argvect[i][1]) {
                        case 'l' :
                                /* make a library, and the next argument
                                 * is a library name */
                                pp_mode = PP_MAKELIB;
                                if (!isLibname(libraryname = argvect[++i])) {
                                        /* didn't qualify as a library name */
                                        fprintf(stderr, "\nError: library name\
 can't have \".\"");
                                        fprintf(stderr, USAGE);
                                        return (PP_ERROR);
                                }
                                break;
                        default :
                                /* undefined option */
                                fprintf(stderr, USAGE);
                                return (PP_ERROR);
                        }
                } else {
                        /* argument is a sourcefile name */
                        if (++sourcefileno == 1) {
                                /* found the first sourcefile */
                                sourcefilevect = argvect + i;
                                last_source_index = i;
                        } else {
                                /* check if the source files list is
                                 * consecutive */
                                if (++last_source_index != i) {
                                        /* not consecutive */
                                        fprintf(stderr, "Error: list of source\
 file names must be consecutive");
                                        fprintf(stderr, USAGE);
                                        return (PP_ERROR);
                                } 
                        }
                } 
        }

        /* check the niumber of source files */
        if (sourcefileno == 0) {
                fprintf(stderr, USAGE);
                return (PP_ERROR);
        }

        /* check whether the source file names are in valid form */
        if(check_sourcefile_name(sourcefilevect, sourcefileno) != PP_OK)
                return(PP_ERROR);
        else 
                return (PP_OK);
}                                                       
        
/*
 * open the target get file and store their file descriptor in the respective
 * global vars 
 *
 *      pp_fnfile = file descriptor for *.fn ( file of function names)
 *      pp_rcvfile = fdesp for *.rcv    ( _rcv routines of rpc function)
 *      pp_callfile = fdesp for *.call  ( _call routines of rpc function)
 *      pp_cfile = fdesp for *.c        ( source code to be compiled by cc )
 */
static
void    open_target_files(commfname)
        char    *commfname;
{
        char    buffer[PP_MAX_FILE_CHAR];
        char    *strptr = buffer;
        
        strptr = strcpy(buffer, commfname);
        strptr = strcat(buffer, ".fn");
        if((pp_fnfile = fopen(strptr, "w")) == NULL)
                sys_error("File '%s': can't open",strptr);

        switch(pp_mode) {
        case PP_MAKELIB :       
                strptr = strcpy(buffer, commfname);
                strptr = strcat(buffer, ".rcv");
                if((pp_rcvfile = fopen(strptr, "w")) == NULL)
                        sys_error("File '%s': can't open",strptr);
                                
                strptr = strcpy(buffer, commfname);
                strptr = strcat(buffer, ".call");
                if((pp_callfile = fopen(strptr, "w")) == NULL)
                        sys_error("File '%s': can't open",strptr);
                        
                fprintf(pp_rcvfile,"/*\n ********** Component file : %s.rcv ***********\n */\n# line 1 \"%s.lf\"\n", commfname, commfname);
                fprintf(pp_callfile,"/*\n ********** Component file : %s.call ***********\n */\n# line 1 \"%s.lf\"\n", commfname, commfname);           
                break;
        case PP_NORMAL :
                strptr = strcpy(buffer, commfname);
                strptr = strcat(buffer, ".c");
                if((pp_cfile = fopen(strptr, "w")) == NULL)
                        sys_error("File '%s': can't open",strptr);
                fprintf(pp_cfile,"/*\n * %s.c\n */\n# line 1 \"%s.df\"\n", commfname, commfname);
                break;
        }
}

/* 
 * close all the target files 
 */
static
void    close_target_files()
{
        switch (pp_mode) {
        case PP_MAKELIB:
                if (fclose(pp_rcvfile) == EOF) 
                        sys_error("File '%s.rcv': error in closing", pp_genname);
                if (fclose(pp_callfile) == EOF) 
                        sys_error("File '%s.call': error in closing",pp_genname);
                break;
        case PP_NORMAL:
                if (fclose(pp_cfile) == EOF) 
                        sys_error("File '%s.c': error in closing", pp_genname);
                break;
        }
        
        if (fclose(pp_fnfile) == EOF) 
                sys_error("File '%s.fn': error in closing", pp_genname);
}

/*DEBUG */
void    print_arg()
{
        int     i;
        
        if(libraryname != NULL) printf("Library name ='%s'\n", libraryname);
        printf("# of source file = %d\nOperation = %d\n", sourcefileno, pp_mode);
        for(i = 0; i < sourcefileno; i++) {
                printf("Sourcefilename :'%s'\n", sourcefilevect[i]);
        }
        fflush(stdout);
}
                
main (argc, argv)
        int     argc;
        char    **argv;
{
        char    *strptr;
        int     i;
        
        if (process_arguments(argc, argv) != PP_OK)
                /* error in the command line */ 
                halt("");
                
        /* run each source file thru the parser */
        for (i = 0; i < sourcefileno; i++) {    
                pp_sourcename = sourcefilevect[i];
                
                /* print the file name being preprocessed */
                printf("%s:\n", pp_sourcename); fflush(stdout);
                
                /* store the generic name of the file for future use */
                strptr = strncpy(pp_genname, pp_sourcename, strlen(pp_sourcename) - 3);
                        
                /* open target files and the source file */
                open_target_files(strptr);
                if (freopen(pp_sourcename, "r", stdin) == NULL){
                        sys_error("can't open source file '%s'", pp_sourcename);
                        exit (1);
                }
                
                /* init variables for each new source file */
                init_fname_tab();
                pp_mfile_included = (pp_shfile_included = PP_FALSE);
                
                /* parse the source file */     
                if (yyparse() != PP_OK) 
                        halt("Parser can't recover from syntax error");
                
                /* post-parse operation */      
                close_target_files();
                
                switch(pp_mode) {
                case PP_NORMAL:
                        if(create_entry_point_files(pp_genname, 
                                pp_mfile_included, pp_shfile_included) != PP_OK)                                                fprintf(stderr,"%s.df : Skipped!\n", pp_genname);
                        break;
                }

        }
        
        /* wrap up */
        switch(pp_mode) {
        case PP_MAKELIB:
                printf("Compiling library files.....\n"); fflush(stdout);
                compile_library_files(libraryname, sourcefilevect, sourcefileno);
                break;
        }
}
