%{
/******************************************************************************
 *
 * pp_lex.l : lexical analyser for specification files in isis_pre
 *
 * description of the various start symbols :
 *	N :
 *		The normal text in the source file. In this mode, lex
 *		will eat up as much text as possible ( till the next <cr> )
 *	RPC_HEAD :
 *	      	RPC's declaration statement and parameters' declaration
 *	RPC_STAT :
 *		RPC's body statement
 *	RPC_REPLY :
 *		RPC's reply to the caller. This start symbol is triggered
 *		by the keyword return in RPC_STAT
 *	RPC_CALL :
 *		RPC function invokation
 *	RPC_ARG :
 *		Arguments when calling an RPC function. This start symbol is 
 *		triggered after the RPC function name is read in RPC_CALL
 *	COMMENT :
 *		Comments in the specification file. 
 *	IMPT_EXPT :
 *		<import> and <export> statements
 *	CONTROL :
 *		control statements (ie. statements beginning with '#')
 *
 *  by : Cheong, Weng Seng
 *       Dept of Computer Science
 *       Cornell University
 *
 *****************************************************************************/
%}

alpha		[a-zA-Z_]
number		[0-9]
alpha_num	[a-zA-Z_0-9]
ws		[ \t\n]*
file_char       [a-z_A-Z0-9/.]

%{
#include <strings.h>
#include <stdio.h>
#include "pp_yacc.h"
#include "pp_const.h"
#include "pp_global.h"
#include "pp_lex_msg.h"

static int	clevel;	/* level of curly braces in the rpc declaration's body */
static int	plevel;	/* level of parentheses */
static int	backslash_in_string = PP_FALSE;
static int	stage;	/* current stage before entering stage COMMENT */
static int	bufcount = 0;		/* # of charaters in the buffer */
static int	entry_count = 0;     	/* number of entrys encountered */
static int	import_count = 0;     	/* number of imports encountered */
static int	export_count = 0;     	/* number of exports encountered */
static int 	im_export_count = 0;    /* number of im/export encountered */

/* text buffer to store token obtained by pp_lex */
typedef struct lexbuf {
	char	buf[PP_BUFSIZE];
} lexbuf;

static 	lexbuf	buf_tab[PP_NO_OF_LEX_BUF];
static	int	current_buf = 0;	/* no of the buffer to be filled */
					 
void 	push_text()
{
	if ((yyleng + bufcount) > PP_BUFSIZE) {
		fprintf(stderr,PP_BUFFER_OVERFLOW_MSG);
		
	} else {
		if (bufcount == 0)
			strcpy(buf_tab[current_buf].buf, yytext);
		else
			strcat(buf_tab[current_buf].buf, yytext);
		yylval.v.text = buf_tab[current_buf].buf;
		yylval.v.len = (bufcount += yyleng);
	}
}

void	no_text()
{
	yylval.v.text = buf_tab[current_buf].buf;
	yylval.v.len = bufcount;
}

#define INIT_BUFFER()	(bufcount = 0)
#define WRAPUP_BUFFER()	bufcount = 0; ++current_buf;current_buf %= PP_NO_OF_LEX_BUF

void	push_token()
{
	if (yyleng > PP_BUFSIZE){ 
		fprintf(stderr, "yylex() read more than buffersize");
	} else {
		strcpy(buf_tab[current_buf].buf, yytext);
		yylval.v.text = buf_tab[current_buf].buf;
		yylval.v.len = yyleng;
		++current_buf;
		current_buf %= PP_NO_OF_LEX_BUF;
	}
}

void	empty_token()	
{ 
	yylval.v.text = NULL;
	yylval.v.len = 0;
}

void	do_increline()
{
	char	*tmp;
	
	for(tmp = index(yytext, '\n'); 
		tmp != 0; pp_lineno++, tmp = index(tmp, '\n'));
}
%}

%e	650
%p	1600
%n	250
%k	250
%a	1600
%o	2500
%start	N RPC_HEAD RPC_STAT RPC_REPLY COMMENT RPC_ARG RPC_CALL IMPT_EXPT CONTROL
%start	STR
%%
<RPC_HEAD>uses			{ empty_token(); return USES; }
<RPC_HEAD>abcast		{ push_token(); return BROADCAST; }
<RPC_HEAD>bcast			{ push_token(); return BROADCAST; }
<RPC_HEAD>cbcast 		{ push_token(); return BROADCAST; }
<RPC_HEAD>gbcast 		{ push_token(); return BROADCAST; }
<RPC_HEAD>register		{ empty_token(); return REGISTER; }
<RPC_HEAD>"("			{ empty_token(); return '('; }
<RPC_HEAD>")"			{ empty_token(); return ')'; }
<RPC_HEAD>","			{ empty_token(); return ','; }
<RPC_HEAD>"="			{ empty_token(); return '='; }
<RPC_HEAD>"["			{ empty_token(); return '['; }
<RPC_HEAD>"]"			{ empty_token(); return ']'; }
<RPC_HEAD>";"			{ empty_token(); return ';'; }
<RPC_HEAD>"/*"			{ stage = RPC_HEAD; BEGIN COMMENT; }
<RPC_HEAD>"{"			{ unput(*yytext); clevel = 0;
				  BEGIN RPC_STAT; }
<RPC_HEAD>"*"			{ empty_token(); return '*'; }	      
<RPC_HEAD>{number}+		{ push_token(); return INTG; }
<RPC_HEAD>{number}+"."{number}*	{ push_token(); return DECIMAL; }
<RPC_HEAD>{alpha}{alpha_num}*/{ws}"("	
				{ push_token(); do_increline(); 
				  return FUNC_NAME; }
<RPC_HEAD>{alpha}{alpha_num}*	{ push_token(); return IDENT; }
<RPC_HEAD>.			;

<RPC_STAT>"$nullreply("{ws}*")"	{ no_text(); WRAPUP_BUFFER();
				  return NULLREPLY; }
<RPC_STAT>"$abortreply("{ws}*")"	
				{ no_text(); WRAPUP_BUFFER();
				  return ABORTREPLY; }
<RPC_STAT>"return"		{ no_text(); WRAPUP_BUFFER(); 
				  plevel = 0; BEGIN RPC_REPLY;
				  return REPLY; }
<RPC_STAT>"/*"			{ stage = RPC_STAT; BEGIN COMMENT; }
<RPC_STAT>"{"			{ clevel++;  
			     	  if (clevel == 1) {
				  	/* signal the start of the rpc body */
					empty_token(); INIT_BUFFER();
					return LEFT_CURLY;
				  } else {
				  	/* just another parenthesis */
					push_text();
				  }
			        }
<RPC_STAT>"}"			{ clevel--; 
				  if (clevel == 0) {
				  	push_text(); WRAPUP_BUFFER();
					BEGIN N;
				  	return RIGHT_CURLY;
				  } else {
				  	push_text();
				  }
				}
<RPC_STAT>";"			{ empty_token(); return ';'; }
<RPC_STAT>"r"			{ push_text(); }
<RPC_STAT>"$"			{ push_text(); }
<RPC_STAT>("*"|"/")		{ push_text(); }
<RPC_STAT>[^r$}{;/*\n]+/";"	{ push_text(); WRAPUP_BUFFER(); 
				  return RPC_STATEMENT; }
<RPC_STAT>[^r$}{;/*\n]+		{ push_text(); }
<RPC_STAT>\n			{ push_text(); pp_lineno++; }

<RPC_REPLY>"("			{ plevel++;
				  if (plevel == 1) {
				  	empty_token(); INIT_BUFFER();
					return '(';
				  } else {
				  	push_text();
				  }
				}
<RPC_REPLY>")"			{ plevel--;
				  if(plevel == 0) {
				  	no_text(); WRAPUP_BUFFER();
					BEGIN RPC_STAT;
					return RET_VAL;
				  } else {
				  	push_text();
				  }
				}
<RPC_REPLY>[^)(\n]+		{ push_text(); }
<RPC_REPLY>\n			{ push_text(); pp_lineno++; }

<RPC_ARG>";"			{ empty_token(); BEGIN N;
				  INIT_BUFFER(); return ';'; }
<RPC_ARG>"("			{ plevel++;
				  if (plevel == 1) {
				  	empty_token(); INIT_BUFFER();
				  	return '(';
				  } else {
				  	push_text();
				  }
				}
<RPC_ARG>")"			{ if (plevel == 1){
				  	empty_token(); INIT_BUFFER();
				  	plevel--;
				  	return ')';
				  } else {
				  	push_text();
					plevel--;
				  }
				}
<RPC_ARG>","			{ if (plevel == 1) {
				  	empty_token(); INIT_BUFFER();
					return ',';
				  } else {
				  	push_text();
				  }
				}
<RPC_ARG>"@"			{ empty_token(); INIT_BUFFER();
				  return '@'; }
<RPC_ARG>"=>"			{ empty_token(); INIT_BUFFER();
				  return ARROW; }
<RPC_ARG>./","			{ push_text();
				  if (*yytext == '(') plevel++;
				  if (*yytext == ')') plevel--;
				  if (plevel == 1) {
				  	WRAPUP_BUFFER();
				  	return EXPR;
				  }
				}
<RPC_ARG>./")"			{ push_text();
				  if (*yytext == '(') plevel++;
				  if (*yytext == ')') plevel--;
				  if (plevel == 1) {
				  	WRAPUP_BUFFER(); return EXPR;
				  }
			        }
<RPC_ARG>.			{ push_text(); }

<RPC_CALL>{ws}"*"*{ws}{alpha}{alpha_num}*{ws}"=" 
				{ push_token(); do_increline();
				  return IDENT_EQUAL; }
<RPC_CALL>{ws}"*"*{ws}{alpha}{alpha_num}*/{ws}"("  	
				{ push_token(); do_increline();
				  plevel = 0; INIT_BUFFER();
				  BEGIN RPC_ARG; return FUNC_NAME; }

<CONTROL>.+			{ if (yytext[yyleng - 1] == '\\') {
					push_text();
		  		  } else {
		  			push_text(); WRAPUP_BUFFER();
					BEGIN N; return CONTROL_STAT;
		  		  }
				}
<CONTROL>\n			{ push_text(); pp_lineno++; }

<IMPT_EXPT>library		{ push_token(); return LIBRARY; }
<IMPT_EXPT>{file_char}*   	{ push_token(); return LIB_NAME; }
<IMPT_EXPT>"<"			{ empty_token(); return '<'; }
<IMPT_EXPT>">"			{ empty_token(); BEGIN N; return '>'; }
<IMPT_EXPT>.			;

<COMMENT>"*/"			{ switch(stage) {
					case N:		BEGIN N;
							break;
				    	case RPC_STAT: 	BEGIN RPC_STAT;
							break;
					case RPC_HEAD:	BEGIN RPC_HEAD;
							break;
					default:	halt("Error: Undefined stage value in lexical analyser\n");
				   }
				}
<COMMENT>.			;

<STR>[^"\n]+			{ push_text();
				  if(yytext[yyleng - 1] == '\\') 
				 	backslash_in_string = PP_TRUE;
				}
<STR>\"				{ push_text();
				  if(backslash_in_string) 
				  	/* backslash no more in effect */
					backslash_in_string = PP_FALSE;
				  else {
				  	BEGIN N;
					return NORMAL_TEXT;
				  }
				}
<STR>\n				{ push_text(); pp_lineno++;
				  if(backslash_in_string) 
				  	/* backslash no more in effect */
					backslash_in_string = PP_FALSE;
				  else
				  	fprintf(stderr,PP_RETURN_IN_STR_MSG, 						pp_lineno - 1);
				}

<N>"$"entry			{ nul_info(); BEGIN RPC_HEAD; 
				  yylval.freq = ++entry_count; 
				  return ENTRY; }
<N>"$"import		    	{ BEGIN IMPT_EXPT;
				  yylval.freq = ++import_count;
				  return IMPORT; }
<N>"$"export		       	{ BEGIN IMPT_EXPT;
				  yylval.freq = ++export_count;
				  return EXPORT; }
<N>"$"im"/"export	       	{ BEGIN IMPT_EXPT;
				  yylval.freq = ++im_export_count;
				  return IM_EXPORT; }
<N>"$"init_rpc{ws}"("{ws}")"{ws}";"		
				{ empty_token(); do_increline();
				  return INIT_RPC; }
<N>"$$"				{ nul_info(); BEGIN RPC_CALL;
				  plevel = 0; INIT_BUFFER();
				  empty_token(); return DOUBLE_DOLLAR; }
<N>"/*"				{ stage = N; BEGIN COMMENT; }
<N>"$"				{ switch(pp_mode) {
				  case PP_NORMAL : 
				  	fprintf(pp_cfile, "$");
				  	break;
				  case PP_MAKELIB:
				   	fprintf(pp_rcvfile, "$");
				  	break;
				  }
			        }
<N>"/"				{ switch(pp_mode) {
				  case PP_NORMAL :
				  	fprintf(pp_cfile, "/");
				  	break;
				  case PP_MAKELIB:
				  	fprintf(pp_rcvfile, "/");
				  	break;
				  }
				}
<N>"*"				{ switch(pp_mode) {
				  case PP_NORMAL : 
				  	fprintf(pp_cfile, "*");
				  	break;
				  case PP_MAKELIB:
				  	fprintf(pp_rcvfile, "*");
				  	break;
				  } 
				}
<N>^#				{ push_text(); BEGIN CONTROL; }
<N>#				{ switch(pp_mode) {
				  case PP_NORMAL : 
				  	fprintf(pp_cfile, "#");
				  	break;
				  case PP_MAKELIB:
				  	fprintf(pp_rcvfile, "#");
				  	break;
				  }
				}
<N>\"				{  /* trap all strings in the normal text 
				    * to  prevent error when $<keyword> 
				    * is in the string */
				   push_text(); 
				   backslash_in_string = PP_FALSE;
				   BEGIN STR;
				}
<N>[^$*#/\n\"]+			{ push_token(); return NORMAL_TEXT; }

\n				{ pp_lineno++; 
				  switch(pp_mode) {
				  case PP_NORMAL : 
				  	fprintf(pp_cfile, "\n");
				  	break;
				  case PP_MAKELIB:	
				  	fprintf(pp_rcvfile, "\n");
				  	fprintf(pp_callfile, "\n");
				  	break;
				  } 
				}
.				{ unput(*yytext);
				  entry_count = 0;
				  import_count = 0;
				  export_count = 0;
				  BEGIN N; }

