/*
 * ISIS System V1.0 (8/15/88).   Report problems to isis-bugs@gvax.cs.cornell.edu 
 * Note: Export restrictions apply.  License required for export to ``non-free world''
 * countries (USDC terminology).  Agreement to respect this export restriction required
 * for export to all other countries.
 */
/******************************************************************************
 *
 * PP_info.c : module for maintaining information of token strings, 
 *              eg. function name, type, etc. , of parsed rpc code
 *
 *  by : Cheong, Weng Seng
 *       Dept of Computer Science
 *       Cornell University
 *
 ****************************************************************************/
#include <stdio.h>
#include <strings.h>

#include "pp_const.h"
#include "pp_errhandle.e"
#include "pp_global.h"
#include "pp_fname_tab.e"

/* parameter  member */
typedef struct para_t {
  char  p_name[PP_MAX_PARA_CHAR + 1];   /* parameter name */
  char  p_type[PP_MAX_TYPE_CHAR + 1];   /* parameter type */
  char  fmt_char;                       /* format char for type */
  bool  reg_var;                        /* true if parameter is
                                         * declared as a register var */
} para_t;

/* rpc function info */
typedef struct {
  char  *f_name;                        /* function name , ptr to fname_tab */
  char  f_type[PP_MAX_TYPE_CHAR + 1];   /* function return value's type */
  char  fmt_char;                       /* format char for function type
                                         * (see ISIS docu. on message
                                         *  field ) */
  para_t        para_tab[PP_MAX_NO_PARA];       /* table of parameter names */
  int   para_no;                        /* num of parameters */
  char  b_type[PP_MAX_BTYPE_CHAR + 1];  /* broadcast type */
}rpc_t;

static  rpc_t   info;
static  int     cur_idx;        /* current parameter index refered 
                                 * in the table */
static  char    temp_paratype_buffer[PP_MAX_TYPE_CHAR + 1];
/* temp buffer to hold parameter type string
 * before inserting it into its proper place
 * in the table (info.para_tab) */

/*
 *  initialize info's components to their initial state
 */
void    nul_info()
{
  register int  i;
  
  info.f_name = NULL; 
  info.f_type[0] = '\0'; 
  info.b_type[0] = '\0';
  info.para_no = 0;
  for(i = 0; i < PP_MAX_NO_PARA; i++) 
    {
      info.para_tab[i].p_name[0] = '\0';
      info.para_tab[i].p_type[0] = '\0';
    }
}

/*
 * assign the format character to the corresponding variable type
 * as documented in ISIS documentation. The format character is used in inserting
 * values into messages.
 * parameter : type_strptr - null-terminated string of variable type
 *
 * At present, only the basic types are supported
 */
char    get_format(type_strptr)
     char       *type_strptr;
{
  if (strcmp(type_strptr,"int") == PP_EQUAL || 
      strcmp(type_strptr,"long") == PP_EQUAL || 
      strcmp(type_strptr,"unsigned") == PP_EQUAL )
    return('d');
  
  if (strcmp(type_strptr, "char") == PP_EQUAL )
    return ('c');

  if (strcmp(type_strptr, "char*") == PP_EQUAL)
    return ('s');
  
  if (strcmp(type_strptr, "address") == PP_EQUAL )
    return('a');
  
  if (strcmp(type_strptr, "void") == PP_EQUAL )
    return('v');
  
  if (strcmp(type_strptr, "short") == PP_EQUAL)
    return('h');

  if (strcmp(type_strptr, "bitvec") == PP_EQUAL)
    return('b');

  if (strcmp(type_strptr, "double") == PP_EQUAL)
    return('f');
  
  return('x');   /* x is the error format character */
}

/*
 *  insert the token string into the data structure info accoprding to its type
 *  parameters : infostr - a null terminated string
 *               length - length of infostr
 *               type - type of token, indication by constant value
 */
int     insertinfo( infostr, length, type)
     char       *infostr;
     int        length, type;
{
  /* insert accordingly */
  switch (type) 
    {
      
      /* inserting function name */
    case PP_F_NAME :    
      info.f_name = insert_tab(infostr);
      if (info.f_name == NULL) halt("Sorry, please recompile with a larger\
 PP_MAX_NO_FUNCTION value");
      break;    
      
      /* inserting function type */
    case PP_F_TYPE :    
      strncpy(info.f_type, infostr, PP_MAX_TYPE_CHAR);
      if((info.fmt_char = get_format(info.f_type)) == 'x') 
        {
          fprintf(stderr, "line %d: Error: Non supported function type\n",
                  pp_lineno);
        }
      break;
      
      /* inserting broadcast type */
    case PP_B_TYPE :    
      strncpy(info.b_type, infostr, PP_MAX_BTYPE_CHAR);
      break;
      
      default : 
        halt("Fatal error: info.c:insertinfo() called with undefined type");
    }
  
  return(PP_OK);
}

/* 
 * insert a parameter name string into the parameter table
 *
 * parameter :  parastr - string with the parameter name
 *              length - length of parastr
 */
int     insertparameter(parastr, length)
     char       *parastr;       /* parameter string */
     int        length;         /* length of parastr */
{
  
  /* check if there is still room for new parameter */
  if (info.para_no  >= PP_MAX_NO_PARA)
    {
      fprintf(stderr, "line %d:(Fatal error) number of parameters\
 greater than %d\n", pp_lineno, PP_MAX_NO_PARA); 
      halt("Please recompile with a larger value of PP_MAX_NO_PARA");
    }
  
  /* copy the string to the next free element in the table */
  strncpy(info.para_tab[info.para_no].p_name, parastr, PP_MAX_PARA_CHAR);
  info.para_no++;
  if (length > PP_MAX_PARA_CHAR)
    {
      fprintf(stderr, "line %d:(Error) parameter name length \
greater than %d\n", pp_lineno, PP_MAX_PARA_CHAR);
      return(PP_ERROR);
    }
  
  return (PP_OK);
}       

/* 
 * return the pointer to the function information string
 *
 * parameter : type - const defining the type of information wanted
 */
char    *getinfo(type)
     int        type;
{
  switch (type) 
    {
    case PP_F_NAME :    return(info.f_name);
    case PP_F_TYPE :    return(info.f_type);
    case PP_B_TYPE :    return(info.b_type);
    case PP_F_FMT  :    return(&info.fmt_char);
      default : halt("Fatal error: info.c:getinfo() called with undefined type");
      return(NULL);
    }
}

/* 
 * get the first parameter from the parameter table, and initialize
 * cur_idx = 0 
 * store the name and type of the first parameter into the variables, 
 * name and type, respectively
 * if no available member, store NULL to name and type 
 * 
 * parameter :  name - pointer to the pointer of the string buffer for param name
 *              type -  (ditto) for param type
 *              fmt -   format character
 *              reg - true if this parameter is a register var
 */
void    firstparameter(name, type, fmt, reg)
     char       **name, **type, *fmt;
     bool       *reg;
{       
  cur_idx = 0;
  if (info.para_no == 0) 
    {
      /* no parameter in the table */
      *name = (*type = NULL);
    } 
  else 
    {
      *name = info.para_tab[cur_idx].p_name;
      *type = info.para_tab[cur_idx].p_type;
      *fmt = info.para_tab[cur_idx].fmt_char;
      *reg = info.para_tab[cur_idx].reg_var;
    }
}

/* 
 * go the the next parameter in the table, cur_idx++
 * store the name and the type into the variables, name and type, respectively
 * if no available member, store NULL to <name> and <type> is undefined
 *
 * parameter :  < same as firstparameter() >
 */
void    nextparameter(name, type, fmt, reg)
     char       **name, **type, *fmt;
     bool       *reg;
{
  cur_idx++;
  if (*(info.para_tab[cur_idx].p_name) == '\0') 
    {
      /* no more parameters */
      *name = (*type = NULL);
    } 
  else 
    {
      *name = info.para_tab[cur_idx].p_name;
      *type = info.para_tab[cur_idx].p_type;
      *fmt = info.para_tab[cur_idx].fmt_char;
      *reg = info.para_tab[cur_idx].reg_var;
    }
}

/* 
 * store the parameter type string into static variable,
 * temp_paratype_buffer, and return the pointer of the static var
 *
 * parameter :  str - string of the parameter type
 *              len - length of str
 */
char    *store_paratype(str, len)
     char       *str;
     int        len;
{
  char  *s;
  
  /* copy the name the variable buffer */
  s = strncpy(temp_paratype_buffer, str, PP_MAX_TYPE_CHAR);
  if (len > PP_MAX_TYPE_CHAR) 
    {
      fprintf(stderr, "line %d:(Error) '%s' is longer than %d\
 characters. It will be truncated\n", pp_lineno, str, PP_MAX_TYPE_CHAR);
    }
  return(s);
}

/* 
 * search for parameter name in the parameter table.
 * If not found, return PP_ERROR
 * Else insert parameter type string into the table and return PP_OK
 * 
 * parameter :  pname - string of the parameter name
 *              ptype - string of the parameter type
 *              plen -  length of ptype
 *              preg - true if parameter is a register variable
 */
int     insert_paratype(pname, ptype, plen, preg)
     char       *ptype, *pname;
     int        plen;
     bool       preg;
{       
  register int  i;
  
  /* search for the parameter name in the table */
  for(i = 0;((i < PP_MAX_NO_PARA) && 
             (strncmp(pname, info.para_tab[i].p_name, PP_MAX_PARA_CHAR) != PP_EQUAL));
      i++);
  
  if (i != PP_MAX_NO_PARA) 
    {
      /* found pname in the table */
      /* copy the parameter type string into the table */
      strncpy(info.para_tab[i].p_type, ptype, PP_MAX_PARA_CHAR);
      if ((info.para_tab[i].fmt_char = get_format(info.para_tab[i].p_type)) == 'x')
        {
          fprintf(stderr, "line %d: Error: Non supported rpc parameter type\n",
                  pp_lineno);
        }
      info.para_tab[i].reg_var = preg;
      return (PP_OK);
    }  
  return (PP_ERROR);
}
