/*
 * ISIS System V1.0 (8/15/88).   Report problems to isis-bugs@gvax.cs.cornell.edu 
 * Note: Export restrictions apply.  License required for export to ``non-free world''
 * countries (USDC terminology).  Agreement to respect this export restriction required
 * for export to all other countries.
 */
/******************************************************************************
 *
 * PP_fname_tab.c : module to manage the table of rpc function names
 *
 ******************************************************************************/
#include <stdio.h>
#include <strings.h>

#include "pp_const.h"
#include "pp_errhandle.e"
#include "pp_global.h"

/* table of function names */
typedef struct {
        char    str[PP_MAX_FUNCTION_CHAR];
}fname;
static fname    fname_tab[PP_MAX_NO_FUNCTION];  
static int      funcno;                 /* # of function names in the table */

/* 
 * init the function name table
 */
void    init_fname_tab()
{
        register int    i;

        funcno = 0;
        for(i =0; i < PP_MAX_NO_FUNCTION; i++){
                *fname_tab[i].str = '\0';
        }
}

/*
 * sort fname_tab[] for the first funcno elements, and delete any duplicate
 * elements.
 *
 * sorting is done with the library routine qsort and strcmp as the comparison
 * routine.
 */
void    sort_tab()
{
        int     num_deleted;
        
        /* sort the fname_tab[] */
        qsort((char *)fname_tab, funcno, sizeof(fname), strcmp);
        
        /* delete any duplicate in fname_tab */
        num_deleted = delete_dup(funcno);
        if (num_deleted > 0) {
                funcno -= num_deleted;
                qsort((char *)fname_tab, funcno, sizeof(fname), strcmp);
        }
}

/*
 * Insert the function name into the table. Return the pointer to 
 * the table entry (as a char *)
 * else if table is full , return NULL
 */
char    *insert_tab(strptr)
        char    *strptr;
{
        if (funcno >= PP_MAX_NO_FUNCTION)
                /* try deleting the duplicate by sorting the table */
                sort_tab();

        if(funcno < PP_MAX_NO_FUNCTION) {
                /* insert into table at index funcno */
                strncpy(fname_tab[funcno].str, strptr, PP_MAX_FUNCTION_CHAR-1);
                return(fname_tab[funcno++].str);
        } else {
                /* table is full */
                fprintf(stderr, "Overflow: Number of rpc function exceeds\
 %d\n", PP_MAX_NO_FUNCTION);
                return(NULL);
        }
}
        
/* 
 * delete any duplicates in fname_tab[], which has a total of count elements.
 * return the number of elements deleted
 */
static
int     delete_dup(count)
        int     count;
{
        int     base, i;
        
        base = count - 1; 
        for(i = base; i > 0; i--) {
                if(strcmp(fname_tab[i].str,fname_tab[i-1].str) == PP_EQUAL) {
                        /* found duplicates */
                        if (i != base) {
                                /* move the base element to ith */
                                 strcpy(fname_tab[i].str,fname_tab[base].str);
                        }
                        *fname_tab[base].str = '\0';
                        base--;
                }
        }
        return(count - base - 1);
}

/* 
 * return the pointer of the ith function name string in the table
 *
 * return NULL if table empty
 */
char    *get_fname(i)
        int     i;
{
        if (i < funcno && i < PP_MAX_NO_FUNCTION) 
                return(fname_tab[i].str);
        else
                return (NULL);
}

/* for debugging purposes */
void    print_tab()
{
        register int i;
        
        for(i = 0; i < funcno; i ++)
                printf("%d : '%s'\n", i, fname_tab[i].str);
