/* Written by Alex Siegel at Cornell University 9/88 */
/* This module is the main program for the game.  It coordinates all */
/* the other components.  It also handles the sending and */
/* receiving of nearly all the messages, data, and updates.  This is a */
/* combination of X windows and isis with a lot of flavor from both. */

#include "gdefs.h"
#include "config.h"
#include "isis/isis.h"
#include <X11/Xlib.h>
#include <time.h>

extern army *loadarmy();
extern update *genupd();
extern char *getenv();

/* Entry number for isis entry points.  Update handles receives update */
/* messages.  Add_army handles adding new armies.  Get_talk handles */
/* receiving messages from other players. */
#define UPDATE_ENTRY 	1
#define ADD_ARMY_ENTRY	2
#define GET_TALK_ENTRY	3

/* REMAP_FREQUENCY is the number of turns before the map gets remade */
/* without external stimulus. */
#define REMAP_FREQUENCY 8

address gaddr;
/* gaddr - isis group address of game group */
army *myarmy,battle[MAXPLAYER];
/* myarmy - pointer to local player's army
   battle - array of armies for entire battle */
address addrs[MAXPLAYER];
/* addrs - isis address of each player */
long lastupd[MAXPLAYER];
/* lastupd - time of last update from each player.  Used for */
/* 			 synchronization of different players */
int numarmy,myindex;
/* numarmy - number of armies in game
   myindex - number of local army */
long mytime;
/* mytime - local time counter */
int gamespeed=3;
/* gamespeed - number of seconds delay between turns */
int armyflag,viewflag,quitflag,portflag,verbflag,remap;
/* armyflag - contains the command line position of the army file
              name, 0 otherwise.
   viewflag - true if the -v option is in use
   quitflag - true if the player wishes to quit
   portflag - value of isis port provided by -p option, 0 otherwise
   verbflag - true if the game is in verbose mode
   remap - number of turns until an automatic remake of the map */

extern Display *disp;
extern Window wind;

int conflict_maintask();
int receive_update();
int group_change();
int group_init();
int send_state();
int receive_state();
int add_army();
int xinput_handler();
int xerrhandle();
int keyhandler();
int get_talk();

main(argc,argv)
int argc;
char **argv;
{
  FILE *inf;
  /* inf - army input file */
  int i,j;
  /* i,j - loop variables */
  char *s;
  /* s - pointer to isis port string in environment table */

  /* Compute values for all game flags */
  armyflag = 0;
  viewflag = False;
  portflag = 0;
  /* Loop through command line arguments */
  for(i=1;i<argc;++i) {
	if(argv[i][0] == '-')
	  /* Loop through option list */
	  for(j=1;j<strlen(argv[i]);++j)
		switch(argv[i][j]) {
		case 'v':
		  viewflag = True;
		  break;
		case 'p':
		  sscanf(argv[i+1],"%d",&portflag);
		  break;
		default:
		  printf("Unknown flag: %c\n",argv[i][j]);
		  exit(-1);
		}
	/* Skip past a port number */
	else if((argv[i][0] >= '0') && (argv[i][0] <= '9') && portflag)
	  continue;
	/* No leading dash or digit, it must be an army name */
	else
	  armyflag = i;
  }
  verbflag = True;

  /* Load in item description data */
  loaddata();
  if(armyflag) {
	/* Load in locally controlled army */
	inf = fopen(argv[armyflag],"r");
	if(inf == NULL) {
	  perror(argv[1]);
	  exit(-1);
	}
	myarmy = loadarmy(inf);
	fclose(inf);
  }

  /* Load port number from environment if none is present in the command string */
  if(! portflag) {
	s = getenv("ISISPORT");
	if(s == NULL) {
	  printf("You must have the environment variable ISISPORT.\n");
	  printf("You may also use the -p <port number> option on the\n");
	  printf("command line.\n");
	  exit(-1);
	}
	sscanf(s,"%d",&portflag);
  }

  /* Initialize isis connection */
  isis_init(portflag);

  /* Tell isis about the various tasks and entries available */
  isis_task(conflict_maintask,"conflict_maintask");
  isis_task(group_change,"group_change");
  isis_entry(UPDATE_ENTRY,receive_update,"receive_update");
  isis_entry(ADD_ARMY_ENTRY,add_army,"add_army");
  isis_entry(GET_TALK_ENTRY,get_talk,"get_talk");
  
  /* Start up main loop of isis */
  isis_mainloop(conflict_maintask);
}

/* X windows event mask */
#define EVMASK ButtonPressMask | ExposureMask | EnterWindowMask

/* Main task of simulation. */
conflict_maintask()
{
  int i,numupd,newarmy;
  /* i - loop variable
	 numupd - number of updates on latest update list
	 newarmy - number of new army */
  update *upd;
  /* upd - pointer to latest update list */
  armyunit *curunit;
  /* curunit - pointer to current army unit */
  register long t;
  /* t - temporary storage use for a swap */

  /* Join the game group */
  gaddr = pg_join("conflict",
		  PG_MONITOR, group_change, 0,
		  PG_XFER,0,send_state,receive_state,
		  PG_INIT, group_init,
		  0);

  /* Flag isis the initialization is done */
  isis_start_done();

  /* If there is an local army present, go through army initialization */
  if(armyflag) {
	/* Gues what number the new army is */
	if(numarmy < MAXPLAYER)
	  newarmy = numarmy;
	/* Use a slot from an old player */
	else
	  for(i=0;i<numarmy;++i)
		if(battle[i].size == 0) {
		newarmy = i;
		break;
	  }
	
	/* Offset my army so that it is not on top of somebody else */
	for(i=0;i<myarmy->size;++i) {
	  curunit = (myarmy->units) + i;
	  switch(newarmy) {
	  case 0:
		curunit->ypos -= 500;
		break;
	  case 1:
		curunit->ypos = -(curunit->ypos) + 500;
		break;
	  case 2:
		curunit->ypos -= 500;
		t = curunit->xpos;
		curunit->xpos = curunit->ypos;
		curunit->ypos = t;
		break;
	  case 3:
		curunit->ypos = -(curunit->ypos) + 500;
		t = curunit->xpos;
		curunit->xpos = curunit->ypos;
		curunit->ypos = t;
		break;
	  }
	}

	/* Initialize myindex */
	myindex = -1;
	/* Broadcast local army to add to everyone else */
	abcast(gaddr,ADD_ARMY_ENTRY,
		   "%l%C",mytime,myarmy->units,myarmy->size * sizeof(armyunit),
		   ALL,"");
	/* Check to see if myindex was not assigned */
	if(myindex == -1) {
	  printf("Sorry, but too many people are playing already.\n");
	  exit(-1);
	}
	
	/* Handler key board commands */
	isis_input(fileno(stdin),keyhandler,0);

	printf("I have entered the fray.\n");
  }

  /* If the tactical map is needed, create it */
  if(viewflag) {
	/* Set the X windows error handler */
	XSetErrorHandler(xerrhandle);
	/* Create the map */
	makemap();
	/* Initialize remap */
	remap = REMAP_FREQUENCY;
	/* Specify to X windows which events are interesting */
	XSelectInput(disp,wind,EVMASK);
	/* Tell isis to flag this process when X events are available */
	isis_input(ConnectionNumber(disp),xinput_handler,0);
  }

  /* Main game loop */
  quitflag = False;
  while(! quitflag) {
	/* Wait gamespeed seconds */
	sleep(gamespeed);
	/* If there is a local army, do one turn */
	if(armyflag) {
	  /* Wait until everyone else has synchronized. */
	  for(i=0;i<numarmy;++i)
		while((lastupd[i] < lastupd[myindex]) &&
			  (pg_rank(gaddr,addrs[i]) >= 0))
		  sleep(1);
	  /* Generate new update list */
	  upd = genupd(myindex,numarmy,battle,&numupd);
	  /* Increment local time counter */
	  lastupd[myindex] ++;
	  /* Print time */
	  if(verbflag && (lastupd[myindex] % 6 == 0))
		printf("%d minutes into the battle.\n",lastupd[myindex]/6);
	  /* Broadcast update to everyone INCLUDING MYSELF */
	  cbcast(gaddr,UPDATE_ENTRY,
			 "%l%C",lastupd[myindex],upd,numupd*sizeof(update),
			 ALL,"");
	  /* Free the update array */
	  free(upd);

	  /* Check to see if I am out of troops to play with */
	  for(i=0;i<battle[myindex].size;++i)
		if(battle[myindex].units[i].count)
		  break;
	  if(i == battle[myindex].size) {
		printf("I'm done for the day.\n");
		confquit(0);
	  }
	}

	/* If map is in use and check to see if it is time to redraw it */
	if(viewflag) {
	  remap--;
	  if(remap == 0) {
		drawmap(battle,numarmy);
		remap = REMAP_FREQUENCY;
	  }
	}
  }

  /* Quit */
  confquit(0);
}

/* Quit the game with exit status specified in arg */
confquit(arg)
int arg;
/* arg - exit status requested */
{
  int i;
  /* i - loop variable */
  armyunit *curunit;
  /* curunit - pointer to current unit */

  /* If there is a map, destroy it */
  if(viewflag)
	killmap();

  /* If there is a local army, display its final status */
  if(armyflag) {
	printf("Final status:\n");
	for(i=0;i<battle[myindex].size;++i) {
	  curunit = (battle[myindex].units) + i;
	  printf("#%d : Strength = %d  Hits = %d",i,curunit->count,curunit->tothits);
	  if(curunit->charges >= 0)
		printf("  Ammo = %d",curunit->charges);
	  printf("  Moral = ");
	  if(curunit->moral <= -9)
		printf("defeated\n");
	  else if(curunit->moral <= -5)
		printf("very poor\n");
	  else if(curunit->moral <= -2)
		printf("poor\n");
	  else if(curunit->moral <= 1)
		printf("normal\n");
	  else if(curunit->moral <= 4)
		printf("good\n");
	  else if(curunit->moral <= 8)
		printf("very good\n");
	  else
		printf("victorious\n");
	}
  }

  /* Exit */
  exit(arg);
}

/* Handle a signal from isis that X input is ready */
xinput_handler(arg)
int arg;
/* arg - isis generic argument */
{
  XEvent xev;
  /* xev - new X event */

  /* Read in the next X event */
  XWindowEvent(disp,wind,EVMASK,&xev);
  /* Redraw the map whever the window is exposed */
  if(xev.type == Expose) {
	drawmap(battle,numarmy);
	remap = REMAP_FREQUENCY;
  }
  /* Quit when the right mouse button is pressed */
  else if((xev.type == ButtonPress) && (xev.xbutton.button == Button3))
	confquit(0);
  /* Raise the window when the mouse enters it */
  else if(xev.type == EnterNotify) {
	XRaiseWindow(disp,wind);
	XFlush(disp);
  }
}

/* Handle a X windows error event */
xerrhandle(disp,err)
Display *disp;
/* disp - display where error occured */
XErrorEvent *err;
/* err - error event */
{
  char buf[100];
  /* buf - buffer for text describing error */
  int len;
  /* len - length of text describing error */

  /* Get error description */
  XGetErrorText(disp,err->error_code,buf,&len);

  /* Print message to user */
  printf("X error!  Op code = %d: %s\n",err->request_code,buf);

  /* Crash */
  exit(-1);
}

/* Isis task to handle group initialization */
group_init()
{
  /* If there is a local army, initialize critical variables */
  if(armyflag) {
	printf("You are the first.  You may select the scenario.\n");
	numarmy = 0;
	mytime = 0;
  }
  /* Otherwise, quit. */
  else {
	printf("Sorry.  Nobody else is playing now.\n");
	confquit(0);
  }
}

/* Isis task to handle group changes */
group_change(gview_p,arg)
groupview *gview_p;
/* gview_p - pointer to group view produced by isis */
int arg;
/* arg - generic isis argument */
{
  int i,j;
  /* i,j - loop variables */

  /* Make sure that everyone is still in the group */
  for(i=0;i<numarmy;++i) {
	/* Of course I am in the group */
	if(armyflag && (i == myindex))
	  continue;
	/* Go through group view member list to see if player i is still */
	/* around */
	for(j=0;j < gview_p->gv_nmemb;++j)
	  if(addr_isequal(gview_p->gv_members[j],addrs[i]))
		break;
	/* If player i is dead, kill his local data storage */
	if((j == gview_p->gv_nmemb) && battle[i].size) {
	  printf("Player %d has dropped out.\n",i);
	  battle[i].size = 0;
	  free(battle[i].units);
	  battle[i].units = NULL;
	  /* Redraw the map */
	  if(viewflag) {
		drawmap(battle,numarmy);
		remap = REMAP_FREQUENCY;
	  }
	}
  }
}

/* Isis task to send the local state to a new group member */
send_state(last_locator)
int last_locator;
/* last_locator - last state message sent */
{
  int i;
  /* i - loop variable */

  /* Send last update times and implicitly the total number of armies */
  xfer_out(0,"%L",lastupd,numarmy);
  /* Send all the locally known armies */
  for(i=0;i<numarmy;++i)
    xfer_out(i+1,"%a%C",addrs[i],battle[i].units,battle[i].size * sizeof(armyunit));
}

/* Isis task to receive a state from an old group member */
receive_state(locator,msg_p)
int locator;
/* locator - number signifying which state description message this is */
message *msg_p;
/* msg_p - pointer to state message */
{
  long size;
  /* size - size of armies array received */
  register int i;
  /* i - loop variable */

  /* Receive time stamp list and number of players */
  if(locator == 0) {
    msg_get(msg_p,"%L",lastupd,&numarmy);
	/* If local army is present, generate a local time which is the */
	/* maximum of everyone else's time */
	if(armyflag) {
	  mytime = lastupd[0];
	  for(i=1;i<numarmy;++i)
		if(lastupd[i] > mytime)
		  mytime = lastupd[i];
	}
  }
  /* Receive an army */
  else {
    msg_get(msg_p,"%a%+C",addrs+(locator-1),&(battle[locator-1].units),&size);
	/* Convert length in characters into length in armyunit structures */
    battle[locator-1].size = size/sizeof(armyunit);
  }
}

/* Isis task to receive a new army from a new player */
add_army(msg_p)
message *msg_p;
/* msg_p - pointer to message containing new army */
{
  long size;
  /* size - length of army units array in character */
  int newarmy,i;
  /* newarmy - number of new army being received 
	 i - loop variable */
  
  /* Figure out who the new army is */
  if(numarmy < MAXPLAYER) {
	newarmy = numarmy;
	numarmy++;
  }
  /* Use a slot from an old player */
  else {
	for(i=0;i<numarmy;++i)
	  if(battle[i].size == 0) {
		newarmy = i;
		break;
	  }
	if(i == numarmy) {
	  printf("Too many people are trying to join!\n");
	  nullreply(msg_p);
	  return;
	}
  }

  /* Extract army from message */
  msg_get(msg_p,"%l%+C",&(lastupd[newarmy]),&(battle[newarmy].units),&size);
  /* Convert length to armyunit structures */
  battle[newarmy].size = size/sizeof(armyunit);
  /* Get address of new player */
  addrs[newarmy] = msg_getsender(msg_p);
  /* If it is my army, then now I know which number player I am.  This */
  /* may seems like a strange way of finding out which number I am, */
  /* but it insures consistency when multiple people are joining */
  /* roughly simultaneously. */
  if(armyflag && addr_ismine(addrs[newarmy])) {
    myindex = newarmy;
    printf("I am army number %d.\n",myindex);
  }
  /* Redraw the map if someone else joins */
  else if(viewflag) {
	drawmap(battle,numarmy);
	remap = REMAP_FREQUENCY;
  }

  /* Reply to signal that message was received and processed */
  nullreply(msg_p);
}

/* Isis task to handles an update message */
receive_update(msg_p)
message *msg_p;
/* msg_p - pointer to message containing update */
{
  int ind,numupd;
  /* ind - index of army which sent update
	 numupd - total number of updates in list */
  int i;
  /* i - loop variable */
  update *upd;
  /* upd - pointer to update list */
  long size;
  /* size - length of update list in characters */

  /* Compute who sent the update */
  ind = -1;
  for(i=0;i<numarmy;++i)
    if(addr_isequal(addrs[i],msg_getsender(msg_p))) {
      ind = i;
      break;
    }
  if(ind == -1) {
    printf("I just got an army update for an army that doesn't exist!\n");
    nullreply(msg_p);
  }
  
  /* Extract the update from the message */
  msg_get(msg_p,"%l%+C",&(lastupd[ind]),&upd,&size);
  /* Convert the length from characters to update structures */
  numupd = size/sizeof(update);
  /* Loop through updates */
  for(i=0;i<numupd;++i) {
	/* Erase unit which update pertains to */
	if(viewflag && disp && wind)
	  drawunit(battle,upd[i].unit,upd[i].index,numarmy);
	/* Execute the update */
	eatupd(upd+i,battle,myindex);
	/* Redraw the unit */
	if(viewflag && disp && wind) {
	  drawunit(battle,upd[i].unit,upd[i].index,numarmy);
	  XFlush(disp);
	}
  }
  /* Free the update list */
  free(upd);
  /* Reply to signal that update was received */
  nullreply(msg_p);
}

/* Isis task to handle messages from other players */
get_talk(msg_p)
message *msg_p;
/* msg_p - pointer to incoming message */
{
  int i;
  /* i - loop variable */
  char buf[100];
  /* buf - buffer to hold new message */

  /* Compute who sent the message */
  for(i=0;i<numarmy;++i)
    if(addr_isequal(addrs[i],msg_getsender(msg_p))) {
      printf("Message from player %d:\n",i);
      break;
    }
  if(i == numarmy)
    printf("Message from somebody:\n");

  /* Extract and print the message */
  msg_get(msg_p,"%C",buf,&i);
  buf[i] = '\0';
  printf(" -> %s\n",buf);

  /* Reply that message was received */
  nullreply(msg_p);
}
