/* Written by Alex Siegel at Cornell University 9/88 */
/* This module handles moving units around and having them attack each */
/* other. */

#include "gdefs.h"
#include "config.h"
#include <math.h>

extern long getval();

extern long minind,minunit,centx,centy;
extern int maxupd,numupd;
extern update *upds;
extern int verbflag;

/* Generate a movement update for a unit depending on where it is
   trying to go. */
genmove(myind,myunit,batt)
int myind,myunit;
/* myind,myunit - player index and unit number */
army *batt;
/* batt - array of armies */
{
  long newx,newy,targx,targy;
  /* newx,newy - new x and y position produced
	 targx,targy - target x and y position */
  armyunit *curunit;
  /* curunit - pointer ot current army unit */
  float dist;
  /* dist - total distance travelled */
  register float dx,dy;
  /* dx,dy - x and y distance to target */
  update *curupd;
  /* curupd - pointer to current update being written */
  register int i,idx,idy;
  /* i - loop variable
	 idx,idy - integer x and y distance to nearest friendly unit */

  /* Get pointer to current unit */
  curunit = (batt[myind].units) + myunit;

  /* Movement modification for negative moral */
  if((curunit->movmode == CHASE) && (curunit->moral < 0)) {
	if(verbflag)
	  printf("Unit %d decides to give up the chase.\n",myunit);
	curunit->movmode = FORWARD;
  }
  if((curunit->movmode == FORWARD) && (curunit->moral < -3)) {
	if(verbflag)
	  printf("Unit %d is battered to halt.\n",myunit);
	curunit->movmode = STAND;
  }
  if((curunit->movmode == STAND) && (curunit->moral < -6)) {
	if(verbflag)
	  printf("Unit %d is forced to retreat.\n",myunit);
	curunit->movmode = RETREAT;
	curunit->targmode = NEAREST;
  }
  if((curunit->movmode == RETREAT) && (curunit->moral < -9)) {
	if(verbflag)
	  printf("Unit %d turns tail and runs.\n",myunit);
	curunit->movmode = RUNAWAY;
	curunit->targmode = NEAREST;
	curunit->attmode = NONE;
  }

  /* Movement modification for postive moral */
  if((curunit->movmode == RUNAWAY) && (curunit->moral > 0)) {
	if(verbflag)
	  printf("Unit %d decides to stop running.\n",myunit);
	curunit->movmode = RETREAT;
  }
  if((curunit->movmode == RETREAT) && (curunit->moral > 3)) {
	if(verbflag)
	  printf("Unit %d has made a stand.\n",myunit);
	curunit->movmode = STAND;
  }
  if((curunit->movmode == STAND) && (curunit->moral > 6)) {
	if(verbflag)
	  printf("Unit %d decides to close in.\n",myunit);
	curunit->movmode = FORWARD;
	curunit->targmode = NEAREST;
	curunit->attmode = ATTACK;
  }
  if((curunit->movmode == FORWARD) && (curunit->moral > 9)) {
	if(verbflag)
	  printf("Unit %d decides to give chase to those slime dogs.\n",myunit);
	curunit->movmode = CHASE;
	curunit->targmode = NEAREST;
	curunit->attmode = ATTACK;
  }

  /* Just return if unit is standing around */
  if(curunit->movmode == STAND)
	return;

  /* Assign default new position */
  newx = curunit->xpos;
  newy = curunit->ypos;

  /* Check to see if the units have gotten bunched up */
  /* Allow a chance for a unit to break free of a pack */
  if(random() % 4 != 0)
	for(i=0;i<batt[myind].size;++i) {
	  if(i == myunit)
		continue;
	  idx = batt[myind].units[i].xpos - newx;
	  if(idx < 0)
		idx = -idx;
	  if(idx > 3)
		continue;
	  idy = batt[myind].units[i].ypos - newy;
	  if(idy < 0)
		idy = -idy;
	  if(idy > 3)
		continue;
	  
	  /* Mill around randomly until the other units have cleared away */
	  newx = newx + (random() % 11) - 5;
	  newy = newy + (random() % 11) - 5;
	  
	  /* User message */
	  if(verbflag)
		printf("Unit %d is disorganized.\n",myunit);
	  break;
	}

  /* Continue if unit is free from other units */
  if(i == batt[myind].size) {
	/* Generate target coordinate depending on mode */
	switch(curunit->targmode) {
	case NEAREST:
	  targx = batt[minind].units[minunit].xpos;
	  targy = batt[minind].units[minunit].ypos;
	  break;
	case FOLLOW:
	  targx = batt[myind].units[curunit->targunit].xpos;
	  targy = batt[myind].units[curunit->targunit].ypos;
	  break;
	case POSIT:
	  targx = curunit->targx;
	  targy = curunit->targy;
	  break;
	case CENTER:
	  targx = centx;
	  targy = centy;
	  break;
	}

	/* Calculate base speed of unit */
	dist = getval(curunit->platf,SPEED);
	/* Modify speed depending on total bulk */
	dist = (dist * getval(curunit->platf,BULK))/
	  (getval(curunit->platf,BULK) + getval(curunit->weapon,BULK) +
	   getval(curunit->armor,BULK));
	/* Modify speed depending on movement mode */
	switch(curunit->movmode) {
	case CHASE:
	  dist = 2.0 * dist;
	  break;
	case FORWARD:
	  break;
	case RETREAT:
	  if(curunit->targmode != FOLLOW)
		dist = -dist;
	  break;
	case RUNAWAY:
	  if(curunit->targmode != FOLLOW)
		dist = -2.0 * dist;
	  else
		dist = 2.0 * dist;
	  break;
	}
	/* Get distance to target */
	dx = curunit->xpos - targx;
	dy = curunit->ypos - targy;
	/* Check to see if unit is close enough to move to right on top of target */
	if((dx*dx + dy*dy <= dist*dist) && (dist > 0.0)) {
	  newx = targx;
	  newy = targy;
	}
	/* Check to see you unit is trying to retreat from a unit that 
	   it is on top of */
	else if(dx==0.0 && dy==0.0 && dist<0.0) {
	  /* Sometime a unit will fail to escape */
	  if(random() % 3) {
		newx = curunit->xpos;
		newy = curunit->ypos;
	  }
	  else {
		newx = (random()%((int) dist)) - ((int) dist)/2 + curunit->xpos;
		newy = (random()%((int) dist)) - ((int) dist)/2 + curunit->ypos;
	  }
	}
	/* Otherwise move part way */
	else if(dist != 0.0) {
	  /* Calculate the ratio of distance traveled to total distance */
	  dist = dist/sqrt(dx*dx + dy*dy);
	  /* Produce rounded new positions */
	  newx = (targx - curunit->xpos)*dist + 0.5 + curunit->xpos;
	  newy = (targy - curunit->ypos)*dist + 0.5 + curunit->ypos;
	}
  }
  
  /* A unit will not run into lava unless forced */
  if((curunit->movmode != RUNAWAY) && (curunit->movmode != RETREAT)) {
	if(newx < -1000)
	  newx = -1000;
	if(newx > 1000)
	  newx = 1000;
	if(newy < -1000)
	  newy = -1000;
	if(newy > 1000)
	  newy = 1000;
  }

  /* If a new position has been reached, generate an update */
  if((newx != curunit->xpos) || (newy != curunit->ypos)) {
	curupd = upds + numupd;
	numupd++;
	curupd->type = POS_UPD;
	curupd->index = myind;
	curupd->unit = myunit;
	curupd->new.pos.x = newx;
	curupd->new.pos.y = newy;
  }
}

/* Generate an attack update if the unit is close enough to attack.
   True is returned if an attack was made, false otherwise. */
int genatt(myind,myunit,batt)
int myind,myunit;
/* myind,myunit - player index and unit number of unit */
army *batt;
/* batt - array of armies */
{
  long range;
  /* range - maximum attacking range */
  armyunit *curunit;
  /* curunit - pointer to current unit */
  float dx,dy,prob,modif;
  /* dx,dy - x and y distance to target unit
	 prob - probability of attack hitting
	 modif - modification factor on probability */
  update *curupd;
  /* curupd - pointer to current update */

  /* Get pointer to current unit */
  curunit = (batt[myind].units) + myunit;

  /* If unit moral is too low, the unit may only defend */
  if(curunit->moral < -5) {
	if(verbflag)
	  printf("Unit %d is too scared to fight.\n",myunit);
	curunit->attmode = DEFEND;
	return(0);
  }

  /* Calculate maximum range */
  range = getval(curunit->platf,RANGE) +
	getval(curunit->weapon,RANGE);
  /* Check to see if no hostile unit is visable */
  if(minind == myind)
	return(0);
  /* Calculate distance to closest enemy unit */
  dx = curunit->xpos - batt[minind].units[minunit].xpos;
  dy = curunit->ypos - batt[minind].units[minunit].ypos;
  /* If enemy unit is out of range, return */
  if(range < sqrt(dx*dx+dy*dy))
	return(0);

  /* Calculate to hit modifier.  No modifier for short range weapons. */
  /* The modifier is basically an exponent for miss probability drop */
  /* off. */
  if(range > 3) {
	/* Maximum modifier is 5.0 */
	if((dx == 0.0) && (dy == 0.0))
	  modif = 5.0;
	else {
	  modif = (range*range)/(dx*dx + dy*dy);
	  if(modif > 5.0)
		modif = 5.0;
	}
  }
  else
	modif = 1.0;

  /* Generate attack update */
  curupd = upds + numupd;
  numupd++;
  curupd->type = HIT_UPD;
  curupd->index = minind;
  curupd->unit = minunit;
  /* If no weapon, base attack on platform statistics */
  if(curunit->weapon == -1) {
	curupd->new.hit.avdam = getval(curunit->platf,DAMAG);
	curupd->new.hit.sptype = getval(curunit->platf,SPTYPE);
	curupd->new.hit.numb = curunit->count * 3;
	prob = getval(curunit->platf,HITPROB) / 100.0;
  }
  else {
	/* Use weapon statistics */
	curupd->new.hit.avdam = getval(curunit->platf,DAMAG)/3 + getval(curunit->weapon,DAMAG);
	curupd->new.hit.sptype = getval(curunit->weapon,SPTYPE);
	curupd->new.hit.numb = curunit->count * getval(curunit->weapon,ATTACKS);
	/* Check to see if the unit has limited charges */
	if(curunit->charges > 0) {
	  /* Decrement from the total number of charges */
	  if(curupd->new.hit.numb > curunit->charges)
		curupd->new.hit.numb = curunit->charges;
	  curunit->charges -= curupd->new.hit.numb;
	  /* If out of charges, use hands after this */
	  if(curunit->charges == 0) {
		printf("Unit %d is out of ammunition.\n",myunit);
		curunit->weapon = -1;
		/* Decrement moral */
		curunit->moral -= 2;
	  }
	}
	/* Calculate base hit probability */
	prob = getval(curunit->platf,HITPROB) * getval(curunit->weapon,HITPROB) / 4000.0;
  }
  /* Modify hit probability with exponential modifier */
  prob = 1.0 - pow((double) (1.0 - prob),(double) modif);
  /* Modify the total number of hits accordingly */
  curupd->new.hit.numb *= prob;
  /* Add one to moral if this unit got an attack off */
  if((curupd->new.hit.numb) && (random() & 1))
	curunit->moral ++;
  /* Return true */
  return(1);
}
