/* Written by Alex Siegel at Cornell University 9/88 */
/* This module generates and eats update lists for armies.  This and */
/* movatt.c form the core of the simulation end of the game. */

#include "gdefs.h"
#include "config.h"
#include <math.h>

extern long getval();
extern int genatt();

long minind,minunit,centx,centy;
/* minind,minunit - unit player index and number of hostile unit which is
  				    closest to the unit under consideration
   centx,centy - center of enemy formation */
int maxupd,numupd;
/* maxupd - current size of update array
   numupd - current number of updates   numupd<=maxupd */
update *upds;
/* upds - current array of updates */
extern int verbflag,armyflag;

/* Generate center position of hostile forces */
getcent(myind,numarm,batt)
int myind,numarm;
/* myind - index of player under consideration
   numarm - total number of armies */
army *batt;
/* batt - array of armies */
{
  register int nu,j,k;
  /* nu - total number of hostile units
	 j,k - loop variabels */
  register armyunit *curunit;
  /* curunit - current unit under consideration */

  /* Compute the average position of hostile units */
  nu = 0;
  centx = centy = 0;
  /* Loop through all armies */
  for(j=0;j<numarm;++j) {
	/* Ignore units in home army */
    if(j == myind)
      continue;
	/* Loop through all units */
    for(k=0;k<batt[j].size;++k) {
	  /* Get pointer to current unit */
      curunit = (batt[j].units) + k;
	  /* Ignore dead units */
	  if(curunit->count == 0)
		continue;
	  /* Add position to total */
      centx += curunit->xpos;
      centy += curunit->ypos;
      nu++;
    }
  }
  /* Get average */
  if(nu == 0) {
	centx = centy = 0;
  }
  else {
	centx /= nu;
	centy /= nu;
  }
}

/* Calculate closest hostile unit within three kilometers */
getmin(myind,myunit,numarm,batt)
int myind,myunit,numarm;
/* myind - number of player
   myunit - unit number of unit under consideration
   numarm - total number of armies */
register army *batt;
/* batt - array of armies */
{
  register int j,k,dx,dy;
  /* j,k - loop variables
	 dx,dy - x and y distances between units */
  long mindist;
  /* mindist - current minimum distance */
  long xpos,ypos;
  /* xpos,ypos - position of unit under consideration */

  /* Get position of unit */
  xpos = batt[myind].units[myunit].xpos;
  ypos = batt[myind].units[myunit].ypos;
  /* Initialize variables */
  mindist = 3000;
  minind = myind;
  minunit = myunit;
  /* Loop through all armies */
  for(j=0;j<numarm;++j) {
	/* Ignore units in home army */
	if(j == myind)
	  continue;
	/* Loop through all units */
	for(k=0;k<batt[j].size;++k) {
	  /* Ignore dead units */
	  if(batt[j].units[k].count == 0)
		continue;
	  /* Calculate the absolute x and y coordinate difference */
	  dx = xpos - batt[j].units[k].xpos;
	  if(dx < 0)
		dx = -dx;
	  dy = ypos - batt[j].units[k].ypos;
	  if(dy < 0)
		dy = -dy;
	  /* Compare the taxi cab geometry distance to the current minimum */
	  if(dx + dy < mindist) {
		/* Assign a new closest hostile unit */
		mindist = dx+dy;
		minind = j;
		minunit = k;
	  }
	}
  }
}

/* Generate a casulty update for a unit */
gencasul(myind,myunit,batt)
int myind,myunit;
/* myind,myunit - player index and unit number of unit to be
                  considered */
army *batt;
/* batt - array of armies */
{
  update *curupd;
  /* curupd - current update being written */
  long minhits;
  /* minhits - minimum allowable hit points for unit size */
  armyunit *curunit;
  /* curunit - pointer to current units */
  int newcnt;
  /* newcnt - new total unit size */
  
  /* Get pointer to current unit */
  curunit = (batt[myind].units) + myunit;

  /* If unit has walked off the end of the world, kill it */
  if((curunit->xpos < -1000) || (curunit->xpos > 1000) ||
	 (curunit->ypos < -1000) || (curunit->ypos > 1000)) {
	printf("Unit %d has walked into the lava fields.\n",myunit);
	curupd = upds + numupd;
	numupd++;
	curupd->type = CNT_UPD;
	curupd->index = myind;
	curupd->unit = myunit;
	curupd->new.count = 0;
	return;
  }

  /* Calculate the minimum required number of hit points for 
	 the number of platforms */
  minhits = curunit->count * getval(curunit->platf,HITS);
  minhits = (3*minhits)/5;
  /* If the unit has too few hit points, reduce the number of platforms */
  if(minhits > curunit->tothits) {
	/* Calculate the proportional reduction in platforms */
	newcnt = (curunit->count * curunit->tothits) / minhits;
	/* A unit can not have negative men */
	if(newcnt < 0)
	  newcnt = 0;
	printf("Unit %d reduced to %d through casualties.\n",myunit,newcnt);
	/* Generate update */
	curupd = upds + numupd;
	numupd++;
	curupd->type = CNT_UPD;
	curupd->index = myind;
	curupd->unit = myunit;
	curupd->new.count = newcnt;

	/* Decrement moral depending on relative severity of casulties */
	if(curunit->count <= 5)
	  curunit->moral -= 2*(curunit->count - newcnt);
	else if(curunit->count <= 10)
	  curunit->moral -= curunit->count - newcnt;
	else
	  curunit->moral -= (curunit->count - newcnt)/2;
  }
}

/* Generate an update list for an army.  A pointer to the allocated
   update list is returned. */
update *genupd(myind,numarm,batt,nupd)
int myind,numarm,*nupd;
/* myind - player index of army
   numarm - total number of armies
   nupd - number of updates generated.  RETURN VALUE! */
army *batt;
/* batt - array of armies */
{
  int i,attres;
  /* i - loop index
	 attres - result of call to genatt */
  armyunit *curunit;
  /* curunit - pointer to current unit */

  /* Initialize array of updates to have 0 updates */
  numupd = 0;
  /* Allow up to 50 updates before resizing array */
  maxupd = 50;
  upds = (update *) malloc(maxupd * sizeof(update));

  /* Generate center of hostile forces if someone needs it */
  for(i=0;i<batt[myind].size;++i)
	if(batt[myind].units[i].targmode == CENTER) {
	  getcent(myind,numarm,batt);
	  break;
	}

  /* Loop through all units */
  for(i=0;i<batt[myind].size;++i) {
	/* Get pointer ot current unit */
	curunit = (batt[myind].units) + i;
	/* Ignore dead units */
	if(curunit->count == 0)
	  continue;

	/* Limit moral to between -10 and +10 */
	if(curunit->moral < -10)
	  curunit->moral = -10;
	if(curunit->moral > 10)
	  curunit->moral = 10;
	/* Allow sanity to slowly return to units */
	if(random() % 5 == 0) {
	  if(curunit->moral < 0)
		curunit->moral ++;
	  if(curunit->moral > 0)
		curunit->moral --;
	}

	/* Generate casulty update for unit */
	gencasul(myind,i,batt);
	/* If unit needs closest enemy, generate it */
	if((curunit->targmode == NEAREST) || (curunit->attmode == ATTACK))
	  getmin(myind,i,numarm,batt);

	/* Generate attack update if necessary */
	attres = 0;
	if(curunit->attmode == ATTACK)
	  attres = genatt(myind,i,batt);

	/* If unit did not attack, it can move */
	if((! attres) || (curunit->movmode != FORWARD))
	  genmove(myind,i,batt);

	/* Make sure there is enough space on the update list */
	if(numupd + 3 > maxupd) {
	  maxupd += 50;
	  upds = (update *) realloc(upds,maxupd * sizeof(update));
	}

  }

  /* Assign the return value for number of updates */
  *nupd = numupd;
  /* Return the update list */
  return(upds);
}

/* Eat an update produced by another player */
eatupd(upd,arm,myindex)
update *upd;
/* upd - pointer update to be eaten */
army *arm;
/* arm - array of armies */
int myindex;
/* myindex - player index */
{
  armyunit *curunit;
  /* curunit - current unit */
  long hard,damag,oldhits;
  /* hard - "hardness" of unit, initial resistance to damage
	 damag - damage sustained during an attack
	 oldhits - old number of hit points before an attack */
  register int i;
  /* i - loop variable */
  static float mult[] = {0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0,
						 1.0, 1.0, 1.0, 1.3, 1.5, 1.7, 1.9, 2.2, 3.0, 5.0};
  /* mult - random damage multiplier */
  
  /* Get pointer to unit */
  curunit = (arm[upd->index].units) + upd->unit;

  /* Change position if it is a simple position update */
  if(upd->type == POS_UPD) {
	curunit->xpos = upd->new.pos.x;
	curunit->ypos = upd->new.pos.y;
  }

  /* Change count if it is a simple count update */
  else if(upd->type == CNT_UPD)
	curunit->count = upd->new.count;

  /* Calculate damage if it is an attack update */
  else if((upd->type == HIT_UPD) && (upd->index == myindex) && armyflag) {
	/* Calculate total hardness of unit */
	hard = getval(curunit->platf,HARD) + getval(curunit->armor,HARD);
	/* Add bonus hardness if applicable */
	if(upd->new.hit.sptype && (upd->new.hit.sptype == getval(curunit->platf,SPTYPE)))
	  hard += getval(curunit->platf,SPBON);
	if(upd->new.hit.sptype && (upd->new.hit.sptype == getval(curunit->armor,SPTYPE)))
	  hard += getval(curunit->armor,SPBON);
	/* Modify hardness depending on current mode */
	if(curunit->movmode == CHASE) {
	  hard = hard/2;
	  upd->new.hit.numb += upd->new.hit.numb/2;
	}
	if(curunit->movmode == STAND) {
	  hard += hard/2;
	  upd->new.hit.numb -= upd->new.hit.numb/3;
	}
	if(curunit->attmode == DEFEND) {
	  hard += hard/2;
	  upd->new.hit.numb -= upd->new.hit.numb/3;
	}
	/* Remember old number of hit points */
	oldhits = curunit->tothits;
	/* Loop through the number of attacks */
	for(i=0;i<upd->new.hit.numb;++i) {
	  /* Calculate damage for this attack */
	  damag = upd->new.hit.avdam * mult[random() % 20];
	  /* Subtract off the hardness */
	  damag -= hard;
	  /* Limit damage because at most one person can die in a single */
	  /* attack */
	  if(damag < 0)
		damag = 0;
	  if(damag > getval(curunit->platf,HITS))
		damag = getval(curunit->platf,HITS);
	  /* Knock off hit points */
	  curunit->tothits -= damag;
	}

	/* User message */
	if((curunit->tothits != oldhits) && verbflag) {
	  printf("Unit %d took %d points of damage.\n",upd->unit,oldhits - curunit->tothits);
	  /* Decrement moral */
	  curunit->moral --;
	}
	else
	  if(verbflag)
		printf("Unit %d resists an attack.\n",upd->unit);

	/* If the unit lost more than 10% hits points, decrement moral by 2. */
	if(((float) curunit->tothits)/((float) oldhits) <= .9) {
	  curunit->moral -= 2;
	  /* User message */
	  if(verbflag)
		printf("Unit %d was hit hard!\n",upd->unit);
	}
	/* People who are trying to get away and still get hit will only try harder */
	if(curunit->movmode == RETREAT)
	  curunit->moral --;
	if(curunit->movmode == RUNAWAY)
	  curunit->moral -= 3;
  }
}
