/* Written by Alex Siegel at Cornell University 9/88 */
/* This module handles creating and drawing the tactical map. */

#include <stdio.h>
#include <X11/Xlib.h>
#include "config.h"
#include "gdefs.h"
#include <math.h>

float xfact,yfact;
/* xfact,yfact - multiplication factors to linearly scale position into
                 screen position */
long xmin,ymin;
/* xmin,ymin - minimum coordinate values amoung live units */
GC armgc;
/* armgc - army graphics descriptor used by X windows */
Font armfont;
/* armfont - font used for the unit labels */
int winsize;
/* winsize - map window size (width and height) */
Display *disp;
/* disp - X windows display descriptor */
Window wind;
/* wind - current X windows map window */

/* Draw a single unit using the current mapping parameters.  Note that units are
   written with exclusive or so that the same routine for drawing can be used
   for erasure. */
drawunit(curarm,unitnum,playnum,numplay)
army *curarm;
/* curarm - pointer to array of armies in battle */
int unitnum;
/* unitnum - number of unit to be drawn */
int playnum;
/* playnum - player number of unit to be drawn */
int numplay;
/* numplay - total number of players in battle */
{
  armyunit *curunit;
  /* curunit - pointer to current army unit */
  int xscr,yscr;
  /* xscr,yscr - window position of unit to be drawn */
  char buf[30];
  /* buf - character buffer used for generating labels */
  XTextItem titem;
  /* titem - X windows text item description */

  /* Get pointer to current unit */
  curunit = (curarm[playnum].units) + unitnum;

  /* Do not draw dead units */
  if(curunit->count == 0)
	return;

  /* Calculate screen position of unit */
  xscr = (curunit->xpos - xmin) * xfact;
  yscr = (curunit->ypos - ymin) * yfact;
  /* If unit is off the screen, redraw the whole map */
  if((xscr<5) || (yscr<5) || (xscr>winsize-5) || (yscr>winsize-5)) {
	drawmap(curarm,numplay);
	return;
  }
  /* Switch on the player number to determine which pattern to use
	 when drawing the unit */
  switch(playnum) {
  case 0:
	/* This draws an X */
    XDrawLine(disp,wind,armgc,xscr-5,yscr-5,xscr+5,yscr+5);
    XDrawLine(disp,wind,armgc,xscr-5,yscr+5,xscr+5,yscr-5);
    break;
  case 1:
	/* This draws a + */
    XDrawLine(disp,wind,armgc,xscr,yscr-5,xscr,yscr+5);
    XDrawLine(disp,wind,armgc,xscr-5,yscr,xscr+5,yscr);
    break;
  case 2:
	/* This draws a < */
    XDrawLine(disp,wind,armgc,xscr+5,yscr+5,xscr-5,yscr);
    XDrawLine(disp,wind,armgc,xscr+5,yscr-5,xscr-5,yscr);
    break;
  case 3:
	/* This draw a > */
    XDrawLine(disp,wind,armgc,xscr-5,yscr+5,xscr+5,yscr);
    XDrawLine(disp,wind,armgc,xscr-5,yscr-5,xscr+5,yscr);
    break;
  default:
	/* This draws a V */
    XDrawLine(disp,wind,armgc,xscr,yscr+5,xscr-5,yscr-5);
    XDrawLine(disp,wind,armgc,xscr,yscr+5,xscr+5,yscr-5);
    break;
  }

  /* Generate text item for unit number label */
  sprintf(buf,"%d",unitnum);
  titem.chars = buf;
  titem.nchars = strlen(buf);
  titem.delta = 0;
  titem.font = armfont;
  /* Draw unit number label */
  XDrawText(disp,wind,armgc,xscr-5,yscr-5,&titem,1);
  /* Draw unit size */
  sprintf(buf,"%d",curunit->count);
  titem.nchars = strlen(buf);
  XDrawText(disp,wind,armgc,xscr-5,yscr+15,&titem,1);
  /* Draw unit division */
  sprintf(buf,"%d",curunit->division);
  titem.nchars = strlen(buf);
  XDrawText(disp,wind,armgc,xscr+7,yscr+5,&titem,1);
}

/* Redraw the entire tactical map */
drawmap(curarm,numplay)
army *curarm;
/* curarm - array of armies involved in battle */
int numplay;
/* numplay - number of players in battle */
{
  XWindowAttributes wattr;
  /* wattr - window attributes of map window */
  long xmax,ymax;
  /* xmax,ymax - maximum x and y coordinate of live units */
  armyunit *curunit;
  /* curunit - pointer to current army unit */
  register int i;
  /* i - loop variables */
  int xscr,yscr,del,av,pl;
  /* xscr,yscr - current screen position */
  char buf[30];
  /* buf - buffer user for generating labels */
  XTextItem titem;
  /* titem - X windows text item used for generating labels */

  /* Clear the window */
  XClearWindow(disp,wind);

  /* Get the window size */
  XGetWindowAttributes(disp,wind,&wattr);

  /* Calculate minimum and maximum x and y coordinates */
  xmin = ymin = 1000000;
  xmax = ymax = -1000000;
  for(pl=0;pl<numplay;++pl)
    for(i=0;i<curarm[pl].size;++i) {
      curunit = (curarm[pl].units) + i;
	  if(curunit->count == 0)
		continue;
      if(xmin > curunit->xpos) xmin = curunit->xpos;
      if(xmax < curunit->xpos) xmax = curunit->xpos;
      if(ymin > curunit->ypos) ymin = curunit->ypos;
      if(ymax < curunit->ypos) ymax = curunit->ypos;
    }

  /* Do not continue if no units are around */
  if(xmin > xmax) {
	printf("It has gotten very lonely.\n");
	return;
  }
  
  /* Square up the relative aspect of x and y */
  if(xmax-xmin > ymax-ymin) {
    del = (xmax-xmin)/2;
    av = (ymin+ymax)/2;
    ymin = av - del;
    ymax = av + del;
  }
  else {
    del = (ymax-ymin)/2;
    av = (xmin+xmax)/2;
    xmin = av - del;
    xmax = av + del;
  }

  /* Allow 10% padding and a little more around border */
  del = (xmax-xmin)/10 + 3;
  xmin -= del;
  xmax += del;
  del = (ymax-ymin)/10 + 3;
  ymin -= del;
  ymax += del;

  /* Calculate the scaling factor to get from coodinates to screen position */
  xfact = ((float) wattr.width)/(xmax-xmin);
  yfact = ((float) wattr.height)/(ymax-ymin);

  /* Draw the units */
  for(pl=0;pl<numplay;++pl)
	for(i=0;i<curarm[pl].size;++i) 
	  drawunit(curarm,i,pl,numplay);

  /* Draw the reference scales */
  titem.chars = buf;
  titem.delta = 0;
  titem.font = armfont;
  del = (xmax - xmin)/20;
  if(del == 0)
	del = 1;
  for(i=xmin+del;i+del<=xmax;i += del) {
	xscr = (i - xmin) * xfact;
	sprintf(buf,"%d",i);
	titem.nchars = strlen(buf);
	XDrawLine(disp,wind,armgc,xscr,0,xscr,9);
	XDrawLine(disp,wind,armgc,xscr,wattr.height-1,xscr,wattr.height-10);
	XDrawText(disp,wind,armgc,xscr,20,&titem,1);
  }

  del = (ymax - ymin)/20;
  if(del == 0)
	del = 1;
  for(i=ymin+del;i+del<=ymax;i += del) {
	yscr = (i - ymin) * yfact;
	sprintf(buf,"%d",i);
	titem.nchars = strlen(buf);
	XDrawLine(disp,wind,armgc,0,yscr,9,yscr);
	XDrawLine(disp,wind,armgc,wattr.width-1,yscr,wattr.width-10,yscr);
	XDrawText(disp,wind,armgc,10,yscr,&titem,1);
  }

  /* Draw center cross hairs */
  xscr = (-xmin) * xfact;
  yscr = (-ymin) * yfact;
  XDrawLine(disp,wind,armgc,xscr-10,yscr,xscr+10,yscr);
  XDrawLine(disp,wind,armgc,xscr,yscr-10,xscr,yscr+10);
  XFlush(disp);
}

/* Create the window for the tactical map and load the font */
makemap()
{
  XGCValues xgcv;
  /* xgcv - structure used by X windows for creating armgc */

  /* Open up the display */
  disp = XOpenDisplay(NULL);
  /* Check to see if the open display succeeded */
  if(disp == NULL) {
	printf("Display open failed.  Check DISPLAY environment variable.\n");
	exit(-1);
  }

  /* Create the window */
  winsize = DisplayWidth(disp,0)-100;
  if(winsize > DisplayHeight(disp,0)-100)
	winsize = DisplayHeight(disp,0)-100;
  wind = XCreateSimpleWindow(disp,DefaultRootWindow(disp),50,50,winsize,winsize,
							 2,WhitePixel(disp,0),BlackPixel(disp,0));
  /* Check to see if the open window succeeded */
  if(wind == 0) {
	printf("Window open failed.\n");
	XCloseDisplay(disp);
	exit(-1);
  }

  /* Name and raise the window */
  XStoreName(disp,wind,"tactical map");
  XMapRaised(disp,wind);

  /* Create armgc */
  xgcv.function = GXxor;
  xgcv.foreground = BlackPixel(disp,0);
  xgcv.background = WhitePixel(disp,0);
  xgcv.line_width = 1;
  armgc = XCreateGC(disp,wind,
					GCFunction | GCForeground | GCBackground | GCLineWidth,
					&xgcv);
  /* Load the labeling font */
  armfont = XLoadFont(disp,"6x10");
  /* Flush all the calls to the server */
  XSync(disp,False);
}

/* Destroy the tactical map */
killmap()
{
  /* Free up everything and shut down connection to X windows */
  XFreeGC(disp,armgc);
  XUnloadFont(disp,armfont);
  XUnmapWindow(disp,wind);
  XDestroyWindow(disp,wind);
  XCloseDisplay(disp);
}
