/*
 *      ISIS release V1.1, Dec. 1988
 *      Export restrictions apply
 */
/*
 *      ISIS distributed systems toolkit: token tool
 */

#include "isis.h"

typedef struct  token token;

struct  token
{
        address t_gaddr;
        char    t_name[32];
        queue   *t_req;
        int     t_flag;
        int     t_watch;
        address t_holder;
        token   *t_next;
};

#define T_INUSE         0x0001          /* Token is in use here */
#define T_HOLD          0x0002          /* Token is still at this process */
#define T_FAIL          0x0004          /* Current holder failed */
#define T_PASS_ON_FAIL  0x0008          /* Pass if current holder fails */
#define T_SENDING       0x0010          /* Sending this token */

static  adesc t_ad ={ sizeof(token), sizeof(token), 4 };

#define t_alloc()       ((token*)mallocate(&t_ad))
#define t_free(tp)      mdeallocate((char*)tp,&t_ad)

static  token   *t_root;
static  send_token(), t_failed(), token_req(), tk_send(), tk_rcv(), token_pass(), t_newholder();

tk_init()
  {
        register tindex;
        isis_entry(GENERIC_TOKEN_REQ, token_req, "isis-tokens:token_req");
        isis_entry(GENERIC_TOKEN_PASS, token_pass, "isis-tokens:token_pass");
        allow_xfers_xd(NULLARG, XD_TOKEN, tk_send, tk_rcv);
        isis_task(t_failed, "isis-tokens:t_failed");
        isis_task(tk_send, "isis-tokens:tk_send");
        isis_task(tk_rcv, "isis-tokens:tk_rcv");
  }

token *
map_token(gaddr, name)
  address gaddr;
  char *name;
  {
        register token *tp;
        register groupview *gv;
        tp = t_root;
        while(tp)
            if(addr_isequal(gaddr, tp->t_gaddr) && strcmp(name, tp->t_name) == 0)
                return(tp);
            else
                tp = tp->t_next;
        gv = pg_getlocalview(gaddr);
        if(gv == 0)
            return((token*)0);
        tp = t_alloc();
        tp->t_gaddr = gaddr;
        tp->t_req = qu_null();
        tp->t_next = t_root;
        t_root = tp;
        strcpy(tp->t_name, name);
        t_newholder(tp, gv->gv_members[0], -1);
        return(tp);
  }

static
tk_send(locator, gaddr)
  address gaddr;
  {
        register token *tp;
        register n = 0;
        for(tp = t_root; tp; tp = tp->t_next)
            if(addr_isequal(tp->t_gaddr, gaddr))
            {
                register pass_on_fail = 0;
                if(tp->t_flag&T_PASS_ON_FAIL)
                    ++pass_on_fail;
                xfer_out(n++, "%a.%s: %a,%d", tp->t_gaddr, tp->t_name, tp->t_holder, pass_on_fail);
            }
  }

static
tk_rcv(locator, mp)
  {
        register token *tp;
        address gaddr, holder;
        char *name;
        int pass_on_fail;
        msg_get(mp, "%a.%-s: %a,%d", &gaddr, &name, &holder, &pass_on_fail);
        tp = t_root;
        while(tp)
            if(addr_isequal(gaddr, tp->t_gaddr) && strcmp(name, tp->t_name) == 0)
                return;
            else
                tp = tp->t_next;
        tp = t_alloc();
        tp->t_next = t_root;
        t_root = tp;
        tp->t_req = qu_null();
        tp->t_gaddr = gaddr;
        strcpy(tp->t_name, name);
        t_newholder(tp, holder, pass_on_fail);
  }

dump_tokens()
  {
        register token *tp = t_root;
        while(tp)
        {
            register queue *qp;
            print("Token ");
            paddr(tp->t_gaddr);
            print("..%s held by ", tp->t_name);
            paddr(tp->t_holder);
            if(tp->t_flag&T_FAIL) print(" (failed)");
            if(tp->t_flag)
            {
                print(" <");
                if(tp->t_flag&T_INUSE) print(" inuse");
                if(tp->t_flag&T_HOLD) print(" local");
                if(tp->t_flag&T_PASS_ON_FAIL) print(" pass-on-fail");
                if(tp->t_flag&T_SENDING) print(" sending");
                print(" >");
            }
            print("\n");
            if(tp->t_req->qu_next != tp->t_req)
                print("  ... pending requests: ");
            for(qp = tp->t_req->qu_next; qp != tp->t_req; qp = qp->qu_next)
            {
                register message *mp = qp->qu_msg;
                paddr(msg_getsender(mp));
                print("[%d] ", msg_getid(mp));
            }
            print("\n");
            tp = tp->t_next;
        }
  }

t_request(gaddr, name, pass_on_fail)
  address gaddr;
  char   *name;
  {
        register rv;
        register token *tp = map_token(gaddr, name);
        int passed;
        if(tp == (token*)0)
        {
            isis_errno = IE_RESTRICTED;
            return(-1);
        }
        if((tp->t_flag&(T_HOLD|T_INUSE)) == T_HOLD)
        {
            tp->t_flag |= T_INUSE;
            return(1);
        }
        if(cbcast(gaddr, GENERIC_TOKEN_REQ, "%a,%s,%d", gaddr, name, pass_on_fail,
            1, "%d", &passed) != 1)
        {
            isis_errno = IE_RESTRICTED;
            return(-1);
        }
        return(passed);
  }

static
token_req(mp)
  message *mp;
  {
        address gaddr;
        char *name;
        register token *tp;
	static send_token();
        if(msg_get(mp, "%a,%-s", &gaddr, &name) != 2)
            panic("token_req");
        tp = map_token(gaddr, name);
        if(tp == (token*)0)
            panic("token_req");
        msg_increfcount(mp);
        qu_add_mp(tp->t_req, name, mp, NULLROUTINE);
        if((tp->t_flag&(T_INUSE|T_HOLD)) == T_HOLD)
            (void)send_token(tp);
  }

t_pass(gaddr, name)
  address gaddr;
  char   *name;
  {
        register token *tp = map_token(gaddr, name);
        if(tp == (token*)0)
            panic("t_pass");
        if((tp->t_flag&(T_INUSE|T_FAIL)) == 0)
        {
            isis_errno = IE_NOTALLOWED;
            return(-1);
        }
        tp->t_flag &= ~T_INUSE;
        return(send_token(tp));
  }

static
send_token(tp)
  register token *tp;
  {
        int passed = 0;
        register queue *qp;
        /* Loop, trying to pass token until queue empty or success */
        if(tp->t_flag&T_SENDING)
            return(passed);
        tp->t_flag |= T_SENDING;
        if((tp->t_flag&T_HOLD) == 0)
            panic("send_token: not holder");
        tp->t_flag &= ~T_HOLD;
        while(!passed && (qp = qu_head(tp->t_req)))
        {
            register message *mp = qp->qu_msg;
            address who;
            int id, pass_on_fail, rv;
            qu_free(qp);
            who = msg_getsender(mp);
            id = msg_getid(mp);
            if(msg_get(mp, "%d", &pass_on_fail) != 1)
                panic("token_send: msg_get failed, fpointer %d", mp->fpointer);
            rv = cbcast(tp->t_gaddr, GENERIC_TOKEN_PASS,
                    "%a,%s,%a,%d,%d", tp->t_gaddr, tp->t_name, who, id, pass_on_fail,
                1,
                    "%d", &passed);
            reply(mp, "%d", passed);
            msg_delete(mp);
        }
        if(passed == 0)
            tp->t_flag |= T_HOLD;
        tp->t_flag &= ~T_SENDING;
        return(passed);
  }

static
token_pass(mp)
  message *mp;
  {
        address new_holder, gaddr;
        register passed = 1;
        register token *tp;
        int his_msgid, pass_on_fail;
        char *name;
	static t_newholder();
        if(msg_get(mp, "%a,%-s,%a,%d,%d", &gaddr, &name, &new_holder, &his_msgid, &pass_on_fail) != 5)
           panic("token_pass had problems with msg_get");
        tp = map_token(gaddr, name);
        if(tp == (token*)0)
           panic("token_pass: group/token-name unknown");
        if(pg_rank(gaddr, new_holder) != -1)
            t_newholder(tp, new_holder, pass_on_fail);
        else
            passed = 0;
        if(addr_ismine(msg_getsender(mp)))
            reply(mp, "%d", passed);
        else
        {
            register queue *qp;
            for(qp = tp->t_req->qu_next; qp != tp->t_req; qp = qp->qu_next)
            {
                register message *mp = qp->qu_msg;
                if(msg_getid(mp) == his_msgid && addr_isequal(new_holder, msg_getsender(mp)))
                {
                    msg_delete(mp);
                    qu_free(qp);
                    break;
                }
            }
        }
  }

address 
t_holder(gaddr, name)
  address gaddr;
  char   *name;
  {
        register token *tp = map_token(gaddr, name);
        if(tp == (token*)0)
            return(NULLADDRESS);
        return(tp->t_holder);
  }

static ignore() {}

static do_send_token(msg, gaddr, how, tp)
  register token *tp;
  message *msg;
  address gaddr;
  {
        send_token(tp);
        cc_terminate("");
  }

static t_failed(tp)
  register token *tp;
  {
        if(tp->t_flag&T_PASS_ON_FAIL)
            coord_cohort((message*)0, tp->t_gaddr, do_send_token, ignore, tp);
        else
           tp->t_flag |= T_FAIL;
  }

static
t_newholder(tp, who, pass_on_fail)
  register token *tp;
  address who;
  {
        who.entry = 0;
        tp->t_holder = who;
        tp->t_flag = 0;
        if(addr_ismine(who))
            if(pass_on_fail != -1)
                tp->t_flag |= T_HOLD|T_INUSE;
            else
                tp->t_flag |= T_HOLD;
        if(pass_on_fail)
            tp->t_flag |= T_PASS_ON_FAIL;
        if(tp->t_watch)
            pg_watch_cancel(tp->t_watch);
        if(pass_on_fail)
            tp->t_watch = pg_watch(tp->t_gaddr, tp->t_holder, W_FAIL, t_failed, tp);
        else
            tp->t_watch = 0;
  }
