#include <stdio.h>
#include <curses.h>
#include <sys/time.h>
#include "struct.h"

init_world() /* initializes the world */
{
  int x, y, z;

  for( x = 0; x < WORLDSIZE; x++ ) {
    for( y = 0; y < WORLDSIZE; y++ ) {
      for( z = 0; z < WORLDSIZE; z++ ) {
        world[ worldcycle ][ x ][ y ][ z ] = (char) 0;
      }
    }
  }
}

init_screen() /* initializes the screen */
{
  initscr(); /* initializes curses */
  noecho();
  raw();
  clear();
  move( 0, 0 );
  refresh();
}

draw_screen() /* draws the screen */
{
  register int x, y, z, den;

  for( x = 0; x < WORLDSIZE; x++ ) {
    for( y = 0; y < WORLDSIZE; y++ ) {
      den = 0;
      for( z = 0; z < WORLDSIZE; z++ ) {
        if( world[ worldcycle ][ x ][ y ][ z ] == (char) 1 ) {
          den++;
        }
      }
      mvaddch( x, y, density[ den ] );
    }
  }
  refresh();
}

calculate_world()
{
  register int x, y, z;
  register int dx, dy, dz;
  register char surr, result;

  for( x = 1; x < WORLDSIZE - 1; x++ ) {
    for( y = 1; y < WORLDSIZE - 1; y++ ) {
      for( z = 1; z < WORLDSIZE - 1; z++ ) {
        surr = 0;
        for( dx = -1; dx <= 1; dx++ ) {
          for( dy = -1; dy <= 1; dy++ ) {
            for( dz = -1; dz <= 1; dz++ ) {
              if( world[ worldcycle ][ x+dx ][ y+dy ][ z+dz ] == 1 ) surr++;
            }
          }
        }
        result = world[ worldcycle ][ x ][ y ][ z ];
        if( result == 1 ) surr--;
        switch( world[ worldcycle ][ x ][ y ][ z ] ) {
          case 0:
            if( surr >= 6 && surr <= 8 ) {
              result = 1;
            } else {
              result = 0;
            }
          break;
          case 1:
            if( surr >= 4 && surr <= 7 ) {
              result = 1;
            } else {
              result = 0;
            }
          break;
        }
        world[ 1-worldcycle ][ x ][ y ][ z ] = result;
      }
    }
  }
  worldcycle = 1 - worldcycle;
}

int save_world()
{
  int x, y, z;
  char fname[ 256 ];
  FILE *fp;

  clear();
  refresh();
  noraw();
  echo();
  printf( "What is the name of the world file to be saved? " );
  scanf( "%s", fname );
  if( ( fp = fopen( fname, "w" ) ) == 0 ) {
    printf( "Error opening world file %s for writing.\n", fname );
    raw();
    noecho();
    return( -1 );
  }
  for( x = 0; x < WORLDSIZE; x++ ) {
    for( y = 0; y < WORLDSIZE; y++ ) {
      for( z = 0; z < WORLDSIZE; z++ ) {
        fwrite( &world[ worldcycle ][ x ][ y ][ z ], 1, 1, fp );
      }
    }
  }
  fclose( fp );
  raw();
  noecho();
  return( 0 );
}

int load_world()
{
  int x, y, z;
  char fname[ 256 ];
  FILE *fp;

  clear();
  refresh();
  noraw();
  echo();
  printf( "What is the name of the world file to be loaded? " );
  scanf( "%s", fname );
  if( ( fp = fopen( fname, "r" ) ) == 0 ) {
    printf( "Error opening world file %s for reading.\n", fname );
    raw();
    noecho();
    return( -1 );
  }
  for( x = 0; x < WORLDSIZE; x++ ) {
    for( y = 0; y < WORLDSIZE; y++ ) {
      for( z = 0; z < WORLDSIZE; z++ ) {
        fread( &world[ worldcycle ][ x ][ y ][ z ], 1, 1, fp );
      }
    }
  }
  fclose( fp );
  raw();
  noecho();
  return( 0 );
}

simu_loop() /* actual mainloop */
{
  char c, abuf[ 128 ];
  register int x, y, z, dx, dy, dz;
  int observe;
  struct timeval wait;
  int read_set;

  observe = 1;
  worldcycle = 0;
  progstate = EDIT;
  x = WORLDSIZE / 2;
  y = WORLDSIZE / 2;
  z = WORLDSIZE / 2;
  while( progstate != END ) {
    while( progstate == EDIT || progstate == RUN ) {
      draw_screen();
      if( observe == 1 ) {
	for( dz = -2; dz <= 2; dz++ ) {
	  if( z+dz >= 0 && z+dz <= WORLDSIZE - 1 ) {
	    for( dy = -4; dy <= 4; dy++ ) {
	      if( y+dy >= 0 && y+dy <= WORLDSIZE - 1 ) {
		for( dx = -4; dx <= 4; dx++ ) {
		  if( x+dx >= 0 && x+dx <= WORLDSIZE - 1 ) {
		    mvaddch( 1 + dy + 10 * ( dz + 2 ), WORLDSIZE + 1 + dx,
			    density[10*world[worldcycle][x+dx][y+dy][z+dz]] );
		  }
		}
	      }
	    }
	  }
	}
	sprintf( abuf, "[%d][%d][%d]       ", x, y, z );
	mvaddstr( 0, WORLDSIZE, abuf );
	move( x, y );
	refresh();
      }
      wait.tv_sec = 0;
      wait.tv_usec = WAITUSECS;
      read_set = 1;
      if( select( 32, &read_set, 0, 0, &wait ) > 0 ) {
        if( read( 0, &c, 1 ) > 0 ) {
          switch( c ) {
	  case '4':
	  case 'j':
	    y--;
	    if( x < 1 ) {
	      x = ( WRAPAROUND == 1 ) ? WORLDSIZE - 2 : 1;
	    }
            break;
	  case '6':
	  case 'l':
	    y++;
	    if( x >= WORLDSIZE - 1 ) {
	      x = ( WRAPAROUND == 1 ) ? 1 : WORLDSIZE - 2;
	    }
            break;
	  case '8':
	  case 'i':
	    x--;
	    if( y < 1 ) {
	      y = ( WRAPAROUND == 1 ) ? WORLDSIZE - 2 : 1;
	    }
            break;
	  case '2':
	  case 'm':
	    x++;
	    if( y >= WORLDSIZE - 1 ) {
	      y = ( WRAPAROUND == 1 ) ? 1 : WORLDSIZE - 2;
	    }
            break;
          case '1':
	  case 'a':
	    z--;
	    if( z < 1 ) {
	      z = ( WRAPAROUND == 1 ) ? WORLDSIZE - 2 : 1;
	    }
            break;
	  case '9':
	  case 'd':
	    z++;
	    if( z >= WORLDSIZE - 1 ) {
	      z = ( WRAPAROUND == 1 ) ? 1 : WORLDSIZE - 2;
	    }
            break;
	  case ' ':
	    world[ worldcycle ][ x ][ y ][ z ] = 1 - world[ worldcycle ][ x ][ y ][ z ];
            break;
          case '.':
	  case '+':
	  case 'x':
              world[ worldcycle ][ x ][ y ][ z ] = (char) 1;
            break;
          case '0':
	  case '-':
	  case 'c':
	    world[ worldcycle ][ x ][ y ][ z ] = (char) 0;
            break;
	  case 'R':
	  case 'r':
	    progstate =  RUN;
            break;
	  case 'E':
	  case 'e':
	    progstate = EDIT;
	    break;
	  case 's':
	  case 'S':
	    save_world();
            break;
	  case 'L':
	  case 'F':
	  case 'f':
	    load_world();
            break;
	  case 'O':
	  case 'o':
	    observe = 1 - observe;
	    break;
	  case 'q':
	  case 'Q':
	    progstate = END;
            break;
          }
        }
      }
      if( progstate == RUN ) calculate_world();
    }
  }
}

end_screen()
{
  refresh();
  noraw();
  echo();
  endwin();
  puts( " " );
}

main()
{
  init_world();
  init_screen();
  simu_loop();
  end_screen();
}
