/* 
 * ms_process.c
 *
 * x-kernel v3.1	12/10/90
 *
 * Copyright (C) 1990  Larry L. Peterson and Norman C. Hutchinson
 */

#include "process.h"
#include "memory.h"
#include "assert.h"

/* Basic Process Implementation */
#ifndef NDEBUG
int nSwitches;
int GotHit;
#endif

/*********************************************************
* Machine specific (Ms_) routines; called from process.c
*********************************************************/

Ms_AllocatePds()
{
  unsigned pdMemory;
  
  /* Allocate 16k per megabyte of physical memory, max 64k */
  pdMemory = min( ( (4*1024*1024) >> 6 ), 0x10000 );
  MaxProcesses = min(pdMemory/sizeof( Process ), 
    (KERNEL_LIMIT - KERNEL_MALLOC_LIMIT) / PROCESSMAXSTACK);
  pdMemory = MaxProcesses * sizeof(Process);
  ProcessDescriptors = (Process *) malloc(pdMemory);
  TRACE2(memoryinit, 1, "  Allocating %d process descriptors (%d bytes).", 
    MaxProcesses, pdMemory);
  EndPds = ProcessDescriptors + MaxProcesses;
}

printxx()
{
  extern char *StackBottom;
  long *p = (long *)StackBottom;
  int i;
  int x = spl7();

  for (i = 0; i < 12; i++) {
    printf("%x ", p[-i]);
  }
  printf("\n");
  splx(x);
}

Ms_DestroyProcess(pd)
register Process *pd;
{
  register VirtualAddress a;

  TRACE2(processcreation, 1, "Destroying process stack from %x to %x",
    pd->ps.stackLimit.u, pd->ps.stackBase.u);
  for (a.u = pd->ps.stackLimit.u; a.u < pd->ps.stackBase.u; a.u += PAGE_SIZE) {
    if (GetPageMap(a.u) != INVALID_PTE) {
      TRACE1(processcreation, 1, "Freeing page at %x", a.u);
      Ms_FreePage(a.u);
      TRACE1(processcreation, 1, "Freed page at %x", a.u);
    }
  }
  TRACE0(processcreation, 1, "Done Ms_DestroyProcess");
}

Ms_CreateProcess( aspace, pd, entry, kernelP, nargs, argv) 
register Aspace *aspace;
register Process *pd;
int (*entry)();
int kernelP;
int nargs, *argv;
/*
 * Perform the machine-specific portion of process creation,
 */
{
  register Ms_ProcessorState *state;
  register MC68020ExceptionStackFrame *frame;
  register VirtualAddress ustack, kstack;
  int r;
  state = &(pd->ps);

#ifdef DEBUGRQ
  pd->queuePtr = (SyncQueue *)0;
#endif
  kstack.u = state->stackBase.u;
  if (kstack.u == 0) {
    kstack.u = PDTOKSTACK(pd);
    assert(kstack.u >= KERNEL_MALLOC_LIMIT && kstack.u <= KERNEL_LIMIT); 
    if ((r = AllocatePage(kstack.u - PAGE_SIZE, 1))) {
      printf("Ms_CreateProcess, AllocatePage returns %x\n", r);
      Kabort("Can't create kernel stack page");
    }
    IFTRACE(processcreation, 1) {
      PageMapEntry p;
      p.u = GetPageMap(kstack.u-PAGE_SIZE);
      printf("Allocated kernel stack page at VA %x, entry %x\n",
	kstack.u-PAGE_SIZE, p.u);
    }
    ustack.u = KTOUSTACK(kstack.u);
    if (!kernelP) {
      if ((r = AllocatePage(ustack.u - PAGE_SIZE, 0))) {
	printf("Ms_CreateProcess, AllocatePage returns %x\n", r);
	Kabort("Can't create user stack page");
      }
      IFTRACE(processcreation, 1) {
	PageMapEntry p;
	p.u = GetPageMap(ustack.u-PAGE_SIZE);
	printf("Allocated user stack page at VA %x, entry %x\n",
	  ustack.u-PAGE_SIZE, p.u);
      }
    }
    state->stackBase.u = state->stackLimit.u = kstack.u;
    state->stackLimit.u -= PROCESSMAXSTACK;
  } else {
    ustack.u = KTOUSTACK(kstack.u);
  }

  /*
   * This puts extra words on the stack.  With optimization (gcc), the trace
   * calls in DestroyProcess can read past the end of the stack at times.
   */ 
  for (r = 0; r < 5; r++) {
    *(--((int *) kstack.u)) = 0;
  }
  if (kernelP) {
    for (r = 1; r <= nargs; r++) {
      *(--((int *) kstack.u)) = argv[nargs - r];
    }
    *(--((Unspec *) kstack.u)) = (Unspec) DestroyProcess;
  } else {
    for (r = 1; r <= nargs; r++) {
      *(--((int *) ustack.u)) = argv[nargs - r];
    }
    *(--((Unspec *) ustack.u)) = (Unspec) DestroyProcess;
  }
    
  frame = (MC68020ExceptionStackFrame *) 
    (kstack.u - 8 /*MC68020ExceptionStackFrame_frame*/);
  frame->sr = kernelP ? 0x2000 : 0x0000;
  frame->pc = (unsigned long) entry;
  frame->format.format = 0;
  frame->format.vector = TRAP0;
  state->regs.r.usp = (Unspec) ustack.u;
  state->regs.r.a7 = (Unspec) frame;
  state->regs.r.a6 = 0;  /* Set initial frame pointer to 0 */
}

Process *readyQueues[32];
unsigned int readyBits;

#ifdef DEBUGRQ
Ms_CheckOutRQ(pd)
Process *pd;
{
  register Process *head, *tail;
  register int i;
  for (i = 0; i < 32; i++) {
    head = readyQueues[i];
    if (head) {
      tail = head;
      do {
	assert (tail != pd);
	if (tail != head) assert (tail->link != tail);
	tail = tail->link;
      } while (tail != head);
    }
  }
}
#endif

Ms_Kabort( str ) register char *str;

/* Kernel abort: print message on console and stop with
 * only refresh interrupt enabled.
 */
{
    printf("Kernel abort: "); 
    printf("%s\r\n",str); 
    StackDump((unsigned *)0);
    CheckOutMemory();
    breakpoint();
  }

Ms_RemoveReady(pd)
Process *pd;
{
  int x = Ms_Spl7();
  register Process *last, *follow, *run;
  register int prio = pd->priority;
  last = readyQueues[prio];
  assert((int)last);
  follow = last;
  run = follow->link;
  do {
    if (run == pd) break;
    follow = run;
    run = run->link;
  } while (run != last);
  assert(run == pd);
  assert(follow->link == pd);
  if (pd->link == pd) {			/* only process */
    assert(readyQueues[prio] == pd);
    readyQueues[prio] = 0;
    assert(readyBits & (1 << prio));	/* dependent -llp */
    readyBits &= ~(1 << prio);		/* dependent -llp */
  } else {
    follow->link = pd->link;
    if (readyQueues[prio] == pd) {	/* last process */
      readyQueues[prio] = follow;
    } else {				/* any other process */
      /* do nothing */
    }
  }
  pd->queuePtr = 0;
  pd->link = 0;
  Ms_Splx(x);
}

/*********************************************************
* Internal routines
*********************************************************/

CheckOutStack(pd, sp)
register Process *pd;
unsigned sp;
{
  assert(sp <= pd->ps.stackBase.u);
  assert(pd->ps.stackLimit.u <= sp); 
}

DisplayProcess(pd, verbose)
register Process *pd;
int verbose;
{
  register MC68020ExceptionStackFrame *frame;
  frame = (MC68020ExceptionStackFrame *) pd->ps.regs.r.a7;

  printf("process %x (%x) prio %d @ %x sp = %x\n", pd->pid, pd, pd->priority,
    frame ? frame->pc : 0xffffffff, pd->ps.regs.r.a7);
  if (verbose) {
    printf("  a7 = %x usp = %x\n", pd->ps.regs.r.a7, pd->ps.regs.r.usp);
    if (frame != 0) {
      frame->format.filler = 0;
      printf("  frame sr = %x pc = %x format = %x\n", frame->sr, frame->pc, 
	*(short *) &frame->format);
    }
  }
}









