/* 
 * machine.c
 *
 * x-kernel v3.1	12/10/90
 *
 * Copyright 1990  Larry L. Peterson and Norman C. Hutchinson
 */

#include "debug.h"
#include <sys/types.h>
#include <sys/file.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <syslog.h>
#include <errno.h>
#include <signal.h>
extern char *sprintf();

int tracesystem=0;
extern char *rom[];
extern int SignalsPossible;
typedef void(*PFV) ();
typedef int(*PFI) ();

struct	int_vector {
  PFI	handler;
/* device is now given by the index
  int	device;
*/
};

int inInterrupt;
int	dispatch();

/* don't know how many sockets we should allow for */
/* update this in [udp,tcp].c too */
#define NUMSOCKETSICANUSE 30
struct	int_vector  ivec[NUMSOCKETSICANUSE+1];
int                 ivec_in_use_mask;
extern	int	errno;

cancelSignalHandler(sock)
int sock;
{
  ivec_in_use_mask &= ~(1 << sock);
  TRACE2(system, 3, "Cancelling handler for %d mask = %x", sock, ivec_in_use_mask);
}

installSignalHandler(sock, f)
int sock;
PFI f;
{
  ivec[sock].handler = f;
  ivec_in_use_mask |= (1 << sock);
  SignalsPossible = 1;
  TRACE2(system, 3, "Setting handler for %d, mask = %x", sock, ivec_in_use_mask);
}

static unsigned int heldmask = 0;

xholdsignals(sock)
{
  TRACE1(system, 5, "Holding file descriptor %d", sock);
  heldmask |= (1 << sock);
}

xreleasesignals(sock)
int sock;
{
  TRACE1(system, 5, "Releasing file descriptor %d", sock);
  heldmask &= ~(1 << sock);
}

/*
 *
 *  Interrupt Handler
 *
 *  The handler is now invoked with a single integer argument
 *  identifying the socket to service, so that more than one
 *  socket may use the same handler.
 */
findsocket()
{
  int	mask, i, n;
  static struct timeval zero = {0, 0};
  for (i = 0; i < 32; i++) {
    mask = (1 << i);
    n = select(32, (fd_set *)&mask, (fd_set *)0, (fd_set *)0, &zero);
    if (n == 1) {
      return (i);
    }
  }
  return(33);
}

fixFileMask()
{
  int	mask, i;
  static struct timeval zero = {0, 0};
  for (i = 0; i < 32; i++) {
    if (ivec_in_use_mask & (1 << i)) {
      mask = (1 << i);
      if (select(32, (fd_set *)&mask, (fd_set *)0, (fd_set *)0, &zero) < 0 && errno == EBADF) {
	TRACE1(system, 3, "Turning off socket #%d", i);
	ivec_in_use_mask &= ~(1 << i);
      }
    }
  }
}

/*ARGSUSED*/
dispatch(interruptNo)
int interruptNo;
{
  register int i;
  int	mask, n, foundany=0;
  static struct timeval zero;

  inInterrupt++;
  while (1) {
    mask = ivec_in_use_mask & ~heldmask;
    if((n = select(NUMSOCKETSICANUSE, (fd_set *)&mask, (fd_set *)0, (fd_set *)0, &zero)) == -1) {
      if (errno == EBADF) {
	TRACE0(system, 3, "EBADF on select in dispatch");
	fixFileMask();
	continue;
      } else {
	perror("select");
	inInterrupt--;
	return;
      }
    } else if(!n) {
      TRACE0(system, 3, "dispatch: can't find any input");
      break;
    } else if(!mask) {
      printf("select returns a zeroed mask! ivecmask %#x, n is %d\n",
	     ivec_in_use_mask, n);
      break;
    } else {
      for(i=0;i<NUMSOCKETSICANUSE;i++){
	if(mask & (1 << i)) {
	  if (! ivec[i].handler) {
	    printf("Null handler for sock %d\n", i);
	  } else {
	    TRACE1(system, 3, "Calling handler for sock %d", i);
	    (*ivec[i].handler)(i);
	    foundany ++;
	  }
	}
      }
    }
  }
  if(!foundany){
    TRACE4(system, 3,
      "unknown interrupt %d (mask %#x ivec %#x real %d) in dispatch\n", 
      interruptNo, mask, ivec_in_use_mask, findsocket());
  }
  inInterrupt--;
}

/********************************************
*
*  Clock Device
*
*********************************************/

#define SIGSTACKSIZE (4 * 1024)
static int signalstack[SIGSTACKSIZE];
struct itimerval i_value, i_zero;
#ifdef vax
#define handlerresulttype int
#endif
#ifdef sun
#define handlerresulttype void
#endif
typedef handlerresulttype (*handlertype)();

onfault(h)
handlertype h;
{
  struct sigvec vec;

  vec.sv_mask         = 0;
  vec.sv_onstack      = 1;
  vec.sv_handler = h;
  sigvec(SIGINT, &vec, (struct sigvec *)0);
  sigvec(SIGSEGV, &vec, (struct sigvec *)0);
  sigvec(SIGBUS, &vec, (struct sigvec *)0);
}
  
init_clock(ih,interval)
PFV	ih;
long	interval;
{
  extern int exit();
  struct sigstack theStackInfo;

  theStackInfo.ss_sp = (caddr_t) &signalstack[SIGSTACKSIZE-2];
  theStackInfo.ss_onstack = 0;
  if (sigstack(&theStackInfo, (struct sigstack *)NULL) < 0) {
    fflush(stdout);  fflush(stderr);
    perror("sigstack");
    abort();
  }

  errno = 0;

  definehandler(SIGPIPE, SIG_IGN);
  definehandler(SIGALRM, (handlertype) ih);
  definehandler(SIGINT, (handlertype) exit);
  definehandler(SIGIO,  (handlertype) dispatch);
  definehandler(SIGURG,  (handlertype) dispatch);

  i_value.it_interval.tv_sec = interval / 1000;
  i_value.it_interval.tv_usec = (interval % 1000)*1000;
  i_value.it_value.tv_sec = i_value.it_interval.tv_sec;
  i_value.it_value.tv_usec = i_value.it_interval.tv_usec;

  if (errno)
	perror("init_clock:");
  return;
}

read_clock(msec)	/* returns the number of msec */
long	*msec;     	/* since sometime in late 1986 */
{
  struct  timeval   time;
  struct  timezone  zone;

  gettimeofday(&time, &zone);
  *msec = (time.tv_sec - 500000000)*1000 + (time.tv_usec / 1000);
  return(0);
}

static handlertype	handlers[NSIG];
static int 		pendings[NSIG];
static int held, pending, limit;

static enable()
{
  /* This is SUPPOSED to be `=', not `==' !!!! */
  TRACE0(system, 5, "Enabling...");
  while (held = pending) {
    int i;
    pending = 0;
    for (i = 0; i <= limit; i++) {
      if (pendings[i]) {
	pendings[i] = 0;
	handlers[i](i);
      }
    }
  }
  TRACE0(system, 5, "Done Enable");
}

static handlerresulttype lhandler(sig)
int sig;
{
  TRACE1(system, 5, "lhandler on sig %d", sig);
  if (held) {
    ++pendings[sig];
    ++pending;
  } else {
    ++held;
    handlers[sig](sig);
    enable();
  }
}

definehandler(sig, handler)
int sig;
handlertype handler;
{
  struct sigvec lvec;

  lvec.sv_mask = -1;
  lvec.sv_onstack = 1;
  if (handler == SIG_IGN) {
    lvec.sv_handler = SIG_IGN;
  } else {
    handlers[sig] = handler;
    lvec.sv_handler = lhandler;
    if (sig > limit) limit = sig;
  }
  sigvec(sig, &lvec, (struct sigvec *)NULL);
}

int spl7()
{
  int res = held;
  held = 1;
  return res;
}

void splx(x)
int x;
{
  if (!x) enable();
}

wait_interrupt()
{
  TRACE0(system, 5, "Waiting for an interrupt");
  sigblock((1<<(SIGIO-1))|(1<<(SIGURG-1))|(1<<(SIGALRM-1)));
  if (pending) {
    sigsetmask(0);
    enable();
  } else {
    sigpause(0);
  }
  TRACE0(system, 5, "Got the interrupt");
}
