/* 
 * upi.c
 *
 * x-kernel v3.1	12/10/90
 *
 * Copyright (C) 1990  Larry L. Peterson and Norman C. Hutchinson
 */

#include <varargs.h>
#include "upi.h"
#include "system.h"
#include "process.h"
#ifndef XSIMUL
#include "memory.h"
#endif

int *checkaddress;

#if 0
#define CHECKADDRESS if (checkaddress) printf(" --> %x\n", *checkaddress);
#else
#define CHECKADDRESS 
#endif

int x_errno;
#ifndef NDEBUG
static char *controlops[] = {
  "getmyaddr",
  "getpeeraddr",
  "getalladdrs",
  "getmaxpacket",
  "getoptpacket",
  "getproto",
  "setproto",
  "setdebug",
  "resolve",
  "rresolve",
  "freeresources"
};
#  define CONTROLMSG(n) ((n) < sizeof controlops / sizeof(char *) ? \
	controlops[n] : "unknown")
#else
#  define CONTROLMSG(n) ""
#endif

/*************************************************
* Uniform Protocol Interface Operations
/*************************************************/

x_init(p)
XObj p;
{
  assert(p->type == Protocol);
  (*p->init)(p);
}


XObj x_open(hlp, llp, participants)
XObj	hlp;
XObj	llp;
Part	*participants;
{
  XObj s;
  assert(llp->type == Protocol);
  assert(hlp->type == Protocol);
  if (!validopen(hlp, llp))  {
    x_errno = INVALID_OPEN;
    TRACE2(protocol, 1, "Invalid open of %s by %s", llp->name, hlp->name);
    return(ERR_XOBJ);
  }
  TRACE2(protocol, 1, "Calling open[%s] by %s", llp->name, hlp->name);
  CHECKADDRESS;
  s = (XObj)(*(llp->open))(llp,hlp, participants);
  TRACE3(protocol, 1, "Open[%s] by %s returns %x", llp->name, hlp->name, s);
  CHECKADDRESS;
  return(s);
}

x_openenable(hlp, llp, participants)
XObj	hlp;
XObj	llp;
Part	*participants;
{
  int ans;
  assert(llp->type == Protocol);
  assert(hlp->type == Protocol);
  TRACE2(protocol, 3, "Calling openenable[%s] by %s", llp->name, hlp->name);
  CHECKADDRESS;
  ans = (*(llp->openenable))(llp,hlp, participants);
  CHECKADDRESS;
  return ans;
}

void x_callopendone(hlp, s, participants)
XObj hlp;
XObj s;
Part *participants;
{
  register Pfs od;
#ifndef XSIMUL
  extern callUserOpenDone();
#endif

  assert(s->type == Session);
  assert(hlp->type == Protocol);
  
  if (!(od = (Pfs) hlp->opendone)) return;
  if (hlp->kernel) {
    TRACE2(protocol, 1, "Calling kernel opendone[%s] by %s", hlp->name, s->myprotl->name);
    (*od)(hlp, s, participants);
  } else {
    TRACE2(protocol, 1, "Calling user opendone[%s] by %s", hlp->name, s->myprotl->name);
  CHECKADDRESS;
#ifndef XSIMUL
    callUserOpenDone(hlp->as, od, s, participants);
#else
    (*od)(s, participants);
#endif
    TRACE2(protocol, 1, "Opendone[%s] by %s returns", hlp->name, s->myprotl->name);
    CHECKADDRESS;
  }
}

XObj x_opendone(hlp, llp, participants)
XObj	hlp;			  
XObj	llp;
Part	*participants;
{
  XObj	s;

  assert(llp->type == Protocol);
  assert(hlp->type == Protocol);
  if (!validopen(hlp, llp))  {
    x_errno = INVALID_OPEN;		
    return(ERR_XOBJ);
  }
  if ((s=(XObj)(*(llp->open))(llp,hlp, participants)) == ERR_XOBJ)
    return(s);
  x_callopendone(hlp, s, participants);
  return(s);
}

x_callclosedone(s)
XObj s;
{
  register Pfi cd;
#ifndef XSIMUL
  extern callUserCloseDone();
#endif
  assert(s->type == Session);
  if (cd = s->up->closedone) {
    if (s->up->kernel) {
      TRACE2(protocol, 1, "Calling kernel closedone[%s] by %s", s->up->name,
	s->myprotl->name);
      (*cd)(s);
    } else {
      TRACE2(protocol, 1, "Calling user closedone[%s] by %s", s->up->name, s->myprotl->name);
#ifndef XSIMUL
      callUserCloseDone(s->up->as, cd, s);
#else
      (*cd)(s);
#endif
    }
  }
}

x_closedone(s)
XObj s;
{
  x_callclosedone(s);
  (*s->close)(s);
}

x_opendisable(hlp, llp, participants)
XObj	hlp;
XObj	llp;
Part	*participants;
{
  assert(llp->type == Protocol);
  assert(hlp->type == Protocol);
  TRACE2(protocol, 1, "Calling opendisable[%s] by %s", llp->name, hlp->name);
  return((*(llp->opendisable))(llp,hlp, participants));
}

x_close(s)
XObj s;
{
  TRACE0(protocol, 3, "x_close: entered");
  assert(x_is_session(s));
  TRACE2(protocol, 3, "Calling close[%s] by %s", s->myprotl->name, s->up->name);
  return((*s->close)(s));
}

#if defined(XSIMUL) || !defined(NDEBUG)

x_demux(s, msg)
XObj 	s;
Msg	msg;
{
#ifndef XSIMUL
  extern callUserDemux();
#endif
  register Pfi demux;

  assert(x_is_session(s));
  TRACE3(protocol, 3, "Calling demux[%s] by %s, %d bytes", s->up->name, s->myprotl->name,
    msg_len(msg));
  IFTRACE(protocol, 7) {
    printf("       Message:\n");
    msg_display(msg, 8);
  }
  if (!(demux = s->up->demux)) return(NULL);
  if (s->up->kernel) {
    return((*demux)(s->up,s, msg));
  } else {
#ifndef XSIMUL
    callUserDemux(s->up->as, demux, s, msg);
    return(NULL);
#else
    int res, len;
    char *buffer;
    len = msg_len(msg);
    buffer = malloc((unsigned)len);

    msg_externalize(msg, buffer);
    res = (*demux)(s, buffer, len);
    free(buffer);
    return(res);
#endif
  }
}

x_push(s, msg, nmsg)
XObj s;
Msg	msg, *nmsg;
{
  assert(x_is_session(s));
  TRACE3(protocol, 3, "Calling push[%s] by %s, %d bytes", s->myprotl->name, s->up->name,
    msg_len(msg));
  IFTRACE(protocol, 7) {
    printf("       Message:\n");
    msg_display(msg, 8);
  }
  return((*s->push)(s, msg, nmsg));
}

x_pop(s, ds, msg)
XObj s, ds;
Msg	msg;
{
  assert(!(int)ds || x_is_session(ds));
  TRACE2(protocol, 3, "Calling pop[%s], %d bytes", s->myprotl->name,
    msg_len(msg));
  return((*s->pop)(s, ds, msg));
}
#endif XSIMUL || !NDEBUG

x_control(s, opcode, buf, len)
XObj	s;
int	opcode;
char	*buf;
int	len;
{
  register Pfi c;
#ifndef XSIMUL
  extern callUserControl();
#endif
  int res;
  if (c = s->control) {
    if (s->kernel) {
      TRACE3(protocol, 1, "Calling kernel control[%s] op %s (%d)",
	s->myprotl->name, CONTROLMSG(opcode), opcode);
      CHECKADDRESS;
      res = c(s, opcode, buf, len);
    } else {
      TRACE3(protocol, 1, "Calling user control[%s] op %s (%d)",
	s->myprotl->name, CONTROLMSG(opcode), opcode);
      CHECKADDRESS;
#ifndef XSIMUL
      res = callUserControl(s->as, c, s, opcode, buf, len);
#else
      res = c(s, opcode, buf, len);
#endif
    }
  } else {
    res = 0;
  }
  TRACE4(protocol, 1, "Control[%s] op %s (%d) returns %d",
	s->myprotl->name, CONTROLMSG(opcode), opcode, res);
  CHECKADDRESS;
  return res;
}

x_getproc(s, p, type)
XObj	s;
XObj	p;
XObjType type;
{
  TRACE2(protocol, 3, "Calling getproc old %s new %s",s->name,p->name);
  if (s->getproc) {
    (*s->getproc)(p,type);
    TRACE0(protocol, 3, "Done getproc");
    return(0);
  } else {
    return(-1);
  }
}

x_instantiateprotl(p)
XObj p;
{
  assert(x_is_protocol(p));
  TRACE1(protocol, 3, "Calling instantiateprotl protl %s",p->name);
  if (p->instantiateprotl) {
    return ((*p->instantiateprotl)(p));
  }
  return(0);
}


x_closeprotl(p)
XObj p;
{
  assert(x_is_protocol(p));
  TRACE1(protocol, 3, "Calling closeprotl protl %s",p->name);
  if (p->close) {
    return ((*p->close)(p));
  }
  return(0);
}


/*************************************************
/* Create and Destroy XObj's Sessions and Protocols 
/*************************************************/

/*
 * x_createsession(hlp, llp, numdown, downs)
 */

XObj x_createsession(va_alist)
va_dcl
{
  va_list ap;
  XObj	hlp;
  XObj	llp;
  int numdown;
  XObj s;
  int i;

  va_start(ap);
  hlp = va_arg(ap, XObj);
  llp = va_arg(ap, XObj);
  numdown = va_arg(ap, int);
  s = x_createxobj(numdown);
  s->name = llp->name;
  s->type = Session;
  s->up = hlp;
  s->index = llp->index;
  s->as = llp->as;
  s->myprotl = llp;
  s->protocolid = llp->protocolid;
  s->instance = llp->instance;
  s->kernel = llp->kernel;
  s->device = llp->device;
  llp->rcnt++;
  x_getproc(llp,s,Session);
  for (i = 0; i < numdown; i++) {
    s->down[i] = va_arg(ap, XObj);
  }
  return s;
}

XObj x_createprotocol(numdown,name,id,instance,kernel,device,getproc)
int numdown;
char *name;
int id;
int instance;
int kernel;
int device;
Pfi getproc;
{
  XObj s;

  s = x_createxobj(numdown);
  s->type = Protocol;
  s->name = name;
  s->myprotl = s;
  s->index = (unsigned int)s;
  s->protocolid = id;
  s->instance = instance;
  s->kernel = kernel;
  s->device = device;
  if (getproc) (*getproc)(s,Protocol);
  return(s);
}


XObj x_createxobj(i)
int i;
{
  XObj s;

  if (i > STD_DOWN) {
    s = (struct xobj *) malloc((unsigned)(sizeof(struct xobj)+((i-STD_DOWN)*sizeof(XObj))));
  } else {
    s = (struct xobj *) malloc(sizeof(struct xobj));
  }
  bzero((char *)s,sizeof(struct xobj));
  s->state= ST_INITIALIZING;
  s->mutex = (int *)malloc(sizeof(Semaphore));
  InitSemaphore((Semaphore *)s->mutex, 1);
  s->rcnt = 1;
  return((XObj)s);
}

user_getproc(p,type)
XObj p;
XObjType type;
{
  if (type == Protocol) {
    p->instantiateprotl = noop;
    p->init = noop;
    p->close = noop;
    p->push = noop;
    p->pop = noop;
  } else {
    p->push = noop;
    p->pop = noop;
    p->instantiateprotl = noop;
    p->init = noop;
    p->close = noop;
  }
  p->open = noop;
  p->openenable = noop;
  p->opendone = noop;
  p->closedone = noop;
  p->opendisable = noop;
  p->demux = noop;
  p->getproc = noop; 
}

XObj x_createuserprotl(demux, opendone, closedone, control)
Pfi demux, closedone, control;
Pfs opendone;
{
  XObj p;
  TRACE0(protocol, 3, "Calling createprotl");  

  p = x_createprotocol(0,"user",0,0,0,0,user_getproc);
  p->demux = demux;
  p->opendone = (Pfi) opendone;
  p->closedone = closedone;
  p->control = control;
#ifndef XSIMUL
  p->as =  (struct _Aspace *)GetAddressableAspace();
#endif
  TRACE1(protocol, 3, "createprotl returns %x", p);
  return(p);
}


x_destroyxobj(s)
XObj	s;
{
  if (s->state > ST_BROKEN)
    free(s->state);
  VAll((Semaphore *)s->mutex);
  free((char *)s->mutex);
  free((char *)s);
}

x_destroysession(s)
XObj s;
{
  assert(s->type == Session);
  if ((--(s->myprotl->rcnt)) < 1) { 
    x_closeprotl(s->myprotl);
  } 
  x_destroyxobj(s);
  
}

x_destroyprotl(s)
XObj s;
{
  assert(s->type == Protocol);
  x_destroyxobj(s);
}

/*************************************************
/* Miscellaneous Routines
/*************************************************/

noop()
{
}

/*ARGSUSED*/
validopen(hlp, llp)    /* perhaps validate based on dependency graph */
XObj	hlp, llp;
{
  return(1);
}


void InitProtocols(deviceP)
int deviceP;
{
  static int nextp;
  if (deviceP) {
    TRACE0(protocol, 3, "initProtocols (device)");
    build_pgraph();
    for (nextp=0; protl_tab[nextp] && protl_tab[nextp]->device; nextp++)  {
      TRACE2(protocol, 1, "initializing protocol %d (%s)", nextp,
	protl_tab[nextp]->name);
      (*protl_tab[nextp]->init)(protl_tab[nextp]);
    }
  } else {
    TRACE0(protocol, 3, "initProtocols (process)");
    for (; protl_tab[nextp]; nextp++)  {
      TRACE2(protocol, 1, "initializing protocol %d %s", nextp,
	protl_tab[nextp]->name);
      (*protl_tab[nextp]->init)(protl_tab[nextp]);
    }
  }
}

/* this may even work. */
int x_probeprotl(name)
char *name;
{
  register char *whom;
  register int i;
  for (i = 0; (whom = protocol_names[i]); i++) {
    if (*whom == *name && !strcmp(whom, name)) return(i);
  }
  return(-1);
}

XObj x_getprotlbyname(name)
char *name;
{
  register XObj p;
  register int i;
  for (i = 0; (p = protl_tab[i]); i++) {
    if (*p->name == *name && !strcmp(p->name, name)) return(p);
  }
  return(NULL);
}

int x_printxobj(p)
XObj p;
{
 	int i;

  	if (!p) printf("\nXOBJ NULL \n");
  	printf("\nXOBJ %x %s \n",p, p->name);
	if (p->type == Protocol) {
  	 	printf("type = Protocol\n");
      	} else {
  	 	printf("type = Session\n");
	}
     	printf("myprotocol = %x %s\n",p->myprotl,p->myprotl->name);
	printf("id = %d instance = %d\n",p->protocolid,p->instance);
	printf("rcnt = %d \n",p->rcnt);
	printf("up = %x ",p->up);
	if (p->up) {
		printf("up->type = %s ", p->type == Protocol ? "Protocol" : "Session");
		printf("up->name = %s\n",p->up->name);
	} else {
		printf("\n");
	}
 		
			
	printf("numdown = %d ",p->numdown);
	for (i=0;i<p->numdown;i++) {
		if (p->down[i]) {
			printf(" %d = %s ",i,p->down[i]->name);
		}
	}
	printf("\n");
}
