/* 
 * intelEthRcv.c
 *
 * x-kernel v3.1	12/10/90
 *
 * Copyright (C) 1990  Larry L. Peterson and Norman C. Hutchinson
 */

#include "xkernel.h"
#include "eth.h"
#include "intelEthType.h"
#include "intelEthRcv.h"
#include "intelEthCmd.h"
#include "intelEth.h"


/*
 * function declarations
 */

extern int eth_demux();
extern void extractMsgFromFrame();


/*
 * static variables
 */

rcvFrameDesc_t *RcvFrameDescHeadPtr;
rcvFrameDesc_t *RcvFrameDescTailPtr;


/*
 * initRcv
 */

initRcv()
{
  
  TRACE0(ie, 5, "initRcv");
  
  initRcvFrameArea();
  
  SCB.rcvUnitCtlCmd = RU_START;
  ALERT_IE_DEV();
  
}				/* end initRcv */


/*
 * initRcvFrameArea
 *
 * We allocate receive buffers that are as big as the maximum ether data size.
 * Hence, we need allocate only as many Receive Buffer Descriptors as
 * Receive Frame Descriptors (each buffer will hold an entire frame).
 * The Receive Frame Descriptors are linked in a circle,
 * and the Receive Buffer Descriptors are linked in a parallel circle.
 * Although the circle of Buf Descs is pointed to only by
 * the first Frame Desc, as frames are received, each Buf Desc
 * will be used by the corresponding Frame Desc.
 */

initRcvFrameArea()
{
  unsigned rfdIx;		/* rcv frame desc index */
  unsigned rbdIx;		/* rcv buf desc index */
  Msg msg;
  char *buffer;
  
  /*
   * Build the list of receive frame descriptors
   */
  
  for (rfdIx = 0; rfdIx < NUM_RFDS; rfdIx++) {
    RcvFrameDescArray[rfdIx].lastAvailRfd = FALSE;
    RcvFrameDescArray[rfdIx].nextRfdOffset =
      IE_OFFSET_FROM_AD(&RcvFrameDescArray[(rfdIx + 1) % NUM_RFDS]);
    RcvFrameDescArray[rfdIx].rbdHeadOffset = NULL_IE_OFFSET;
    RcvFrameDescArray[rfdIx].suspendWhenDone = FALSE;
    RcvFrameDescArray[rfdIx].used = FALSE;
  }
  
  RcvFrameDescHeadPtr = &RcvFrameDescArray[0];
  RcvFrameDescTailPtr = &RcvFrameDescArray[NUM_RFDS - 1];
  RcvFrameDescTailPtr->lastAvailRfd = TRUE;
  
  /* the sysControlBlk has a ptr to the whole list of rcvFrameDescs */
  SCB.rcvFrameDescHeadOffset = IE_OFFSET_FROM_AD(RcvFrameDescHeadPtr);
  
  
  
  /*
   * Build the list of receive buffer descriptors, and the messages/buffers
   * they describe
   */
  
  for (rbdIx = 0; rbdIx < NUM_RBDS; rbdIx++) {
    RcvBufDescArray[rbdIx].nextRbdOffset =
      IE_OFFSET_FROM_AD(&RcvBufDescArray[(rbdIx + 1) % NUM_RBDS]);
    /* make the rbd point to the buffer of a Msg */
    msg_make_allstack(msg, MAX_ETH_DATA_SZ, (char *) 0, 0);
    buffer = msg.stack->stack;
    assert(msg.stack->ref.ref > 0);
    RcvBufDescArray[rbdIx].bufPtr = IE_AD_FROM_AD(buffer);
    RcvBufDescArray[rbdIx].msg = msg;
    RcvBufDescArray[rbdIx].bufSzLoByte = LO_BYTE_OF_2(MAX_ETH_DATA_SZ);
    RcvBufDescArray[rbdIx].bufSzHiByte = HI_BYTE_OF_2(MAX_ETH_DATA_SZ);
    /* there is no lastAvailRbd, because RFDs will run out first! */
    RcvBufDescArray[rbdIx].lastAvailRbd = FALSE;
    RcvBufDescArray[rbdIx].used = FALSE;
    RcvBufDescArray[rbdIx].lastBufOfFrame = FALSE;
  }
  
  /* the head rcvFrameDesc has a ptr to the whole list of rcvBufDescs */
  RcvFrameDescHeadPtr->rbdHeadOffset = IE_OFFSET_FROM_AD(&RcvBufDescArray[0]);
  
}				/* end initRcvFrameArea */


/*
 * freeRcvBufs
 */

freeRcvBufs()
{
  unsigned rbdIx;
  
  
  for (rbdIx = 0; rbdIx < NUM_RBDS; rbdIx++) {
    msg_free(RcvBufDescArray[rbdIx].msg);
  }
  
}				/* end freeRcvBufs */


/*
 * handleRcvdFrame
 */

handleRcvdFrame()
{
  
  /* process each used Receive Frame */
  while (RcvFrameDescHeadPtr->used) {
    
    assert(RcvFrameDescHeadPtr->rcvSuccessful);
    
    /* ack this frame reception (do we have to ack each frame?) (now?) */
    SCB.ackFrameRcvd = 1;
    ALERT_IE_DEV();
    
    extractMsgFromFrame(RcvFrameDescHeadPtr);
    
    /* former head, since used and now available again, becomes tail */
    RcvFrameDescHeadPtr->used = FALSE;
    RcvFrameDescHeadPtr->lastAvailRfd = TRUE;	/* prepare to become tail */
    RcvFrameDescTailPtr->lastAvailRfd = FALSE;	/* no longer true tail */
    RcvFrameDescTailPtr = RcvFrameDescHeadPtr;
    RcvFrameDescHeadPtr = (rcvFrameDesc_t *)
      AD_FROM_IE_OFFSET(RcvFrameDescHeadPtr->nextRfdOffset);
  }
  
}				/* end handleRcvdFrame */


/*
 * extractMsgFromFrame
 */

void extractMsgFromFrame(rcvFrameDescPtr)
     rcvFrameDesc_t *rcvFrameDescPtr;
{
  rcvBufDesc_t *rcvBufDescPtr;
  ETHhost srcEthAd, destEthAd;
  ethType_t ethPktType;
  unsigned ethDataSz;
  Msg newMsg;
  Msg rcvdMsg;
  char *buffer;
  
  /*
   * Extract info from the frame descriptor.
   */
  rcvBufDescPtr = (rcvBufDesc_t *)
    AD_FROM_IE_OFFSET(rcvFrameDescPtr->rbdHeadOffset);
  assert(rcvBufDescPtr->used);
  assert(rcvBufDescPtr->lastBufOfFrame);
  srcEthAd = rcvFrameDescPtr->srcEthAd;
  destEthAd = rcvFrameDescPtr->destEthAd;
  ethPktType = rcvFrameDescPtr->ethPktType;
  TRACE4(ie, 4, "extractMsgFromFrame: Packet from %x%x%x type %x",
	 srcEthAd.high, srcEthAd.mid, srcEthAd.low, ethPktType);
  rcvdMsg = rcvBufDescPtr->msg;
  assert(rcvdMsg.stack->ref.ref > 0);
  assert(rcvdMsg.stack->stack ==
	 (char *) AD_FROM_IE_AD(rcvBufDescPtr->bufPtr));
  ethDataSz = (unsigned) CAT_BYTES(rcvBufDescPtr->byteCntHiByte,
				   rcvBufDescPtr->byteCntLoByte);
  
  
  /*
   * Recycle the frame descriptor and buffer descriptor, but with a new
   * buffer (in the belly of a new msg).
   */
  msg_make_allstack(newMsg, MAX_ETH_DATA_SZ, (char *) 0, 0);
  buffer = newMsg.stack->stack;
  assert(newMsg.stack->ref.ref > 0);
  rcvBufDescPtr->msg = newMsg;
  rcvBufDescPtr->bufPtr = IE_AD_FROM_AD(buffer);
  rcvBufDescPtr->used = FALSE;
  
  
  /*
   * Arrange the passing-up of the Msg with padding removed.
   */
  assert(rcvdMsg.stack->ref.ref > 0);
  rcvdMsg.stack->base = rcvdMsg.stack->stack + ethDataSz - 1;
  rcvdMsg.stack->size = ethDataSz;
  msg_push(rcvdMsg, (int) ethDataSz);
  assert(rcvdMsg.stack->ref.ref > 0);
  if (CreateKernelProcess(eth_demux, 5,
			  (sizeof rcvdMsg + 3) / 4 + 1 + 2 * ((sizeof srcEthAd + 3) / 4),
			  rcvdMsg, ethPktType, srcEthAd, destEthAd) == 0) {
    msg_free(rcvdMsg);
  }
  return;
  
}				/* end extractMsgFromFrame */
