;;; -*- Mode:LISP; Package:PCL; Base:10; Syntax:Common-Lisp -*-
;;;
;;; *************************************************************************
;;; Copyright (c) 1985, 1986, 1987, 1988, 1989, 1990 Xerox Corporation.
;;; All rights reserved.
;;;
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted.  Any distribution of this
;;; software or derivative works must comply with all applicable United
;;; States export control laws.
;;; 
;;; This software is made available AS IS, and Xerox Corporation makes no
;;; warranty about the software, its performance or its conformity to any
;;; specification.
;;; 
;;; Any person obtaining a copy of this software is requested to send their
;;; name and post office or electronic mail address to:
;;;   CommonLoops Coordinator
;;;   Xerox PARC
;;;   3333 Coyote Hill Rd.
;;;   Palo Alto, CA 94304
;;; (or send Arpanet mail to CommonLoops-Coordinator.pa@Xerox.arpa)
;;;
;;; Suggestions, comments and requests for improvements are also welcome.
;;; *************************************************************************
;;;

(in-package 'pcl)

(defun make-final-one-index-accessor-dfun (gf type index table)
  (let ((cache (fill-dfun-cache table nil 1 #'one-index-limit-fn)))
    (make-one-index-accessor-dfun gf type index cache)))				

(defun make-final-n-n-accessor-dfun (gf type table)
  (let ((cache (fill-dfun-cache table t 1 #'n-n-accessors-limit-fn)))
    (make-n-n-accessor-dfun gf type cache)))

(defun fill-dfun-cache (table valuep nkeys limit-fn &optional cache)
  (let ((cache (or cache (get-cache nkeys valuep limit-fn
				    (+ (hash-table-count table) 3)))))
    (maphash #'(lambda (classes value)
		 (setq cache (fill-cache cache
					 (class-wrapper classes)
					 value
					 t)))
	     table)
    cache))

(defun make-final-checking-dfun (generic-function function
						  classes-list new-class)
  (let ((metatypes (arg-info-metatypes (gf-arg-info generic-function))))
    (if (every #'(lambda (mt) (eq mt 't)) metatypes)
	(values #'(lambda (&rest args)
		    (invoke-emf function args))
		nil (default-method-only-dfun-info))
	(let ((cache (make-final-ordinary-dfun-internal 
		      generic-function nil #'checking-limit-fn 
		      classes-list new-class)))
	  (make-checking-dfun generic-function function cache)))))

(defun make-final-caching-dfun (generic-function classes-list new-class)
  (let ((cache (make-final-ordinary-dfun-internal 
		generic-function t #'caching-limit-fn
		classes-list new-class)))
    (make-caching-dfun generic-function cache)))

(defun make-final-constant-value-dfun (generic-function classes-list new-class)
  (let ((cache (make-final-ordinary-dfun-internal 
		generic-function :constant-value #'caching-limit-fn
		classes-list new-class)))
    (make-constant-value-dfun generic-function cache)))

(defun make-final-ordinary-dfun-internal (generic-function valuep limit-fn
							   classes-list new-class)
  (let* ((arg-info (gf-arg-info generic-function))
	 (nkeys (arg-info-nkeys arg-info))
	 (new-class (and new-class
			 (equal (type-of (gf-dfun-info generic-function))
				(cond ((eq valuep t) 'caching)
				      ((eq valuep :constant-value) 'constant-value)
				      ((null valuep) 'checking)))
			 new-class))
	 (cache (if new-class
		    (copy-cache (gf-dfun-cache generic-function))
		    (get-cache nkeys (not (null valuep)) limit-fn 4))))
      (make-emf-cache generic-function valuep cache classes-list new-class)))

(defun real-use-dispatch-dfun-p (gf &optional (caching-p (use-caching-dfun-p gf)))
  (unless caching-p
    (let* ((methods (generic-function-methods gf))
	   (arg-info (gf-arg-info gf))
	   (mt (arg-info-metatypes arg-info))
	   (nreq (length mt)))
      ;;Is there a position at which every specializer is eql or non-standard?
      (dotimes (i nreq nil)
	(when (not (eq 't (nth i mt)))
	  (let ((some-std-class-specl-p nil))
	    (dolist (method methods)
	      (let ((specl (nth i (method-specializers method))))
		(when (and (not (eql-specializer-p specl))
			   (let ((sclass (specializer-class specl)))
			     (or (null (class-finalized-p sclass))
				 (member *the-class-standard-object*
					 (class-precedence-list sclass)))))
		  (setq some-std-class-specl-p t))))
	    (unless some-std-class-specl-p
	      (return-from real-use-dispatch-dfun-p t))))))))

(defun make-dispatch-dfun (gf)
  (values (get-dispatch-function gf) nil (dispatch-dfun-info)))

(defun get-dispatch-function (gf)
  (let ((methods (generic-function-methods gf)))
    (function-funcall (get-secondary-dispatch-function1 gf methods nil nil nil 
							nil nil t)
		      nil nil)))

(defun make-final-dispatch-dfun (gf)
  (make-dispatch-dfun gf))

(defun update-dispatch-dfuns ()
  (dolist (gf (gfs-of-type '(dispatch initial-dispatch)))
    (dfun-update gf #'make-dispatch-dfun)))


(defun make-final-dfun (gf &optional classes-list)
  (multiple-value-bind (dfun cache info)
      (make-final-dfun-internal gf classes-list)
    (set-dfun gf dfun cache info)))

(defun make-final-dfun-internal (gf &optional classes-list)
  (let ((methods (generic-function-methods gf)) type
	(new-class *new-class*) (*new-class* nil)
	specls all-same-p)
    (cond ((null methods)
	   (values
	    #'(lambda (&rest args)
		(apply #'no-applicable-method gf args))
	    nil
	    (no-methods-dfun-info)))
	  ((setq type (cond ((every #'standard-reader-method-p methods)
			     'reader)
			    ((every #'standard-writer-method-p methods)
			     'writer)))
	   (with-eq-hash-table (table)
	     (multiple-value-bind (table all-index first second size no-class-slots-p)
		 (make-accessor-table gf type table)
	       (if table
		   (cond ((= size 1)
			  (let ((w (class-wrapper first)))
			    (make-one-class-accessor-dfun gf type w all-index)))
			 ((and (= size 2) (or (integerp all-index) (consp all-index)))
			  (let ((w0 (class-wrapper first))
				(w1 (class-wrapper second)))
			    (make-two-class-accessor-dfun gf type w0 w1 all-index)))
			 ((or (integerp all-index) (consp all-index))
			  (make-final-one-index-accessor-dfun 
			   gf type all-index table))
			 (no-class-slots-p
			  (make-final-n-n-accessor-dfun gf type table))
			 (t
			  (make-final-caching-dfun gf classes-list new-class)))
		   (make-final-caching-dfun gf classes-list new-class)))))
	  ((and (not (and (every #'(lambda (specl) (eq specl *the-class-t*))
				 (setq specls (method-specializers (car methods))))
			  (setq all-same-p
				(every #'(lambda (method)
					   (and (equal specls
						       (method-specializers method))))
				       methods))))
		(use-constant-value-dfun-p gf))
	   (make-final-constant-value-dfun gf classes-list new-class))
	  ((use-dispatch-dfun-p gf)
	   (make-final-dispatch-dfun gf))
	  ((and all-same-p (not (use-caching-dfun-p gf)))
	   (let ((emf (get-secondary-dispatch-function gf methods nil)))
	     (make-final-checking-dfun gf emf classes-list new-class)))
	  (t
	   (make-final-caching-dfun gf classes-list new-class)))))



(defvar dfun-count nil)
(defvar dfun-list nil)
(defvar *minimum-cache-size-to-list*)

(defun list-dfun (gf)
  (let* ((sym (type-of (gf-dfun-info gf)))
	 (a (assq sym dfun-list)))
    (unless a
      (push (setq a (list sym)) dfun-list))
    (push (generic-function-name gf) (cdr a))))

(defun list-all-dfuns ()
  (setq dfun-list nil)
  (map-all-generic-functions #'list-dfun)
  dfun-list)

(defun list-large-cache (gf)
  (let* ((sym (type-of (gf-dfun-info gf)))
	 (cache (gf-dfun-cache gf)))
    (when cache
      (let ((size (cache-size cache)))
	(when (>= size *minimum-cache-size-to-list*)
	  (let ((a (assoc size dfun-list)))
	    (unless a
	      (push (setq a (list size)) dfun-list))
	    (push (let ((name (generic-function-name gf)))
		    (if (eq sym 'caching) name (list name sym)))
		  (cdr a))))))))

(defun list-large-caches (&optional (*minimum-cache-size-to-list* 130))
  (setq dfun-list nil)
  (map-all-generic-functions #'list-large-cache)
  (setq dfun-list (sort dfun-list #'< :key #'car))
  (mapc #'print dfun-list)
  (values))


(defun count-dfun (gf)
  (let* ((sym (type-of (gf-dfun-info gf)))
	 (cache (gf-dfun-cache gf))
	 (a (assq sym dfun-count)))
    (unless a
      (push (setq a (list sym 0 nil)) dfun-count))
    (incf (cadr a))
    (when cache
      (let* ((size (cache-size cache))
	     (b (assoc size (third a))))
	(unless b 
	  (push (setq b (cons size 0)) (third a)))
	(incf (cdr b))))))

(defun count-all-dfuns ()
  (setq dfun-count (mapcar #'(lambda (type) (list type 0 nil))
			   '(ONE-CLASS TWO-CLASS DEFAULT-METHOD-ONLY
			     ONE-INDEX N-N CHECKING CACHING 
			     DISPATCH)))
  (map-all-generic-functions #'count-dfun)
  (mapc #'(lambda (type+count+sizes)
	    (setf (third type+count+sizes)
		  (sort (third type+count+sizes) #'< :key #'car)))
	dfun-count)
  (mapc #'(lambda (type+count+sizes)
	    (format t "~&There are ~4d dfuns of type ~s"
		    (cadr type+count+sizes) (car type+count+sizes))
	    (format t "~%   ~S~%" (caddr type+count+sizes)))
	dfun-count)
  (values))

(defun gfs-of-type (type)
  (unless (consp type) (setq type (list type)))
  (let ((gf-list nil))
    (map-all-generic-functions #'(lambda (gf)
				   (when (memq (type-of (gf-dfun-info gf)) type)
				     (push gf gf-list))))
    gf-list))
