#ifndef	SCHEDULER_H
#define	SCHEDULER_H

/*$Header: Scheduler.h,v 2.204 89/10/07 23:21:25 keith Stab $*/

/* Scheduler.h -- declarations for the Process Scheduler

	THIS SOFTWARE FITS THE DESCRIPTION IN THE U.S. COPYRIGHT ACT OF A
	"UNITED STATES GOVERNMENT WORK".  IT WAS WRITTEN AS A PART OF THE
	AUTHOR'S OFFICIAL DUTIES AS A GOVERNMENT EMPLOYEE.  THIS MEANS IT
	CANNOT BE COPYRIGHTED.  THIS SOFTWARE IS FREELY AVAILABLE TO THE
	PUBLIC FOR USE WITHOUT A COPYRIGHT NOTICE, AND THERE ARE NO
	RESTRICTIONS ON ITS USE, NOW OR SUBSEQUENTLY.

Author:
	K. E. Gorlen
	Computer Systems Laboratory, DCRT
	National Institutes of Health
	Bethesda, MD 20892

$Log:	Scheduler.h,v $
 * Revision 2.204  89/10/07  23:21:25  keith
 * Pre-release
 * 
 * Revision 2.203  89/08/08  15:24:07  keith
 * Pre-release
 * 
 * Revision 2.202.1.1  89/07/01  21:55:51  keith
 * Base revision for R2.00 MI version
 * 
 * Revision 2.202  89/06/22  20:55:51  keith
 * Base revision for AT&T C++ R2.0 release (Cycle 20)
 * 
 * Revision 2.201.1.3  89/06/21  23:55:47  keith
 * Add private copy constructor.
 * 
 * Revision 2.201.1.2  89/06/01  23:17:18  keith
 * Remove base class argument from DECLARE_MEMBERS.
 * 
 * Revision 2.201.1.1  89/05/19  15:43:57  keith
 * Add base class arg to DECLARE_MEMBERS.
 * Place enums and typedefs in classes.
 * 
 * Revision 2.201  89/05/12  11:41:06  keith
 * Release for R2.0 Beta test.
 * 
 * Revision 2.200.1.1  89/04/24  17:17:58  keith
 * Working revision for R2.0 Beta 6++
 * 
 * Revision 2.200  89/04/17  23:31:08  keith
 * Base revision for R2.0 Beta 6.
 * 
 * Revision 2.121  89/02/16  11:08:57  keith
 * Base revision for C++ R1.2.1 compatible version.
 * 
*/
#include "LinkedList.h"
#include "StackProc.h"
#include "nihclconfig.h"
#ifdef HAVE_SELECT
#include <sys/time.h>
#endif

class HeapProc;

class Scheduler : public NIHCL {
public:			// type definitions
	enum setjmp_val { schedule_process=0, resume_current_process, resume_new_process };
public:			// static member functions
	static Process* MAIN_PROCESS(int priority =0);
	static unsigned char activePriority()	{ return active_process->priority(); }
	static Process& activeProcess()		{ return *active_process; }
	static bool astActive()			{ return ast_level != 0; }
	static const char* className();
	static void dumpOn(ostream& strm =cerr);
	static void initialize(stackTy* bottom, int priority =0);
	static void printOn(ostream& strm =cout);
	static void terminateActive()		{ activeProcess().terminate(); }
	static void restoreProcessStack();
	static void schedule();
	static void yield();
public:			// static member variables
	static JMP_BUF switcher;	// environment for stack switcher
private:		// static member variables
	static Process* active_process;
	static Process* previous_process;
	static StackProc* main_stack_process;
	static stackTy* main_stack_bottom;
	static unsigned runCount;	// total # of runnable processes 
	static LinkedList runList[MAXPRIORITY+1];
#ifdef HAVE_SELECT
	static LinkedList selectList;	// Process waiting for select
	static struct timeval selectTimeout;	// timeout on select
	static int selectfd();			// do select(2) system call
	void timeOutSelect()	{ selectTimeout.tv_sec = selectTimeout.tv_usec = 0; }
#endif
protected:		// static member variables
	static int ast_level;	// AST nesting level 
	Scheduler() {}		// to prevent construction of instances
private:		// member functions
	Scheduler(const Scheduler&) {}	// to prevent copy of instances
	static void addProcess(Process& p);
	friend Process;
	friend StackProc::StackProc(stackTy* bottom, int priority);
	friend void StackProc::switchFrom(HeapProc*);
	friend void StackProc::switchFrom(StackProc*);
};

inline void Scheduler::initialize(stackTy* bottom, int priority)
{
	Scheduler::main_stack_bottom = bottom;
	Scheduler::active_process = new StackProc(bottom,priority); /* create MAIN process */
}

inline void Scheduler::restoreProcessStack()
{

	StackProc* active = (StackProc*)Scheduler::active_process;

#if STACK_GROWS_DOWN

	unsigned long size = active->stack_bottom - active->stack_top;
	alloca((size + Scheduler::main_stack_bottom - active->stack_bottom) * sizeof(stackTy));
	Process::copyStack(active->stack_save, active->stack_top+1, size);

#else

	unsigned long size = active->stack_top - active->stack_bottom;
	alloca((size + active->stack_bottom - Scheduler::main_stack_bottom) * sizeof(stackTy));
	copyStack(active->stack_save, active->stack_bottom, size);

#endif

	Scheduler::main_stack_process = active;
/* the following will cause "longjmp botch" on some machines */
	LONGJMP(active->env, resume_new_process);
}

/*
	main() must call MAIN_PROCESS first thing to initialize the scheduler.
*/

#define MAIN_PROCESS(priority) \
	Scheduler::initialize((stackTy*)alloca(sizeof(stackTy)), priority); \
	if (_SETJMP(Scheduler::switcher) != 0) {  /* longjmp() here to restore a process's stack */ \
		Scheduler::restoreProcessStack(); \
 	} \

class AST_LEVEL : public Scheduler {
public:
	AST_LEVEL()	{ ast_level++; }
	~AST_LEVEL()	{ ast_level--; }
};

// AST/signal handlers must begin with an AST_ENTER
#define AST_ENTER	AST_LEVEL ast_level_dummy

#endif
