/* OIOTbl.c -- Implementation of tables used for Object I/O

	THIS SOFTWARE FITS THE DESCRIPTION IN THE U.S. COPYRIGHT ACT OF A
	"UNITED STATES GOVERNMENT WORK".  IT WAS WRITTEN AS A PART OF THE
	AUTHOR'S OFFICIAL DUTIES AS A GOVERNMENT EMPLOYEE.  THIS MEANS IT
	CANNOT BE COPYRIGHTED.  THIS SOFTWARE IS FREELY AVAILABLE TO THE
	PUBLIC FOR USE WITHOUT A COPYRIGHT NOTICE, AND THERE ARE NO
	RESTRICTIONS ON ITS USE, NOW OR SUBSEQUENTLY.

Author:
	K. E. Gorlen
	Bg. 12A, Rm. 2033
	Computer Systems Laboratory
	Division of Computer Research and Technology
	National Institutes of Health
	Bethesda, Maryland 20892
	Phone: (301) 496-1111
	uucp: uunet!nih-csl!keith
	Internet:keith@alw.nih.gov
	July, 1988

Function:
	
Tables to keep track of multiple references to objects and classes.
Used by storeOn()/readFrom().

Modification History:

$Log:	OIOTbl.c,v $
 * Revision 2.204  89/10/07  23:20:25  keith
 * Pre-release
 * 
 * Revision 2.203  89/08/08  15:23:03  keith
 * Pre-release
 * 
 * Revision 2.202.1.2  89/07/03  23:08:57  keith
 * Change assocAt() to return 0 instead of nil when assocation
 * not found.
 * 
 * Revision 2.202.1.1  89/07/01  21:54:58  keith
 * Base revision for R2.00 MI version
 * 
 * Revision 2.202  89/06/22  20:54:56  keith
 * Base revision for AT&T C++ R2.0 release (Cycle 20)
 * 
 * Revision 2.201.1.1  89/06/21  10:51:27  keith
 * Change 2nd arg of DO to class name.
 * Replace downward casts from Object by calls to castdown().
 * 
 * Revision 2.201  89/05/12  11:18:43  keith
 * Release for R2.0 Beta test.
 * 
 * Revision 2.200  89/05/12  10:48:30  keith
 * Initial revision for R2.0.
 * 
 * Revision 2.200.1.1  89/04/24  17:20:18  keith
 * Working revision for R2.0 Beta 6++
 * 
 * Revision 2.200  89/04/17  23:32:33  keith
 * Base revision for R2.0 Beta 6.
 * 
 * Revision 2.121  89/02/16  11:11:03  keith
 * Base revision for C++ R1.2.1 compatible version.
 * 
 * Revision 2.1  89/02/08  13:22:50  ted
 * Initial version after renaming from oops
 * 
 * Revision 1.2  88/07/15  15:45:44  keith
 * Simplify Object I/O.
 * 
 * Revision 1.1  88/07/14  18:11:15  keith
 * Initial revision
 * 

*/

#include "OIOTbl.h"
#include "AssocInt.h"

extern const int NIHCL_STMBRREF,NIHCL_STREFMBR;

StoreOnTbl::StoreOnTbl(unsigned size)
	: d(size), c(Class::dictionary.size())
{
	d.add(*new AssocInt(*Object::nil,0));	// add nil object
	objNum = classNum = 0;
}

StoreOnTbl::~StoreOnTbl()
{
	DO(d,AssocInt,asc) delete asc; OD
	DO(c,Class,cl) cl->number(0); OD
}

bool StoreOnTbl::add(const Object& obj, int& num)
{
	AssocInt* asc = (AssocInt*)d.assocAt(obj);
	if (asc == 0) {	// object has not been stored
		d.add(*new AssocInt((Object&)obj, (num = ++objNum)));
		return YES;	// need to store object
	}
// return number of stored object
	num = (int)Integer::castdown(asc->value())->value(); 
	if (num == -1) setError(NIHCL_STREFMBR,DEFAULT,&obj,obj.className());
	return NO;		// don't need to store object -- use reference
}

bool StoreOnTbl::addClass(const Class* cp, unsigned short& num)
{
	if ((num = cp->number()) == 0) { 
		c.add(*(Class*)cp);
		((Class*)cp)->number(num = ++classNum);
		return YES;	// need to store class
	}
	return NO;	// don't need to store class -- use reference
}

void StoreOnTbl::addMember(const Object& obj)
{
	AssocInt* asc = (AssocInt*)d.assocAt(obj);
	if (asc == 0) {			// object has not been stored 
		d.add(*new AssocInt((Object&)obj,-1));	// add member object to IdentDict
		return;
	}
// Error -- tried to store a member object that has already been stored
   	int objectNum = (int)Integer::castdown(asc->value())->value(); 
	setError(NIHCL_STMBRREF,DEFAULT,objectNum,&obj,obj.className());
}

ReadFromTbl::ReadFromTbl(unsigned size)
	: obs(size), cls(Class::dictionary.size())
{
	obs.add(*Object::nil);		// nil is @0
	cls.add(*Object::nil);
}

int ReadFromTbl::add(Object& obj)
{
	int objectNum = obs.size();
	if (objectNum == obs.capacity()) obs.reSize(objectNum+1024);
	obs.add(obj);
	return objectNum;
}

int ReadFromTbl::add(const Class* cl)
{
	int classNum = cls.size();
	cls.add(*(Class*)cl);
	return classNum;
}

Object* ReadFromTbl::objectAt(int i)
{
	return obs[i];
}

const Class* ReadFromTbl::classAt(int i)
{
	return Class::castdown(cls[i]);
}
