// This may look like C code, but it is really -*- C++ -*-
// 
// Copyright (C) 1988 University of Illinois, Urbana, Illinois
// Copyright (C) 1989 University of Colorado, Boulder, Colorado
// Copyright (C) 1990 University of Colorado, Boulder, Colorado
//
// written by Dirk Grunwald (grunwald@foobar.colorado.edu)
//
#ifndef	HardwareContext_h
#define	HardwareContext_h
#pragma once

#include <stream.h>
#include <assert.h>
#include <AwesimeConfig.h>
#include <setjmp.h>

extern long MagicStackMarker;

typedef void (*voidFuncP)();

class CpuMultiplexor;
class SingleCpuMux;
class MultiCpuMux;
class SimulationMultiplexor;
class SingleSimMux;

typedef unsigned long HardwareContextQuad;

#define HARDWARE_CONTEXT_CHECK_NONE 0
#define HARDWARE_CONTEXT_CHECK_SWITCH 1
#define HARDWARE_CONTEXT_CHECK_MPROTECT 2

class HardwareContext {
    //
    //	The following two fields are machine dependent & their order
    //	should not be changed.
    //
    HardwareContextQuad sp;
    HardwareContextQuad pc;
    HardwareContextQuad out0;

    void** stackBase;	// bottom of stack
    void** stackEnd;	// maximum depth of stack? (actually, a MIN value)
    unsigned stackMax;	// maximum depth of stack? (actually, a MIN value)
    unsigned stackSize;	// stack size in units of void*
    void** stackMallocAt;

    //
    // point to MagicStackMarker to check for corruption
    //
    long *stackCheck;
    unsigned checkStackLimits;

    friend class Thread;

    friend class CpuMultiplexor;
    friend class SingleCpuMux;
    friend class MultiCpuMux;

    friend class SimulationMultiplexor;
    friend class SingleSimMux;
    
    //
    //	Accessed by friend classes only
    //
    void switchContext(HardwareContext *to);
    void magicSwitchTo(HardwareContext *to);
    void **getSp();
    
    void **mallocAt();
    void buildReturnFrame(void * returnThis, voidFuncP returnAddress);
    void stackOverflow();
    
    //
    // never allocated by anything other than friend classes
    //
    HardwareContext(int checked, unsigned stackSize);
    
    void reclaimStack();

public:
    HardwareContext();
    
    long maxStackDepth();
    void checkStack(int overage = 0);
    void classPrintOn(ostream& strm);
};

static inline
HardwareContext::HardwareContext()
{
    int NotReached = 0;
    assert( NotReached );
}

static inline ostream&
operator<<(ostream& strm, HardwareContext& ob)
{
    ob.classPrintOn(strm);
    return strm;
}

static inline long
HardwareContext::maxStackDepth()
{
    return( long(stackMax) );
}

static inline void **
HardwareContext::getSp()
{
    register int theStackPointer asm("%sp");
    return( (void **) theStackPointer );
}


static inline void
HardwareContext::checkStack(int overage)
{
    unsigned depth = stackBase - getSp() + overage;
    if (stackMax < depth) {
	stackMax = depth;
    }
    if ( stackMax >= stackSize || *stackCheck != MagicStackMarker ) {
	stackOverflow();
    }
}

#endif	HardwareContext_h
