// This may look like C code, but it is really -*- C++ -*-
// 
// Copyright (C) 1988 University of Illinois, Urbana, Illinois
// Copyright (C) 1989 University of Colorado, Boulder, Colorado
// Copyright (C) 1990 University of Colorado, Boulder, Colorado
//
// written by Dirk Grunwald (grunwald@foobar.colorado.edu)
//
#ifndef Gate_h
#define Gate_h
#pragma once

//
//	Gate.h
//	The gate is initially closed. Gates can be open or closed,
//	everyone waits when it's open, everyone passes when it's
//	open. Unlike events, Gates don't fiddle with the state --
//	it's controlled by external tasks.
// 

#include <CpuMultiplexor.h>
#include <ReserveByException.h>
#include <FifoScheduler.h>
#include <SpinLock.h>

typedef enum {GateLocked, GateUnlocked} GateState;

class Gate : public ReserveByException {
    SpinLock lock;
    GateState pState;

    FifoScheduler pPileOfThreads;
    
    void hiddenWait();
    
private:
    virtual int reserveByException(Thread *byWho);
    
public:
    
    Gate(GateState startState = GateLocked);
    virtual ~Gate();
    
    //
    //	Number of taskings waiting
    //
    int size();
    //
    //	GateLocked or GateUnlocked
    //
    GateState state();

    void wait();
    void open();
    void close();
};

static inline
Gate::Gate(GateState startState)
{
    pState = startState;
}


static inline GateState
Gate::state()
{
    lock.reserve();
    GateState s = pState;
    lock.release();
    return( s );
}


static inline int
Gate::size()
{
    lock.reserve();
    int s = pPileOfThreads.size();
    lock.release();
    return(s);
}

static inline void
Gate::wait()
{
    lock.reserve();
    GateState s = pState;
    lock.release();
    if ( s == GateLocked ) {
	hiddenWait();
    }
}

static inline void
Gate::close()
{
    lock.reserve();
    pState = GateLocked;
    lock.release();
}

#endif /* Gate_h */
