//
// Copyright (C) 1991 Texas Instruments Incorporated.
//
// Permission is granted to any individual or institution to use, copy, modify,
// and distribute this software, provided that this complete copyright and
// permission notice is maintained, intact, in all copies and supporting
// documentation.
//
// Texas Instruments Incorporated provides this software "as is" without
// express or implied warranty.
//
// Created: MBN 08/29/89 -- Initial design and implementation
// Updated: MBN 10/11/89 -- Changed "current_position" to "curpos" 
// Updated: LGO 02/02/90 -- Re-wrote practically everything
// Updated: MJF 03/12/90 -- Added group names to RAISE
//
// The  Queue class is publicly derived from the Generic class  and is used
// to implement  non-type specific  functionality  for the  parameterized Queue
// class. In this  manner, code common to all  instances of the Queue class can
// be shared to reduce code replication. The Queue class implements  a circular
// buffer of  a   user-specified type.   This is accomplished   by   using  the
// parameterized type capability  of C++. The Queue  will grow  dynamically  as
// necessary with the amount of growth determined by the value of an allocation
// size slot. Fixed length  queues  are also supported by  setting the value of
// the allocation size slot to INVALID.
//

#ifndef BASE_QUEUEH				// If no class definition
#include <cool/Base_Queue.h>				// Include definition file
#endif


// void set_growth_ratio(float) -- Set growth percentage of this Queue
// Input:                          Float ratio and character string type
// Output:                         None

void Queue::set_growth_ratio (float ratio, const char* Type) {
#if ERROR_CHECKING
  if (ratio < 0.0)				// If non-positive growth
    RAISE (Error, SYM(Queue), SYM(Negative_Ratio),
	   "Queue<%s>::set_growth_ratio(): Negative growth ratio %f",
	   Type, ratio);
#endif
  this->growth_ratio_s = ratio;			// Set growth ratio
}


// void set_alloc_size(int) -- Set the default allocation size growth rate
// Input:                      Integer size and character string type
// Output:                     None

void Queue::set_alloc_size (int size, const char* Type) {
#if ERROR_CHECKING
  if (size < 0 && size != INVALID)		// If index out of range
    RAISE (Error, SYM(Queue), SYM(Negative_Size),
	   "Queue<%s>::set_alloc_size(): Negative growth size %d",
	   Type, size);
#endif
  this->alloc_size = size;			// Set growth size
}


// Queue () -- Empty constructor for the Queue class
// Input:      None
// Output:     None

Queue::Queue () {
  this->limit = 0;				// Initialize size
  this->in = this->out = 0;			// Intialize first/last pointer
  this->alloc_size = QUEUE_MEM_BLK_SZ;		// Default
}


// Queue (long) -- Constructor that specifies number of elements
// Input:          Integer number of elements
// Output:         None

Queue::Queue (long n) {
  this->limit = n;				// Element capacity
  this->in = this->out = 0;			// Intialize first/last pointer
  this->alloc_size = QUEUE_MEM_BLK_SZ;		// Default
}


// Queue (Queue&) -- Constructor for reference to another Queue
// Input:            Queue reference
// Output:           None

Queue::Queue (const Queue& s) {
  alloc_size = s.alloc_size;			// Set alloc size
  if (alloc_size == INVALID)
    alloc_size = QUEUE_MEM_BLK_SZ;
  this->in = s.in;				// Set in
  this->out = s.out;				// Set out
  this->limit = s.limit;			// Set limit
}


// ~Queue -- Destructor for Queue class that frees up storage
// Input:    None
// Output:   None

Queue::~Queue () {
}


// look_error -- Error message for parameterized Queue<Type>::look() method
// Input:        Character string of type
// Output:       None

void Queue::look_error (const char* Type) {
  RAISE (Error, SYM(Queue), SYM(No_Elements),
	 "Queue<%s>::look(): No elements in queue", Type);
}


// value_error -- Error message for parameterized Queue<Type>::value() method
// Input:         Character string of type
// Output:        None

void Queue::value_error (const char* Type) {
  RAISE (Error, SYM(Queue), SYM(Invalid_Cpos),
	 "Queue<%s>::value(): Invalid current position", Type);
}


// resize_error -- Error message for parameterized Queue<Type>::resize() method
// Input:          Character string of type
// Output:         None

void Queue::resize_error (const char* Type) {
  RAISE (Error, SYM(Queue), SYM(Static_Size),
	 "Queue<%s>::resize(): Static-size queue", Type);
}


// assign_error -- Error message for parameterized Queue<Type>::operator=()
// Input:          Character string of type
// Output:         None

void Queue::assign_error (const char* Type) {
  RAISE (Error, SYM(Queue), SYM(Static_Size),
	 "Queue<%s>::operator=(): Static-size queue", Type);
}
