/*
 * Copyright (c) 1990 Paul Pomes
 * Copyright (c) 1990 University of Illinois Board of Trustees
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of Illinois, Urbana.  In addition, redistribution
 * and use must conform to the terms listed in the Copying file in
 * this directory.
 *
 * The name of the University may not be used to endorse or promote products
 * derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#ifndef lint
static char rcsid[] = "@(#)$Header: /usr/local/src/cso/oed2/RCS/misc.c,v 2.0.0.1 90/03/30 14:51:08 paul Exp $";
#endif /* lint */

#include	"oed2.h"

/*
 * Miscellaneous routines:
 *
 * GetCount -- Extract number of matches from Pat response stream
 * PrintFile -- Print contents of a file
 * PagerOpen -- Open a output filter for writing
 * PagerClose -- Close an output filter
 * Finis -- kill of children and exit
 * Usage -- print how to use message
 * Error -- Print error message on open streams
 * Warning -- Print error message on open streams
 * Fatal -- Print error message with program name, then exit
 * Malloc -- a malloc with error checking
 * Realloc -- realloc with error checking
 */
/*
 * GetCount -- Extract number of matches from Pat response stream
 *
 *	Parameters:
 *		None
 *	Returns:
 *		 Number of matches
 *	Side Effects:
 *		None
 */

GetCount ()
{
	int		i;
	int		Iterations;
	char		Stuff[MAXBLK+1];
	int		StuffCnt = MAXBLK;
	char		*Cpnt;

	Cpnt = Stuff;
	for (Iterations = 60, StuffCnt = MAXBLK; StuffCnt > 0 && Iterations;) {
		errno = 0;
		i = fread (Cpnt, 1, StuffCnt, FromPat);
		if (i == 0 && (errno == EWOULDBLOCK || errno == 0)) {
			if (ferror (FromPat))
				clearerr (FromPat);
			Iterations--;
			sleep (1);
			continue;
		}
		Iterations = 60;
		StuffCnt -= i;
		Cpnt += i;
		*Cpnt = CHNULL;
		if (nequal ((Cpnt-3), ">> ", 3))
			break;
	}
	if (tTd(16,3)) {
		printf ("GetCount: From pat =%s=\n", Stuff);
		(void) fflush (stdout);
	}
	if (Iterations == 0)
		Fatal ("GetCount: 60 failed reads in a row, it's time to die");

	/* Handle error messages by printing the mesg and returning 0 */
	if (equal (Stuff, "Interrupted>> ") ||
	    nequal (Stuff, "Command expressed incorrectly", 29)) {
		Error ("GetCount: Pat returned error msg: %s", Stuff);
		return (0);
	}

	if ((Cpnt = index (Stuff, ':')) == CPNULL)
		Warning ("GetCount: index() failed to find ':' in =%s=", Stuff);
	i = atoi (++Cpnt);
	return (i);
} /* GetCount */
/*
 * PrintFile -- Print contents of a file
 *
 *	Print the contents of the argument file if it exists.
 *
 *	Parameters:
 *		OutFile -- Name of file to print
 *	Returns:
 *		none
 *	Side effects:
 *		Prints error messages if warranted.
 *	Deficiencies:
 */

PrintFile (OutFile)
char	*OutFile;
{
	FILE		*FilePnt;
	FILE		*Out;			/* output file pointer */
	FILE		*PagerStrm = stdout;	/* pager file pointer */
	int		i;
	char		Line[MAXSTR+1];
	char		Name[MAXSTR];
	extern FILE	*PagerOpen();

	/* Can we trust our calling routine?  Naw */
	if (OutFile == CPNULL || *OutFile == CHNULL) {
		Warning ("PrintFile: Handed a NULL pointer or string");
		return;
	}

	/* Create the full file name unless Outfile is root'ed */
	if (*OutFile == '/')
		(void) strcpy (Name, OutFile);
	else
		(void) sprintf (Name, "%s/%s", OED2LIB, OutFile);

	/* See if it's OK to read the file if it exists. */
	if (access (Name, F_OK) == -1)
		return;
	if (access (Name, R_OK) == -1) {
		Warning ("PrintFile: Can't access file %s: %s", Name, ERR);
		return;
	}

	if (*Pager != CHNULL) {
		if ((PagerStrm = PagerOpen (Pager)) == FILENULL) {

			/* This shouldn't happen */
			Error ("PrintFile: PagerOpen(%s): %s", Pager, ERR);
			*Pager = CHNULL;
			PagerStrm = stdout;
		}
		(void) signal (SIGPIPE, SIG_IGN);
	}
	(void) fflush (stdout);
	Out = PagerStrm;

	/* It's there and OK, read it and weep. */
	if ((FilePnt = fopen (Name, "r")) == FILENULL)

		/* Something is seriously wrong if we're here. */
		Fatal ("PrintFile: fopen of %s file", Name);

	while (fgets (Line, MAXSTR, FilePnt) != CPNULL) {
		if (*Line == '#')
			continue;
		fputs (Line, Out);
	}
	(void) fclose (FilePnt);
	(void) fflush (Out);
	if (*Pager != CHNULL) {
		i = PagerClose (Out);
		(void) signal (SIGPIPE, SIG_DFL);
		if (tTd(17,1))
			printf ("PrintFile: PagerClose (Pager) returned %d\n", i);
	}
	(void) fflush (stdout);
} /* PrintFile */
/*
 * PagerOpen -- Open an output filter
 *
 *	Create a pipe, fork, and return the stream descriptor of the child's
 *	input stream.
 *
 *	Parameters:
 *		Pgm -- Name of pager program to use
 *	Returns:
 *		Stream descriptor or FILENULL if exec fails
 *	Side effects:
 *		None
 *	Deficiencies:
 */

FILE *
PagerOpen (Pgm)
char *Pgm;
{
	int		P1[2];			/* pipe for child */
	int		Pid;			/* pid from vfork() */

	/* Flush pending I/O, set up pipes and fork */
	if (tTd(17,1))
		printf ("PagerOpen: Using %s\n", Pgm);
	(void) fflush (stdout);
	if (pipe (P1) == -1)
		Fatal ("PagerOpen: pipe (P1): %s", ERR);
	if ((Pid = vfork()) == -1)
		Fatal ("PagerOpen: Pgm vfork (): %s", ERR);
	if (tTd(17,1))
		printf ("PagerOpen: Pid %d (0 in child)\n", Pid);

	if (Pid == 0)	/* child */
	{
		/* Close and dup */
		if (close (0) == -1)
			Fatal ("PagerOpen: close (0): %s", ERR);
		if (fcntl (P1[0], F_DUPFD, 0) == -1)
			Fatal ("PagerOpen: fcntl (0): %s", ERR);

		/* Close all open file descriptors */
		(void) close (P1[0]);
		(void) close (P1[1]);

		execl("/bin/sh", "sh", "-c", Pgm, CPNULL);
		Warning ("PagerOpen: execl(\"%s\"): %s", Pgm, ERR);
		_exit (1);
	}

	/* Close the descriptors not used in the parent. */
	if (close (P1[0]) == -1)
		Fatal ("PagerOpen: close (P1[0]) after Pgm vfork(): %s", ERR);
	if (P1[1] >= MAX_PIDS)
		Fatal ("PagerOpen: file descriptor value (%d) too large for PidTab", P1[1]);
	PidTab[P1[1]] = Pid;
	return (fdopen(P1[1], "w"));
} /* PagerOpen */
/*
 * PagerClose -- Close an output filter
 *
 *	Parameters:
 *		str -- Stream descriptor to close
 *	Returns:
 *		status of child or -1
 *	Side effects:
 *		Prints error messages if warranted.
 *	Deficiencies:
 */

PagerClose (str)
FILE	*str;
{
	int		Omask;			/* for signal handling */
	int		Child;			/* pid of child from PidTab */
	int		Pid;			/* temp var for wait() call */
	union wait	Status;			/* exit status of child */
	int		i;

	Child = PidTab[fileno (str)];
	PidTab[fileno (str)] = -1;
	(void) fclose (str);
	if (Child == -1)
		return (-1);
	Omask = sigblock (sigmask (SIGINT) | sigmask (SIGQUIT)
			| sigmask (SIGHUP));
	while ((Pid = wait (&Status)) != Child && Pid != -1)
		for (i = 0; i < MAX_PIDS; i++)
			if (PidTab[i] == Pid)
				PidTab[i] = -1;
	(void) sigsetmask (Omask);
	return (Pid == -1 ? -1 : Status.w_status);
} /* PagerClose */
/*
 * Finis -- kill of children and exit
 *
 *	Parameters:
 *		none
 *	Returns:
 *		none, uses ExitStat for exit value
 *	Side effects:
 *		program terminates
 *	Deficiencies:
 */

Finis ()
{
#ifndef REMOTE
	if ((ToPat != FILENULL && PidTab[fileno(ToPat)] > 0) && ExitStat == 0)
		fprintf (stderr, "%s: Finis(): Pat running & ExitStat == 0", MyName);
	if (ToPat != FILENULL && PidTab[fileno(ToPat)] > 0)
		(void) kill (PidTab[fileno(ToPat)], SIGTERM);
#endif /* ! REMOTE */
	exit (ExitStat);
} /* Finis */
/*
 * Usage -- Print how to use message and perhaps copyright message
 *
 *	Parameters:
 *		full_text -- prints the copyright statement if set
 *	Returns:
 *		none
 *	Side effects:
 *		none
 *	Deficiencies:
 */

Usage (full_text)
int	full_text;
{
	register int	which = 0;		/* current line */
	extern char	Version[];

	if (full_text)
		PrintFile ("license");
	fprintf (stdout, "oed2 %s, patchlevel %d\n", Version, PATCHLEVEL);
	while (usage[which] != CPNULL) {
		fprintf (stdout, usage[which++], MyName);
		(void) putc ('\n', stdout);
	}
	(void) putc ('\n', stdout);
	(void) fflush (stdout);
	return;
} /* Usage */
/*
 * Error -- Print error message on open streams
 *
 * Print an error message to stderr and open streams.  Message is preceded
 * by "<MyName>: " using global char *MyName, and followed by a newline.
 *
 *	Parameters:
 *		message -- printf format string
 *		arg1-4 --printf arguments
 *	Returns:
 *		None
 *	Side effects:
 *		None
 *	Deficiencies:
 */

/* VARARGS 1 */
Error (message, arg1, arg2, arg3, arg4)
char	*message;
long	arg1, arg2, arg3, arg4;
{
	char	Str[MAXSTR], *Spnt;

	(void) sprintf (Str, "%s: ", MyName);
	Spnt = index (Str, CHNULL);
	(void) sprintf (Spnt, message, arg1, arg2, arg3, arg4);
	Spnt = index (Str, CHNULL);
	(void) sprintf (Spnt, "\n");
	if (RdrStrm != FILENULL)
		fputs (Str, RdrStrm);
	else if (TeeStrm != FILENULL)
		fputs (Str, TeeStrm);
	fputs (Str, stderr);
} /* Error */
/*
 * Warning -- Print error message on open streams
 *
 * Print an error message to stderr and open streams.  Message is preceded
 * by "<MyName>: " using global char *MyName, and followed by a newline.
 * If lflag is set, issue a syslog (LOG_ERR,...) message as well.
 *
 *	Parameters:
 *		message -- printf format string
 *		arg1-4 -- printf arguments
 *	Returns:
 *		None
 *	Side effects:
 *		None
 *	Deficiencies:
 */

/* VARARGS 1 */
Warning (message, arg1, arg2, arg3, arg4)
char	*message;
long	arg1, arg2, arg3, arg4;
{
	char	Str[MAXSTR], *Spnt;

	(void) sprintf (Str, "%s: ", MyName);
	Spnt = index (Str, CHNULL);
	(void) sprintf (Spnt, message, arg1, arg2, arg3, arg4);
	if (lflag)
		syslog (LOG_ERR, Str);
	Spnt = index (Str, CHNULL);
	(void) sprintf (Spnt, "\n");
	if (RdrStrm != FILENULL)
		fputs (Str, RdrStrm);
	else if (TeeStrm != FILENULL)
		fputs (Str, TeeStrm);
	fputs (Str, stderr);
} /* Warning */
/*
 * Fatal -- Print error message with program name, then exit
 *
 * Print an error message to stderr and exit nonzero.  Message is preceded
 * by "<MyName>: " using global char *MyName, and followed by a newline.
 *
 *	Parameters:
 *		message -- printf format string
 *		arg1-4 -- printf arguments
 *	Returns:
 *		none, exit (1)
 *	Side effects:
 *		program terminates
 *	Deficiencies:
 */

/* VARARGS 1 */
Fatal (message, arg1, arg2, arg3, arg4)
char	*message;
long	arg1, arg2, arg3, arg4;
{
	char	Str[MAXSTR], *Spnt;

	(void) sprintf (Str, "%s: ", MyName);
	Spnt = index (Str, CHNULL);
	(void) sprintf (Spnt, message, arg1, arg2, arg3, arg4);
	if (lflag)
		syslog (LOG_ERR, Str);
	Spnt = index (Str, CHNULL);
	(void) sprintf (Spnt, "\n");
	fputs (Str, stderr);
	Finis ();
} /* Fatal */
/*
 * Malloc -- a malloc with error checking
 *
 *	Parameters:
 *		size -- number of bytes to get
 *	Returns:
 *		(char *) of first char of block, or
 *		calls Fatal() upon error
 *	Side effects:
 *		none
 *	Deficiencies:
 */

char	*malloc ();
char	*Malloc ();

char *
Malloc (size)
unsigned	size;	/* bytes to get */
{
	char	*cp;	/* pointer to memory */

	if (tTd(12,2))
		printf ("Malloc: request %u bytes\n", size);
	if ((cp = malloc (size)) == CPNULL)
	    Fatal ("malloc %u bytes failed", size);
	return (cp);
} /* Malloc */
/*
 * Realloc -- realloc with error checking
 *
 *	Parameters:
 *		ptr -- pointer to existing data
 *		size -- number of bytes to get
 *
 *	Returns:
 *		(char *) of first char of block, or
 *		Fatal() if any error
 *
 *	Side Effects:
 *		none
 */

char *
Realloc (ptr, size)
char		*ptr;
unsigned	size;
{
		char	*cp;		/* pointer to memory */
	extern	char	*realloc();

	if (tTd(12,2))
		printf ("Realloc: request %u bytes\n", size);
	if ((cp = realloc (ptr, size)) == CPNULL)
		Fatal ("realloc %u bytes failed", size);
	return (cp);
}
