/*
 * oed2 -- provide a "nice" front-end to Pat/OED2
 *
 * N.B., program is installed as ox2 on non-UIUC machines.
 *
 * oed2 acts as front-end to the Pat program when used with the OED2 data.
 * Pat is a very fast text searching program with no knowledge of the
 * data's internal structure.  Oed2, however, does know that the OED2 data
 * is made up of entries that have multiple sub-components and provides
 * commands for manipulating those components.
 *
 * Commands to oed2 are a superset of Pat's.  Pure Pat commands are passed
 * through w.o. interpretation.  Oed2 commands are expanded to multiple Pat
 * commands and then passed through to Pat.
 *
 * Oed2 also tags Pat's output with line numbers for reference in later
 * commands.
 *
 * Usage: oed2 [-v] [-d X.Y] [-D] [-l] [-m mode] [-p pager] [-f file] [-t file]
 *	[-s host] [word]
 *
 * -v	Issue version, usage and copyright information.
 *
 * -d X.Y  Set debug/trace for facility X to severity Y
 *
 * -D	Operate as a daemon, disallow local file opens, and any other 
 *	potentially dangerous operation.  Undocumented in the man page and
 *	usage messages.
 *
 * -l	Use syslog for recording errors.  This is undocumented in the man
 *	page and usage messages.
 *
 * -m mode	Printing mode where mode is one of s(imple), r(off), n(roff),
 *		2(vt200), or p(at).  r prints with n/troff commands embedded,
 *		n pipes the former through nroff.  Default is -m n.
 *		Selection mode where mode is one of a(ll), e(tymology),
 *		d(efinitions), and/or q(uotations).
 *
 * -p pager	Select an alternate pager program.  The full pathname must
 *		be used.
 *
 * -f file	Re-direct output to the named file.
 *
 * -t file	Duplicate output to the named file.
 *
 * -s host	Use alternate Pat/OED2 server at host (iff -DREMOTE)
 *
 * word		Non-interactive lookup of "word".
 *
 * Copyright (c) 1990 Paul Pomes
 * Copyright (c) 1990 University of Illinois Board of Trustees
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of Illinois, Urbana.  In addition, redistribution
 * and use must conform to the terms listed in the Copying file in
 * this directory.
 *
 * The name of the University may not be used to endorse or promote products
 * derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#ifndef lint
static char rcsid[] = "@(#)$Header: /usr/local/src/cso/oed2/RCS/main.c,v 2.0.0.3 90/04/23 10:28:24 paul Exp $";
#endif /* lint */

#define		MAIN
#include	"oed2.h"

/* Used to prevent confusion in vi's showmatch function. */
#define		LEFT_BRACE		('{')

/* Release version number (declared in Version.c) */
extern char	Version[];

/* How program was invoked (argv[0]) for error messages */
char		*MyName;

/* Operating as daemon if set (-D) */
int		Daemon = 0;

/* Use syslog for errors if set (-l) */
int		lflag = 0;

/* Trace vector for debugging flags (from sendmail) */
unsigned char	*tTvect;
unsigned char	tTdvect[100];
int		tTsize;

/* How we die.  Assume horribly unless reset */
int		ExitStat = 1;

/* Select and print modes */
unsigned int	Mode = (unsigned int) 0;
unsigned int	PrtMode = (unsigned int) 0;

/* Block of offsets used by prt #N commands, collected by PatCapture() */
unsigned long	PatOffsets[MAX_OFFSETS];

/* Table of active process IDs from fork(). */
int		PidTab[MAX_PIDS];

#ifdef REMOTE
/* What server to contact */
char		Oed2Host[MAXSTR];

/* How to invoke oed2/ox2 */
char *usage[] = {
	"usage: %s [-v] [-d X.Y] [-m mode] [-p pager] [-f outfile] [-t teefile]",
	"\t[-s host] [word]",
	CPNULL,
};

#else /* ! REMOTE */
/* Name of OED2 data file */
char		*Oed2Name = OED2;

/* How to invoke oed2/ox2 */
char *usage[] = {
	"usage: %s [-v] [-d X.Y] [-m mode] [-p pager] [-f outfile] [-t teefile]",
	"\t[word]",
	CPNULL,
};
#endif /* REMOTE */

/* Streams for Pat I/O */
FILE		*ToPat, *FromPat;

/* Alternate output streams */
char		RdrName[MAXSTR];		/* for re-directed output */
FILE		*RdrStrm = FILENULL;		/* for re-directed output */
char		TeeName[MAXSTR];		/* for tee'ed output */
FILE		*TeeStrm = FILENULL;		/* for tee'ed output */

/* What display pager program to use */
char		Pager[MAXSTR];

/* Forward declaration to keep lint and gcc happy */
static		InitPat();

main (argc, argv)
int	argc;
char	**argv;
{
	extern int	optind;		/* from getopt () */
	extern char	*optarg;	/* from getopt () */
	int		option;		/* option "letter" */
	union wait	status;		/* exit status of Pat */
	register char	*Cpnt;		/* fast character pointer */
	int		i;		/* the ubiquitous i */
	time_t		T0, T1;		/* start and end times */
#ifndef	REMOTE
	int		Pid;		/* value from fork() */
	int		P1[2], P2[2];	/* Parent/child pipes for Pat */
#endif	/* REMOTE */
	extern char	*getenv();
	extern unsigned int SetMode();
	extern time_t	time();

	/*
	 * BASENAME:  Full string, or past '/' if any:
	 */

	MyName = ((MyName = rindex (*argv, '/')) == CPNULL) ? *argv : (MyName + 1);
	(void) time (&T0);
#ifdef REMOTE
	(void) strcpy (Oed2Host, PAT_HOST);
#endif	/* REMOTE */

	/* Initialize the PidTab table */
	for (i = 0; i < MAX_PIDS; i++)
		*(PidTab + i) = -1;

	/*
	 * Settle on a pager.  The PAGER environment variable overrules
	 * the default pager defined in oed2.h as MYPAGER.  A -p pager argument
	 * in either OED2/OX2 environment variable or on the oed2 command line
	 * will override MYPAGER.
	 */
	if ((Cpnt = getenv ("PAGER")) != CPNULL && *Cpnt != CHNULL)
		(void) strcpy (Pager, Cpnt);
	else
		(void) strcpy (Pager, MYPAGER);
		
	/*
	 * PARSE ARGUMENTS: do environment first, then command line
	 */

	*RdrName = *TeeName = CHNULL;
	if ((Cpnt = getenv ("OED2")) != CPNULL && *Cpnt != CHNULL) {
		if (ScanArgs (Cpnt, CPNULL) < 0) {
			Error ("main: Format problems with OED2 environment variable, skipping");
			PrtMode = (unsigned int) 0;
		}
	}
	else if ((Cpnt = getenv ("OX2")) != CPNULL && *Cpnt != CHNULL) {
		if (ScanArgs (Cpnt, CPNULL) < 0) {
			Error ("main: Format problems with OX2 environment variable, skipping");
			PrtMode = (unsigned int) 0;
		}
	}


	while ((option = getopt (argc, argv, "vd:Dlm:p:f:t:s:")) != EOF)
		switch (option) {
		    case 'v':
			Usage (1);
			break;

		    case 'd':
			if (! tTd(1,1)) {
				tTsetup (tTdvect, sizeof (tTdvect), "0-99.1");
				tTflag ("");
			}
			tTflag (optarg);
			Error ("%s", Version);
			break;

		    case 'D':
			Daemon++;
			lflag++;
			break;

		    case 'l':
			lflag++;
			break;

		    case 'm':
			switch (*optarg) {

			    /* Formats are mutually exclusive ... */
			    case 's':
			    case 'r':
			    case 'n':
			    case 'p':
			    case '2':
				Mode &= ~FORMAT_MASK;
				Mode |= SetMode (*optarg);
				break;

			    /* while selections are not. */
			    case 'a':
			    case 'e':
			    case 'd':
			    case 'q':
				Mode |= SetMode (*optarg);
				break;

			    default:
				Error ("main: Unknown value (%c) for -m.", *optarg);
				break;
			}
			break;

		    case 'p':
			(void) strcpy (Pager, optarg);
			if (tTd(17,1))
				printf ("main: Pager program is %s\n", Pager);
			break;

		    case 'f':
			(void) strcpy (RdrName, optarg);
			if (tTd(13,1))
				printf ("main: Re-direct file is %s\n", RdrName);
			break;

		    case 't':
			(void) strcpy (TeeName, optarg);
			if (tTd(14,1))
				printf ("main: Tee file is %s\n", TeeName);
			break;

#ifdef	REMOTE
		    case 's':
			(void) strcpy (Oed2Host, optarg);
			if (tTd(2,1))
				printf ("main: Server host is %s\n", Oed2Host);
			break;
#endif	/* REMOTE */

		    default:
			Error ("main: Bad option letter (%c)\n", option);
			Usage (0);
			exit(1);
		}

	if (lflag)
#ifdef LOG_LOCAL4
		openlog(MyName, LOG_PID | LOG_ODELAY, LOG_LOCAL4);
#else /* ! LOG_LOCAL4 */
		openlog(MyName, LOG_PID);
#endif /* LOG_LOCAL4 */

	/* No re-directs or tees if we're a daemon.  Done in PrtCmd() too */
	if (Daemon && (*RdrName != CHNULL || *TeeName != CHNULL)) {
		Error ("main: Can't open re-direct or tee files in remote operation.\n");
		*RdrName = *TeeName = CHNULL;
		Mode &= ~P_FILE;
		Mode &= ~P_TEE;
	}

	/* Open RdrName or TeeName if now non-null */
	if (*RdrName != CHNULL && *TeeName != CHNULL) {
		Error ("main: Can't set both -f %s and -t %s.  Both are ignored.  Check the\nOED2 or OX2 environment variables and command line arguments for conflicts.",
		    RdrName, TeeName);
		*RdrName = *TeeName = CHNULL;
		Mode &= ~P_FILE;
		Mode &= ~P_TEE;
	}
	else if (*RdrName != CHNULL) {
		if ((RdrStrm = fopen (RdrName, "a")) == FILENULL) {
			Error ("main: Can't open %s for appending: %s", RdrName, ERR);
			*RdrName = CHNULL;
			Mode &= ~P_FILE;
		}
		else {
			if (tTd(13,1))
				printf ("main: Re-direct file is %s opened\n", RdrName);
			Mode |= P_FILE;
		}
	}
	else if (*TeeName != CHNULL) {
		if ((TeeStrm = fopen (TeeName, "a")) == FILENULL) {
			Error ("main: Can't open %s for appending: %s", TeeName, ERR);
			*TeeName = CHNULL;
			Mode &= ~P_TEE;
		}
		else {
			if (tTd(14,1))
				printf ("main: Tee file is %s opened.\n", TeeName);
			Mode |= P_TEE;
		}
	}

	/* Only use the Pager if stdout isn't re-directed. */
	if (equal (Pager, "null") || ! isatty (fileno (stdout))) {
		*Pager = CHNULL;
		if (tTd(17,1))
			printf ("main: Null pager or stdout re-directed, Pager disabled\n");
	}

	/* Is it executable? */
	{
		char Cpy[MAXSTR], *CpyPnt;

		(void) strcpy (Cpy, Pager);
		CpyPnt = index (Cpy, ' ');
		if (CpyPnt == CPNULL)
			CpyPnt = index (Cpy, '\t');
		if (CpyPnt != NULL)
			*CpyPnt = CHNULL;
		if (access (Cpy, F_OK) == -1 || access (Cpy, X_OK) == -1) {
			Error ("main: %s: %s\n\tThe full pathname, e.g., /usr/ucb/more, should be used.", Cpy, ERR);
			*Pager = CHNULL;
		}
	}

	/* Skip options */
	argc -= optind;
	argv += optind;

	/*
	 * Rationalize PrtMode settings from environment with Mode settings
	 * from command line.
	 *
	 * Print modes from Mode have precedence since file opens above
	 * modified Mode.  Any SELECT or FORMAT modes in Mode overrule those
	 * in PrtMode.
	 *
	 * These priorities are reversed (PrtMode favored over Mode) in the
	 * PrtCmd() routine.
	 */
	if (tTd(1,1))
		printf ("main: Initial Mode 0x%x, PrtMode 0x%x words before merge\n",
			Mode, PrtMode);
	Mode = ((Mode&FORMAT_MASK) ? (Mode&FORMAT_MASK) : (PrtMode&FORMAT_MASK))
	     | ((Mode&SELECT_MASK) ? (Mode&SELECT_MASK) : (PrtMode&SELECT_MASK))
	     | (Mode&PRINT_MASK);
	if (tTd(1,1))
		printf ("main: Mode word after merge 0x%x\n", Mode);

	/* Set default modes */
	if (! (Mode & FORMAT_MASK))
		Mode |= F_NROFF;
	if (! (Mode & SELECT_MASK))
		Mode |= S_ALL;
	if (! (Mode & PRINT_MASK))
		Mode |= P_STD;
	if (tTd(1,1))
		printf ("main: Mode word after defaults 0x%x\n", Mode);
#ifdef	REMOTE
	if (argc == 0) {

		/* Reach out and frob the remote Pat server. */
		printf ("Contacting pat/OED2 server on %s...", Oed2Host);
		(void) fflush (stdout);
	}
	ContactPat ();
#else	/* ! REMOTE */
	if (argc == 0) {

		/* Reach out and touch Pat directly. */
		printf ("Creating pat/OED2 background process...");
		(void) fflush (stdout);
	}

	/*
	 * Verify that Pat and the data file are here.  This helps to guard
	 * against installation of a version that uses local Pat/OED2 on a
	 * host that should run with -DREMOTE.
	 */
	if (access (PAT_PGM, F_OK) == -1 || access (PAT_PGM, X_OK) == -1) {
		(void) putc ('\n', stderr);
		Fatal ("main: %s not found or not executable.  Check permissions,\n\tPAT_PGM in oed2.h, or whether %s should be re-compiled with -DREMOTE.", PAT_PGM, MyName);
	}
	if (access (Oed2Name, F_OK) == -1 || access (Oed2Name, R_OK) == -1) {
		(void) putc ('\n', stderr);
		Fatal ("main: %s not found.  Check file permissions, OED2 in oed2.h,\n\tor whether %s should be re-compiled with -DREMOTE.",
		    Oed2Name, MyName);
	}

	/* Running locally, set up pipes and vfork */
	if (pipe (P1) == -1)
		Fatal ("pipe (P1) to pat: %s", ERR);
	if (pipe (P2) == -1)
		Fatal ("pipe (P2) to pat: %s", ERR);

	if ((Pid = vfork()) == -1)
		Fatal ("pat vfork(): %s", ERR);

	if (Pid == 0)	/* child */
	{
		/* Close and dup, close and dup */
		if (close (0) == -1)
			Fatal ("Pat close(0): %s", ERR);
		if (fcntl (P1[0], F_DUPFD, 0) == -1)
			Fatal ("Pat fcntl(0): %s", ERR);
		if (close (1) == -1)
			Fatal ("Pat close(1): %s", ERR);
		if (fcntl (P2[1], F_DUPFD, 1) == -1)
			Fatal ("Pat fcntl(1): %s", ERR);

		/* Close all open file descriptors */
		(void) close (P1[0]);
		(void) close (P1[1]);
		(void) close (P2[0]);
		(void) close (P2[1]);

		/* Fire up Pat with the OED2 data */
		execl (PAT_PGM, "pat", Oed2Name, CPNULL);
		(void) putc ('\n', stderr);
		Fatal ("%s execl: %s", PAT_PGM, ERR);
	}

	/* Set non-blocking output from child */
	if (fcntl (P2[0], F_SETFL, FNDELAY) == -1)
		Fatal ("pat fcntl(FNDELAY): %s", ERR);

	/* Assign slots in PidTab indexed by file descriptor */
	PidTab[P1[1]] = PidTab[P2[0]] = Pid;

	/* Assign stream descriptors to pipe components */
	if ((ToPat = fdopen (P1[1], "w")) == FILENULL)
		Fatal ("pat fdopen(P1[1]): %s", ERR);
	(void) setbuf (ToPat, CPNULL);
	if ((FromPat = fdopen (P2[0], "r")) == FILENULL)
		Fatal ("pat fdopen(P2[0]): %s", ERR);
#endif	/* REMOTE */
	InitPat ();
	if (argc == 0)
		printf ("done.\n\n");
	if (argc > 0)
		while (argc--) {
			PrtCmd (*argv, 'l');
			argv++;
		}
	else {
		char	Word[MAXSTR];

		bzero ((char *) PatOffsets, sizeof (PatOffsets));
		PrintFile ("motd");

		/* Loop on standard input */
		while (1) {
			char	*Wp;

			printf ("%s> ", MyName);
			(void) fflush (stdout);
			if (fgets (Word, MAXSTR, stdin) == CPNULL)
				break;
			if ((Wp = index (Word, '\n')) != CPNULL)
				*Wp = CHNULL;
			if (strlen (Word) == 0)
				continue;
			if (equal (Word, "quit") ||
			    equal (Word, "exit") ||
			    equal (Word, "done") ||
			    equal (Word, "stop"))
				break;
			if (equal (Word, "help") || equal (Word, "?")) {
				PrintFile ("help");
				continue;
			}
			if (equal (Word, "man")) {
				char	ManPage[15];

				(void) strcpy (ManPage, MyName);
				(void) strcat (ManPage, ".0");
				PrintFile (ManPage);
				continue;
			}
			if (equal (Word, "license")) {
				PrintFile ("license");
				continue;
			}
			if (equal (Word, "copyright")) {
				PrintFile ("agreement");
				continue;
			}
			if (equal (Word, "motd")) {
				PrintFile ("motd");
				continue;
			}
			if (nequal (Word, "mode", 4)) {
				if (nequal (Word, "modes", 5))
					ModeCmd (&Word[5]);
				else
					ModeCmd (&Word[4]);
				continue;
			}
			if (nequal (Word, "prt ", 4)) {
				PrtCmd (&Word[3], 'l');
				continue;
			}
			if (nequal (Word, "find ", 5)
			 || nequal (Word, "prtl ", 5)) {
				PrtCmd (&Word[4], 'l');
				continue;
			}
			if (nequal (Word, "prtv ", 5)) {
				PrtCmd (&Word[4], 'v');
				continue;
			}
			if (nequal (Word, "prtvl ", 6)
			 || nequal (Word, "prtlv ", 6)) {
				PrtCmd (&Word[5], 'b');
				continue;
			}

			/* Send anything else to pat after #N expansion */
			i = 0;
			if ((Cpnt = index (Word, '#')) != CPNULL) {
				char	WordCpy[MAXSTR];
				char	WordTmp[MAXSTR];

				(void) strncpy (WordTmp, Word, (Cpnt - Word));
				Cpnt++;

				/* Convert any number if present */
				while (*Cpnt >= '0' && *Cpnt <= '9') {
					i = (i * 10) + (*Cpnt - '0');
					Cpnt++;
				}

				/* check for problems */
				if (i == 0) {
					Error ("main: No number follows # in \"%s\"", Word);
					continue;
				}
				if (i > MAX_OFFSETS) {
					Error ("main: #%d out of allowable offset range of 1 to %d", i, MAX_OFFSETS);
					continue;
				}

				if (PatOffsets[i-1] == (unsigned long) 0) {
					Error ("main: No offset assigned to #%d yet", i);
					continue;
				}
				/* Reduce i by 1 to start array at 0 */
				i--;
				(void) sprintf (WordCpy, "%s%lu%s", WordTmp,
				    PatOffsets[i], Cpnt);
				(void) strcpy (Word, WordCpy);
			}

			/* DisAllow resetting of SaveFile */
			if ((Cpnt = index (Word, LEFT_BRACE)) != CPNULL) {
				Cpnt++;

				/* Skip leading whitespace */
				while (isspace (*Cpnt))
					Cpnt++;
				if (nequal (Cpnt, "SaveFile", 8)) {
					Error ("main: SaveFile cannot be reset");
					continue;
				}
			}
			if (tTd(16,1))
				printf ("main: Sending to pat =%s=\n", Word);
			if (fprintf (ToPat, "%s\n", Word) == EOF)
#ifdef REMOTE
				Fatal ("main: Remote pat process died: %s", ERR);
#else /* ! REMOTE */
				Fatal ("main: Child pat process died: %s", ERR);
#endif /* REMOTE */

			/* Capture and print any output */
			(void) PatCapture (0);
		}
	}

	/*
	 * FINISH UP:
	 */
	if (tTd(16,1))
		printf ("main: Sending quit command to pat\n");
	(void) fprintf (ToPat, "quit\n");

	/* Wait for all children */
	while (wait (&status) != -1 )
		;

	/* Close up shop */
	if (FromPat != FILENULL) {
		PidTab[fileno(FromPat)] = -1;
		(void) fclose (FromPat);
	}
	if (ToPat != FILENULL) {
		PidTab[fileno(ToPat)] = -1;
		(void) fclose (ToPat);
	}

	/* Collect statistics, parent and kids */
	(void) time (&T1);
	if (lflag) {
		char		Results[MAXSTR];
		struct rusage	rusage;

		(void) getrusage (RUSAGE_SELF, &rusage);
		(void) sprintf (Results, "Parent: %ld:%02ldr %ld:%02ldu %ld:%02lds %ldKb, ",
		    (long) (T1-T0)/60, (long) (T1-T0)%60,
		    rusage.ru_utime.tv_sec/60, rusage.ru_utime.tv_sec%60,
		    rusage.ru_stime.tv_sec/60, rusage.ru_stime.tv_sec%60,
		    rusage.ru_maxrss);
		(void) getrusage (RUSAGE_CHILDREN, &rusage);
		(void) sprintf (index (Results, CHNULL), "Kids: %ld:%02ldu %ld:%02lds %ldKb",
		    rusage.ru_utime.tv_sec/60, rusage.ru_utime.tv_sec%60,
		    rusage.ru_stime.tv_sec/60, rusage.ru_stime.tv_sec%60,
		    rusage.ru_maxrss);
		syslog (LOG_INFO, Results);
	}
#ifndef REMOTE
	PidTab[0] = 0;
#endif /* ! REMOTE */
	ExitStat = 0;
	Finis ();
} /* main */
/*
 *  InitPat -- Customize pat behavior
 *
 *	Once the remote or local pat process has been successfully started,
 *	customize its behavior for the role of a background process.
 *
 *	Parameters:
 *		None
 *	Returns:
 *		None
 *	Side Effects:
 *		None
 */
static char	*PatParms[] = {
	"{SaveFile \"/dev/null\"}",
	"{PrintMode 2}",
	"{PrintLength 59}",
	CPNULL
};

static
InitPat ()
{
	int	i;
	char	**ParmPnt;
	char	InBlk[MAXBLK+1], *Cpnt;

	/* Read from Pat until we see a prompt */
	for (ParmPnt = PatParms, Cpnt = InBlk, *Cpnt = CHNULL; ;) {
		errno = 0;
		i = fread (Cpnt, 1, MAXBLK, FromPat);
		if (tTd(16,3) && i == 0)
			printf ("InitPat: Read 0 in greeting read, errno %d\n", errno);
		if (i == 0 && (errno == EWOULDBLOCK || errno == 0)) {
			if (ferror (FromPat))
				clearerr (FromPat);
			sleep (1);
			continue;
		}
		Cpnt += i;
		*Cpnt = CHNULL;
		if (tTd(16,3))
			printf ("InitPat: Read =%s=\n", InBlk);
		if (nequal ((Cpnt-3), ">> ", 3)) {
			if (*ParmPnt == CPNULL)
				break;
			if (tTd(16,1))
				printf ("InitPat: Sending to pat =%s=\n", *ParmPnt);
			fprintf (ToPat, "%s\n", *ParmPnt);
			ParmPnt++;
			Cpnt = InBlk;
			*Cpnt = CHNULL;
		}
	}
}
