/*
 * Copyright (c) 1990 Paul Pomes
 * Copyright (c) 1990 University of Illinois Board of Trustees
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of Illinois, Urbana.  In addition, redistribution
 * and use must conform to the terms listed in the Copying file in
 * this directory.
 *
 * The name of the University may not be used to endorse or promote products
 * derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#ifndef lint
static char rcsid[] = "@(#)$Header: /usr/local/src/cso/oed2/RCS/ScanArgs.c,v 2.0.0.2 90/04/23 09:42:33 paul Exp $";
#endif /* lint */

#include	"oed2.h"

/*
 *  ScanArgs -- Set global PrtMode according to argument string
 *
 *	Examine the Src string for -m arguments and set the appropriate
 *	bit in PrtMode.  If -f or -t options are present and Path is
 *	NULL, copy the file argument to the appropriate global variable.
 *	If Path is not NULL, then ScanArgs was invoked by PrtCmd to set
 *	one time arguments.
 *
 *	Parameters:
 *		Src -- character string with arguments
 *		Path -- char pointer to optional filename buffer
 *	Returns:
 *		0 if no problems
 *		-1 if any errors or less than perfect formatted Src string
 *	Side Effects:
 *		Clears and sets global PrtMode.
 *		May change RdrName and TeeName
 *		May copy a pathname to Path if Path != CPNULL.
 *	Deficiencies
 *		Relies on value of Path (equal or not equal to CPNULL) to
 *		determine whether to change some globals.  At program
 *		initialization,  ScanArgs() is called with Path == CPNULL
 *		to process the OED2/OX2 environment variable and is allowed
 *		to change most globals (except -v).  If Path != CPNULL, then
 *		ScanArgs() was called by PrtCmd() and isn't allowed to
 *		change any global other than PrtMode.
 */

ScanArgs (Src, Path)
char	*Src, *Path;
{
	char			*Cpnt;
	char			Cpy[MAXSTR];
	extern char		Version[];
	extern unsigned int	SetMode();
	extern char		*strtok();

	if (tTd(10,1))
		printf ("ScanArgs: Scanning %s\n", Src);

	/* Handle the null string and string w.o. arguments */
	if (Src == CPNULL)
		return (0);
	if (index (Src, '-') == CPNULL)
		return (0);

	/* Make a copy of Src as strtok() will munge it */
	(void) strncpy (Cpy, Src, MAXSTR-1);
	Cpnt = strtok (Cpy, " \t");
	do {
		/* No more arguments or tokens */
		if (Cpnt == CPNULL || *Cpnt != '-')
			return (0);

		/*
		 * Hyphen'ed arguments are single letters only.  Don't let
		 * users develop the bad habit of running them together.
		 */
		if (*(Cpnt+2) != CHNULL) {
			Error ("ScanArgs: Must have spaces between arguments (%c%c) and values (%s),\n\t%s ignored",
				*Cpnt, *(Cpnt+1), (Cpnt+2), Cpnt);
			return (-1);
		}
		switch (*(Cpnt+1)) {

		    case 'd':
			Cpnt = strtok (CPNULL, " \t");
			if (Cpnt == CPNULL) {
				Error ("ScanArgs: No argument found for -d.");
				return (-1);
			}
			if (! tTd(1,1)) {
				tTsetup (tTdvect, sizeof (tTdvect), "0-99.1");
				tTflag ("");
			}
			tTflag (Cpnt);
			Error ("ScanArgs: %s", Version);
			break;

		    case 'l':
			if (Path != CPNULL) {
				Error ("ScanArgs: Can't set syslog'ing with prt command, %s ignored.", Cpnt);
				break;
			}
			lflag++;
			break;

		    case 'v':
			Usage (1);
			break;

		    case 'm':
			Cpnt = strtok (CPNULL, " \t");
			if (Cpnt == CPNULL) {
				Error ("ScanArgs: No argument found for -m.");
				return (-1);
			}
			*Cpnt = (isupper(*Cpnt)) ? tolower(*Cpnt) : *Cpnt;
			if (tTd(10,2))
				printf ("ScanArgs: Switch with %c\n", *Cpnt);
			switch (*Cpnt) {

			    /* Formats are mutually exclusive ... */
			    case 's':
			    case 'r':
			    case 'n':
			    case 'p':
			    case '2':
				PrtMode &= ~FORMAT_MASK;
				PrtMode |= SetMode (*Cpnt);
				break;

			    /* while selections are not. */
			    case 'a':
			    case 'e':
			    case 'd':
			    case 'q':
				PrtMode |= SetMode (*Cpnt);
				break;

			    default:
				Error ("ScanArgs: Unknown mode character (%c)", *Cpnt);
				break;
			}
			break;

		    case 'p':
			Cpnt = strtok (CPNULL, " \t");
			if (Cpnt == CPNULL) {
				Error ("ScanArgs: No argument found for -p.");
				return (-1);
			}
			if (Path != CPNULL) {
				Error ("ScanArgs: Can't set pager with prt command, -p %s ignored.", Cpnt);
				break;
			}
			(void) strcpy (Pager, Cpnt);
			if (tTd(17,1))
				printf ("ScanArgs: Pager program is %s\n", Pager);
			break;

		    case 'f':
			Cpnt = strtok (CPNULL, " \t");
			if (Cpnt == CPNULL) {
				Error ("ScanArgs: No argument found for -f.");
				return (-1);
			}
			if (equal (Cpnt, "null")) {
				Error ("ScanArgs: Use \"mode file null\" to turn off re-direction, -f null ignored.");
				break;
			}
			if (Path != CPNULL) {
				(void) strcpy (Path, Cpnt);
				if (tTd(10,1))
					printf ("ScanArgs: Path =%s=\n", Path);
				PrtMode |= P_PFILE;
			}
			else {
				(void) strcpy (RdrName, Cpnt);
				if (tTd(10,1) || tTd(13,1))
					printf ("ScanArgs: RdrName =%s=\n", RdrName);
				PrtMode |= P_FILE;
			}
			break;

		    case 't':
			Cpnt = strtok (CPNULL, " \t");
			if (Cpnt == CPNULL) {
				Error ("ScanArgs: No argument found for -t.");
				return (-1);
			}
			if (Path != CPNULL) {
				Error ("ScanArgs: Can't set tee file with prt command (use -f file),\n\t-t %s ignored.", Cpnt);
				break;
			}
			if (equal (Cpnt, "null")) {
				Error ("ScanArgs: Use \"mode tee null\" to turn off tee'ing, -t null ignored.");
				break;
			}
			(void) strcpy (TeeName, Cpnt);
			if (tTd(10,1) || tTd(14,1))
				printf ("ScanArgs: TeeName =%s=\n", TeeName);
			PrtMode |= P_TEE;
			break;

#ifdef REMOTE
		    case 's':
			Cpnt = strtok (CPNULL, " \t");
			if (Cpnt == CPNULL) {
				Error ("ScanArgs: No argument found for -s.");
				return (-1);
			}
			if (Path != CPNULL) {
				Error ("ScanArgs: Can't set server with prt command, -s %s ignored.", Cpnt);
				break;
			}
			(void) strcpy (Oed2Host, Cpnt);
			if (tTd(2,1))
				printf ("ScanArgs: Server host is %s\n", Oed2Host);
			break;
#endif /* REMOTE */

		    default:
			Error ("ScanArgs: Bad argument %s.", Cpnt);
			return (-1);
		}
	} while ((Cpnt = strtok (CPNULL, " \t")) != CPNULL);
	return (0);
} /* ScanArgs */
