/*
 * Copyright (c) 1990 Paul Pomes
 * Copyright (c) 1990 University of Illinois Board of Trustees
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of Illinois, Urbana.  In addition, redistribution
 * and use must conform to the terms listed in the Copying file in
 * this directory.
 *
 * The name of the University may not be used to endorse or promote products
 * derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#ifndef lint
static char rcsid[] = "@(#)$Header: /usr/local/src/cso/oed2/RCS/PrtCmd.c,v 2.0 90/03/10 12:19:36 paul Exp $";
#endif /* lint */

#include	"oed2.h"
#include	<setjmp.h>

/*
 *  PrtCmd -- Handle the oed2 "prt" command
 *
 *	Examine the command string for any temporary mode settings
 *	then format and send a lookup query to pat.
 *
 *	Parameters:
 *		FromUser - pointer to char string with user's command.
 *		SearchType - char value of search to do.
 *		  l: use <LF>, v: use <VF>, b: use <LF> + <VF>
 *	Returns:
 *		 None
 *	Side Effects:
 *		Any mode flags will modify global PrtMode
 *		Exits if any errors
 */

/*
 * How to print output: PRINTF handles does printf according to the
 * following priority rules:  If PrtStrm is non-null, all output goes
 * to it only.  It PrtStrm is null, then examine RdrStrm and TeeStrm.
 * If RdrStrm is non-null, print to it only.  If TeeStrm is non-null,
 * print to both it and stdout.
 */
#define		PRINTF(X,Y)		\
{ \
	if (PrtStrm != FILENULL) \
		fprintf (PrtStrm, X,Y); \
	else { \
		if (RdrStrm != FILENULL) \
			fprintf (RdrStrm, X,Y); \
		else { \
			if (TeeStrm != FILENULL) \
				fprintf (TeeStrm, X,Y); \
			if (PagerStrm != FILENULL) { \
				fprintf (PagerStrm, X,Y); \
			} \
			else \
				fprintf (stdout, X,Y); \
		} \
	} \
}

#define		PAGER_OPEN	\
{ \
	if (*Pager != CHNULL) { \
		if ((PagerStrm = PagerOpen (Pager)) == FILENULL) { \
			Error ("PrtCmd: PagerOpen(%s): %s", Pager, ERR); \
			*Pager = CHNULL; \
		} \
		if (signal (SIGPIPE, SIG_IGN) != SIG_IGN) \
			(void) signal (SIGPIPE, DeadPipe); \
		if (signal (SIGINT, SIG_IGN) != SIG_IGN) \
			(void) signal (SIGINT, DeadPipe); \
	} \
}

#define		PAGER_CLOSE	\
{ \
	if (PagerStrm != FILENULL) { \
		(void) fflush (PagerStrm); \
		i = PagerClose (PagerStrm); \
		if (tTd(17,1)) \
			printf ("PrtCmd: PagerClose (Pager) returned %d\n", i); \
		if (signal (SIGPIPE, SIG_IGN) != SIG_IGN) \
			(void) signal (SIGPIPE, SIG_DFL); \
		if (signal (SIGINT, SIG_IGN) != SIG_IGN) \
			(void) signal (SIGINT, SIG_DFL); \
	} \
}

/* Prepare for SIGPIPE */
static jmp_buf	env;

/* Forward declaration to keep gcc and lint happy. */
static		DeadPipe();

PrtCmd (FromUser, SearchType)
char	*FromUser, SearchType;
{
	char		*Cpnt;		/* J random char pointer */
	int		MatchCnt;	/* number of matches reported by Pat */
	int		i;		/* good ol' i */
	int		LoopCnt = 0;	/* counter for match printing */
	int		SkipNext;	/* skip next token if set */
	int		BaseIndex;	/* First PatOffsets entry for match */
	char		*av[10];	/* argument vector for PipeExec() */
	char		*MatchRaw;	/* pointer for LoadMatch() return */
	char		*MatchFmt = CPNULL; /* pointer for FmtEntry() return */
	char		PrtName[MAXSTR]; /* possible output file for -f */
	char		SaveString[MAXSTR+1]; /* first part of previous match */
	FILE		*PrtStrm = FILENULL;	/* output file pointer */
	FILE		*PagerStrm = FILENULL;	/* pager file pointer */
	extern char	*FmtEntry();
	extern char	*LoadMatch();
	extern char	*PipeExec();
	extern char	*strtok();
	extern FILE	*PagerOpen();

	if (tTd(9,1))
		printf ("PrtCmd: Input =%s=\n", FromUser);
	*PrtName = CHNULL;

	/* Scan for arguments after initializing PrtMode */
	PrtMode = (unsigned int) 0;
	if (ScanArgs (FromUser, PrtName) < 0)
		return;

	/* Fill out PrtMode with pieces parts of Mode (fused!) */
	PrtMode = ((PrtMode&FORMAT_MASK) ? (PrtMode&FORMAT_MASK) : (Mode&FORMAT_MASK))
	  | ((PrtMode&SELECT_MASK) ? (PrtMode&SELECT_MASK) : (Mode&SELECT_MASK))
	  | ((PrtMode&PRINT_MASK) ? (PrtMode&PRINT_MASK) : (Mode&PRINT_MASK));
	if (tTd(9,1))
		printf ("PrtCmd: PrtMode 0x%x\n", PrtMode);

	/*
	 * Is there a word to look up?  Skip over any arguments.
	 * This routine assumes a well formatted argument string since
	 * ScanArgs() has already passed on it.
	 */
	for (SkipNext = 0, Cpnt = FromUser; *Cpnt != CHNULL; Cpnt++) {
		if (isspace(*Cpnt))
			continue;
		if (*Cpnt != '-')
			break;
		Cpnt++;
		switch (*Cpnt) {
		    case 'f':
		    case 'm':
		    case 't':
		    case 'd':
			SkipNext++;
			break;

		    default:
			SkipNext = 0;
			break;
		}
		while (! isspace(*Cpnt))
			Cpnt++;
		if (! SkipNext)
			continue;
		while (isspace(*Cpnt))
			Cpnt++;
		while (! isspace(*Cpnt))
			Cpnt++;
	}
		
	if (Cpnt == CPNULL || *Cpnt == CHNULL) {
		Error ("PrtCmd: Please specify a search word\n");
		return;
	}

	/* No re-directs if we're a daemon. */
	if (Daemon && *PrtName != CHNULL) {
		Error ("PrtCmd: Can't open re-direct file in remote operation.\n");
		*PrtName = CHNULL;
		PrtMode &= ~P_PFILE;
	}

	/* Use a temporary file? */
	if (*PrtName != CHNULL) {
		if (tTd(15,1))
			printf ("PrtCmd: Opening PrtName %s\n", PrtName);
		if ((PrtStrm = fopen (PrtName, "a")) == FILENULL) {
			Error ("Can't open %s for appending: %s", PrtName, ERR);
			*PrtName = CHNULL;
			PrtMode &= ~P_PFILE;
		}
	}
	*SaveString = CHNULL;

	/* Iterate over lookup arguments */
	while (*Cpnt != CHNULL) {
		char	Word[MAXSTR];
		char	SendStr[MAXSTR];
		char	*Mpnt;

		while (isspace(*Cpnt))
			Cpnt++;
		if (*Cpnt == CHNULL)
			break;

		/* Handle a quoted phrase.  Leave the leading " as a flag. */
		if (*Cpnt == '"') {
			Mpnt = Word;
			*Mpnt++ = *Cpnt++;
			while (*Cpnt != CHNULL) {
				if (*Cpnt == '"') {
					*Mpnt = CHNULL;
					break;
				}
				*Mpnt++ = *Cpnt++;
			}
			if (*Cpnt != '"') {
				Error ("PrtCmd: Unbalanced quotes around =%s=", Word);
				break;
			}
			else
				Cpnt++;
			if (tTd(9,1))
				printf ("PrtCmd: Looking up =%s=\n", Word);
		}
		else {
			/* Single word lookup */
			Mpnt = Word;
			while (*Cpnt != CHNULL && *Cpnt != ' ')
				*Mpnt++ = *Cpnt++;
			*Mpnt = CHNULL;
			if (tTd(9,1))
				printf ("PrtCmd: Looking up =%s=\n", Word);
			i = 0;

			/* Handle #N lookups */
			if ((Mpnt = index (Word, '#')) != CPNULL) {
				Mpnt++;

				/* Convert any number if present */
				while (*Mpnt >= '0' && *Mpnt <= '9') {
					i = (i * 10) + (*Mpnt - '0');
					Mpnt++;
				}

				/* check for problems */
				if (i == 0) {
					Error ("PrtCmd: No number follows # in \"%s\"", Word);
					continue;
				}
				if (i > MAX_OFFSETS) {
					Error ("PrtCmd: #%d out of allowable offset range of 1 to %d", i, MAX_OFFSETS);
					continue;
				}

				if (PatOffsets[i-1] == (unsigned long) 0) {
					Error ("PrtCmd: No offset assigned to #%d yet", i);
					continue;
				}
				/* Reduce i by 1 to start array at 0 */
				i--;
				(void) sprintf (Word, "[%lu]", PatOffsets[i]);
			}
		}
		if (*Word == '[')
			(void) sprintf (SendStr, "docs E including %s\n", Word);
		else if (*Word == '"') {
			switch (SearchType) {
			    case 'l':
				(void) sprintf (SendStr, "docs E including \"<LF>%s\"\n", &Word[1]);
				break;

			    case 'v':
				(void) sprintf (SendStr, "docs E including \"<VF>%s\"\n", &Word[1]);
				break;

			    case 'b':
				(void) sprintf (SendStr, "docs E including ( \"<LF>%s\" + \"<VF>%s\")\n", &Word[1], &Word[1]);
				break;
			
			    default:
				Fatal ("PrtCmd: Unknown SearchType (%c)", SearchType);
			}
		}
		else {
			switch (SearchType) {
			    case 'l':
				(void) sprintf (SendStr, "docs E including (\"<LF>%s</LF>\" within docs HL)\n", Word);
				break;

			    case 'v':
				(void) sprintf (SendStr, "docs E including (\"<VF>%s</VF>\" within docs VL)\n", Word);
				break;

			    case 'b':
				(void) sprintf (SendStr, "docs E including ((\"<LF>%s</LF>\" within docs HL) + (\"<VF>%s</VF>\" within docs VL))\n", Word, Word);
				break;
			
			    default:
				Fatal ("PrtCmd: Unknown SearchType (%c)", SearchType);
			}
		}
		if (tTd(16,1))
			printf ("PrtCmd: Sending to pat =%s=\n", SendStr);
		if (fprintf (ToPat, "%s", SendStr) == EOF)
#ifdef REMOTE
			Fatal ("PrtCmd: Remote pat process died: %s", ERR);
#else /* ! REMOTE */
			Fatal ("PrtCmd: Child pat process died: %s", ERR);
#endif /* REMOTE */

		/* See how many matches resulted.  More than 20 lose. */
		if ((MatchCnt = GetCount ()) == 0) {
			Error ("%s%s not found in OED2\n",
			    Word, (*Word == '"') ? "\"" : "");
			continue;
		}
		if (MatchCnt >= MAX_MATCH) {
			Error ("Too many matches (%d >= %d), skipping",
			    MatchCnt, MAX_MATCH);
			continue;
		}
		if (tTd(9,1))
			printf ("PrtCmd: Found %d matches\n", MatchCnt);
		if (MatchCnt > 1)
			PRINTF ("%d matches\n", MatchCnt)
		else
			PRINTF ("%d match\n", MatchCnt)

		/* Issue a pr command to collect the offsets via PatCapture */
		if (tTd(16,1))
			printf ("PrtCmd: Sending to pat =pr\n=\n");
		if (fprintf (ToPat, "pr\n") == EOF)
#ifdef REMOTE
			Fatal ("PrtCmd: Remote pat process died: %s", ERR);
#else /* ! REMOTE */
			Fatal ("PrtCmd: Child pat process died: %s", ERR);
#endif /* REMOTE */
		if ((BaseIndex = PatCapture (1)) == -1)
			Fatal ("PrtCmd: PatCapture returned -1 for %s", Word);

		/*
		 * If the user types a q to more(1) while it's midstream,
		 * SIGPIPE is generated.  Assume the user has seen enough
		 * and empty the pipeline from pat into the bit bucket.
		 */
		if (setjmp (env)) {
			PAGER_CLOSE
			if (MatchFmt != CPNULL)
				(void) free (MatchFmt);
			return;
		}
		PAGER_OPEN
		for (LoopCnt = 1; LoopCnt <= MatchCnt; LoopCnt++) {
			int	TmpIndex;

			/* Generate a single OED2 entry */
			TmpIndex = (BaseIndex + (LoopCnt-1)) % MAX_OFFSETS;
			if (tTd(16,1))
				printf ("PrtCmd: Sending to pat =pr.docs.E [%lu]\n=\n",
					PatOffsets[TmpIndex]);
			if (fprintf (ToPat, "pr.docs.E [%lu]\n", PatOffsets[TmpIndex]) == EOF)
#ifdef REMOTE
				Fatal ("PrtCmd: Remote pat process died: %s", ERR);
#else /* ! REMOTE */
				Fatal ("PrtCmd: Child pat process died: %s", ERR);
#endif /* REMOTE */

			/* Collect the entry and format it. */
			MatchRaw = LoadMatch (1);
			if (strncmp (MatchRaw, SaveString, MAXSTR))
				(void) strncpy (SaveString, MatchRaw, MAXSTR);
			else {
				Error ("\nMatch %d duplicate, suppressed\n", LoopCnt);
				(void) free (MatchRaw);
				continue;
			}
			MatchFmt = FmtEntry (MatchRaw);
			if (PrtMode & F_NROFF) {

				/*
				 * Exec the best nroff around.  The two below
				 * overstrike for bold characters, and
				 * backspace/underline for italics.  This shows
				 * up great when viewed with less(1).
				 */
#if defined(sequent)
				av[0] = "/bin/att";
				av[1] = "att";
				av[2] = "nroff";
				av[3] = CPNULL;
#else /* ! sequent */
# if defined(mips)
			/* Brain-dead MIPS programmers moved things around */
				av[0] = "/usr/bsd43/bin/nroff";
# else /* ! mips */
				av[0] = "/usr/bin/nroff";
# endif /* mips */
				av[1] = "nroff";
				av[2] = "-Tcrt";
				av[3] = CPNULL;
#endif	/* sequent or whatevers */
				MatchFmt = PipeExec (av, MatchFmt);
			}

			/* Warning: do NOT place PipeExec between PAGER_* */
			if (MatchCnt > 1)
				PRINTF ("\nMatch %d\n", LoopCnt)
			PRINTF ("%s", MatchFmt)
			(void) free (MatchFmt);
			MatchFmt = CPNULL;
		}
		PAGER_CLOSE
	}
	if (PrtStrm != FILENULL) {
		if (tTd(15,1))
			printf ("PrtCmd: Closing PrtName %s\n", PrtName);
		(void) fflush (PrtStrm);
		(void) fclose (PrtStrm);
	}
} /* PrtCmd */
/*
 *  DeadPipe -- Handle a dead pipe or interrupt signal
 *
 *	Parameters:
 *		FromUser - pointer to char string with user's command.
 *	Returns:
 *		 None
 *	Side Effects:
 *		Executes a longjmp
 */

static
DeadPipe ()
{
	longjmp (env, 1);
}
