/*
 * Copyright (c) 1990 Paul Pomes
 * Copyright (c) 1990 University of Illinois Board of Trustees
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of Illinois, Urbana.  In addition, redistribution
 * and use must conform to the terms listed in the Copying file in
 * this directory.
 *
 * The name of the University may not be used to endorse or promote products
 * derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#ifndef lint
static char rcsid[] = "@(#)$Header: /usr/local/src/cso/oed2/RCS/PipeExec.c,v 2.0 90/03/10 12:19:34 paul Exp $";
#endif /* lint */

#include	"oed2.h"

/*
 * PipeExec -- Send a buffer through a pipe, store the output in another
 *
 *	Parameters:
 *		Pgm -- array of char pointer with name of program to execute
 *			and its arguments.
 *		Txt -- char pointer to input block, null terminated
 *	Returns:
 *		Pointer to the output read from the child
 *	side effects:
 *		Forks and execv's Pgm
 *		Forks a child to write to Pgm process
 *		Allocates a new malloc'ed buffer
 *		Frees space pointed to by the Txt pointer
 *	deficiencies:
 *
 */

char *
PipeExec (Pgm, Txt)
char	*Pgm[], *Txt;
{
	int		P1[2];			/* pipe for child */
	int		P2[2];			/* pipe for child */
	int		PgmPid;			/* Pgm child's pid */
	int		ChildPid;		/* PipeExec child's pid */
	int		Len;			/* length of Txt block */
	int		i, j;			/* good ol' i and friend */
	char		*InTxt;			/* Txt pointer */
	int		InLen;			/* how much to fwrite */
	char		*Entry = CPNULL;	/* loaded entry block */
	char		*EntryPnt = CPNULL;	/* pointer into entry block */
	unsigned int	Size = MAXBLK;		/* requested size of Entry */
	int		ReadCnt = 0;		/* number of characters read */
	union wait	status;			/* exit status of child */
	extern int	Finis();
	extern char	*Malloc(), *Realloc();

	/* Basic error checking. */
	if (Pgm[0] == CPNULL || *Pgm[0] == CHNULL)
		Fatal ("PipeExec: No program specified");
	if (access (Pgm[0], F_OK) == -1 || access (Pgm[0], X_OK) == -1)
		Fatal ("PipeExec: %s: %s", Pgm[0], ERR);
	if (tTd(8,1))
		printf ("PipeExec: Creating pipe to %s\n", Pgm[0]);

	/* Flush pending I/O, set up pipes and fork */
	(void) fflush (stdout);
	if (pipe (P1) == -1)
		Fatal ("PipeExec: pipe (P1): %s", ERR);
	if (pipe (P2) == -1)
		Fatal ("PipeExec: pipe (P2): %s", ERR);
	if ((PgmPid = vfork()) == -1)
		Fatal ("PipeExec: Pgm vfork (): %s", ERR);
	if (tTd(8,1))
		printf ("PipeExec: PgmPid %d (0 in child)\n", PgmPid);

	if (PgmPid == 0)	/* child */
	{
		/* Close and dup */
		if (close (0) == -1)
			Fatal ("PipeExec: close (0): %s", ERR);
		if (fcntl (P1[0], F_DUPFD, 0) == -1)
			Fatal ("PipeExec: fcntl (0): %s", ERR);

		/* Do it again, Sam */
		if (close (1) == -1)
			Fatal ("PipeExec: close (1): %s", ERR);
		if (fcntl (P2[1], F_DUPFD, 1) == -1)
			Fatal ("PipeExec: fcntl (1): %s", ERR);

		/* Close all open file descriptors */
		(void) close (P1[0]);
		(void) close (P1[1]);
		(void) close (P2[0]);
		(void) close (P2[1]);

		execv (Pgm[0], &Pgm[1]);
		Warning ("PipeExec: execv(\"%s\"): %s", Pgm[0], ERR);
		_exit (1);
	}

	/* Close the descriptors not used in the parent. */
	if (close (P1[0]) == -1)
		Fatal ("PipeExec: close (P1[0]) after Pgm vfork(): %s", ERR);
	if (close (P2[1]) == -1)
		Fatal ("PipeExec: close (P2[1]) after Pgm vfork(): %s", ERR);

	/* Assign slots in PidTab indexed by file descriptor */
	PidTab[P1[1]] = PgmPid;

	/* Setup to write the Txt block to the child */
	Len = strlen (Txt);
	InTxt = Txt;
	InLen = (Len > MAXBLK) ? MAXBLK : Len;

	/* Fork again and let child write to Pgm while parent reads. */
	(void) fflush (stdout);
	if ((ChildPid = fork()) == -1)
		Fatal ("PipeExec: child fork (): %s", ERR);
	if (tTd(8,1))
		printf ("PipeExec: ChildPid %d (0 in child)\n", ChildPid);

	if (ChildPid == 0)	/* child */
	{
		/* Close the descriptors not used in the child. */
		if (close (P2[0]) == -1)
			Fatal ("PipeExec: close (P2[0]) in Child: %s", ERR);

		for (;;) {
			if ((i = write (P1[1], InTxt, InLen)) == -1)
				Fatal ("PipeExec: write to Pgm, %s", ERR);
			Len -= i;
			if (tTd(8,2))
				printf ("PipeExec: Tried %d bytes, wrote %d\n",
					InLen, i);
			if (Len > 0) {
				InTxt += i;
				InLen = (Len > MAXBLK) ? MAXBLK : Len;
			}
			else {
				(void) free (Txt);
				break;
			}
		}
		if (tTd(8,1))
			printf ("PipeExec: Child exit\n");
		_exit(0);
	}

	/* Close the descriptors not used in the parent. */
	if (close (P1[1]) == -1)
		Fatal ("PipeExec: close (P1[1]) in parent: %s", ERR);

	/* Assign slots in PidTab indexed by file descriptor */
	PidTab[P2[0]] = ChildPid;

	/* Setup for read from Pgm. */
	EntryPnt = Entry = Malloc (Size);
	for (;;) {
		int	k;

		k = wait3 (&status, WNOHANG, (struct rusage *) NULL);
		if (tTd(8,1))
			printf ("PipeExec: wait3 got %d\n", k);
		if (k == ChildPid) {
			PidTab[P2[0]] = -1;
			ChildPid = -2;
			continue;
		}
		else if (k == PgmPid) {
			PidTab[P1[1]] = -1;
			PgmPid = -2;
		}
		else if (k > 0) {
			for (i = 0; i < MAX_PIDS; i++)
				if (PidTab[i] == k)
					PidTab[i] = -1;
		}
#ifdef REMOTE
		else if (k == -1)
			Fatal ("PipeExec: wait3 returned -1 (-DREMOTE)");
#else /* ! REMOTE */
		else if (k == -1)
			Fatal ("PipeExec: wait3 returned -1 (what about pat?)");
#endif /* REMOTE */
		if (ReadCnt == (MAXBLK - 1)) {
			j = EntryPnt - Entry;
			Entry = Realloc (Entry, (Size += ((unsigned) MAXBLK)));
			EntryPnt = Entry + j;
			ReadCnt = 0;
		}
		if ((j = read (P2[0], EntryPnt, ((MAXBLK-1)-ReadCnt))) == EOF)
			break;
		if (j == 0) {
			if (PgmPid == -2 || PidTab[P2[0]] == -1)
				break;
			else
				sleep (1);
		}
		if (tTd(8,2))
			printf ("PipeExec: Tried %d bytes, read %d\n",
				((MAXBLK - 1) - ReadCnt), j);
		ReadCnt += j;
		EntryPnt += j;
		*EntryPnt = CHNULL;
	}

	if (tTd(8,3)) {
		printf ("PipeExec: Result =%s=\n", Entry);
		(void) fflush (stdout);
	}
	return (Entry);
} /* PipeExec */
