/*
 * Copyright (c) 1990 Paul Pomes
 * Copyright (c) 1990 University of Illinois Board of Trustees
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of Illinois, Urbana.  In addition, redistribution
 * and use must conform to the terms listed in the Copying file in
 * this directory.
 *
 * The name of the University may not be used to endorse or promote products
 * derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#ifndef lint
static char rcsid[] = "@(#)$Header: /usr/local/src/cso/oed2/RCS/PatCapture.c,v 2.0 90/03/10 12:19:32 paul Exp $";
#endif /* lint */

#include	"oed2.h"
#include	<setjmp.h>

/*
 *  PatCapture -- Capture output from pat, label offsets
 *
 *	Examine the output returned by pat for patterns of the form
 *	"\n[ ]12323, ...".  Store the number returned in a circular
 *	buffer and augment the number in the text with #<index> where
 *	<index> is the array position of the number stored.  If 12323
 *	was stored in PatOffsets[234], the output would look like 
 *	"\n #234: 12323, ...".
 *
 *	Parameters:
 *		Silent - set if no output is to be produced.
 *	Returns:
 *		Index of first new entry into PatOffsets or
 *		-1 if none are found.
 *	Side Effects:
 *		Modifies the global PatPntr.
 */

/*
 * How to print output: PRINTF handles does printf according to the
 * following priority rules:  If RdrStrm is non-null, print to it only.
 * If TeeStrm is non-null, print to both it and stdout.
 */
#define		PRINTF(X,Y,Z)		\
{ \
	if (RdrStrm != FILENULL) \
		fprintf (RdrStrm, X,Y,Z); \
	else { \
		if (TeeStrm != FILENULL) \
			fprintf (TeeStrm, X,Y,Z); \
		if (PagerStrm != FILENULL) \
			fprintf (PagerStrm, X,Y,Z); \
		else \
			fprintf (stdout, X,Y,Z); \
	} \
}

#define		PAGER_OPEN	\
{ \
	if (*Pager != CHNULL) { \
		(void) fflush (stdout); \
		if ((PagerStrm = PagerOpen (Pager)) == FILENULL) { \
			Error ("PatCapture: PagerOpen(%s): %s", Pager, ERR); \
			*Pager = CHNULL; \
		} \
		if (signal (SIGPIPE, SIG_IGN) != SIG_IGN) \
			(void) signal (SIGPIPE, DeadPipe); \
		if (signal (SIGINT, SIG_IGN) != SIG_IGN) \
			(void) signal (SIGINT, DeadPipe); \
	} \
}

#define		PAGER_CLOSE	\
{ \
	if (PagerStrm != FILENULL) { \
		(void) fflush (PagerStrm); \
		i = PagerClose (PagerStrm); \
		if (tTd(17,1)) \
			printf ("PatCapture: PagerClose (Pager) returned %d\n", i); \
		if (signal (SIGPIPE, SIG_IGN) != SIG_IGN) \
			(void) signal (SIGPIPE, SIG_DFL); \
		if (signal (SIGINT, SIG_IGN) != SIG_IGN) \
			(void) signal (SIGINT, SIG_DFL); \
	} \
}

/* Prepare for SIGPIPE */
static jmp_buf	env;

/* Forward declaration to keep gcc and lint happy. */
static		DeadPipe();

PatCapture (Silent)
int	Silent;
{
	static int	PatIndex = 0;		/* current open index */
	int		RetIndex;		/* PatIndex value on entry */
	static char	*Entry = CPNULL;	/* loaded entry block */
	char		*EntryPnt = CPNULL;	/* pointer into entry block */
	char		*LastPnt;		/* position of newline char */
	register char	*Cpnt;			/* working char pointer */
	unsigned int	Size = MAXBLK;		/* requested size of Entry */
	unsigned long	TmpOffset;		/* working copy of offset */
	int		ReadCnt = 0;		/* number of characters read */
	int		i, j;			/* good ol' i and friend */
	FILE		*PagerStrm = FILENULL;	/* for terminal display */
	extern char	*Malloc(), *Realloc();
	extern FILE	*PagerOpen();

	RetIndex = PatIndex;

	/* Malloc an initial chunk. */
	EntryPnt = Entry = Malloc (Size);

	/* Fill'er up.  Stop when a newline is found. */
	for (;;) {
		if (ReadCnt == (MAXBLK - 1)) {
			i = EntryPnt - Entry;
			Entry = Realloc (Entry, (Size += ((unsigned) MAXBLK)));
			EntryPnt = Entry + i;
			ReadCnt = 0;
		}
		errno = 0;
		j = fread (EntryPnt, 1, ((MAXBLK - 1) - ReadCnt), FromPat);
		if (j == 0) {
			if (errno == EWOULDBLOCK || errno == 0) {
				if (ferror (FromPat))
					clearerr (FromPat);
				sleep (1);
				continue;
			}
			else
				Fatal ("PatCapture: fread: %s", ERR);
		}
		ReadCnt += j;
		EntryPnt += j;
		*EntryPnt = CHNULL;
		if (nequal ((EntryPnt-3), ">> ", 3)) {
			if (tTd(7,3) || tTd(16,3)) {
				printf ("PatCapture: Pat returned: =%s=\n", Entry);
				(void) fflush (stdout);
			}
			EntryPnt -= 3;
			*EntryPnt = CHNULL;
			ReadCnt = EntryPnt - Entry;
			break;
		}
	}

	/*
	 * Read a line at a time from the Entry buffer.  Examine the 
	 * start of each for optional whitespace, then digits followed by
	 * a comma.  Store the number represented by digits in PatOffsets
	 * and print the string with digits replaced by #<offset> where
	 * <offset> is the index into PatOffsets for the number.
	 */
	if (Entry == CPNULL || *Entry == CHNULL)
		return (-1);
	if (setjmp (env)) {
		(void) free (Entry);
		PAGER_CLOSE
		return ((RetIndex == PatIndex) ? -1 : RetIndex);
	}
	if (! Silent)
		PAGER_OPEN
	for (EntryPnt = Entry; EntryPnt < (Entry + ReadCnt);) {
		TmpOffset = (unsigned long) 0;
		if ((LastPnt = index (EntryPnt, '\n')) == CPNULL) {
			Error ("PatCapture: No terminal newline");
			break;
		}
		*LastPnt = CHNULL;
		Cpnt = EntryPnt;

		/* Skip leading whitespace */
		while (isspace (*Cpnt))
			Cpnt++;

		/* Convert any number if present */
		while (*Cpnt >= '0' && *Cpnt <= '9') {
			TmpOffset = (TmpOffset * (unsigned long)10)
				  + (*Cpnt - '0');
			Cpnt++;
		}

		/*
		 * Only if the number was followed by a comma does
		 * PatOffsets get updated.
		 */
		if (TmpOffset > 0 && *Cpnt == ',') {
			PatOffsets[PatIndex] = TmpOffset;
			if (! Silent)
				PRINTF("#%d %s\n", PatIndex+1, EntryPnt)
			PatIndex++;
			PatIndex = PatIndex % MAX_OFFSETS;
		}
		else if (! Silent)
			PRINTF("%s%s", EntryPnt, "\n")
		EntryPnt = LastPnt + 1;
	}
	PAGER_CLOSE
	(void) free (Entry);
	return ((RetIndex == PatIndex) ? -1 : RetIndex);
} /* PatCapture */
/*
 *  DeadPipe -- Handle a dead pipe or interrupt signal
 *
 *	Parameters:
 *		FromUser - pointer to char string with user's command.
 *	Returns:
 *		 None
 *	Side Effects:
 *		Executes a longjmp
 */

static
DeadPipe ()
{
	longjmp (env, 1);
}
