/*
 * Copyright (c) 1990 Paul Pomes
 * Copyright (c) 1990 University of Illinois Board of Trustees
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of Illinois, Urbana.  In addition, redistribution
 * and use must conform to the terms listed in the Copying file in
 * this directory.
 *
 * The name of the University may not be used to endorse or promote products
 * derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#ifndef lint
static char rcsid[] = "@(#)$Header: /usr/local/src/cso/oed2/RCS/ModeCmd.c,v 2.0 90/03/10 12:19:30 paul Exp $";
#endif /* lint */

#include	"oed2.h"

/*
 *  ModeCmd -- Handle the oed2 "mode" command
 *
 *	Examine the argument string for valid mode settings and set the
 *	global Mode variable.  Resolve conflicts in settings.
 *
 *	Parameters:
 *		FromUser - pointer to char string with user's command.
 *	Returns:
 *		 None
 *	Side Effects:
 *		Modifies Mode global variable
 */

ModeCmd (FromUser)
char	*FromUser;
{
	char		Cstr[MAXSTR];	/* copy of argument string */
	char		*Cpnt;		/* handy pointers are everywhere */
	unsigned int	MyMode;		/* mode temp before changing global */
	extern unsigned int	SetMode();
	extern char	*strtok();

	if (tTd(6,1))
		printf ("ModeCmd: Input string =%s=\n", FromUser);
	if (FromUser == CPNULL || *FromUser == CHNULL) {
		PrintModes();
		return;
	}

	/*
	 * Copy the string, break it into tokens, and run the tokens through
	 * the mill.
	 */
	(void) strcpy (Cstr, FromUser);
	if ((Cpnt = strtok (Cstr, " \t")) == CPNULL) {
		PrintModes();
		return;
	}
	MyMode = (unsigned int) 0;
	do {
		if (Cpnt == CPNULL || *Cpnt == CHNULL)
			return;

		/* Skip leading '-' if not the only char in token. */
		Cpnt = (*Cpnt == '-' && *(Cpnt+1) != CHNULL) ? Cpnt+1 : Cpnt;
		*Cpnt = (isupper(*Cpnt)) ? tolower(*Cpnt) : *Cpnt;
		if (tTd(6,2))
			printf ("ModeCmd: Switch with %c\n", *Cpnt);
		switch (*Cpnt) {

		    /* Silly people */
		    case 'v':
			Usage (1);
			break;

		    /* Handle modes */
		    case 's':
		    case 'r':
		    case 'n':
		    case 'p':
		    case '2':
			MyMode &= ~FORMAT_MASK;
			MyMode |= SetMode (*Cpnt);
			break;

		    case 'a':
		    case 'e':
		    case 'd':
		    case 'q':
			MyMode |= SetMode (*Cpnt);
			break;

		    case 'F':
		    case 'f':
			Cpnt = strtok (CPNULL, " \t");
			if (Cpnt == CPNULL) {
				Error ("ModeCmd: No argument found for f.");
				break;
			}
			if (Daemon) {
				Error ("ModeCmd: Can't open re-direct file in remote operation.\n");
				break;
			}
			if (equal (Cpnt, "null")) {
				if (tTd(6,2) || tTd(13,1))
					printf ("ModeCmd: Closing re-direct file");
				if (tTd(13,2))
					printf ("ModeCmd: RdrName %s, RdrStrm %s, Mode 0x%x\n",
					 RdrName,
					 (RdrStrm != FILENULL) ? "Open": "Null",
					 Mode);
				if (RdrStrm == FILENULL || *RdrName == CHNULL ||
				    ! (Mode & P_FILE))
					Fatal ("ModeCmd: Inconsistent state for re-direct close");
				(void) fclose (RdrStrm);
				RdrStrm = FILENULL;
				*RdrName = CHNULL;
				Mode &= ~P_FILE;
				if (tTd(13,2))
					printf ("ModeCmd: Re-direct file closed.");
				break;
			}
			if (RdrStrm != FILENULL) {
				Error ("ModeCmd: Must close current re-direct file %s before opening new one.\n\tUse \"mode f null\"", RdrName);
				break;
			}
			if (TeeStrm != FILENULL) {
				Error ("ModeCmd: Must close tee file %s prior to opening re-direct (-f) file.\n\tUse \"mode t null\"", TeeName);
				break;
			}
			(void) strcpy (RdrName, Cpnt);
			if (tTd(6,2))
				printf ("ModeCmd: RdrName =%s=\n", RdrName);
			if (tTd(13,2))
				printf ("ModeCmd: Before file open with RdrName %s, RdrStrm %s, Mode 0x%x\n",
				 RdrName,
				 (RdrStrm != FILENULL) ? "Open": "Null",
				 Mode);
			if (RdrStrm != FILENULL || (Mode & P_FILE))
				Fatal ("ModeCmd: Inconsistent state for re-direct open");
			if ((RdrStrm = fopen (RdrName, "a")) == FILENULL) {
				Error ("ModeCmd: Open failed on %s: %s", RdrName, ERR);
				*RdrName = CHNULL;
				break;
			}
			Mode |= P_FILE;
			break;

		    case 'T':
		    case 't':
			Cpnt = strtok (CPNULL, " \t");
			if (Cpnt == CPNULL) {
				Error ("ModeCmd: No argument found for t.");
				break;
			}
			if (Daemon) {
				Error ("ModeCmd: Can't open tee file in remote operation.\n");
				break;
			}
			if (equal (Cpnt, "null")) {
				if (tTd(6,2) || tTd(14,1))
					printf ("ModeCmd: Closing tee file");
				if (tTd(14,2))
					printf ("ModeCmd: TeeName %s, TeeStrm %s, Mode 0x%x\n",
					 TeeName,
					 (TeeStrm != FILENULL) ? "Open": "Null",
					 Mode);
				if (TeeStrm == FILENULL || *TeeName == CHNULL ||
				    ! (Mode & P_TEE))
					Fatal ("ModeCmd: Inconsistent state for tee close");
				(void) fclose (TeeStrm);
				TeeStrm = FILENULL;
				*TeeName = CHNULL;
				Mode &= ~P_TEE;
				if (tTd(14,2))
					printf ("ModeCmd: Tee file closed.");
				break;
			}
			if (TeeStrm != FILENULL) {
				Error ("ModeCmd: Must close current tee file %s before opening new one.\n\tUse \"mode t null\"", RdrName);
				break;
			}
			if (RdrStrm != FILENULL) {
				Error ("ModeCmd: Must close re-direct file %s prior to opening tee (-t) file.\n\tUse \"mode f null\"", RdrName);
				break;
			}
			(void) strcpy (TeeName, Cpnt);
			if (tTd(6,2))
				printf ("ModeCmd: TeeName =%s=\n", TeeName);
			if (tTd(14,2))
				printf ("ModeCmd: Before file open with TeeName %s, TeeStrm %s, Mode 0x%x\n",
				 TeeName,
				 (TeeStrm != FILENULL) ? "Open": "Null",
				 Mode);
			if (TeeStrm != FILENULL || (Mode & P_TEE))
				Fatal ("ModeCmd: Inconsistent state for tee open");
			if ((TeeStrm = fopen (TeeName, "a")) == FILENULL) {
				Error ("ModeCmd: Open failed on %s: %s", TeeName, ERR);
				*TeeName = CHNULL;
				break;
			}
			Mode |= P_TEE;
			break;

		    default:
			Error ("ModeCmd: Bad argument %s, ignored", Cpnt);
			break;
		}
	} while ((Cpnt = strtok (CPNULL, " \t")) != CPNULL);
	Mode = ((MyMode&FORMAT_MASK) ? (MyMode&FORMAT_MASK) : (Mode&FORMAT_MASK))
	  | ((MyMode&SELECT_MASK) ? (MyMode&SELECT_MASK) : (Mode&SELECT_MASK))
	  | (Mode&PRINT_MASK);
	return;
} /* ModeCmd */
/*
 *  PrintModes -- Print modes now in effect
 *
 *	Also make consistency checks between variables and Mode word.
 *
 *	Parameters:
 *		None
 *	Returns:
 *		 None
 *	Side Effects:
 *		Exits if state variables are inconsistent
 */

PrintModes()
{
	int	Count = 0;

	if (tTd(6,1))
		;
	if (Mode & S_ALL)
		printf ("\tSelecting all sections of entry for printing.\n");
	else {
		printf ("\tSelecting ");
		if (Mode & S_ETYM) {
			printf ("etymology ");
			Count++;
		}
		if (Mode & S_DEF) {
			printf ("definition ");
			Count++;
		}
		if (Mode & S_QUOTE) {
			printf ("quotation ");
			Count++;
		}
		printf ("section%s of entry for printing.\n",
			(Count > 1) ? "s" : "");
	}
	Count = 0;
	printf ("\tOutput format is ");
	if (Mode & F_PAT) {
		printf ("raw OED2 data with line breaks.\n");
		Count++;
	}
	if (Mode & F_ROFF) {
		printf ("converted OED2 data with embedded n/troff codes.\n");
		if (Count++)
			Fatal ("PrintModes: Multiple formats selected in Mode (0x%x)", Mode);
	}
	if (Mode & F_NROFF) {
		printf ("converted OED2 data formatted with nroff.\n");
		if (Count++)
			Fatal ("PrintModes: Multiple formats selected in Mode (0x%x)", Mode);
	}
	if (Mode & F_SIMPLE) {
		printf ("converted OED2 data with added line breaks.\n");
		if (Count++)
			Fatal ("PrintModes: Multiple formats selected in Mode (0x%x)", Mode);
	}
	if (Mode & F_VT200) {
		printf ("converted OED2 data using VT200 character sets.\n");
		if (Count++)
			Fatal ("PrintModes: Multiple formats selected in Mode (0x%x)", Mode);
	}

	printf ("\tOutput files: ");
	if (tTd(6,2)) {
		printf ("PrintModes: RdrName %s, RdrStrm %s, Mode 0x%x\n",
		 RdrName,
		 (RdrStrm != FILENULL) ? "Open": "Null",
		 Mode);
		printf ("PrintModes: TeeName %s, TeeStrm %s\n",
		 TeeName,
		 (TeeStrm != FILENULL) ? "Open": "Null");
	}
	if (RdrStrm == FILENULL) {
		if (*RdrName != CHNULL || (Mode & P_FILE))
			Fatal ("PrintModes: Inconsistent re-direct close state.");
	}
	else {
		if (*RdrName == CHNULL || ! (Mode & P_FILE))
			Fatal ("PrintModes: Inconsistent re-direct open state.");
		else if (TeeStrm != FILENULL || *TeeName != CHNULL || (Mode & P_TEE))
			Fatal ("PrintModes: Re-direct and tee files open.");
		else
			printf ("re-direct into %s\n", RdrName);
	}
	if (TeeStrm == FILENULL) {
		if (*TeeName != CHNULL || (Mode & P_TEE))
			Fatal ("PrintModes: Inconsistent tee close state.");
	}
	else {
		if (*TeeName == CHNULL || ! (Mode & P_TEE))
			Fatal ("PrintModes: Inconsistent tee open state.");
		else if (RdrStrm != FILENULL || *RdrName != CHNULL || (Mode & P_FILE))
			Fatal ("PrintModes: Tee and re-direct files open.");
		else
			printf ("tee into %s and to ", TeeName);
	}
	if (*RdrName == CHNULL) {
		printf ("stdout");
		if (*Pager != CHNULL)
			printf (" paged with %s", Pager);
		printf ("\n");
	}
} /* PrintModes */
