/*
 * Copyright (c) 1990 Paul Pomes
 * Copyright (c) 1990 University of Illinois Board of Trustees
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of Illinois, Urbana.  In addition, redistribution
 * and use must conform to the terms listed in the Copying file in
 * this directory.
 *
 * The name of the University may not be used to endorse or promote products
 * derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#ifndef lint
static char rcsid[] = "@(#)$Header: /usr/local/src/cso/oed2/RCS/FmtEntry.c,v 2.0.0.1 90/03/30 14:37:06 paul Exp $";
#endif /* lint */

#include	"oed2.h"

/*
 * FmtEntry -- PrettyPrint entry with either 'roff codes, simple format, or pat
 *
 *	Take an OED2 entry and format it in one of several ways.
 *	Strange #define's are used for good effect.
 *
 *	Parameters:
 *		InBlk -- char pointer to block with null terminated OED2 entry
 *	Returns:
 *		Pointer to start of formatted text block
 *	Side effects:
 *		Malloc()'s a chunk of space for formatted entry that must be
 *		   freed later.
 *		free()'s memory pointed to by InBlk
 *	Deficiencies:
 *		It's complex and wonderful, ripe for further break-up.
 */

/*
 * Add a string to the output block IFF we are not skipping to the end of
 * a tag.  Keep track of output line length.  Because it doesn't check
 * line length, output lines can easily exceed 65 characters before a
 * LineCnt check is made.  Anyone with strong feelings about this is
 * welcome to code the change.
 */
#define		STRADD(Aleph,DoWah)	\
{ \
	/* just to avoid a lint complaint */ \
	int DoCnt = DoWah; \
	char *floop = Aleph; \
	if (! SkipTag) \
		while (*floop != CHNULL) { \
			if (DoCnt) { \
				if (*floop == '\n') \
					CharCnt = 0; \
				else \
					CharCnt++; \
			} \
			*RoffPnt++ = *floop++; \
		} \
}

/*
 * Change font to one of (R)oman, (I)talic, or (B)old iff current font is
 * the (P)revious font.  Change to (P)revious font iff current font is not
 * (P)revious.  The NewFont variable is used to keep lint happy since it
 * otherwise bitches about constants in compare operations.  Do nothing if
 * SkipTag is in force.
 *
 * Font changes only make sense for ROFF and VT200 output mode (NROFF is the
 * same as ROFF in this routine).
 */
#define		FONT(Beta)		\
{ \
	if (! SkipTag) { \
		 if ((CurFont == 'P' && Beta != 'P') \
		  || (CurFont != 'P' && Beta == 'P')) { \
			char	Brief[10]; \
			CurFont = Beta; \
			if (MyMode & F_ROFF) { \
				(void) sprintf (Brief, "\\f%c", Beta); \
				STRADD(Brief,1) \
			} \
			else if (MyMode & F_VT200) { \
				switch (Beta) { \
				    case 'P': \
					STRADD("\033[0m",0) \
					break; \
				    case 'I': \
					STRADD("\033[4m",0) \
					break; \
				    case 'B': \
					STRADD("\033[1m",0) \
					break; \
				    default: \
					STRADD("\033[0m",0) \
					break; \
				} \
			} \
		} \
	} \
}

/* Initial Roff for controlling 'roff format */
static char	*RoffInit[] = {
	".br\n",
	".if t .sp 1i\n",
	".hy 14\n",
	".na\n",
	".if n .ll 72n\n",
	".de hd\n",
	"'sp 1i\n",
	"..\n",
	".de fo\n",
	"'bp\n",
	"..\n",
	".wh 0 hd\n",
	".wh -1i fo\n",
	".\\\" Stolen from /usr/lib/tmac/tmac.s\n",
	".ds ' \\h'\\w'e'u/5'\\z\\'\\h'-\\w'e'u/5'\n",
	".ds ` \\h'\\w'e'u/5'\\z\\`\\h'-\\w'e'u/5'\n",
	".ds ^ \\h'\\w'o'u/10'\\z^\\h'-\\w'e'u/10'\n",
	".ds - \\h'\\w'e'u/5'\\z\\(rn\\h'-\\w'e'u/5'\n",
	".ds , \\h'\\w'c'u/5'\\z,\\h'-\\w'e'u/5'\n",
	".ds : \\h'\\w'u'u/5'\\z\"\\h'-\\w'e'u/5'\n",
	".ds ^ \\h'\\w'n'u/10'\\z^\\h'-\\w'e'u/10'\n",
	".ds ~ \\h'\\w'n'u/10'\\z~\\h'-\\w'e'u/10'\n",
	".ds C \\h'\\w'c'u/5'\\v'-.6m'\\s-4\\zv\\s+4\\v'.6m'\\h'-\\w'c'u/5'\n",
	CPNULL
};

/* Sorted list of greek characters used in OED2.  Do not re-order! */
static struct {
	char	*roff;
	char	*vt200;
} GreekTab[] = {
	"\\(*a",	"Ta",	/* a -> alpha */
	"\\(*b",	"Tb",	/* b -> beta */
	"\\(*c",	"Tx",	/* c -> xi */
	"\\(*d",	"Td",	/* d -> delta ??? */
	"\\(*e",	"Te",	/* e -> epsilon ??? */
	"\\(*f",	"Tf",	/* f -> phi */
	"\\(*g",	"Tg",	/* g -> gamma */
	"\\(*y",	"Th",	/* h -> eta */
	"\\(*i",	"Ti",	/* i -> iota */
	"\\(ts",	"As",	/* j -> terminal sigma */
	"\\(*k",	"Tk",	/* k -> kappa */
	"\\(*l",	"Tl",	/* l -> lambda ??? */
	"\\(*m",	"Tm",	/* m -> mu ??? */
	"\\(*n",	"Av",	/* n -> nu */
	"\\(*o",	"Ao",	/* o -> omicron */
	"\\(*p",	"Tp",	/* p -> pi */
	"\\(*h",	"Tj",	/* q -> theta */
	"\\(*r",	"Tr",	/* r -> rho */
	"\\(*s",	"Ts",	/* s -> sigma */
	"\\(*t",	"Tt",	/* t -> tau */
	"\\(*u",	"Ty",	/* u -> upsilon */
	"UnMap",	"AUnMap", /* v -> un-mapped */
	"\\(*w",	"Tw",	/* w -> omega */
	"\\(*x",	"Tc",	/* x -> chi ??? */
	"UnMap",	"AUnMap", /* y -> un-mapped */
	"\\(*z",	"Tz",	/* z -> zeta ??? */
};

char *
FmtEntry (InBlk)
char	*InBlk;
{
	char		TagStr[25];		/* tag string */
	char		SymStr[25];		/* symbol string */
	char		*TagPnt = TagStr;	/* pointer for collecting tag */
	char		*SymPnt = SymStr;	/* pointer for collecting sym */
	struct TAG	*CurTag;		/* current tag */
	struct SYM	*CurSym;		/* current symbol */
	int		InTag = 0;		/* set if inside a tag */
	int		InSym = 0;		/* set if inside a symbol */
	int		InGrk = 0;		/* set if printing greek */
	int		EndTag = 0;		/* set if a closing tag */
	int		SmallCaps = 0;		/* whether caps are lowered */
	int		SkipTag = 0;		/* skip the data for this tag */
	int		CharCnt;		/* # of chars since newline */
	unsigned int	MyMode;			/* local copy of PrtMode */
	int		Action;			/* final switch variable */
	char		NewFont;		/* new output font (lint bug) */
	char		CurFont = 'P';		/* current output font */
	register char	*InPnt = InBlk;		/* InBlk pointer */
	char		*RoffBlk;		/* output block */
	register char	*RoffPnt;		/* and pointer */
	extern struct TAG *FindTag();
	extern struct SYM *FindSymbol();
	extern char	*Malloc();

	/* At least one format mode must be set */
	MyMode = PrtMode;
	if (! (MyMode & FORMAT_MASK)) {
		Warning ("FmtEntry: No format action set in mode word (0x%x), assuming NROFF", MyMode);
		MyMode |= F_NROFF;
	}

	/* F_ROFF and F_NROFF are all the same to us */
	if (MyMode & F_NROFF)
		MyMode |= F_ROFF;

	if (tTd(4,1))
		printf ("FmtEntry: InBlk length is %d\n", strlen (InBlk));
	if (tTd(4,3)) {
		printf ("FmtEntry: Pre-fmt entry: =%s=\n", InBlk);
		(void) fflush (stdout);
	}

	/*
	 * The output block is almost always guaranteed smaller than the
	 * input block.  To salve my conscience about lack of error checking
	 * while RoffBlk is stuffed, RoffBlk is made 4K larger than InBlk.
	 */
	RoffPnt = RoffBlk = Malloc ((unsigned) (CharCnt = strlen (InBlk) + MAXBLK));
	bzero (RoffPnt, CharCnt);
	CharCnt = 0;

	/*
	 * In PAT format, just add line breaks every so often.
	 */
	if (MyMode & F_PAT) {
		while (*InPnt != CHNULL) {
			if (CharCnt > 65 && (*InPnt == ' ' || *InPnt == '>')) {
				if (*InPnt == ' ')
					*RoffPnt++ = '\n';
				else if (*InPnt == '>') {
					*RoffPnt++ = '>';
					*RoffPnt++ = '\n';
				}
				InPnt++;
				CharCnt = 0;
			}

			/* Suppress leading blanks after a line break */
			else if (CharCnt == 0 && *InPnt == ' ')
				InPnt++;
			else {
				*RoffPnt++ = *InPnt++;
				CharCnt++;
			}
		}

		/* Tack on a end newline if the previous character wasn't. */
		if (RoffPnt != RoffBlk && *(RoffPnt - 1) != '\n')
			*RoffPnt++ = '\n';
		*RoffPnt++ = CHNULL;

		/* We're done with InBlk, free the memory */
		(void) free (InBlk);
		if (tTd(4,1))
			printf ("FmtEntry: RoffBlk (F_PAT) length is %d\n", strlen (RoffBlk));
		if (tTd(4,3)) {
			printf ("FmtEntry: Post-fmt (F_PAT) entry: =%s=\n", RoffBlk);
			(void) fflush (stdout);
		}

		/* All done, return pointer to converted block. */
		return (RoffBlk);
	}

	/*
	 * Put the entry into 'roff or VT2XX format.  First insert the init
	 * string for 'roff format, then iterate through InBlk.
	 */
	if (MyMode & F_ROFF) {
		char **Rpnt;

		for (Rpnt = RoffInit; *Rpnt != CPNULL; Rpnt++)
			STRADD(*Rpnt,1)
	}

	/*
	 * This is complex and wonderful and should probably be re-written.
	 * Let this be an example of how code evolves (you should have seen
	 * it before it was released -- icck).
	 */
	for (; *InPnt != CHNULL;) {

		/* If not parsing a tag or symbol... */
		if (! InTag && ! InSym) {

			/* Insert a newline if line has grown */
			if (CharCnt > 79) {
				char	*WrkPnt = RoffPnt;

				if (tTd(4,4))
					printf ("FmtEntry: retro newline (%d)\n", CharCnt);

				/* This has the potential of going negative. */
				while (*WrkPnt != ' ')
					WrkPnt--;
				*WrkPnt = '\n';
				CharCnt = RoffPnt - WrkPnt;
			}

			/* Skip leading spaces after newline */
			if (CharCnt == 0 && *InPnt == ' '){
				if (tTd(4,5))
					printf ("FmtEntry: Skip after nl\n");
				InPnt++;
				continue;
			}

			/*
			 * Be careful of leading . and ' chars
			 * as n/troff will use them as commands.
			 */
			if (MyMode & F_ROFF && CharCnt == 0 &&
			  (*InPnt == '.' || *InPnt == '\047')) {
				if (tTd(4,5))
					printf ("FmtEntry: Insulating\n");
				STRADD("\\&",1)
				CharCnt = 2;
			}

			/* Have we found a tag or symbol? */
			if (*InPnt == '<') {
				if (tTd(4,5))
					printf ("FmtEntry: InTag true\n");
				InTag++;
				InPnt++;
				continue;
			}

			/* No, but a previous tag may have set conditionals. */

			/* Don't print text between some tags. */
			if (SkipTag) {
				InPnt++;
				continue;
			}

			/* Set to decode a symbol */
			if (*InPnt == '&') {
				if (tTd(4,5))
					printf ("FmtEntry: InSym true\n");
				InSym++;
				InPnt++;
				continue;
			}

			/*
			 * If SmallCaps is set, convert all lower case letters
			 * to upper case before adding to output block.
			 */
			if (SmallCaps && islower(*InPnt)) {
				*RoffPnt = toupper (*InPnt);
				RoffPnt++; InPnt++;
				CharCnt++;
				continue;
			}

			/* Print greek character equivalents */
			if (InGrk) {
				int	GrkTemp;

				if (islower (*InPnt) && (MyMode & F_ROFF)) {
					GrkTemp = *InPnt - 'a';
					STRADD(GreekTab[GrkTemp].roff,1)
				}
				if (islower (*InPnt) && (MyMode & F_VT200)) {
					GrkTemp = *InPnt - 'a';
					if (*GreekTab[GrkTemp].vt200 == 'T') {
						STRADD("\033(>",0)
						STRADD(&GreekTab[GrkTemp].vt200[1],1)
						STRADD("\033(B",0)
					}
					else
						STRADD(&GreekTab[GrkTemp].vt200[1],1)
				}
				else {
					*RoffPnt++ = *InPnt;
					CharCnt++;
				}
				InPnt++;
				continue;
			}

			/* Nothing special to do, just copy. */
			*RoffPnt++ = *InPnt++;
			CharCnt++;
			continue;
		}
		if (InTag) {

			/* Found end of a tag */
			if (*InPnt == '>') {
				*TagPnt = CHNULL;
				TagPnt = TagStr;
				if (tTd(4,5))
					printf ("FmtEntry: Looking up %s\n", TagStr);

				/* Look it up and get its actions */
				if ((CurTag = FindTag (TagStr)) == TAGNULL) {
					if (tTd(4,5))
						printf ("FmtEntry: Unknown tag %s\n", TagStr);
					Error ("FmtEntry: Unknown tag %s\n", TagStr);
					EndTag = InTag = 0;
					InPnt++;
					continue;
				}

				/*
				 * Perform tag actions, set conditionals.
				 * Assume that the tag contents are to be
				 * skipped unless overridden.  This allows
				 * for concatenation of select modes (e.g.,
				 * defs and quotes but no etymologies).  Once
				 * a non-SKIP action has been found, punt on
				 * the remaining checks.
				 */
				if (CurTag->Value == TAG_gk)
					InGrk = EndTag ? 0 : 1;
				Action = SKIP;
				if (MyMode & S_ALL)
					Action = EndTag ? CurTag->all_e
							: CurTag->all_s;
				if (MyMode & S_ETYM && Action == SKIP) 
					Action = EndTag ? CurTag->etym_e
							: CurTag->etym_s;
				if (MyMode & S_DEF && Action == SKIP) 
					Action = EndTag ? CurTag->def_e
							: CurTag->def_s;
				if (MyMode & S_QUOTE && Action == SKIP) 
					Action = EndTag ? CurTag->quote_e
							: CurTag->quote_s;
				switch (Action) {
				    case NO_OP:		/* Do nothing */
					break;

				    case PAREN:		/* (text) */
					if (EndTag)
						STRADD(")",1)
					else {
						if (RoffPnt > RoffBlk &&
						   *(RoffPnt - 1) != ' ' &&
						   *(RoffPnt - 1) != '\n')
							STRADD(" ",1)
						STRADD("(",1)
					}
					break;

				    case FONTB:		/* Change to Bold */
					NewFont = 'B';
					FONT(NewFont)
					break;

				    case FONTI:		/* Change to Italic */
					NewFont = 'I';
					FONT(NewFont)
					break;

				    case FONTIP:	/* Italics, paren */
					NewFont = 'B';
					FONT(NewFont)
					STRADD("(",1)
					break;

				    case FONTP:		/* Change to Previous */
					NewFont = 'P';
					FONT(NewFont)
					break;

				    case FONTPS:	/* Paren, prev font */
					STRADD(") ",1)
					NewFont = 'P';
					FONT(NewFont)
					break;

				    case FONTR:		/* Change to Roman */
					NewFont = 'R';
					FONT(NewFont)
					break;

				    case NEWL:		/* Print a newline */
					if (RoffPnt > (RoffBlk+3) &&
					   strncmp ((RoffPnt-3), "\n  ", 3) &&
					   strncmp ((RoffPnt-2), "\n\n", 2))
						STRADD("\n",1)
					break;

				    case PARA:		/* Start a paragraph */
					if (RoffPnt > (RoffBlk+3) &&
					   nequal ((RoffPnt-3), "\n  ", 3))
						break;
					if (RoffPnt > RoffBlk &&
					  *(RoffPnt-1) == '\n')
						STRADD("  ",1)
					else
						STRADD("\n  ",1)
					break;

				    case SPACE:		/* Print a space */
					if (RoffPnt > RoffBlk &&
					   *(RoffPnt - 1) != ' ' &&
					   *(RoffPnt - 1) != '\n')
						STRADD(" ",1)
					break;

				    case SKIP:		/* Set/clear SkipTag */
					if (EndTag) {
						if (SkipTag == CurTag->Value)
							SkipTag = 0;
					}
					else if (SkipTag == 0)
						SkipTag = CurTag->Value;
					break;

				    case UP:		/* Superscript start */
#ifdef notdef
	/* don't enable unless UP/L_SQ are made into separate actions */
					if (MyMode & F_ROFF)
						STRADD("\\u",1)
					else {
#endif /* notdef */
						if (RoffPnt > RoffBlk &&
						   *(RoffPnt - 1) != ' ' &&
						   *(RoffPnt - 1) != '\n')
							STRADD(" ",1)
						STRADD("[",1)
					break;

				    case DOWN:		/* Superscript end */
#ifdef notdef
	/* don't enable unless DOWN/R_SQ are made into separate actions */
					if (MyMode & F_ROFF)
						STRADD("\\d",1)
					else {
#endif /* notdef */
						if (RoffPnt > RoffBlk &&
						   *(RoffPnt - 1) != ' ' &&
						   *(RoffPnt - 1) != '\n')
							STRADD(" ",1)
						STRADD("]",1)
					break;

				    case S_C:	       /* Set/clear SmallCaps */
					if (EndTag) {
						if (SmallCaps == CurTag->Value){
							if (MyMode & F_ROFF)
								STRADD("\\s0",1)
							SmallCaps = 0;
						}
					}
					else if (SmallCaps == 0) {
						if (MyMode & F_ROFF)
							STRADD("\\s-1",1)
						SmallCaps = CurTag->Value;
					}
					break;

				    case SLASH:	       /* Print slash */
					if (! EndTag)
						STRADD("/",1)
					break;

				    default:
					Warning ("FmtEntry: Unknown command (%u) for tag %s, MyMode 0x%x", Action, CurTag->String, MyMode);
					break;
				}

				/* Clear tag state variables */
				EndTag = InTag = 0;
				if (tTd(4,5))
					printf ("FmtEntry: InTag EndTag false\n");
				InPnt++;
			}

			/* Set if this is an ending tag. */
			else if (*InPnt == '/') {
				if (tTd(4,5))
					printf ("FmtEntry: EndTag true\n");
				EndTag++;
				InPnt++;
			}

			/* Still collecting the tag string. */
			else
				*TagPnt++ = *InPnt++;
			continue;
		}
		if (InSym) {

			/* Found end of a symbol */
			if (*InPnt == '.') {
				*SymPnt = CHNULL;
				SymPnt = SymStr;
				if (tTd(4,5))
					printf ("FmtEntry: Looking up %s\n", SymStr);

				/*
				 * Special case the ae and oe ligatures if
				 * SmallCaps is true.
				 */
				if (SmallCaps && (equal (SymStr, "ae") ||
						  equal (SymStr, "oe"))) {
					if (islower (SymStr[0]))
						SymStr[0] = toupper (SymStr[0]);
					if (islower (SymStr[1]))
						SymStr[1] = toupper (SymStr[1]);
				}

				/* Look it up and get its strings */
				if ((CurSym = FindSymbol (SymStr)) == SYMNULL) {
					if (tTd(4,5))
						printf ("FmtEntry: Unknown symbol %s\n", SymStr);
					if (lflag)
						syslog (LOG_ERR, "FmtEntry: Unknown symbol %s", SymStr);
					InSym = 0;
					InPnt++;
					continue;
				}

				/*
				 * Add symbol printable equivalent to output
				 * block.
				 */
				if (MyMode & F_ROFF)
					STRADD(CurSym->RoffStr,1)
				else if (MyMode & F_SIMPLE)
					STRADD(CurSym->SimpleStr,1)
				else if (MyMode & F_VT200 &&
					 CurSym->vt200Str != CPNULL) {
					switch (*CurSym->vt200Str) {
					    
					    case CHNULL:
						break;

					    /* multi-national character set */
					    case 'M':
						STRADD("\033(<",0)
						break;

					    /* special graphics character set */
					    case 'S':
						STRADD("\033(0",0)
						break;

					    /* technical character set */
					    case 'T':
						STRADD("\033(>",0)
						break;

					    /* ascii character set */
					    case 'A':
						break;

					    /* shouldn't happen */
					    default:
						Error ("FmtEntry: Unknown VT200 key %c\n", *CurSym->vt200Str);
						break;
					}
					if (*CurSym->vt200Str != CHNULL) {
						STRADD(&CurSym->vt200Str[1],1)
						STRADD("\033(B",0)
					}
				}

				/* Clear symbol state variables */
				InSym = 0;
				if (tTd(4,5))
					printf ("FmtEntry: InSym false\n");
				InPnt++;
			}

			/* Still collecting the symbol string. */
			else
				*SymPnt++ = *InPnt++;
		}
	}

	/* Tack on a end newline if the previous character wasn't. */
	if (RoffPnt > RoffBlk && *(RoffPnt - 1) != '\n')
		*RoffPnt++ = '\n';
	*RoffPnt++ = CHNULL;

	/* We're done with InBlk, free the memory */
	(void) free (InBlk);
	if (tTd(4,1))
		printf ("FmtEntry: RoffBlk (F_ROFF|F_VT200) length is %d\n", strlen (RoffBlk));
	if (tTd(4,3)) {
		printf ("FmtEntry: Post-fmt (F_ROFF|F_VT200) entry: =%s=\n", RoffBlk);
		(void) fflush (stdout);
	}

	/* All done, return pointer to converted block. */
	return (RoffBlk);
} /* FmtEntry */
